/*
 * eeprom_m24m01.c
 *
 *  Created on: 15/10/2021
 *      Author: alexrayne <alexraynepe196@gmail.com>
 * ----------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
 * ------------------------------------------------------------------
     STM M24M01 1Mbit I2C eeprom on i2c_masterx SSP
 */

#include "flash_i2c_hal.h"
#include <ssp_hal.h>
#include <flash_hal.h>
#include "ptx.h"


/////////////////////////////////////////////////////////////////////////////
#include <trace_probes.h>

#ifndef trace_flash_rd
trace_need(flash_wr)
trace_need(flash_wrwe)
trace_need(flash_rd)
#endif



/////////////////////////////////////////////////////////////////////////////

//< размер данных,для которого предпочитается ожидать блокирующего завершения SSP транзакции,
//  для больших буферов - возвращается BUSY/ptWAITING, и ведется ожидание ниткой приложения
#ifndef SSP_TRANSACTION_WAIT_TH
#define SSP_TRANSACTION_WAIT_TH     8
#endif


DevResult flashi2c_write_enable(FLASH_Device* self, bool onoff){
    (void)self; (void)onoff;
    return DEV_NOT_IMPLEMENTED;
}
DevResult flashi2c_erase_all(FLASH_Device* self)               { (void)self; return DEV_NOT_IMPLEMENTED; }
PTResult flashi2c_erase_sectors(FLASH_Device* self, flash_addr_t sec, unsigned len) {
    (void)self; (void)sec; (void)len;
    return DEV_NOT_IMPLEMENTED;
}



DevResult flashi2c_init(I2CFlash_GenDevice* this){
    flash_init(&this->flash);
    this->flash.select      = flashi2c_select;
    this->flash.erase_all   = flashi2c_erase_all;
    this->flash.state       = flashi2c_state;
    this->flash.read        = flashi2c_read_one;
    this->flash.write       = flashi2c_write;
    this->flash.write_enable= flashi2c_write_enable;
    this->flash.erase_sectors = flashi2c_erase_sectors;
    this->flash.flush       = flashi2c_flush;
    return DEV_OK;
}



PTResult flashi2c_flush(FLASH_Device* self){
    I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;

    return  sspdev_wait_flush(&this->ssp, 0);// self->wait_ready(self, self->info->burn_page_ticks);
};



FlashState flashi2c_state(FLASH_Device* self){
    I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;

    // touch last SLA-device, checking it online

    sspio_cs_hold(&this->ssp.io, false);
    SSPResult ok = sspdev_touch(&this->ssp);

    if ( ok == SSP_errBUSY ) {
        // если последняя операция с портом была неблокирующая, дождусь пока порт станет готов к обмену
        DevResult res = sspdev_wait_trx(&this->ssp);
        if ( res != DEV_OK ){
            return res;
        }
        ok = sspdev_touch(&this->ssp);
    }

    DevResult res = sspdev_wait_trx(&this->ssp);
    if (res == DEV_OK) {
        // device ready
        this->flash.status_cache &= (FLASH_sWELOCK | FLASH_sWE);
        return this->flash.status_cache |FLASH_sSTATUS;
    }
    else if (res == SSP_errNOACK ){
        // device ready
        this->flash.status_cache &= (FLASH_sWELOCK | FLASH_sWE | FLASH_sBUSY);
        return this->flash.status_cache |FLASH_sSTATUS;
    }

    return res;
}


DevResult  flashi2c_select(FLASH_Device* self, unsigned csid){
    I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;
    sspio_set_cs(&this->ssp.io, csid);
    return DEV_OK;
}


PTResult  flashi2c_read_one(FLASH_Device* self, flash_addr_t addr, void* dst, unsigned len){
    DevResult ok  = self->select_addr(self, addr);
    if (ok != DEV_OK) return AS_PTRESULT(ok);

    if (len <= SSP_TRANSACTION_WAIT_TH )
    {
        return AS_PTRESULT( flashi2c_read_page(self, dst, addr, len) );
    }

    //прерванный цикл только ожидает завершения SSP
    flash_cycles_abort(self);

    return flashi2c_ask_page(self, dst, addr, len);
}

PTResult  flashi2c_read_pages(FLASH_Device* self, flash_addr_t addr, void* dst, unsigned len){
    //I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;

    // if Flash is per-byte writing (pagesie ==1), then suppose that it can
    //      continue read though all chip
    if ( flash_in_single_page(self, addr, len) ){
        return flashi2c_read_one(self, addr, dst, len);
    }

    return flash_read_pages(self, addr, dst, len, (flash_page_op)&flashi2c_ask_page);
}

PTResult flashi2c_write(FLASH_Device* self, flash_addr_t addr, const void* src, unsigned len){
    return flash_write_pages(self, addr, src, len, (flash_page_op)&flashi2c_post_page);
}



// базовые операции со страницами. Они не обязаны корректно пересекать
//  границы страниц
PTResult flashi2c_post_page(FLASH_Device* self, void* src, flash_page_t page, unsigned len){
    I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;

    trace_flash_wr_on();

    flashi2c_force_busy(this);

    // flash burn starts after CS disabled
    sspio_cs_hold(&this->ssp.io, false);

    flash_busy_start(self);
    int writen = sspdev_postData(&this->ssp, page, (const void*)src, len);

    trace_flash_wr_off();

    if (writen == 0)
        return ptWAITING;
    if (writen >= 0)
        return (writen == (int)len)? ptOK :ptNOK;
    else
        return writen;
}

DevResult flashi2c_write_page(FLASH_Device* self, void* src, flash_page_t page, unsigned len){
    I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;

    flashi2c_force_busy(this);

    trace_flash_wr_on();
    // flash burn starts after CS disabled
    sspio_cs_hold(&this->ssp.io, false);

    flash_busy_start(self);
    int writen = sspdev_writeData(&this->ssp, page, (const void*)src, len);

    trace_flash_wr_off();

    if (writen >= 0)
        return (writen == (int)len)? DEV_OK:DEV_NOK;
    else
        return writen;
}

PTResult flashi2c_ask_page(FLASH_Device* self, void* dst, flash_page_t page, unsigned len){
    I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;

    trace_flash_rd_on();
    int readen = sspdev_askData(&this->ssp, page, dst, len);
    trace_flash_rd_off();
    if (readen == 0)
        return ptWAITING;
    if (readen > 0)
        return (readen == (int)len)? ptOK :ptNOK;
    else
        return readen;
}

DevResult flashi2c_read_page(FLASH_Device* self, void* dst, flash_page_t page, unsigned len){
    I2CFlash_GenDevice* this = (I2CFlash_GenDevice*)self;

    trace_flash_rd_on();
    int readen = sspdev_readData(&this->ssp, page, dst, len);
    trace_flash_rd_off();
    if (readen >= 0)
        return (readen == (int)len)? DEV_OK:DEV_NOK;
    else
        return readen;
}

