/*
 * r_iic_master_devaddr.h
 *
 *  Created on: 9/04/2021
 *      Author: alexraynepe196@gmail.com
 * ----------------------------------------------------------------------
 * драйвер устройства i2c с внутренней адресацией
 */

#ifndef BSP_R_I2C_MASTERX_H_
#define BSP_R_I2C_MASTERX_H_

#include <c_compat.h>
#include <mcu-chip.h>
// FSP generated sources
#include "r_i2c_master_api.h"


#ifdef __cplusplus
namespace i2c {
extern "C" {
#endif



enum {
    // это событие передается после обновления внутреней машины I2C, I2C_status поменялся.
    //      требуется I2C_wait для продолжения работы,
    I2C_MASTER_EVENT_BUSY     = 4,  ///< A transfer has events, and need to poll
};


typedef fsp_err_t I2C_Status;
typedef enum I2CStatusID{
    I2C_OK        = FSP_SUCCESS ,
    I2C_BUSY      = 700 ,
    I2C_NOSLA     = 701 ,   //SLA not acknowledges
    I2C_ABORT     = FSP_ERR_ABORTED,
    //, I2C_ERROR    = 0x01
    //, I2C_TIMEOUT  = 0x03
    //, I2C_ERRCRC   = 0x05   //use for aside api
}  I2CStatusID;


typedef struct {
    i2c_master_instance_t*  port;
    //uint32_t        rate;
    //unsigned long   TOinterval;
} I2C_basic;

typedef I2C_basic* i2cHandle;

i2cHandle I2C_init_master(/*I2C_REGS_TYPE*/ const void* io);  /*, i2c_descriptor* handle*/
I2C_Status I2C_close_master(i2cHandle h);

I2C_Status I2C_status(i2cHandle port);
I2C_Status I2C_abort(i2cHandle port);

#define  I2C_wait(...)  I2C_status(__VA_ARGS__)

/// @brief handle on i2c transfer finish
typedef void (* I2C_on_event)(i2cHandle h, i2c_master_event_t ev, void* ctx);

void I2C_assign_handler(i2cHandle port, I2C_on_event, void* ctx);


typedef enum {

    i2co_XRECV      = 0 ,
    i2co_XSEND      = 1 ,

    //!< делает XRECV без рестарта (отменяет рестарт фрейма) после передачи адреса.
    //! TODO: RA FSP i2c_master can't implement it
    //i2co_OneFrame   = 4,

    //!< XSEND отсылает фрейм данных после рестрата (форсирует рестарт фрейма) после адресной части
    //! TODO: RA FSP i2c_master всегда делает только так
    //i2co_XFrame     = 8 ,

    //!< подавляет освобождение шины после транзакции. обычный запрос освобождает шину после передачи данных.
    //! @note RA FSP i2c_master позволяет это делать рестартом
    i2co_LockBus    = 0x10 ,

    //< I2C_xfer2 provide Addr-len here
    i2co_ADDR       = 0xc0 ,
    i2co_ADDR_Pos   = 6 ,
    i2co_ADDR1      = 0<< i2co_ADDR_Pos,
    i2co_ADDR2      = 1<< i2co_ADDR_Pos,
    i2co_ADDR3      = 2<< i2co_ADDR_Pos,
    i2co_ADDR4      = 3<< i2co_ADDR_Pos,
} i2c_io_option;



I2C_Status I2C_xfer(i2cHandle port, unsigned char SLA
          , const void* addr, unsigned alen
          , void* data, unsigned dlen
          , i2c_io_option mode
          );

/** операции и2ц с выделеным адресом. адресная часть - передается как нормальные данные
 * */
/** тоже самое что и miic_transfer(port, SLA, addr, alen, dara, dlen)
 * \sa miic_transfer
 */
INLINE
I2C_Status I2C_read(i2cHandle port, unsigned char SLA
          , const void* addr, unsigned alen
          , void* data, unsigned dlen
          )
{
    return I2C_xfer(port, SLA, addr, alen, data, dlen, i2co_XRECV);
}

INLINE
I2C_Status I2C_write(i2cHandle port, unsigned char SLA
          , const void* addr, unsigned alen
          , void* data, unsigned dlen
          )
{
    return I2C_xfer(port, SLA, addr, alen, data, dlen, i2co_XSEND);
}



/// @brief this operations continues on last SLA, with no adress - device provide adress on it`s own
I2C_Status I2C_xfer_next(i2cHandle port, void* data, unsigned dlen, i2c_io_option mode );

INLINE
I2C_Status I2C_read_next(i2cHandle port, void* data, unsigned dlen )
{
    return I2C_xfer_next(port, data, dlen, i2co_XRECV);
}

INLINE
I2C_Status I2C_write_next(i2cHandle port, void* data, unsigned dlen)
{
    return I2C_xfer_next(port, data, dlen, i2co_XSEND);
}



/// @brief alternative simplified API for regs adress in 4bytes
///
static inline
uint32_t I2C_MODE( uint8_t sla, uint8_t mode) {
    return (sla << 8) | mode ;
}

I2C_Status I2C_xfer2(i2cHandle port, uint32_t SLA_mode, uint32_t addr , void* data, unsigned dlen );



/// @brief API for registers set with 1byte target register index
INLINE
I2C_Status I2C_read_regs(i2cHandle port, unsigned char SLA, uint8_t addr, void* data, unsigned dlen )
{
    return I2C_xfer2(port, I2C_MODE(SLA, (i2co_XRECV|i2co_ADDR1) ), addr, data, dlen);
}

INLINE
I2C_Status I2C_write_regs(i2cHandle port, unsigned char SLA  , uint8_t addr, void* data, unsigned dlen)
{
    return I2C_xfer2(port, I2C_MODE(SLA, (i2co_XSEND|i2co_ADDR1) ), addr, data, dlen);
}




/// @brief this operations continues on last SLA
I2C_Status I2C_xfer_continue(i2cHandle port, uint32_t SLA_mode, uint32_t addr ,void* data, unsigned dlen );

INLINE
I2C_Status I2C_read_continue(i2cHandle port, uint8_t addr, void* data, unsigned dlen )
{
    return I2C_xfer_continue(port, I2C_MODE(0, (i2co_XRECV|i2co_ADDR1) ), addr, data, dlen);
}

INLINE
I2C_Status I2C_write_continue(i2cHandle port, uint8_t addr, void* data, unsigned dlen)
{
    return I2C_xfer_continue(port, I2C_MODE(0, (i2co_XSEND|i2co_ADDR1) ), addr, data, dlen);
}




#ifdef __cplusplus
}
} // namespace i2c {
#endif



#endif /* BSP_R_I2C_MASTERX_H_ */
