using System;
using System.Text;
using System.Security.Cryptography;
using System.Diagnostics;
using System.Collections.Generic;

namespace mdc5{
	struct SaveArgument{
		public string Name{
			get; set;
		}
		public int Side{
			get; set;
		}
		public int Length{
			get; set;
		}
		public int Offset{
			get; set;
		}
	}
	class GameImage : Interface.Logput
	{
		public const int PAGESIZE = 0x2000;
		//imagefile.Fds[] m_diskimage;
		//imagefile.Nes[] m_romimage;
		imagefile.GameImage[] m_gameimage;
		Bios m_biosimage;
		
		public GameImage()
		{
			Debug.Assert(PAGESIZE == (Manage.SIZE + Bios.EXTRA_SIZE));
		}
		
		bool BiosLoad(string rom_image, string rom_patch)
		{
			{
				byte [] bios;
				if(Static.Utility.binary_load(rom_image, out bios) == false){
					OnLogAdd(rom_image + " open error!");
					return false;
				}
				m_biosimage = new Bios();
				if(m_biosimage.rom_load(bios) == false){
					OnLogAdd(rom_image + " checksum error!");
					return false;
				}
			}
			{
				string [] recoard;
				if(Static.Utility.text_load(rom_patch, out recoard) == false){
					OnLogAdd(rom_patch + " open error!");
					return false;
				}
				if(m_biosimage.patch_load(recoard) == false){
					OnLogAdd(rom_patch + " recoard syntax error!");
					return false;
				}
			}
			return true;
		}
		bool image_load(mdc5.Script script, bool do_patch, bool patchlog, imagefile.GameImage g)
		{
			if(g.Load(script) == false){
				OnLogAdd(script.ImageFilename + " open error!");
				return false;
			}

			if(script.PatchData.Length != 0 && do_patch == true){
				if(patchlog == true){
					OnLogAdd("patching " + script.GameCode + "...");
				}
				foreach(RomRecoard.MotorolaSRecoard t in script.PatchData){
					string log;
					if(t.Valid == true){
						if(g.PatchManual(t, out log) != imagefile.GameImage.patchresult.OK){
							OnLogAdd(log);
							return false;
						}
						if(patchlog == true){
							OnLogAdd(log);
						}
					}
				}
			}
			if((g.Type == mdc5.Script.imagetype.disk) && (do_patch == true)){
				string [] log;
				g.PatchAuto(out log);
				if(patchlog == true){
					OnLogAdd(log);
				}
			}
			return true;
		}

		bool ImageLoad(mdc5.Script [] image, bool do_patch, bool patchlog)
		{
			List<imagefile.GameImage> gameimage = new List<imagefile.GameImage>();
			foreach(mdc5.Script t in image){
				switch(t.ImageType){
				case mdc5.Script.imagetype.disk:{
					imagefile.Fds fds = new imagefile.Fds();
					if(image_load(t, do_patch, patchlog, fds) == false){
						return false;
					}
					gameimage.Add(fds);
					}break;
				case mdc5.Script.imagetype.rom:{
					imagefile.Nes nes = new imagefile.Nes();
					if(image_load(t, do_patch, patchlog, nes) == false){
						return false;
					}
					gameimage.Add(nes);
					}break;
				default:
					break;
				}
			}
			m_gameimage = gameimage.ToArray();
			return true;
		}
		bool Link(bool make_bin, bool make_nes, int capacity, string fileprefix)
		{
			Debug.Assert(m_gameimage != null);
			Debug.Assert(m_biosimage != null);
			Debug.Assert((capacity % PAGESIZE) == 0);
			Manage m = new Manage(m_gameimage.Length, capacity);
			foreach(imagefile.GameImage d in m_gameimage){
				m.Append(d);
			}
			byte [] rom;
			{
				string [] linklog;
				rom = m.Link(m_biosimage.RomBottom, m_biosimage.RomExtra, out linklog);
				OnLogAdd(linklog);
			}
			int page_max = capacity / PAGESIZE;
			if((m.TotalPage > page_max) && (capacity != Manage.CAPACITY_AUTO)){
				OnLogAdd(String.Format("ROM capacity over! {0}/{1} pages.", m.UsedPage, page_max));
				return false;
			}
			if(m.TotalPage > 0x80){
				OnLogAdd(String.Format("ROM capacity 8Mbit over! {0}/{1} pages.", m.UsedPage, page_max));
				return false;
			}
			OnLogAdd(String.Format("linked {0}/{1} pages.", m.UsedPage, m.TotalPage));
			if(make_bin == true){
				string file = fileprefix + ".bin";
				Static.Utility.binary_save(file, rom);
				OnLogAdd(file + " created.");
			}
			if(make_nes == true){
				imagefile.Nes nes = new imagefile.Nes();
				byte [] ppurom = new byte[0];
				bool t = nes.ImageSet(rom, 0x8000*2, ppurom, imagefile.Nes.mapper.ExROM);
				Debug.Assert(t);
				string file = fileprefix + ".nes";
				Static.Utility.binary_save(file, nes.ImageGet());
				OnLogAdd(file + " created.");
			}
			
			return true;
		}
		public bool BatchLink(Interface.Konfig c, mdc5.Script [] s, string output_prefix, bool do_patch, bool patchlog)
		{
			string rom_image = c.PathGet("romimage.in") + "/disksys.rom";
			string rom_patch = c.PathGet("rompatch.in") + "/bios.sr";
			if(BiosLoad(rom_image, rom_patch) == false){
				return false;
			}
			if(ImageLoad(s, do_patch, patchlog) == false){
				return false;
			}
			return Link(c.OutputBin, c.OutputNes, c.RomCapacity, c.PathGet("romimage.out") + "/" + output_prefix);
		}
	}
	class Manage{
		readonly static string sigature = "mdc5 0.02";
		const int SIZEOF_6502_POINTER = 2;
		public const int SIZE = 0x0800;
		int m_game_offset = 0;
		int m_page_offset = 1; // | 0x80; //0 は Extra 専用, 0x80 は mmc5 ROM bit
		int m_page_num = 0;
		readonly int m_romsize;
		public const int CAPACITY_AUTO = 0;
		readonly Dictionary<char, byte> m_bgname;
		ManageGame [] m_gamedata;
		const string BGNAME_PATTERN = (
			"0ァゥェォッャュっゃゅ" +
			"!！「」ー-?゛゜" +
			"いろはにほへと" + "ちりぬるを" +
			"わかよたれそ" +   "つねならむ" +
			"うのおくやま" + "けふこえて" +
			"あさきゆめみし" + "ょひもせすん" +
			"イロハニホヘト" + "チリヌルヲ" +
			"ワカヨタレソ" +   "ツネナラム" +
			"ウィノオクヤマ" + "ケフコエテ" +
			"アサキユメミシ" + "ョヒモセスン"
		);
		
		public int UsedPage
		{
			get{return m_page_offset + 1;}
		}
		public int TotalPage
		{
			get{return m_page_num;}
		}
		public Manage(int gamecount, int romsize)
		{
			m_romsize = romsize;
			m_gamedata = new ManageGame[gamecount];
			m_bgname = new Dictionary<char, byte>();
			m_bgname.Add('\0', 0);
			//0x01-0x09 '1' から '9'
			for(int i = 1; i < 10; i++){
				char c = Convert.ToChar('0' + i);
				m_bgname.Add(c, (byte) i);
			}
			//0x0a-0x23 'A' から 'Z'
			for(int i = 0; i <= 'Z' - 'A'; i++){
				char c = Convert.ToChar('A' + i);
				m_bgname.Add(c, (byte) (i + 0x0a));
			}
			m_bgname.Add(' ', 0x24);
			m_bgname.Add(',', 0x25);
			m_bgname.Add('.', 0x26);
			for(int i = 0; i < BGNAME_PATTERN.Length; i++){
				char c = Convert.ToChar(BGNAME_PATTERN.Substring(i, 1));
				m_bgname.Add(c, (byte) (i + 0x88));
			}
		}
		public void Append(imagefile.GameImage d)
		{
			m_gamedata[m_game_offset] = new ManageGame(d.Type, m_bgname);
			m_gamedata[m_game_offset].Set(d, ref m_page_offset);
			m_game_offset += 1;
		}
		
		byte [] rom_new(int romsize, int usepage, out int pagenum)
		{
			int usesize = usepage * GameImage.PAGESIZE;
			if(usesize >= romsize){ //使用ページ数を算出するために自動に変更
				romsize = CAPACITY_AUTO;
			}
			if(romsize == CAPACITY_AUTO){
				romsize = 0x10000;
				while(romsize < usesize){
					romsize *= 2;
				}
			}
			pagenum = romsize / GameImage.PAGESIZE;
			return Static.Utility.memory_new_fill(romsize, 0xff);
		}
		void copy(byte [] input, ref int offset, ref byte []concat)
		{
			for(int i = 0; i < input.Length; i++){
				concat[offset + i] = input[i];
			}
			offset += input.Length;
		}
		
		public byte [] Link(byte [] bios_buttom, byte [] bios_extra, out string [] log)
		{
			byte [] rom = rom_new(m_romsize, m_page_offset + 1, out m_page_num); //+1 は buttom 分, page_offset に extra 分は含まれている
			int offset = 0;
			copy(manage_make(m_gamedata, out log), ref offset, ref rom);
			copy(bios_extra, ref offset, ref rom);
			
			foreach(ManageGame t in m_gamedata){
				copy(t.Data, ref offset, ref rom);
			}
			offset = rom.Length - GameImage.PAGESIZE;
			copy(bios_buttom, ref offset, ref rom);
			Debug.Assert(offset == rom.Length);
			return rom;
		}
		
		public static void m6502_u16_set(int address, ref byte [] rom, ref int offset)
		{
			rom[offset++] = (byte) (address & 0xff);
			rom[offset++] = (byte) ((address >> 8) & 0xff);
		}
		//offset に影響がない版
		public static void m6502_u16_set(int address, ref byte [] rom, int offset)
		{
			rom[offset++] = (byte) (address & 0xff);
			rom[offset++] = (byte) ((address >> 8) & 0xff);
		}
		
		byte [] manage_make(ManageGame [] game, out string [] log)
		{
			byte [] manage = Static.Utility.memory_new_fill(SIZE, 0xff);
			int cpu_address = 0xc000;
			int offset = 0;
			//0xc000 0x10: sigunature
			foreach(char t in sigature.ToCharArray()){
				manage[offset] = (byte) t;
				offset++;
			}
			manage[offset++] = (byte) '\0';
			Debug.Assert(offset < 0x10);
			for(int i = offset; i < 0x10; i++){
				manage[i] = 0xff;
			}
			offset = 0x10;
			
			//0xc010 1: gamenum
			manage[offset++] = (byte) game.Length;
			//0xc011 2 * gamenum: gameinfomation struct pointer
			//ポインタ表の配列だけ作ってあとまわし
			int[] gameinfo_address = new int[game.Length];
			offset += game.Length * SIZEOF_6502_POINTER;

			//0xc011 2 * gamenum, gamenum*sizeof(struct info)
			int gameinfo_offset = 0;
			ManageSave saveram = new ManageSave();
			List <string> loglist = new List<string>();
			foreach(ManageGame g in game){
				gameinfo_address[gameinfo_offset++] = offset + cpu_address;
				g.RomMake(cpu_address, ref manage, ref offset, saveram, loglist);
			}
			log = loglist.ToArray();
			//ポインタ表を埋める
			offset = 0x11;
			foreach(int t in gameinfo_address){
				m6502_u16_set(t, ref manage, ref offset);
			}
			return manage;
		}
	}
	class ManageGame{
		int[] m_sideoffset;
		SaveArgument[] m_save;
		byte[] m_data;
		string m_name;
		readonly Dictionary<char, byte> m_bgname;
		readonly mdc5.Script.imagetype m_type;
		public byte [] Data{
			get{ return m_data;}
		}
		public ManageGame(mdc5.Script.imagetype type, Dictionary<char, byte> bgname)
		{
			m_type = type;
			m_bgname = bgname;
		}
		public void Set(imagefile.GameImage d, ref int offset)
		{
			m_data = d.ToRomImage(GameImage.PAGESIZE, ref offset, out m_sideoffset, out m_save);
			m_name = d.Name;
		}
		void name_set(string str, Dictionary<char, byte> d, ref byte [] rom, ref int offset)
		{
			//文字の値が見つからないエラー処理はいれないので、script取り込み時に弾くこと.
			int i;
			for(i = 0; i < str.Length; i++){
				Debug.Assert(d.ContainsKey(str[i]));
				rom[offset++] = d[str[i]];
			}
			//4byte alignment にして 6502 の処理時間を稼ぐ
			if((i % 4) != 0){
				int pad = 4 - str.Length % 4;
				for(i = 0; i < pad; i++){
					rom[offset++] = d[' '];
				}
			}
			rom[offset++] = d['\0'];
		}
		public void RomMake(int cpu_address, ref byte [] manage, ref int offset, ManageSave saveram, List <string> loglist)
		{
			switch(m_type){
			case mdc5.Script.imagetype.disk:
				//0 1: diskside count
				manage[offset++] = (byte) m_sideoffset.Length;
				//2 8: sideoffset
				for(int i = 0; i < m_sideoffset.Length; i++){
					byte page = (byte) m_sideoffset[i];
					page |= 0x80; //bit7 MMC5 ROM bit
					manage[offset++] = page;
				}
				break;
			case mdc5.Script.imagetype.rom:
				manage[offset++] = (byte) 0x80; //ROM assign
				for(int i = 0; i < m_sideoffset.Length; i++){
					manage[offset++] = (byte) m_sideoffset[i];
				}
				break;
			default:
				Debug.Assert(false);
				break;
			}
			Debug.Assert(m_sideoffset.Length < 8);
			for(int i = m_sideoffset.Length; i < 8; i++){
				manage[offset++] = 0xff;
			}
			//0xa 1: savedata count
			manage[offset++] = (byte) m_save.Length;
			//0xb 2: savedata description pointer
			int save_offset = offset;
			manage[offset++] = 0xaa; //dummy
			manage[offset++] = 0x55;
			//0x11 2: boot screen display name pointer
			int name_offset = offset;
			manage[offset++] = 0xaa; //dummy
			manage[offset++] = 0x55;
			//savedata
			if(m_save.Length != 0){
				//pointer set
				Manage.m6502_u16_set(cpu_address + offset, ref manage, save_offset);
				foreach(SaveArgument t in m_save){
					String s = String.Format(
						"saveram allocated 0x{0:x4}:{1}", 
						saveram.Offset, t.Name
					);
					loglist.Add(s);
					manage[offset++] = (byte) t.Side;
					//diskoffset は pagenum と pageoffset に分割する
					manage[offset++] = (byte) (t.Offset / 0x2000);
					Manage.m6502_u16_set(t.Offset & 0x1fff, ref manage, ref offset);
					Manage.m6502_u16_set(t.Length, ref manage, ref offset);
					Manage.m6502_u16_set(saveram.Append(t), ref manage, ref offset);
				}
			}
			//bootscreen data
			Manage.m6502_u16_set(cpu_address + offset, ref manage, name_offset);
			name_set(m_name, m_bgname, ref manage, ref offset);
		}
	}
	class ManageSave{
		int m_offset = 0x10; //0-0xf はテスト, checksum に使用
		List <mdc5.SaveArgument> m_data = new List<mdc5.SaveArgument>();
		public int Offset{ //log 専用 method
			get {return m_offset;}
		}
		public int Append(mdc5.SaveArgument s)
		{
			int ret = m_offset;
			m_data.Add(s);
			m_offset += s.Length;
			return ret;
		}
	}
	class Bios{
		public const int EXTRA_SIZE = 0x1800;
		readonly static byte [] BIOS_HASH = {
			0x57, 0xfe, 0x1b, 0xde, 0xe9, 0x55, 0xbb, 0x48, 
			0xd3, 0x57, 0xe4, 0x63, 0xcc, 0xbf, 0x12, 0x94, 
			0x96, 0x93, 0x0b, 0x62
		};
		byte [] m_rom_bottom, m_rom_extra;

		public byte [] RomBottom{
			get{return m_rom_bottom;}
		}
		public byte [] RomExtra{
			get{return m_rom_extra;}
		}
		public bool rom_load(byte [] rom)
		{
			SHA1CryptoServiceProvider sf = new SHA1CryptoServiceProvider();
			byte [] result = sf.ComputeHash(rom);
			Debug.Assert(result.Length == BIOS_HASH.Length);
			for(int i = 0; i < result.Length; i++){
				if(result[i] != BIOS_HASH[i]){
					return false;
				}
			}
			m_rom_bottom = rom;
			m_rom_extra = Static.Utility.memory_new_fill(EXTRA_SIZE, 0xff);
			return true;
		}
		void patch(RomRecoard.MotorolaSRecoard r, uint offset, ref byte [] data)
		{
			offset = r.Address - offset;
			for(int i = 0; i < r.Data.Length; i++){
				data[offset + i] = r.Data[i];
			}
		}
		public bool patch_load(string [] recoard)
		{
			foreach(string t in recoard){
				RomRecoard.MotorolaSRecoard r = new RomRecoard.MotorolaSRecoard(t);
				if(r.Error == true){
					return false;
				}
				if(r.Valid == false){
					continue;
				}
				uint address_end = r.Address + (uint) r.Data.Length - 1;
				if(r.Address >= 0xe000){
					if(address_end >= 0x10000){
						return false;
					}
					patch(r, 0xe000, ref m_rom_bottom);
				}else if(r.Address >= 0xc800){
					if(address_end >= 0xe000){
						return false;
					}
					patch(r, 0xc800, ref m_rom_extra);
				}else{
					return false;
				}
			}
			return true;
		}
	}
}
