/*
base program http://musyozoku211.blog118.fc2.com/?q=bpe
bpe.c
*/
using System;
using System.Diagnostics;
using System.Collections.Generic;
using hardware;

namespace Archiver.Bpe{
	class PairTable{
		protected const int TABLE_SIZE = 0x100;
		//protected 
		public const int SIGNED_TABLE_UNIT = 3;
	}
	class SignedList {
		Queue<byte> m_count, m_main, m_sub;
		public int Length{
			get {
				Debug.Assert(m_count.Count == m_main.Count);
				Debug.Assert(m_count.Count == m_sub.Count);
				return m_count.Count;
			}
		}
		public SignedList()
		{
			m_count = new Queue<byte>();
			m_main = new Queue<byte>();
			m_sub = new Queue<byte>();
		}
		public void Add(byte data, byte mainindex, byte subindex)
		{
			m_count.Enqueue(data);
			m_main.Enqueue(mainindex);
			m_sub.Enqueue(subindex);
		}
		public void Write(List<byte> data)
		{
			if(m_main.Count == 0){
				return;
			}
			Debug.Assert(m_main.Count < 0x100);
			data.Add((byte) m_main.Count);
			int byteoffset = 0;
			while(m_count.Count != 0){
				data.Add(m_count.Dequeue());
				data.Add(m_main.Dequeue());
				data.Add(m_sub.Dequeue());
				byteoffset += 3;
				if(byteoffset == 0x10 - 1){
					data.Add(0); //dummy for alignment 0x10
					byteoffset = 0;
				}
			}
		}
	}
	class SignedTable : PairTable{
		byte [] m_main, m_sub;
		public SignedTable()
		{
			m_main = new byte [TABLE_SIZE];
			m_sub = new byte [TABLE_SIZE];
			for(int i = 0; i < TABLE_SIZE; i++){
				m_main[i] = (byte) i;
				m_sub[i] = 0;
			}
		}
		public bool Read(Queue<byte> input_data, Block.type type)
		{
			if(type != Block.type.Raw){
				if(input_data.Count == 0){
					return false;
				}
				int pts = input_data.Dequeue();
				int alignmentoffset = 0;
				for(int i = 0; i < pts; i++){
					if(input_data.Count < SIGNED_TABLE_UNIT){
						return false;
					}
					int index = input_data.Dequeue();
					m_main[index] = input_data.Dequeue();
					m_sub[index] = input_data.Dequeue();
					alignmentoffset += 3;
					if(alignmentoffset == 0x10 - 1){
						input_data.Dequeue(); //dummy for alignment 0x10
						alignmentoffset = 0;
					}
				}
			}
			return true;
		}
		public byte MainGet(int index){
			return m_main[index];
		}
		public byte SubGet(int index){
			return m_sub[index];
		}
	}
	class EncodePair : PairTable{
		bool [] m_table;
		public EncodePair()
		{
			m_table = new bool[TABLE_SIZE];
			for(int i = 0; i < TABLE_SIZE; i++){
				m_table[i] = false;
			}
		}
		public void Load(byte [] block)
		{
			for(int i = 0; i < block.Length; i++){
				int index = block[i];
				m_table[index] = true;
			}
		}
		public bool UnusedCharFind(ref byte unused)
		{
			foreach(bool t in m_table){
				unused += 1;
				if(m_table[unused] == false){
					return true;
				}
			}
			return false;
		}
		public void FoundCharSet(byte data)
		{
			int index = (int) data;
			m_table[index] = true;
		}
	}
	class PairCount : PairTable{
		ushort [] m_data;
		public PairCount()
		{
			m_data = new ushort[TABLE_SIZE * TABLE_SIZE];
			for(int i = 0; i < m_data.Length; i++){
				m_data[i] = 0;
			}
		}
		int index_get(byte c0, byte c1)
		{
			return (c0 << 8) | c1;
		}
		public ushort Increment(byte c0, byte c1)
		{
			int index = index_get(c0, c1);
			m_data[index] += 1;
			return m_data[index];
		}
		public void Clear(byte c0, byte c1)
		{
			int index = index_get(c0, c1);
			m_data[index] = 0;
		}
	}
	class Buffer{
		byte [] m_work;
		readonly int m_size;
		int m_offset = 0;
		public Buffer(int size)
		{
			m_size = size;
			m_work = new byte [size];
		}
		public void Load(byte [] data, ref int offset)
		{
			System.Buffer.BlockCopy(data, offset, m_work, 0, m_size);
			offset += m_size;
		}
		public void Load(Queue<byte> data)
		{
			for(int i = 0; i < m_size; i++){
				m_work[i] = data.Dequeue();
			}
		}
		public int Offset{
			get {return m_offset;}
		}
		public byte Shift(){
			return m_work[m_offset++];
		}
	}
	class Contents{
		byte [] m_data;
		public int Length{
			get{return m_data.Length;}
		}
		public void Write(List<byte> block)
		{
			foreach(byte t in m_data){
				block.Add(t);
			}
		}
		public void Encode(byte [] block, EncodePair pairtable, SignedList signed_list)
		{
			PairCount paircount = new PairCount();
			int size = block.Length; //??????
			byte encode_data = 0xff;
			while(true){
				if(pairtable.UnusedCharFind(ref encode_data) == false){
					break;
				}
				/* ペア出現用カウンタにカウントし、最もよく出るペア(c1,c2)を得る */
				ushort maxcount = 0;
				byte c0 = 0, c1 = 0;
				for(int i = 0; i < size - 1; i++){
					byte cn0 = block[i + 0];
					byte cn1 = block[i + 1];
					ushort tcount = paircount.Increment(cn0, cn1);
					if(maxcount < tcount){
						c0 = cn0;
						c1 = cn1;
						maxcount = tcount;
					}
					if(i < size - 2){
						if((cn0 == cn1) && (cn1 == block[i + 2])){
							i++;
						}
					}
				}
				/* ペア(c0,c1)の出現回数が3以下なら終了 */
				if(maxcount <= 3){
					break;
				}
				/* ペア(c0,c1)を encode_data に置き換える
				同時にペア出現用カウンタの中身を0に戻す */
				paircount.Clear(c0, c1);
				int read_offset = 0, write_offset = 0;
				while(read_offset < size - 1){
					if(
						(block[read_offset] == c0) && 
						(block[read_offset+1] == c1)
					){
						if(block.Length - read_offset >= 3){
							paircount.Clear(block[read_offset + 1], block[read_offset + 2]);
						}else{
							//この分岐は元のプログラムのバグのような...
							paircount.Clear(block[read_offset + 1], 0);
						}
						block[write_offset++] = encode_data;
						read_offset += 2;
					}else{
						paircount.Clear(block[read_offset + 0], block[read_offset + 1]);
						block[write_offset++] = block[read_offset++];
					}
				}
				if(read_offset == size - 1){ //これのせいでデータが破壊されるぽい
					block[write_offset++] = block[read_offset];
				}

				size = write_offset;//????
				
				pairtable.FoundCharSet(encode_data);
				signed_list.Add(encode_data, c0, c1);
			}
			m_data = new byte[size];
			System.Buffer.BlockCopy(block, 0, m_data, 0, size);
		}
		public bool Decode(int bufsize, int blocksize, Buffer workbuf, SignedTable table)
		{
			List <byte> rowdata = new List <byte>();
			Stack<byte> stack = new Stack<byte>();
			while((workbuf.Offset < blocksize) || (stack.Count > 0)){
				byte ch;
				if(stack.Count == 0){
					ch = workbuf.Shift();
				}else{
					ch = stack.Pop();
				}
				while(true){
					if(ch == table.MainGet(ch)){
						Debug.Assert(stack.Count < bufsize);
						Debug.Assert(rowdata.Count < bufsize);
						rowdata.Add(ch);
						break;
					}
					stack.Push(table.SubGet(ch));
					if(stack.Count >= 0x80){
						return false;
					}
					ch = table.MainGet(ch);
				}
			}
			m_data = rowdata.ToArray();
			return true;
		}
	}
	class Block{
		public enum type{
			Raw, Compress, Backlog
		};
		protected const int BLOCK_HEADER_SIZE = 2;
		protected const int HEAD_TYPE_MASK = 0xc000;
		protected const int HEAD_BLOCKSIZE_MASK = ~HEAD_TYPE_MASK;
		protected const int HEAD_TYPE_RAW = 0;
		protected const int HEAD_TYPE_COMPRESS = 0x8000;
		protected const int HEAD_TYPE_BACKLOG = 0xc000;
	}
	class DecodeBlock : Block{
		readonly type m_type;
		readonly int m_blocksize;
		SignedTable m_table;
		Buffer m_workbuf;
		public DecodeBlock(Queue<byte> data)
		{
			int headcode = hardware.m6502.word_dequeue(data);
			m_blocksize = headcode & HEAD_BLOCKSIZE_MASK;
			switch(headcode & HEAD_TYPE_MASK){
			case HEAD_TYPE_RAW:
				m_type = type.Raw;
				break;
			case HEAD_TYPE_COMPRESS:
				m_type = type.Compress;
				break;
			case HEAD_TYPE_BACKLOG:
				m_type = type.Backlog;
				break;
			}
		}
		int pts_get(int num)
		{
			int pts = 0, alingment_offset = 0;
			while(num != 0){
				pts += PairTable.SIGNED_TABLE_UNIT;
				alingment_offset += PairTable.SIGNED_TABLE_UNIT;
				if(alingment_offset == 0x10 - 1){
					pts++;
					alingment_offset = 0;
				}
				num--;
			}
			return pts;
		}
		public bool DataSet(int mdcoffset, Queue<byte> data_queue, byte [] data_all)
		{
			m_table = new SignedTable();

			m_workbuf = new Buffer(m_blocksize);
			if(m_type == type.Backlog){
				if(data_queue.Count < hardware.mmc5.OFFSET_SIZE){
					return false;
				}
				//Debug.Assert(mdcoffset != 0, "mdcoffset undefined!");
				int offset = hardware.mmc5.offset_dequeue(data_queue, mdcoffset);
				m_workbuf.Load(data_all, ref offset);
				Queue<byte> backdata = new Queue<byte>();
				int pts = pts_get(data_all[offset]) + 1; //offset++ はやらない
				for(int i = 0; i < pts; i++){
					backdata.Enqueue(data_all[offset + i]);
				}
				if(m_table.Read(backdata, m_type) == false){
					return false;
				}
			}else{
				//read block data
				if(data_queue.Count < m_blocksize){
					return false;
				}
				m_workbuf.Load(data_queue);
				if(m_table.Read(data_queue, m_type) == false){
					return false;
				}
			}
			return true;
		}
		public bool Write(int buffer_size, List<byte> out_data)
		{
			Contents context = new Contents();
			if(context.Decode(buffer_size, m_blocksize, m_workbuf, m_table) == false){
				return false;
			}
			context.Write(out_data);
			return true;
		}
	}
	class EncodeBlock : Block{
		type m_type;
		int m_blocksize, m_self_offset, m_backlog_offset;
		List<byte> m_blockdata = new List<byte>();
		public int Length{
			get {return m_blockdata.Count + BLOCK_HEADER_SIZE;}
		}
		public int Offset{
			get {return m_self_offset;}
		}
		void block_set(type au_type, int size)
		{
			m_type = au_type;
			m_blocksize = size;
		}
		public void BacklogSet(EncodeBlock back_block, int current_offset)
		{
			m_self_offset = current_offset;
			m_backlog_offset = back_block.Offset + BLOCK_HEADER_SIZE;
			block_set(type.Backlog, back_block.m_blocksize);
			for(int i = 0; i < hardware.mmc5.OFFSET_SIZE; i++){
				m_blockdata.Add(0x77);
			}
		}
		public void Write(List<byte> out_data)
		{
			int block_header = 0;
			switch(m_type){
			case type.Compress:
				block_header = HEAD_TYPE_COMPRESS;
				break;
			case type.Backlog:
				block_header = HEAD_TYPE_BACKLOG;
				break;
			case type.Raw:
				block_header = HEAD_TYPE_RAW;
				break;
			}
			Debug.Assert(m_blocksize < 0x8000);
			block_header |= m_blocksize;
			hardware.m6502.word_add(out_data, block_header);
			foreach(byte t in m_blockdata.ToArray()){
				out_data.Add(t);
			}
		}
		public void Encode(byte [] raw, int size, int container_offset)
		{
			m_self_offset = container_offset;
			EncodePair pairtable = new EncodePair();
			pairtable.Load(raw);
			SignedList list = new SignedList();
			Contents context = new Contents();
			context.Encode(raw, pairtable, list);
			
			type au_type = type.Raw;
			if(list.Length != 0){
				au_type = type.Compress;
			}
			block_set(au_type, context.Length);
			context.Write(m_blockdata);
			list.Write(m_blockdata);
		}
		public void MdcOffsetSet(int offset)
		{
			m_self_offset += offset;
			if(m_type != type.Backlog){
				return;
			}
			m_backlog_offset += offset;
			hardware.mmc5.rom_offset_set(0, m_backlog_offset, m_blockdata);
		}
	}
	class Container {
		public enum type{CBR, VBR};
		readonly type m_type;
		readonly int m_buffer_size;
		const int BUFFER_SIZE_SIZE = 1;
		EncodeBlock [] m_block;
		int m_mdcoffset = 0;
		public int BufferSize{
			get {return m_buffer_size;}
		}
		public Container(int bufsize, type t_type)
		{
			m_buffer_size = bufsize;
			m_type = t_type;
		}

		public bool DecodeTest(byte [] rowdata)
		{
			return DecodeTest(EncodeDataGet(), rowdata);
		}
		public bool DecodeTest(byte [] bpedata, byte [] rowdata)
		{
			byte [] decodedata;
			if(Decode(bpedata, out decodedata) == false){
				return false;
			}
			return Archiver.Duplicate.hash_get(rowdata) == Archiver.Duplicate.hash_get(decodedata);
		}
		public bool Decode(byte [] bpedata, out byte [] rowdata)
		{
			Queue<byte> input_data = new Queue<byte>(bpedata);
			for(int i = 0; i < BUFFER_SIZE_SIZE; i++){ //buffersizeを捨てる
				input_data.Dequeue();
			}
			List<byte> out_data = new List<byte>();
			rowdata = null;
			while(input_data.Count != 0){
				DecodeBlock block = new DecodeBlock(input_data);
				if(block.DataSet(m_mdcoffset, input_data, bpedata) == false){
					return false;
				}
				block.Write(m_buffer_size, out_data);
			}
			rowdata = out_data.ToArray();
			return true;
		}

		public bool Encode(byte [] rowdata, bool backstream)
		{
			DuplicateMap [] map;
			int block_count = rowdata.Length / m_buffer_size; //切り落とす前の長さを取っておく
			if((rowdata.Length % m_buffer_size) != 0){
				block_count += 1;
			}
			Debug.Assert(rowdata.Length != 0);
			if(backstream == true){
				int rowsize = rowdata.Length;
				Duplicate d = new Duplicate(m_buffer_size);
				if(d.MapMake(rowdata, out map) == false){
					Debug.Assert(false);
					return false;
				}
				Debug.Assert(map.Length == block_count);
			}else{
				map = new DuplicateMap[block_count];
				for(int i = 0; i < map.Length; i++){
					map[i] = new DuplicateMap();
					map[i].Block = i;
					map[i].Original = true;
				}
			}

			int bpeoffset = BUFFER_SIZE_SIZE;
			Debug.Assert((mmc5.PAGESIZE % m_buffer_size) == 0);
			m_block = new EncodeBlock[block_count];
			for(int block_offset = 0; block_offset < block_count; block_offset++){
				EncodeBlock tblock = new EncodeBlock();
				if(map[block_offset].Original == false){
					int back_block_index = map[block_offset].Block;
					tblock.BacklogSet(m_block[back_block_index], bpeoffset);
				}else{
					int rowoffset = block_offset * m_buffer_size;
					int blocksize = rowdata.Length - rowoffset;
					blocksize = Math.Min(blocksize, m_buffer_size);
					byte [] page = new byte [blocksize];
					System.Buffer.BlockCopy(
						rowdata, rowoffset,
						page, 0, blocksize
					);
					tblock.Encode(page, blocksize, bpeoffset);
				}
				m_block[block_offset] = tblock;
				bpeoffset += tblock.Length;
			}
			return true;
		}
		
		public int EncodedLength()
		{
			int length = BUFFER_SIZE_SIZE;
			foreach(EncodeBlock t in m_block){
				length += t.Length;
			}
			return length;
		}
		public void BacklogOffsetSet(int offset)
		{
			m_mdcoffset = offset;
			foreach(EncodeBlock t in m_block){
				t.MdcOffsetSet(offset);
			}
		}
		public byte [] EncodeDataGet()
		{
			List<byte> data = new List<byte>();
			byte blocksize = (byte) (mmc5.PAGESIZE / m_buffer_size);
			if(m_type == type.CBR){
				blocksize |= 0x80;
			}
			data.Add(blocksize);
			foreach(EncodeBlock t in m_block){
				t.Write(data);
			}
			return data.ToArray();
		}
	}
}
