/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * MICSのエンジンおよびMICSで規定するインターフェースの定義などのパッケージ
 */
package net.wasamon.mics;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.LinkedList;

import net.wasamon.mics.gui.MicsArchitectureViewer;

import org.w3c.dom.Node;

/**
 * シミュレーションエンジン。 対象とするシミュレーションに必要な要素の集合を保持し、 ExecutableElementの並列実行をシミュレーションする。
 * このエンジン自身もまたExecutableElementであり、 MICSシミュレーションを階層的に定義することができる。
 * 
 * @author Takefumi MIYOSHI
 * 
 */
public class MicsCompositeElement extends MicsElement implements Runnable,
		ExecutableElement, DataBuffer, MicsViewable {

	/**
	 * MicsElement
	 */
	private Hashtable<String, MicsElement> element;

	/**
	 * DataBuffer
	 */
	private Hashtable<String, DataBuffer> ram;

	/**
	 * Channel
	 */
	private Hashtable<String, Channel> channel;

	/**
	 * ChannelConnectable
	 */
	private Hashtable<String, ChannelConnectable> channelConnectable;

	/**
	 * 有しているDataBufferインスタンスと、全体におけるオフセットを管理するクラス
	 */
	private class DataBufferElement {
		DataBuffer data;
		int offset;

		public DataBufferElement(DataBuffer d, int o) {
			data = d;
			offset = o;
		}
	}

	private DataBufferElement[] dataBufferElements;

	private MicsArchitectureViewer viewer;

	private class MicsEngineElement extends Thread {
		private ExecutableElement proc;

		int step;

		int tmpCycle;

		public MicsEngineElement(ExecutableElement _proc) {
			this.proc = _proc;
		}

		public void decrement() {
			if (step > 0) {
				step--;
			}
		}

		public void addCycle(int v) {
			tmpCycle += v;
		}

		public void updateCycle() {
			if (isExecutable()) { //
				step = tmpCycle;
				tmpCycle = 0;
			}
		}

		public boolean isExecutable() {
			return step == 0;
		}

		public void run() {
			try {
				ExecInfo info = proc.exec_first();
				if (info != null && info.isTerminatable() == false) { // まだ終了可能状態ではない
					setDoNextCycle();
				}
			} catch (MicsException e) {
				System.err.println(e);
			}
		}

	}

	/**
	 * MicsElement
	 */
	private MicsEngineElement[] executableElements;

	private void init() {
		activeFlag = false;
		executableFlag = false;
		cycle = 0;
	}

	public MicsCompositeElement() {
		init();
	}

	public void clear() {
		element = new Hashtable<String, MicsElement>();
		ram = new Hashtable<String, DataBuffer>();
		channel = new Hashtable<String, Channel>();
		channelConnectable = new Hashtable<String, ChannelConnectable>();
		executableElements = null;
		init();
	}

	public void reset() {
		if (activeFlag == false) { // 動作中はリセットできない
			for (int i = 0; i < executableElements.length; i++) {
				executableElements[i].proc.reset();
			}
			init();
		} else {
			System.out.println("cannot reset processors while engin is active.");
		}
	}

	public void stop() {
		executableFlag = false;
		System.out.println("engine is stopped at " + cycle);
		activeFlag = false;
	}

	public String getProcStatusString() throws MicsException {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < executableElements.length; i++) {
			sb.append(executableElements[i].proc.toString() + "\n");
		}
		return sb.toString();
	}

	private void stepCycle() {
		cycle++;
		for (int i = 0; i < executableElements.length; i++) {
			executableElements[i].decrement();
		}
	}

	private void updateCycle() {
		// cycle++;
		for (int i = 0; i < executableElements.length; i++) {
			executableElements[i].updateCycle();
		}
	}

	public boolean step() throws MicsException, ExecutableElementException,
			ConfigErrorException, MicsException, DataBufferException {
		ExecInfo info1;
		ExecInfo info2;
		stepCycle();
		info1 = exec_first();
		info2 = exec_second();
		//System.out.println("update cycle");
		updateCycle();
		return (!info1.isTerminatable()) || (!info2.isTerminatable());
	}

	private boolean activeFlag;

	private boolean executableFlag;

	private int cycle = 0;
	
	public String getDescription(){
		return "TODO";
	}
	
	public void run() {
		if (activeFlag) {
			System.out.println("cannot start while engin is active.");
			return;
		}
		activeFlag = true;
		Calendar cal = Calendar.getInstance();
		System.out.println("Starting of Execution All at " + cal.getTime());
		long start_time = cal.getTimeInMillis();
		executableFlag = true;
		try {
			while (step() && executableFlag)
				;
			activeFlag = false;
		} catch (MicsException e) {
			System.out.println("Program Terminated at " + cal.getTime()
					+ " abnormally.");
			System.out.println(e.getMessage() + " at " + cycle);
			return;
		}
		cal = Calendar.getInstance();
		if (executableFlag) {
			Mics.getInstance().shell().setMicsEngineEnableState(true);
			System.out.println("Program Terminated at " + cal.getTime());
			long timespan = cal.getTimeInMillis() - start_time;
			System.out.print("Execution Cycle: " + cycle + " Time: " + timespan
					+ "ms");
			if (timespan > 0) {
				System.out.println(" C/S: "
						+ (double) ((double) cycle / (double) timespan) + "kHz");
			} else {
				System.out.println("");
			}
		}
	}

	private boolean fDoNextCycle;

	synchronized public void setDoNextCycle() {
		fDoNextCycle = true;
	}

	LinkedList<MicsEngineElement> threadList = new LinkedList<MicsEngineElement>();

	public ExecInfo exec_first() throws MicsException {
		fDoNextCycle = false;
		for (int i = 0; i < executableElements.length; i++) {
			MicsEngineElement element = executableElements[i];
			if (element.isExecutable()) {
				//System.out.println("exec_first: " + element.proc.id());
				ExecInfo info = element.proc.exec_first();
				if (info != null) {
					fDoNextCycle |= !info.isTerminatable();
				}
			} else {
				fDoNextCycle = true;
			}
		}
		return new ExecInfo(!fDoNextCycle, 0);
	}

	public ExecInfo exec_second() throws MicsException {
		boolean flag = false;
		for (int i = 0; i < executableElements.length; i++) {
			MicsEngineElement element = executableElements[i];
			if (element.isExecutable()) {
				//System.out.println("exec_second: " + element.proc.id());
				ExecInfo info = element.proc.exec_second();
				if (info != null) {
					element.addCycle(info.getCycle());
					if (info.isTerminatable() == false) {
						flag = true;
					}
				}
			} else {
				flag = true;
			}
		}
		return new ExecInfo(!flag, 0);
	}

	public String[] getConnectedElements() {
		/*
		 * ArrayList connections = new ArrayList(); Iterator it =
		 * element.iterator(); while(it.hasNext()){ MicsElement e0 =
		 * (MicsElement)(it.next()); String[] c0 = e0.getConnectedElements(); if(c0 !=
		 * null){ for(int i = 0; i < c0.length; i++){ if(hasMicsElement(c0[i]) ==
		 * false){ connections.add(c0[i]); } } } } return
		 * (String[])connections.toArray(new Object[connections.size()]);
		 */
		return null;
	}

	public String getImagePath() {
		// TODO Auto-generated method stub
		return null;
	}

	private void initializeElement(String base, Node[] nodes)
			throws MicsException {
		clear();
		ArrayList<MicsEngineElement> executableElementsArray = new ArrayList<MicsEngineElement>();
		ArrayList<MicsElement> elementArray = new ArrayList<MicsElement>();
		ArrayList<Channel> channelArray = new ArrayList<Channel>();
		ArrayList<ChannelConnectable> channelConnectableArray = new ArrayList<ChannelConnectable>();
		ArrayList<DataBuffer> ramArray = new ArrayList<DataBuffer>();
		ArrayList<DataBufferElement> dataBufferElementsArray = new ArrayList<DataBufferElement>();
		for (int i = 0; i < nodes.length; i++) {
			Node n = nodes[i];
			String classtype = "";
			try {
				classtype = getAttributeAsString(n, "class");
				String id = getAttributeAsString(n, "id");
				Class c = ClassLoader.getSystemClassLoader().loadClass(classtype);
				MicsElement r = (MicsElement) (c.newInstance());
				r.initialize_base(this, base, id, n);
				elementArray.add(r);
				if (r instanceof ExecutableElement) {
					executableElementsArray.add(new MicsEngineElement(
							(ExecutableElement) r));
				}
				if (r instanceof Channel) {
					channelArray.add((Channel) r);
				}
				if (r instanceof ChannelConnectable) {
					channelConnectableArray.add((ChannelConnectable) r);
				}
				if (r instanceof DataBuffer) {
					ramArray.add((DataBuffer) r);
					int offset = 0;
					if (dataBufferElementsArray.size() > 0) {
						DataBufferElement element = (DataBufferElement) (dataBufferElementsArray
								.get(dataBufferElementsArray.size() - 1));
						offset = element.offset + element.data.size();
					}
					dataBufferElementsArray.add(new DataBufferElement((DataBuffer) r,
							offset));
				}
			} catch (ClassNotFoundException e) {
				e.printStackTrace();
				throw new MicsException("no such class: " + classtype);
			} catch (InstantiationException e) {
				e.printStackTrace();
				throw new MicsException("cannot instantiation: " + classtype);
			} catch (IllegalAccessException e) {
				e.printStackTrace();
				throw new MicsException("illegal access: " + classtype);
			}
		}
		executableElements = executableElementsArray.toArray(new MicsEngineElement[executableElementsArray.size()]);
		for(Channel c0: channelArray){
			channel.put(c0.id(), c0);
		}
		for(ChannelConnectable c0: channelConnectableArray){
			channelConnectable.put(c0.id(), c0);
		}
		for(DataBuffer d0: ramArray){
			ram.put(d0.id(), d0);
		}
		dataBufferElements = dataBufferElementsArray.toArray(new DataBufferElement[dataBufferElementsArray.size()]);
		for(MicsElement e0: elementArray){
			element.put(e0.id(), e0);
		}
//		for(MicsElement e0: elementArray){
//			e0.initialize();
//		}
	}

	public void initialize(String base, Node node) throws MicsException {
		initializeElement(base, getNamedNodeArray(node, "element"));
		viewer = Mics.getInstance().shell().getMicsArchitectureViewerData(this);
	}

	public MicsElement[] getMicsElementArray() {
		return element.values().toArray(new MicsElement[element.size()]);
	}

	public MicsElement getMicsElement(String id) throws MicsException {
		return element.get(id);
	}

	// TODO アドホックすぎる。チャネルの管理のコードをちゃんとかかないとだめなんだと思う
	public DataBuffer getDataBuffer(String id) throws MicsException {
		DataBuffer d = ram.get(id);
		if ((d == null || !id.equals(d.id())) && composite != null) {
			d = composite.getDataBuffer(id);
		}
		return d;
	}

	// TODO アドホックすぎる。チャネルの管理のコードをちゃんとかかないとだめなんだと思う
	public Channel getChannel(String id) throws MicsException {
		Channel c = channel.get(id);
		if ((c == null || !id.equals(c.id())) && composite != null) {
			c = composite.getChannel(id);
		}
		return c;
	}

	// TODO アドホックすぎる。チャネルの管理のコードをちゃんとかかないとだめなんだと思う
	public ChannelConnectable getChannelConnectable(String id) throws MicsException {
		ChannelConnectable cc = channelConnectable.get(id);
		if ((cc == null || !id.equals(cc.id())) && composite != null) {
			cc = composite.getChannelConnectable(id);
		}
		return cc;
	}

	public void shutdown() {
		if (element != null) {
			for(MicsElement el: element.values()){
				el.shutdown();
			}
		}
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		DataBufferElement d = searchDataBuffer(offset);
		if (d != null)
			d.data.dump(offset - d.offset, len, writer);

	}

	public MicsDataPacket read(MicsDataPacket data) {
		return null;
	}

	public int size() {
		if (dataBufferElements.length == 0) {
			return 0;
		} else {
			DataBufferElement d = (DataBufferElement) (dataBufferElements[dataBufferElements.length - 1]);
			return d.offset + d.data.size();
		}
	}

	public String toString(int addr, int length) {
		DataBufferElement d = searchDataBuffer(addr);
		if (d != null) {
			return d.data.toString(addr - d.offset, length);
		} else {
			return null;
		}
	}

	public void write(MicsDataPacket data) {
	}

	public void write(int offset, InputStream reader) throws DataBufferException {
		DataBufferElement d = searchDataBuffer(offset);
		if (d != null) {
			d.data.write(offset - d.offset, reader);
		}
	}

	private DataBufferElement searchDataBuffer(int offset) {
		for (int i = 0; i < dataBufferElements.length - 1; i++) {
			DataBufferElement d0 = dataBufferElements[i];
			DataBufferElement d1 = dataBufferElements[i + 1];
			if (d0.offset <= offset && offset < d1.offset) {
				return d0;
			}
		}
		if (dataBufferElements.length > 0) {
			return dataBufferElements[dataBufferElements.length - 1];
		} else {
			return null;
		}
	}

	public void show() {
		if (viewer != null) {
			viewer.show();
		}
	}

}
