/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;

import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import net.wasamon.mics.architecturemaker.unit.HardwareUnit;
import net.wasamon.mics.architecturemaker.unit.Mediator;

/**
 * @author Masayuki Morisita
 */
public class ActionButtonPanel extends JPanel {
	private RoundedCornerButton newButton;
	private RoundedCornerButton openButton;
	private RoundedCornerButton saveButton;
	private RoundedCornerButton convertButton;
	private RoundedCornerButton redoButton;
	private RoundedCornerButton undoButton;
	private RoundedCornerButton selectAllButton;
	private RoundedCornerButton cutButton;
	private RoundedCornerButton copyButton;
	private RoundedCornerButton pasteButton;
	private RoundedCornerButton zoomInButton;
	private RoundedCornerButton zoomOutButton;
	private RoundedCornerButton adjustButton;
	private RoundedCornerButton disconnectButton;
	private RoundedCornerButton deleteButton;
	private RoundedCornerButton helpButton;
	private RoundedCornerTextField scaleField;
	private Mediator mediator;
	private static ActionButtonPanel instance = new ActionButtonPanel();

	private ActionButtonPanel() {
		setLayout(new BorderLayout());

		mediator = Mediator.getInstance();

		JPanel toolButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		JPanel helpButtonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		add(toolButtonPanel, "West");
		add(helpButtonPanel, "East");

		newButton = new RoundedCornerButton("",        newImageIcon("icons/button/file_bl_b.gif"), getBackground());
		openButton = new RoundedCornerButton("",       newImageIcon("icons/button/file_xml_b.gif"), getBackground());
		saveButton = new RoundedCornerButton("",       newImageIcon("icons/button/save_b.gif"), getBackground());
		convertButton = new RoundedCornerButton("",    newImageIcon("icons/button/thunder.gif"), getBackground());
		undoButton = new RoundedCornerButton("",       newImageIcon("icons/button/undo_b.gif"), getBackground());
		redoButton = new RoundedCornerButton("",       newImageIcon("icons/button/redo_r.gif"), getBackground());
		selectAllButton = new RoundedCornerButton("",  newImageIcon("icons/button/select_all.gif"), getBackground());
		cutButton = new RoundedCornerButton("",        newImageIcon("icons/button/cut_b.gif"), getBackground());
		copyButton = new RoundedCornerButton("",       newImageIcon("icons/button/copy_w.gif"), getBackground());
		pasteButton = new RoundedCornerButton("",      newImageIcon("icons/button/paste_all_y.gif"), getBackground());
                zoomInButton = new RoundedCornerButton("",     newImageIcon("icons/button/magnify_r.gif"), getBackground());
                zoomOutButton = new RoundedCornerButton("",     newImageIcon("icons/button/reduce_b.gif"), getBackground());
		adjustButton = new RoundedCornerButton("",     newImageIcon("icons/button/grid_g.gif"), getBackground());
		disconnectButton = new RoundedCornerButton("", newImageIcon("icons/button/con_off_w.gif"), getBackground());
		deleteButton = new RoundedCornerButton("",     newImageIcon("icons/button/delete_r.gif"), getBackground());
		helpButton = new RoundedCornerButton("",       newImageIcon("icons/button/help_w.gif"), getBackground());
                scaleField = new RoundedCornerTextField("100%");
                scaleField.setColumns(3);
                scaleField.setEditable(false);


		newButton.setToolTipText("New");
		openButton.setToolTipText("Open");
		saveButton.setToolTipText("Save");
		convertButton.setToolTipText("Convert");
		undoButton.setToolTipText("Undo");
		redoButton.setToolTipText("Redo");
		selectAllButton.setToolTipText("Select All");
		cutButton.setToolTipText("Cut");
		copyButton.setToolTipText("Copy");
		pasteButton.setToolTipText("Paste");
                zoomInButton.setToolTipText("Zoom In");
                zoomOutButton.setToolTipText("Zoom Out");
		adjustButton.setToolTipText("Adjust Position");
		disconnectButton.setToolTipText("Disconnect to Each Other");
		deleteButton.setToolTipText("Delete");
		helpButton.setToolTipText("I Can't Help You");

		toolButtonPanel.add(newButton);
		toolButtonPanel.add(openButton);
		toolButtonPanel.add(saveButton);
		toolButtonPanel.add(convertButton);
                toolButtonPanel.add(zoomOutButton);
                toolButtonPanel.add(scaleField);
                toolButtonPanel.add(zoomInButton);
		toolButtonPanel.add(undoButton);
		toolButtonPanel.add(redoButton);
		toolButtonPanel.add(selectAllButton);
		toolButtonPanel.add(cutButton);
		toolButtonPanel.add(copyButton);
		toolButtonPanel.add(pasteButton);
		toolButtonPanel.add(adjustButton);
		toolButtonPanel.add(disconnectButton);
		toolButtonPanel.add(deleteButton);

		helpButtonPanel.add(helpButton);

		openButton.setEnabled(false);
		setActionButtonEnabled(true, true, true, false, false, false, false,
				false, false, false, false);

		// actionListener
		newButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (mediator.modified()) {
					int answer = JOptionPane
							.showConfirmDialog(
									null,
									"Do you save this config file\nbefore cleaning the canvas ?",
									"", JOptionPane.YES_NO_CANCEL_OPTION,
									JOptionPane.QUESTION_MESSAGE, null);
					if (answer == 0) {
						saveXML();
						mediator.setModification(false);
					} else if (answer == 1) {
						mediator.clearAllUnits();
						mediator.convert();
						mediator.setModification(false);
					}
				}
			}
		});

		openButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				JFileChooser fileChooser = new JFileChooser();
				int option = fileChooser.showOpenDialog(null);
				if (option == JFileChooser.APPROVE_OPTION) {
					File file = fileChooser.getSelectedFile();
				} else if (option == JFileChooser.CANCEL_OPTION) {
				} else if (option == JFileChooser.ERROR_OPTION) {
				}
			}
		});

		saveButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (saveXML()) {
					mediator.setModification(false);
				}
			}
		});

		convertButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				mediator.convert();
			}
		});

                zoomInButton.addActionListener(new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
			    int scale = mediator.getCanvasScale();
			    if (scale < 200) {
				scale *= 2;
			    }
			    scaleField.setText(Integer.toString(scale) + "%");
			    mediator.setCanvasScale(scale);
                        }
		    });

                zoomOutButton.addActionListener(new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
			    int scale = mediator.getCanvasScale();
			    if (25 < scale) {
				scale /= 2;
			    }
			    scaleField.setText(Integer.toString(scale) + "%");
			    mediator.setCanvasScale(scale);
                        }
		    });


		selectAllButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				mediator.selectAll();
			}
		});

		cutButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				HardwareUnit[] selectedUnits = mediator.getSelectedUnits();
				mediator.copyUnits(selectedUnits);
				mediator.deleteUnits(selectedUnits);
			}
		});

		copyButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				mediator.copyUnits(mediator.getSelectedUnits());
			}
		});

		pasteButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				mediator.pasteUnits();
			}
		});

		disconnectButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				HardwareUnit[] selectedUnits = mediator.getSelectedUnits();
				mediator.disconnectUnits(selectedUnits);
			}
		});

		deleteButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				HardwareUnit[] selectedUnits = mediator.getSelectedUnits();
				mediator.deleteUnits(mediator.getSelectedUnits());
			}
		});

		adjustButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				mediator.adjustUnitPositions(mediator.getSelectedUnits());
			}
		});

	}
	
	public static ActionButtonPanel getInstance() {
		return instance;
	}
	
	private ImageIcon newImageIcon(String name){
		URL url = getClass().getClassLoader().getResource(name);
		if(url != null){
			return new ImageIcon(url);
		}else{
			return new ImageIcon();
		}
	}


	private boolean saveXML() {
		JFileChooser fileChooser = new JFileChooser();
		int option = fileChooser.showSaveDialog(null);

		if (option == JFileChooser.APPROVE_OPTION) {
			try {
				FileOutputStream outputStream = new FileOutputStream(
						fileChooser.getSelectedFile());
				mediator.printXML(outputStream);
				outputStream.close();
			} catch (FileNotFoundException ex) {
				JOptionPane.showMessageDialog(fileChooser, ex.toString());
				return false;
			} catch (SecurityException ex) {
				JOptionPane.showMessageDialog(fileChooser, ex.toString());
				return false;
			} catch (IOException ex) {
				JOptionPane.showMessageDialog(fileChooser, ex.toString());
				return false;
			}

			File file = fileChooser.getSelectedFile();
		} else if (option == JFileChooser.CANCEL_OPTION) {
			return false;
		} else if (option == JFileChooser.ERROR_OPTION) {
			return false;
		}

		return true;
	}

    private void setActionButtonEnabled(boolean zi, boolean zo, boolean ud, boolean rd, boolean sa, boolean ct, 
					boolean cp, boolean pt,boolean ad, boolean dc, boolean dt) {
	zoomInButton.setEnabled(zi);
	zoomOutButton.setEnabled(zo);
	undoButton.setEnabled(ud);
	redoButton.setEnabled(rd);
	selectAllButton.setEnabled(sa);
	cutButton.setEnabled(ct);
	copyButton.setEnabled(cp);
	pasteButton.setEnabled(pt);
	adjustButton.setEnabled(ad);
	disconnectButton.setEnabled(dc);
	deleteButton.setEnabled(dt);
    }

	public void updateButtonsState() {
		int selectedUnitsCount = mediator.getSelectedUnitsCount();
		boolean sa = (mediator.getUnitsCount() == 0) ? false : true;
		boolean ps = !mediator.copyBufferIsEmpty();

		if (selectedUnitsCount == 0) {
		    setActionButtonEnabled(true, true, false, false, sa, false, false,
					ps, false, false, false);
		} else if (selectedUnitsCount == 1) {
			setActionButtonEnabled(true, true, false, false, sa, true, true,
					ps, true, false, true);
		} else {
			setActionButtonEnabled(true, true, false, false, sa, true, true,
					ps, true, true, true);
		}
	}

	public void updateButtonsState(boolean flag) {
	    setActionButtonEnabled(flag, flag, flag, flag, flag, flag, flag, flag, flag, flag, flag);
	}
}
