/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker.unit;

import java.lang.NullPointerException;
import java.lang.Integer;
import java.util.regex.Pattern;
import java.util.ArrayList;

/**
 * @author Masayuki Morisita
 */
public abstract class HardwareUnit extends MicsNode {
	AppearanceAttribute appearance;
	ArrayList<HardwareUnit> connectedUnits;
	
	public HardwareUnit(String id, String unitClass, int x, int y, int width,
			int height, String imageFile) {
		appearance = new AppearanceAttribute(x, y, width, height, imageFile);
		connectedUnits = new ArrayList<HardwareUnit>();
		tag = "element";
		indent = "  ";

		setID(id);
		conditionTable.put("class", Pattern.compile(".{0}&&[^.]"));
		setUnitClass(unitClass);

		childList.add(appearance);
	}

	public boolean inRange(int x, int y) {
		int width = appearance.getWidth();
		int height = appearance.getHeight();
		int left = appearance.getX() - width / 2;
		int top = appearance.getY() - height / 2;
		return x >= left && x < left + width && y >= top && y < top + height;
	}

	public String getID() {
		return get("id");
	}

	public void setID(String id) {
		set("id", id);
	}

	public String getUnitClass() {
		return get("class");
	}

	private void setUnitClass(String unitClass) {
		setValue("class", unitClass);
	}

	public HardwareUnit[] getConnectedUnitList() {
		return connectedUnits.toArray(new HardwareUnit[connectedUnits.size()]);
	}

	public void setImage(String imageFile) {
		appearance.set("image", imageFile);
	}


	public String getImageFile() {
	    return appearance.get("image");
	}

	public boolean isConnectedWith(HardwareUnit unit) {
		return connectedUnits.contains(unit);
	}

	public void disconnect(HardwareUnit unit) {
		remove(unit);
		unit.remove(this);

		unit.connectedUnits.remove(this);
		connectedUnits.remove(unit);
	}

	public boolean connect(HardwareUnit unit) {
		if (unit == null || !isConnectable(unit) || !unit.isConnectable(this)) {
			return false;
		} else if (isConnectedWith(unit)) {
			return true;
		}

		register(unit);
		unit.register(this);

		connectedUnits.add(unit);
		unit.connectedUnits.add(this);

		return true;
	}

	public HardwareUnit getConnectedUnit(String id) {
		for (int i = 0; i < connectedUnits.size(); ++i) {
			HardwareUnit unit = connectedUnits.get(i);
			if (id.equals(unit.getID())) {
				return unit;
			}
		}

		return null;
	}

	public abstract String getBaseID();

	public abstract void setBaseID(String baseID);

	public abstract String getInformation();

	public abstract void register(HardwareUnit unit);

	public abstract void remove(HardwareUnit unit);

	public abstract boolean isConnectable(HardwareUnit unit);

	public abstract HardwareUnit copy();

	public ArrayList<HardwareUnit> connectedUnits(){
		return connectedUnits;
	}
	public AppearanceAttribute appearance(){
		return appearance;
	}
 
	
}
