/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker.unit;

import java.io.OutputStream;
import java.io.IOException;
import java.lang.NullPointerException;
import java.util.HashMap;

import java.awt.Image;
import javax.swing.ImageIcon;

import net.wasamon.mics.architecturemaker.ActionButtonPanel;
import net.wasamon.mics.architecturemaker.CanvasViewPort;
import net.wasamon.mics.architecturemaker.ConfigFileEditor;

/**
 * @author Masayuki Morisita
 */
public class Mediator {
	private HardwareUnit[] copyBuffer;
	private boolean modified = false;
    private HashMap<AppearanceAttribute, Image> imageTable = new HashMap<AppearanceAttribute, Image>();
    public final static String undefinedImageFile = "icons/unit/pw_hatena1_8.gif";
    private static Mediator instance = new Mediator();

	private Mediator() {
	}

	public static Mediator getInstance() {
		return instance;
	}

	public void setValue(MicsNode node, String key, String value) {
	    if(key.equals("image") && node instanceof AppearanceAttribute){
		setImage((AppearanceAttribute)node, value);
	    }
	    else if((key.equals("width") || key.equals("height")) && node instanceof AppearanceAttribute){
		HardwareUnitManager.getInstance().setValue(node, key, value);
		setImage((AppearanceAttribute)node, value);
	    }
	    else{
		HardwareUnitManager.getInstance().setValue(node, key, value);
	    }

	    update(false, true, true);
	}

    public Image getImage(AppearanceAttribute appearance){
	return imageTable.get(appearance);
    }

    public void setImage(AppearanceAttribute appearance, String imageFile){

	Image temp = null;
	try {
		temp = (new ImageIcon(getClass().getClassLoader().getResource(imageFile))).getImage();
		appearance.setValue("image", imageFile);
	} catch (NullPointerException e) {
		temp = (new ImageIcon(getClass().getClassLoader().getResource(undefinedImageFile))).getImage();
		appearance.setValue("image", undefinedImageFile);
	}

	imageTable.put(appearance, (new ImageIcon(temp.getScaledInstance((int)(appearance.getWidth() * (getCanvasScale() / 100.0)), 
									 (int)(appearance.getHeight() * (getCanvasScale() / 100.0)),
									 Image.SCALE_SMOOTH))).getImage());
    }

	public void removeChild(MicsNode node, String element) {
		node.removeChild(element);
		update(true, true, true);
	}

	public void addUnit(HardwareUnit unit) {
		HardwareUnitManager.getInstance().addUnit(unit);
		HardwareUnitManager.getInstance().clearSelectedUnits();
		HardwareUnitManager.getInstance().select(unit);
		setImage(unit.appearance, unit.appearance.get("image"));
		update(true, true, true);
	}

	public HardwareUnit[] getAllUnits() {
		return HardwareUnitManager.getInstance().getAllUnits();
	}

	public HardwareUnit[] getSelectedUnits() {
		return HardwareUnitManager.getInstance().getSelectedUnits();
	}

	public void clearAllUnits() {
		HardwareUnitManager.getInstance().clearAllUnits();
		copyBuffer = null;
		update(true, true, true);
	}

	public void clearSelectedUnits() {
		HardwareUnitManager.getInstance().clearSelectedUnits();
		update(true, true, true);
	}

	public void deselect(HardwareUnit unit) {
		HardwareUnitManager.getInstance().deselect(unit);
		update(true, true, true);
	}

	public void select(HardwareUnit unit) {
		HardwareUnitManager.getInstance().select(unit);
		update(true, true, true);
	}

	public void selectAll() {
		HardwareUnitManager.getInstance().selectAll();
		update(true, true, true);
	}

	public boolean copyBufferIsEmpty() {
		return copyBuffer == null;
	}

	public void copyUnits(HardwareUnit[] units) {
		if (units != null && units.length != 0) {
			copyBuffer = HardwareUnitManager.getInstance().copy(units);
		}
		update(true, false, false);
	}

	public void pasteUnits() {
		if (copyBufferIsEmpty() || copyBuffer.length == 0) {
			copyBuffer = null;
			return;
		}

		for (int i = 0; i < copyBuffer.length; ++i) {
		    HardwareUnit unit = copyBuffer[i];
			HardwareUnitManager.getInstance().addUnit(unit);
			setImage(unit.appearance, unit.appearance.get("image"));
		}

		HardwareUnitManager.getInstance().clearSelectedUnits();
		for (int i = copyBuffer.length - 1; i >= 0; --i) {
			HardwareUnitManager.getInstance().select(copyBuffer[i]);
		}

		copyBuffer = null;
		update(true, true, true);
	}

	public void deleteUnits(HardwareUnit[] units) {
		HardwareUnitManager.getInstance().deleteUnits(units);
		update(true, true, true);
	}

	public void addInitFile(HardwareUnit unit, String file, String offset) {
		HardwareUnitManager.getInstance().addInitFile(unit, file, offset);
		update(false, true, false);
	}

	public int getUnitsCount() {
		return HardwareUnitManager.getInstance().getUnitsCount();
	}

	public int getSelectedUnitsCount() {
		return HardwareUnitManager.getInstance().getSelectedUnitsCount();
	}

	public boolean selectedUnit(HardwareUnit unit) {
		return HardwareUnitManager.getInstance().selectedUnit(unit);
	}

	public void disconnectUnits(HardwareUnit[] units) {
		HardwareUnitManager.getInstance().disconnectUnits(units);
		update(false, false, true);
	}

	public int getCanvasWidth() {
		return CanvasViewPort.getInstance().getSize().width;
	}

	public int getCanvasHeight() {
		return CanvasViewPort.getInstance().getSize().height;
	}

	public int getCanvasScale() {
		return CanvasViewPort.getInstance().getScale();
	}

	public void setCanvasScale(int scale) {
		CanvasViewPort.getInstance().setScale(scale);
		HardwareUnit[] units = getAllUnits();
		for(int i = 0; i < units.length; ++i){
		    setImage(units[i].appearance, units[i].getImageFile());
		}
		update(false, false, true);
	}

	public void adjustUnitPositions(HardwareUnit units[]) {
		CanvasViewPort.getInstance().adjustUnitPositions(units);
		update(false, true, true);
	}

	public void printXML(OutputStream outputStream) throws IOException {
		HardwareUnitManager.getInstance().printXML(outputStream);
	}

	public void convert() {
		ConfigFileEditor.getInstance().convert();
	}

	public HardwareUnit getUnit(String id) {
		return HardwareUnitManager.getInstance().getUnit(id);
	}

	public boolean modified() {
		return modified;
	}

	public void setModification(boolean modified) {
		this.modified = modified;
	}

	public void update(boolean btn, boolean prpt, boolean cvs) {
		if (btn) {
			ActionButtonPanel.getInstance().updateButtonsState();
		}

		if (prpt) {
			HardwareUnitPropertyTable.getInstance().showPropertyTable();
		}

		if (cvs) {
			CanvasViewPort.getInstance().repaint();
		}
		modified = true;
	}
}
