/* 
 *    Copyright 2007 MICS Project
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mics.architecturemaker.unit;

import java.util.ArrayList;
import java.util.HashMap;
import java.io.OutputStream;
import java.io.IOException;
import java.lang.StringBuilder;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.util.regex.PatternSyntaxException;
import java.util.Iterator;
import java.util.Map;

/**
 * @author Masayuki Morisita
 */
public abstract class MicsNode {
	HashMap<String, StringBuilder> attributeTable;
	HashMap<String, Pattern> conditionTable;
	ArrayList<MicsNode> childList;
	String tag;
	String indent;

	public MicsNode() {
		tag = new String();
		indent = new String();
		attributeTable = new HashMap<String, StringBuilder>();
		conditionTable = new HashMap<String, Pattern>();
		childList = new ArrayList<MicsNode>();
	}

	public String get(String key) {
		return getValue(key).toString();
	}

	public void set(String key, String value) {
		if (check(key, value)) {
			setValue(key, value);
		}
	}

	public boolean check(String key, String value) {
		Pattern pattern = conditionTable.get(key);
		if (pattern == null) {
			return true;
		}

		return pattern.matcher(value).matches();
	}

	protected StringBuilder getValue(String key) {
		return attributeTable.get(key);
	}

	protected void setValue(String key, String value) {
		if (attributeTable.get(key) == null) {
			attributeTable.remove(key);
			attributeTable.put(key, new StringBuilder(value));
			return;
		}

		StringBuilder current = attributeTable.get(key);
		current.delete(0, current.length());
		current.append(value);
	}

	protected void setValue(String key, StringBuilder value) {
		attributeTable.remove(key);
		attributeTable.put(key, value);
	}

	public void printXML(OutputStream outputStream) throws IOException {
		String output = "";

		output += indent + "<" + tag;
		Iterator<Map.Entry<String, StringBuilder>> iterator = attributeTable
				.entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry<String, StringBuilder> entry = iterator.next();
			output += " " + entry.getKey() + "=\"" + entry.getValue() + "\"";
		}

		if (childList.size() == 0) {
			output += "/>\n";
			outputStream.write(output.getBytes());
			return;
		}

		output += ">\n";
		outputStream.write(output.getBytes());

		for (int i = 0; i < childList.size(); ++i) {
			MicsNode child = childList.get(i);
			child.printXML(outputStream);
		}

		output = indent + "</" + tag + ">\n";
		outputStream.write(output.getBytes());
	}

	public String getTag() {
		return tag;
	}

	public void removeChild(String element) {
		if (element == null) {
			return;
		}

		for (int i = 0; i < childList.size(); ++i) {
			MicsNode child = childList.get(i);
			String cElement = child.tag + " ";
			Iterator<Map.Entry<String, StringBuilder>> iterator = child.attributeTable
					.entrySet().iterator();
			while (iterator.hasNext()) {
				Map.Entry<String, StringBuilder> entry = iterator.next();
				cElement += entry.getKey() + "=\"" + entry.getValue() + "\" ";
			}
			if (element.equals(cElement)) {
				childList.remove(child);
				return;
			}
		}
	}
	
	public ArrayList<MicsNode> childList(){
		return childList;
	}
	
	public HashMap<String, StringBuilder> attributeTable(){
		return attributeTable;
	}

}
