/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */


/**
 * MICSのエンジンおよびMICSで規定するインターフェースの定義などのパッケージ
 */
package net.wasamon.mics;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.StringTokenizer;

import net.wasamon.mics.gui.MicsArchitectureViewer;

/**
 * CUIによるMICS操作インターフェース
 * @author Takefumi MIYOSHI
 *
 */
public class MicsPrompt implements MicsShell {

	private static String[] parseInput(String line) throws IOException {
		String[] cmd = null;
		StringTokenizer st = new StringTokenizer(line, " ");
		cmd = new String[st.countTokens()];
		for (int i = 0; i < cmd.length; i++)
			cmd[i] = st.nextToken();
		return cmd;
	}

	public void exec(File f) throws MicsException {
		try {
			InputStreamReader fis = new InputStreamReader(new FileInputStream(f));
			exec(new BufferedReader(fis), "");
			fis.close();
			Mics.getInstance().shutdown(0);
		} catch (IOException e) {
			throw new MicsException(e);
		}
	}

	public void exec() throws MicsException {
		try {
			exec(new BufferedReader(new InputStreamReader(System.in)), "MICS> ");
			System.out.println();
			Mics.getInstance().shutdown(0);
		} catch (IOException e) {
			throw new MicsException(e);
		}
	}

	private void exec(BufferedReader buf, String prompt) throws IOException {
		String line = "";
		System.out.print(prompt);
		while ((line = buf.readLine()) != null) {
			if (execLine(line) == false)
				break;
			System.out.print(prompt);
		}
		buf.close();
	}

	public static boolean execLine(String line) throws IOException {
		String[] cmd = parseInput(line);
		boolean flag = true;
		try {
			if (cmd.length == 0 || cmd == null) {
			} else if (cmd[0].equals("version")) {

				System.out.println(Mics.printVersion());
			} else if (cmd[0].equals("quit")) {

				flag = false;
			} else if (cmd[0].equals("load")) {

				Mics.getInstance().loadFile(cmd);
			} else if (cmd[0].equals("l")) {

				Mics.getInstance().loadFile(cmd);
			} else if (cmd[0].equals("echo")) {
				for (int i = 1; i < cmd.length; i++) {
					System.out.print(cmd[i] + " ");
				}
				System.out.println();
			} else if (cmd[0].equals("run")) {
				System.out.println("run");
				Mics.getInstance().execEngine();
			} else if (cmd[0].equals("xrun")) {
				Mics.getInstance().execEngineBlock();
			} else if (cmd[0].equals("step") || cmd[0].equals("s")) {
				if (cmd.length == 2) {
					Mics.getInstance().execStepByStep(Integer.parseInt(cmd[1]));
				} else {
					Mics.getInstance().execStepByStep();
				}
			} else if (cmd[0].equals("reset")) {

				Mics.getInstance().reset();
			} else if (cmd[0].equals("config")) {

				if (cmd.length > 1) {
					Mics.getInstance().configTargetArchitecture(new File(cmd[1]));
				} else {
					println("no input file");
				}
			} else if (cmd[0].equals("mem")) {

				Mics.getInstance().readMemory(cmd);
			} else if (cmd[0].equals("m")) {

				Mics.getInstance().readMemory(cmd);
			} else if (cmd[0].equals("list")) {
				MicsElement[] elems = Mics.getInstance().getMicsElementArray();
				for (int i = 0; i < elems.length; i++) {
					println(elems[i].getInfo());
				}
			} else if (cmd[0].equals("help")) {
				println("version     print version");
				println("quit        quit this program");
				println("list        print all architecture elements");
				println("load        load data to memory");
				println("l           load data to memory");
				println("run         run your program");
				println("xrun        run your program and block user controll");
				println("step        run your program step by step");
				println("s           run your program step by step");
				println("reset       reset program counter");
				println("config      config architecture");
				println("mem         print memory data");
				println("m           print memory data");
				println("echo        print argument and new line");
			} else {
				println("unknwon command: " + cmd[0]);
			}
		} catch (MicsException e) {

			System.out.println(e.getMessage());
		} catch (NumberFormatException e) {

			System.out.println("argument syntax error.");
		} finally {
		}
		return flag;
	}

	public static void println(String mesg) {
		System.out.println(mesg);
	}

	public void setMicsEngineEnableState(boolean flag) {
		// TODO Auto-generated method stub

	}
	
	public MicsArchitectureViewer getMicsArchitectureViewerData(MicsCompositeElement composite){
		return null;
	}
}
