/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * MICSを使用する上でのツール群
 */
package net.wasamon.mics.tools;

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.wasamon.mjlib.util.GetOpt;
import net.wasamon.mjlib.util.GetOptException;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * MICSシミュレーションの要素や属性値などを記述したスクリプトから初期化メソッドやHTMLドキュメントを生成する
 * // TODO 現在のバージョンと整合性がとれていない
 * @author Takefumi MIYOSHi
 *
 */
public class MakeInitialize {

	private String src;

	private static final int OUTPUT_JAVA = 0;

	private static final int OUTPUT_HTML = 1;

	private InitElement root;

	private String className;

	private class InitElementAttr {
		String name, type;

		public InitElementAttr(String name, String type) {
			this.name = name;
			this.type = type;
		}
	}

	private class InitElement {
		/**
		 * ArrayList for InitElementAttr
		 */
		ArrayList initElementAttrArray;

		/**
		 * ArrayList for InitElement
		 */
		ArrayList initElementChildArray;

		String name;

		String execStmt;

		boolean repeatableFlag;

		public InitElement() {
			initElementAttrArray = new ArrayList();
			initElementChildArray = new ArrayList();
			repeatableFlag = false;
			name = "root";
		}

		public InitElement(String name, boolean flag) {
			this();
			repeatableFlag = flag;
			this.name = name;
		}

		public void addAttr(InitElementAttr attr) {
			initElementAttrArray.add(attr);
		}

		public void addChild(InitElement child) {
			initElementChildArray.add(child);
		}

		public void addExecStmt(String str) {
			execStmt = str;
		}

		public String toString(int offset) {
			String str = "";
			String tab = "";
			for (int i = 0; i < offset; i++) {
				tab += " ";
			}
			str += tab + "name = " + name + "\n";
			str += tab + "exec = {" + execStmt + "}" + "\n";
			{
				Iterator i = initElementAttrArray.iterator();
				while (i.hasNext()) {
					InitElementAttr attr = (InitElementAttr) (i.next());
					str += tab + "attr: " + attr.name + "/" + attr.type + "\n";
				}
			}
			{
				Iterator i = initElementChildArray.iterator();
				while (i.hasNext()) {
					InitElement child = (InitElement) (i.next());
					str += child.toString(offset + 1);
				}
			}
			str += "\n";
			return str;
		}

		Pattern pattern = Pattern.compile("(\\$[0-9a-zA-Z]+)");

		public String toJavaSource(int offset) {
			String str = "";
			String tab = "";
			for (int i = 0; i < offset; i++) {
				tab += "_";
			}
			str += "{\n";
			String varPrefix = "init_var_";
			if ("root".equals(name) == false) {
				varPrefix += tab + name + "_";
			}
			if ("root".equals(name) == false) {
				if (repeatableFlag) {
					str += "ArrayList<Node> " + varPrefix + "obj = ";
					str += "XMLParser.getNamedNodeArrayList(n, \"" + name + "\");\n";
					str += "Iterator<Node> it = " + varPrefix + "obj.iterator();\n";
					str += "while(it.hasNext()){\n";
					str += "Node " + varPrefix + "node = it.next();\n";
				} else {
					str += "Node " + varPrefix + "node = ";
					str += "XMLParser.getNamedNode(n, \"" + name + "\");\n";
				}
			} else {
				str += "Node " + varPrefix + "node = n;\n";
			}
			{
				Iterator i = initElementAttrArray.iterator();
				while (i.hasNext()) {
					InitElementAttr attr = (InitElementAttr) (i.next());
					str += attr.type + " " + varPrefix + attr.name + ";\n";
					str += varPrefix + attr.name + " = ";
					boolean flag = "int".equals(attr.type);
					if (flag) {
						str += "DataUtil.parseInt(";
					}
					str += "XMLParser.getAttribute(" + varPrefix + "node, \"" + attr.name
							+ "\").getNodeValue()";
					if (flag) {
						str += ")";
					}
					str += ";\n";
				}
			}
			Matcher matcher = pattern.matcher(execStmt);
			StringBuffer sb = new StringBuffer();
			int c = 1;
			while (matcher.find()) {
				matcher
						.appendReplacement(sb, varPrefix + matcher.group(c).substring(1));
			}
			matcher.appendTail(sb);
			str += sb.toString() + "\n";
			if (repeatableFlag) {
				str += "}\n";
			}
			{
				Iterator i = initElementChildArray.iterator();
				while (i.hasNext()) {
					InitElement child = (InitElement) (i.next());
					str += child.toJavaSource(offset + 1);
				}
			}
			str += "}\n";
			return str;
		}

		public String toHTMLTable() {
			String str = "";
			str += "<table border=\"1\" width=\"100%\">\n";
			if ("root".equals(name) == false) {
				str += "<tr><th>Name</th><td>" + name + "</td></tr>\n";
			}
			{
				Iterator i = initElementAttrArray.iterator();
				while (i.hasNext()) {
					InitElementAttr attr = (InitElementAttr) (i.next());
					str += "<tr>";
					str += "<th>Attribute</th><td>";
					str += "<table border=\"1\" width=\"100%\"><tr>";
					str += "<th>name</th><td>" + attr.name + "</td>";
					str += "<th>type</th><td>" + attr.type + "</td>";
					str += "</tr></table>";
					str += "</td>";
					str += "</tr>";
				}
			}
			{
				Iterator i = initElementChildArray.iterator();
				while (i.hasNext()) {
					InitElement child = (InitElement) (i.next());
					str += "<tr>";
					str += "<th>child</th>";
					str += "<td>\n";
					str += child.toHTMLTable();
					str += "</td></tr>";
				}
			}
			str += "</table>\n";
			return str;
		}

	}

	public MakeInitialize(String file) throws Exception {
		src = file;
		File f = new File(file);
		if (f.exists() == false) {
			throw new Exception("no such file: " + file);
		}
		try {
			root = new InitElement();
			Node top = XMLParser.getTopNode(f);
			className = XMLParser.getAttribute(top, "name").getNodeValue();
			makeInitElement(top, root);
		} catch (XMLParserException e) {
			throw new Exception("syntax error: " + e.getMessage());
		}
	}

	private void makeInitElement(Node n, InitElement element)
			throws XMLParserException {
		makeAttribute(XMLParser.getNamedNodeArray(n, "attribute"), element);
		makeChild(XMLParser.getNamedNodeArray(n, "child"), element);
		makeExecStmt(XMLParser.getNamedNode(n, "exec"), element);
	}

	private void makeExecStmt(Node node, InitElement element)
			throws XMLParserException {
		element.addExecStmt(XMLParser.getNodeText(node));
	}

	private void makeChild(Node[] nodes, InitElement element)
			throws XMLParserException {
		for (int i = 0; i < nodes.length; i++) {
			Node n = nodes[i];
			String child_name = XMLParser.getAttribute(n, "name").getNodeValue();
			String s = XMLParser.getAttribute(n, "repeatable").getNodeValue();
			boolean repeatableFlag = false;
			if ("true".equals(s)) {
				repeatableFlag = true;
			}
			InitElement elm = new InitElement(child_name, repeatableFlag);
			makeInitElement(n, elm);
			element.addChild(elm);
		}
	}

	private void makeAttribute(Node[] nodes, InitElement element)
			throws XMLParserException {
		for (int i = 0; i < nodes.length; i++) {
			Node n = nodes[i];
			String attr_name = XMLParser.getAttribute(n, "name").getNodeValue();
			String attr_type = XMLParser.getAttribute(n, "type").getNodeValue();
			element.addAttr(new InitElementAttr(attr_name, attr_type));
		}
	}

	public String toString() {
		String str = "";
		str += "class = " + className + "\n";
		str += root.toString(0);
		return str;
	}

	public String toJavaSource() {
		String str = "";
		str += "/* generated automatically by MakeInitialize from " + src + " */\n";
		str += "public void initialize(String id, Node n) throws MicsException{\n";
		str += "super.initialize(id, n);\n";
		str += " try{\n";
		str += root.toJavaSource(0);
		str += "}catch(NumberFormatException e){\n";
		str += "throw new MicsException(\"configuration syntax error: " + className
				+ "\");\n";
		str += "}catch(XMLParserException e){\n";
		str += "throw new MicsException(\"configuration syntax error: " + className
				+ "\");\n";
		str += "}\n";
		str += "}\n";
		return str;
	}

	public String toHTMLTable() {
		String str = "";
		str += "<!-- generated automatically by MakeInitialize from " + src
				+ " -->\n";
		str += "<table border=\"1\">";
		str += "<tr><td>" + className + "</td></tr>";
		str += "<tr><td>" + root.toHTMLTable() + "</td></tr>";
		str += "</table>";
		return str;
	}

	public String getText(int mode) {
		if (mode == OUTPUT_HTML) {
			return toHTMLTable();
		} else {
			return toJavaSource();
		}
	}

	private static void usage() {
		System.out.println("usage: MakeInitialize [-o java/html] source");
		System.exit(0);
	}

	public static void main(String[] args) throws GetOptException {
		GetOpt opt = new GetOpt("o:", "", args);
		if (opt.getArgs().length < 1) {
			usage();
		}
		int mode = OUTPUT_JAVA;
		if (opt.flag("o")) {
			String value = opt.getValue("o");
			if ("java".equals(value)) {
				mode = OUTPUT_JAVA;
			} else if ("html".equals(value)) {
				mode = OUTPUT_HTML;
			} else {
				usage();
			}
		}
		for (int i = 0; i < opt.getArgs().length; i++) {
			try {
				MakeInitialize mi = new MakeInitialize(opt.getArgs()[i]);
				PrintWriter writer = new PrintWriter(new FileWriter(new File(opt
						.getArgs()[i]
						+ ".output")), true);
				writer.println(mi.getText(mode));
				writer.flush();
			} catch (Exception e) {
				System.out.println(e.getMessage());
			}
		}
	}

}
