/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.file;

import java.io.File;
import java.util.ArrayList;
import java.util.Stack;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

/**
 * ファイル名に関する文字列を操作するためのユーティリティ
 *
 * @author Takefumi MIYOSHI
 */
public class FileName{

  public static final String version = "FileName 0.1";
  public static final String author = "Takefumi MIYOSHI (miyoshi@ae.titech.ac.jp)";
  public static final String copyright = "(c) 2003/06/27 All right reserved.";
  
  /**
   * ファイル名から拡張子を返す.<br>
   * ファイル名が↓のようなときextを返す.<br>
   *  (1) /hogehoge/fefe.ext<br>
   *  (2) /hoge/fefe.fefe/gaso.ext<br>
   * @param file ファイル名
   */
  public static String getExt(String file){
    StringTokenizer st = new StringTokenizer(file, ".");
    String[] inputs = new String[st.countTokens()];
    for(int i = 0; st.hasMoreElements(); i++){
      inputs[i] = st.nextToken();
    }
    return inputs[inputs.length-1]; // 最後の一個を返す
  }
  
  /**
   * ワイルドカードを展開する
   * @File pdir 親ディレクトリ
   * @param input 入力
   * return 展開したファイル名の配列。ただし、この時ファイル名は絶対パスになる
   */
  public static Object[] expandWildExpression(File pdir, String input){
    String absname = toAbsPathName(pdir.getAbsolutePath(), input); // とりあえずパスを結合
    String name = getLast(absname);
    int index = name.indexOf('*');
    if(index < 0){
      String[] strs = new String[1];
      strs[0] = absname;
      return strs;
    }
    /*(replaceAllについて)
      String.replaceAll(arg0, arg1)
      arg0は正規表現
      arg1は変換する文字列
      replaceAllはarg0にマッチする文字列の各部分文字列に対し置換を行う
    */
    // メタキャラクタのキャンセル
    name = name.replaceAll("\\.", "\\\\.");
    name = name.replaceAll("\\^", "\\\\^");
    name = name.replaceAll("\\$", "\\\\$");
    name = name.replaceAll("\\?", "\\\\?");
    name = name.replaceAll("\\+", "\\\\+");
    name = name.replaceAll("\\|", "\\\\|");

    // * -> .* な変換
    name = name.replaceAll("\\*", ".*");

    Pattern pattern = Pattern.compile(name);

    String[] files = pdir.list();

    ArrayList list = new ArrayList();

    for(int i = 0; i < files.length; i++){
      if((pattern.matcher(files[i])).matches()){
        list.add(toAbsPathName(pdir.getAbsolutePath(), files[i]));
      }
    }
    return (list.toArray());
  }

  /**
   * 与えられた文字列が通常のファイル名として適当であるかどうかを判定する.
   * 通常ファイルでないもの
   *  - ワイルドカードを含む文字列
   * @param str 判定する文字列
   * @return 適当/不適当
  */
  public static boolean isRegularFileName(String str){
    int result = -1;
    result = str.indexOf('*');
    if(result < 0){
      return true; // この場合ファイル名は通常ファイル名である
    }
    return false;  // 通常ファイル名ではない
  }

  public static String toAbsDirName(String parent, String current){
    String absdir;
    // 先頭が'/'で始まっているかどうか
    if(current.charAt(0) == '/'){
      absdir = current;
    }
    else{
      absdir = parent.concat(current);
    }
    // .や..を圧縮する
    absdir = toParse(absdir);
    // 最後が'/'でおわっているかどうか
    if(absdir.charAt(absdir.length()-1) == '/'){
    }
    else{
      absdir = absdir.concat("/");
    }
    //System.out.println("absdir: " + absdir);
    return absdir;
  }

  public static String toAbsPathName(String pdir, String input){
    // 絶対パスかどうかの判断
    if(input.charAt(0) == '/'){
      return toParse(input);
    }

    else{
      return toParse(pdir.concat(input));
    }
  }

  /**
   * ./や../を圧縮する
   */
  public static String toParse(String path){
    String paths[] = trim(path);
    if(paths.length == 0){
      return "/";
    }
    Stack stack = new Stack();
    for(int i = 0; i < paths.length; i++){
      if(paths[i].equals(".")){ ; }
      else if(paths[i].equals("..")){
        if(stack.empty() == true){;}
        else{stack.pop();}
      }
      else{
        stack.push(paths[i]);
      }
    }
    String str = "/";
    while(stack.empty() == false){
      str = ((String)(stack.pop())).concat(str);
      str = "/".concat(str);
    }
    return str;
  }

  /**
   * 文字列からファイルパーザ用の文字列配列を得る.
   * @param filename ファイル名
   * @return String[]
   */
  public static String[] trim(String filename){
    StringTokenizer st = new StringTokenizer(filename.trim(), "/");
    String[] names;
    names = new String[st.countTokens()];
    for(int i = 0; st.hasMoreElements(); i++){
      names[i] = st.nextToken();
    }
    return names;
  }


  /** 空白を含む文字列を再構成する */
  public static String getLongFileName(String[] inputs, int offset){
    String concatname = new String("");
    while(true){
      concatname = concatname.concat(inputs[offset]);
      if(offset == inputs.length-1){
        break;
      }
      else{
        offset++;
        concatname = concatname.concat(" ");
      }
    }
    return concatname;
  }

  /**
   * パス名から最後の部分だけを切りだす
   */
  public static String getLast(String str){
    StringTokenizer st = new StringTokenizer(str.trim(), "/");
    String names = "/";
    for(int i = 0; st.hasMoreElements(); i++){
      names = st.nextToken();
    }
    return names;
  }

  public static void main(String args[]){
    System.out.println(version);
    System.out.println(author);
    System.out.println(copyright);
  }
}
