/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.file;

import java.io.File;
import java.net.URI;

/**
 * Unixのファイルシステムにアクセスするためのクラス
 *
 * @author Takefumi MIYOSHI
 *
 */
public class UnixFile extends File{
    
  public static final String version = "UnixFile 0.1";
  public static final String author = "Takefumi MIYOSHI (miyoshi@ae.titech.ac.jp)";
  public static final String copyright = "(c) 2003/06/27 All right reserved.";

  public UnixFile(File parent, String child) throws NullPointerException{
    super(parent, child);
  }

  public UnixFile(String pathname) throws NullPointerException{
    super(pathname);
  }

  public UnixFile(String parent, String child) throws NullPointerException{
    super(parent, child);
  }

  public UnixFile(URI uri) throws NullPointerException, IllegalArgumentException{
    super(uri);
  }

  /** RWX mask for owner */
  public static final int S_IRWXU = 0000700;
  /** R for owner */
  public static final int S_IRUSR = 0000400;
  /** W for owner */
  public static final int S_IWUSR = 0000200;
  /** X for owner */
  public static final int S_IXUSR = 0000100;

  /** RWX mask for group */
  public static final int S_IRWXG = 0000070;
  /** R for group */
  public static final int S_IRGRP = 0000040;
  /** W for group */
  public static final int S_IWGRP = 0000020;
  /** X for group */
  public static final int S_IXGRP = 0000010;

  /** RWX mask for other */
  public static final int S_IRWXO = 0000007;
  /** R for other */
  public static final int S_IROTH = 0000004;
  /** W for other */
  public static final int S_IWOTH = 0000002;
  /** X for other */
  public static final int S_IXOTH = 0000001;

  /** set user id on execution */
  public static final int S_ISUID = 0004000;
  /** set group id on execution */
  public static final int S_ISGID = 0002000;
  /** sticky bit */
  public static final int S_ISVTX = 0001000;

  /**
   * chmodシステムコールのネイティブメソッド
   * @param path 絶対パス形式のファイル名
   * @param mode Unix的には、mode_tとして定義してある
   * @return 不成功なら-1を返す
   */
  public native static int chmod(String path, int mode);

  /**
   * chownシステムコールのネイティブメソッド
   * @param path  絶対パス形式のファイル名
   * @param owner Unix的には、uid_tとして定義してある
   * @param group Unix的には、gid_tとして定義してある
   * @return 不成功なら-1を返す
   */
  public native static int chown(String path, int owner, int group);

  /**
   * statシステムコールを呼びだし、uidを取得するネイティブメソッド
   * @param path  絶対パス形式のファイル名
   * @return uid
   */
  public native static int getUid(String path);

  /**
   * statシステムコールを呼びだし、gidを取得するネイティブメソッド
   * @param path  絶対パス形式のファイル名
   * @return gid
   */
  public native static int getGid(String path);

  /**
   * statシステムコールを呼びだし、modeを取得するネイティブメソッド
   * @param path  絶対パス形式のファイル名
   * @return modeを示す値(unix的には、mode_t)
   */
  public native static int getMode(String path);


  static {
    System.loadLibrary("UnixFile");
  }

  /**
   * このパスに対して、指定したモードでchmodを実行する
   * @param mode モード
   * @return 成功/不成功
   */
  public boolean chmod(int mode) throws SecurityException{
    if(chmod(this.getAbsolutePath(), mode) == -1){
      return false;
    }
    return true;
  }

  /**
   * このパスに対して、指定したモードでchownを実行する
   * @param uid ユーザID
   * @param gid グループID
   * @return 成功/不成功
   */
  public boolean chown(int uid, int gid){
    if(chown(this.getAbsolutePath(), uid, gid) == -1){
      return false;
    }
    return true;
  }

  /**
   * このパスのユーザIDを取得する
   * @return uid
   */
  public int getUid(){
    return getUid(this.getAbsolutePath());
  }

  /**
   * このパスのグループIDを取得する
   * @return gid
   */
  public int getGid(){
    return getGid(this.getAbsolutePath());
  }

  /**
   * このパスのパーミッションモードを取得する
   * @return mode
   */
  public int getMode(){
    return getMode(this.getAbsolutePath());
  }

  public static void main(String args[]){
    System.out.println(version);
    System.out.println(author);
    System.out.println(copyright);
    if(args.length == 1){
      UnixFile file = new UnixFile(args[0]);
      System.out.print(file.getName());
      System.out.print(" uid:" + file.getUid());
      System.out.print(" gid:" + file.getGid());
      System.out.print(" mode:" + Integer.toOctalString(file.getMode()));
      System.out.println();
    }
  }

}
