/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.print;

/**
 * フォーマット付きのprintクラス
 *
 * @author Takefumi MIYOSHI
 */
public class PrintFormat{

  public static final String version = "PrintFormat 0.2";
  public static final String author = "Takefumi MIYOSHI (miyoshi@ae.titech.ac.jp)";
  public static final String copyright = "(c) 2003/06/27 All right reserved.";

  /** 文字列のための識別子 */
  public static final char STRING = 's';
  /** 数字のための識別子. JavaのtoString()実装に依存 */
  public static final char DECIMALSTRING = 'd';
  /** 文字のための識別子 */
  public static final char CHARACTER = 'c';
  /** 数字を16進表記するための識別子 */
  public static final char HEXSTRING = 'x';
  /** 数字を8進表記するための識別子 */
  public static final char OCTALSTRING = 'o';
  /** 数字をビット表記するための識別子 */
  public static final char BINARYSTRING = 'b';
  /** '%'を表示するための識別子 */
  public static final char PARCENT = '%';

  private static String toString(Object obj, char format) throws FormatException{
    if(obj instanceof String){
      return toString((String)obj, format);
    }
    if(obj instanceof Integer){
      return toString((Integer)obj, format);
    }
    if(obj instanceof Short){
      return toString((Short)obj, format);
    }
    if(obj instanceof Long){
      return toString((Long)obj, format);
    }
    if(obj instanceof Byte){
      return toString((Byte)obj, format);
    }
    return obj.toString();
  }

  private static String toString(String obj, char format) throws FormatException{
    String str = null;
    switch(format){
    case STRING: // %dが指定されている場合
      str = obj.toString();
      break;
    case DECIMALSTRING: // %dが指定されている場合
      try{
	str = (Integer.decode((String)obj)).toString();
      }catch(NumberFormatException e){
	throw new FormatException();
      }
      break;
    case HEXSTRING: // %xが指定されている場合
      try{
	str = Integer.toHexString((Integer.decode(obj)).intValue());
      }catch(NumberFormatException e){
	throw new FormatException();
      }
      break;
    case OCTALSTRING: // %oが指定されている場合
      try{
	str = Integer.toOctalString((Integer.decode(obj)).intValue());
      }catch(NumberFormatException e){
	throw new FormatException();
      }
      break;
    case BINARYSTRING: // %bが指定されている場合
      try{
	str = Integer.toBinaryString((Integer.decode(obj)).intValue());
      }catch(NumberFormatException e){
	throw new FormatException();
      }
      break;
    }
    if(str == null){
      throw new FormatException();
    }
    return str;
  }

  private static String toString(Integer obj, char format) throws FormatException{
    String str = null;
    switch(format){
    case DECIMALSTRING: // %dが指定されている場合
      str = obj.toString();
      break;
    case HEXSTRING: // %xが指定されている場合
      str = Integer.toHexString(obj.intValue());
      break;
    case BINARYSTRING: // %xが指定されている場合
      str = Integer.toBinaryString(obj.intValue());
      break;
    }
    if(str == null){
      throw new FormatException();
    }
    return str;
  }

  private static String toString(Byte obj, char format) throws FormatException{
    String str = null;
    int val = 0;
    switch(format){
    case DECIMALSTRING: // %dが指定されている場合
      str = obj.toString();
      break;
    case HEXSTRING: // %xが指定されている場合
      val = obj.intValue();
      str = Integer.toHexString(val);
      if(val < 0){
	str = str.substring(6,8);
      }
      break;
    case BINARYSTRING: // %xが指定されている場合
      val = obj.intValue();
      str = Integer.toBinaryString(val);
      if(val < 0){
	str = str.substring(24,32);
      }
      break;
    }
    if(str == null){
      throw new FormatException();
    }
    return str;
  }

  private static String toString(Long obj, char format) throws FormatException{
    String str = null;
    switch(format){
    case DECIMALSTRING: // %dが指定されている場合
      str = obj.toString();
      break;
    case HEXSTRING: // %xが指定されている場合
      str = Long.toHexString(obj.longValue());
      break;
    case BINARYSTRING: // %xが指定されている場合
      str = Long.toBinaryString(obj.longValue());
      break;
    }
    if(str == null){
      throw new FormatException();
    }
    return str;
  }

  private static String toString(Short obj, char format) throws FormatException{
    String str = null;
    switch(format){
    case DECIMALSTRING: // %dが指定されている場合
      str = obj.toString();
      break;
    case HEXSTRING: // %xが指定されている場合
      str = Integer.toHexString(obj.shortValue());
      break;
    case BINARYSTRING: // %xが指定されている場合
      str = Integer.toBinaryString(obj.shortValue());
      break;
    }
    if(str == null){
      throw new FormatException();
    }
    return str;
  }

  /**
   * オブジェクトから、指定した長さの文字列を生成する
   * @param obj 文字列化するオブジェクト
   * @param len 長さ
   * @param format フォーマット
   * @param rightflag 右に寄せるかどうかのフラグ
   */
  private static String makeString(Object obj, int len, char format, boolean rightflag) throws FormatException{
    StringBuffer sb = new StringBuffer(0);
    String str = toString(obj, format);

    if(rightflag == true){
      for(int i = 0; i < len - str.length(); i++){
	if(format == DECIMALSTRING || format == BINARYSTRING || format == HEXSTRING || format == OCTALSTRING){
	  sb.append('0'); // 10進以外の数字の右寄せの場合には0でパディングする
	}else{
	  sb.append(' '); // 数字以外の右寄せ時には空白を追加
	}
      }
      sb.append(str);
    }else{
      sb.append(str);
      for(int i = 0; i < len - str.length(); i++){
	sb.append(' '); // 空白を追加
      }
    }
    if(str == null){
      throw new FormatException();
    }
    return sb.toString();
  }

  /**
   * 部分文字列をobjで指定するオブジェクトでおきかえる.
   * 与えられた文字列に対し破壊的な代入
   * @param pattern 文字列
   * @param offset オフセット
   * @param objs オブジェクトの配列
   * @return おきかえた後の文字列ポインタの位置
   */
  private static int convert(String pattern, int offset, Object[] objs, int index, StringBuffer sb) throws FormatException{
    int p = offset+1;
    char c;
    boolean rightflag = false;
    
    StringBuffer numstr = new StringBuffer(0);

    c = pattern.charAt(p);
    if(c == '0'){ // %のあとに0が続く場合
      rightflag = true;
    }

    while(p < pattern.length()){
      c = pattern.charAt(p);
      if(Character.isDigit(c)){ // 数字が続いている場合
	numstr.append(c);
	p++; // 次の文字を指す
      }
      else{
	p++; // この文字も次のパーズの対象外なので
	if(c == PARCENT){
	  sb.append('%');
	}
	else{
	  int len = 0;
	  if("".equals(numstr.toString()) == false){
	    try{
	      len = Integer.parseInt(numstr.toString()); // 長さをとりだす
	    }catch(NumberFormatException e){
	      throw new FormatException();
	    }
	  }
	  String newstr = makeString(objs[index], len, c, rightflag);
	  sb.append(newstr);
	}
	return p; // 次に読むポイント.
      }
    }

    throw new FormatException();

  }

  /**
   * 引数に一つint型をもてるprintf(String pattern, Object[] obj)の限定メソッド.
   * @param pattern 文字列
   * @param value int型の数値
   * @return 文字列の指定個所を入れかえた新しい文字列
   */
  public static String print(String pattern, int value) throws FormatException{
    Integer obj = new Integer(value);
    Object[] objs = {obj};
    return print(pattern, objs);
  }

  /**
   * 引数に一つStringオブジェクトをもてるprintf(String pattern, Object[] obj)の限定メソッド.
   * @param pattern 文字列
   * @param str Stringオブジェクト
   * @return 文字列の指定個所を入れかえた新しい文字列
   */
  public static String print(String pattern, String str) throws FormatException{
    Object[] objs = {str};
    return print(pattern, objs);
  }

  /**
   * 引数に二つStringオブジェクトをもてるprintf(String pattern, Object[] obj)の限定メソッド.
   * @param pattern 文字列
   * @param str1 Stringオブジェクト
   * @param str2 Stringオブジェクト
   * @return 文字列の指定個所を入れかえた新しい文字列
   */
  public static String print(String pattern, String str1, String str2) throws FormatException{
    Object[] objs = {str1, str2};
    return print(pattern, objs);
  }

  /**
   * 引数に一つbyte型をもてるprintf(String pattern, Object[] obj)の限定メソッド.
   * @param pattern 文字列
   * @param value byte型の数値
   * @return 文字列の指定個所を入れかえた新しい文字列
   */
  public static String print(String pattern, byte value) throws FormatException{
    Byte obj = new Byte(value);
    Object[] objs = {obj};
    return print(pattern, objs);
  }

  /**
   * 引数に一つbyte型をもてるprintf(String pattern, Object[] obj)の限定メソッド.
   * @param pattern 文字列
   * @param value byte型の数値
   * @return 文字列の指定個所を入れかえた新しい文字列
   */
  public static String print(String pattern, long value) throws FormatException{
    Long obj = new Long(value);
    Object[] objs = {obj};
    return print(pattern, objs);
  }

  /**
   * 部分文字列をobjで指定するオブジェクトでおきかえる.
   * いわゆるC言語でいうところのprintfの実装<br>
   * print("fefe %10s fefe", gaso)  -> "fefe gaso       fefe"<br>
   * print("fefe %010s fefe", gaso) -> "fefe       gaso fefe"<br>
   * print("fefe 0x%02x fefe", 100) -> "fefe 0x64 fefe"<br>
   * @param pattern 文字列
   * @param obj オブジェクト
   * @return 文字列の指定個所を入れかえた新しい文字列
   */
  public static String print(String pattern, Object[] obj) throws FormatException{
    StringBuffer sb = new StringBuffer(0);

    int point = 0;
    int offset = 0;
    char c;

    while(offset < pattern.length()){
      try{
	c = pattern.charAt(offset);
      }catch(IndexOutOfBoundsException e){
	throw new FormatException(e.toString());
      }
      if(c == '%'){
	try{
	  offset = convert(pattern, offset, obj, point, sb);
	}catch(NullPointerException e){
	  throw new FormatException();
	}catch(IndexOutOfBoundsException e){
	  throw new FormatException();
	}
	point++;
      }
      else{
	sb.append(c);
	offset++;
      }
    }
    return sb.toString();
  }

  public static void main(String[] args) throws FormatException{
    if(args.length == 0){
      System.out.println(version);
      System.out.println(author);
      System.out.println(copyright);
      return;
    }
    Object[] objs = new Object[args.length - 1];
    for(int i = 0; i < args.length - 1; i++){
      objs[i] = args[i+1];
    }
    System.out.println(PrintFormat.print(args[0], objs));
    return;
  }

  /**
   * 8bitの数字から，その値す16進数の文字列を生成する
   * たとえば，255から"ff"など．
   * 値はサイズ2のbyteの配列に格納して返される
   * @param value 文字列に変換したい値
   * @return 与えられた値を示す文字列のbyte配列
   */
  public static byte[] hex_to_a(byte b){
    byte r[] = new byte[2];
    int i = ((int)(b >> 4)& 0x0f);
    if(i > 9){
      r[0] = (byte)(0x60 + (i - 9));
    }else{
      r[0] = (byte)(0x30 + i);
    }
    i = ((int)b & 0x0f);
    if(i > 9){
      r[1] = (byte)(0x60 + (i - 9));
    }else{
      r[1] = (byte)(0x30 + i);
    }
    return r;
  }

}
