/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.util;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * 引数の解析を行い、その結果を保持するクラス。GetOpt("hvf:", "prefix:", args)などとして利用<br>
 * <ul>
 * <li>最初の引数で一文字のオプションを、第二引数ではロングオプションを設定。
 * <li>":"を末尾につけることで、そのオプションは引数をとることができる。この引数は空白または"="で識別される。
 * <li>また、オプションに"::"をつけると、その後の文字列すべてを、そのオプションの引数として処理する。
 * - や --の次、また、-ではじまらない文字列からを引数として保持する。
 * 
 * @author Takefumi MIYOSHI
 * 
 */
public class GetOpt {

	private String[] args;

	private Hashtable opts;

	/** 引数つきのオプションを保持するのリスト */
	private Hashtable opt_with_arg;

	/** 引数つきではないオプションフラグのリスト */
	private Hashtable opt_flag;

	/** 解析の結果不明だったリスト */
	private ArrayList unknown;

	private String opt_with_arg_rest = "";

	private boolean result = true;

	/**
	 * コンストラクタ
	 * 
	 * @param sp
	 *            解析したい一文字オプションの連続(-vとか)
	 * @param lps
	 *            解析したいロングオプションのcommma separate羅列
	 * @param ptn
	 *            解析すべき文字列の配列
	 */
	public GetOpt(String sp, String lps, String ptn[]) {
		this(sp, lps, ptn, 0);
	}

	/**
	 * コンストラクタ
	 * 
	 * @param sp
	 *            解析したい一文字オプションの連続(-vとか)
	 * @param lps
	 *            解析したいロングオプションのcommma separate羅列
	 * @param ptn
	 *            解析すべき文字列の配列
	 * @param offset
	 *            解析すべき文字のオフセット
	 */
	public GetOpt(String sp, String lps, String ptn[], int offset) {
		args = new String[0];
		opt_with_arg = new Hashtable();
		opt_flag = new Hashtable();
		opts = new Hashtable();
		unknown = new ArrayList();

		StringTokenizer st = new StringTokenizer(lps, ",");
		String lp[] = new String[st.countTokens()];
		for (int i = 0; i < lp.length; i++) {
			lp[i] = st.nextToken();
		}

		makeOptList(sp, lp);

		analyze(ptn, offset);

	}

	/**
	 * デバッグ用コンストラクタ
	 * 
	 */
	public GetOpt(String sp, String lps, String ptn[], boolean flag) {
		this(sp, lps, ptn);

		String[] o = getAllOpt();
		String[] a = getArgs();

		for (int i = 0; i < o.length; i++) {
			System.out.println("Option " + o[i]);
			if (flag(o[i])) {
				try {
					System.out.println(" Value=" + getValue(o[i]));
				} catch (GetOptException e) {
					System.out.println(" Value=");
				}
			}
		}
		if (a != null) {
			for (int i = 0; i < a.length; i++)
				System.out.println("Argument " + a[i]);
		}
	}

	/**
	 * 引数すべてに対し指定したパターンがあるかどうか判定する
	 * 
	 * @param ptn 引数の配列
	 * @param offset 解析する引数のオフセット
	 * 
	 */
	private void analyze(String[] ptn, int offset) {
		int i = offset;
		for (i = offset; i < ptn.length; i++) {
			if (ptn[i].charAt(0) != '-') {
				break;
			}
			if ((ptn[i].equals("-") == true) || (ptn[i].equals("--") == true)) {
				break;
			}

			if (opt_with_arg_rest.equals(ptn[i].substring(1))) {
				String flag = ptn[i].substring(1);
				String rest = "";
				i += 1;
				while (true) {
					rest += ptn[i];
					if (i == ptn.length - 1) {
						break;
					} else {
						rest += " ";
						i += 1;
					}
				}
				opts.put(flag, rest);
			}

			if (ptn[i].charAt(0) == '-') {
				if (ptn[i].charAt(1) == '-') {
					i += analy_longopt(ptn[i].substring(2), ptn, i);
				} else {
					i += analy_shortopt(ptn[i].substring(1), ptn, i);
				}
			}
		}

		args = setArgs(ptn, i);
	}

	/**
	 * 与えられたショートオプションとロングオプションから 引数解析のためのテーブルを生成する
	 */
	private boolean makeOptList(String sp, String[] lp) {
		int i = 0;
		while (i < sp.length()) {
			if (sp.length() > (i + 1) && sp.charAt(i + 1) == ':') { // もし文字の後に':'が続いていた場合引数を伴う
				if (sp.length() > (i + 2) && sp.charAt(i + 2) == ':') { // もう一つ続いていたらラスト
					opt_with_arg_rest = sp.substring(i, i + 1);
					i += 3;
				} else {
					opt_with_arg.put(sp.substring(i, i + 1), new Boolean(true));
					i += 2;
				}
			} else {
				opt_flag.put(sp.substring(i, i + 1), new Boolean(true));
				i += 1;
			}
		}
		i = 0;
		while (i < lp.length) {
			if (lp[i].charAt(lp[i].length() - 1) == ':') { // 最終の文字が':'なら引数を伴う
				opt_with_arg.put(lp[i].substring(0, lp[i].length() - 1), new Boolean(true));
			} else {
				opt_flag.put(lp[i], new Boolean(true));
			}
			i += 1;
		}
		return true;
	}

	/**
	 * パターンに該当するフラグオプションがあるかどうか
	 * 
	 * @param ptn
	 *            パターン文字列
	 * @return 該当オプションがあるかどうか
	 */
	private int analy_shortopt(String ptn, String arg[], int offset) {
		int add = 0;
		for (int i = 0; i < ptn.length(); i++) {
			String flag = ptn.substring(i, i + 1);
			if (opt_flag.containsKey(flag)) {
				opts.put(flag, new Boolean(true));
				add += 0;
			} else if (opt_with_arg.containsKey(flag)) {
				if (arg.length > offset + 1) {
					opts.put(flag, arg[offset + 1]);
					add += 1;
				} else {
					result = false;
					add += 0;
				}
			}
		}
		return add;
	}

	/**
	 * 引数つきオプションの解析 hoge=fefe または hoge fefe をオプション hoge と、その引数 fefe と解析
	 * 
	 * @param ptn
	 *            ためすパターン
	 * @param arg
	 *            オプション字列の配列(引数かもしれないから)
	 * @param offset
	 *            現在のパタンの配列中のオフセット
	 * @return 該当するオプションがあったかどうか
	 */
	private int analy_longopt(String ptn, String arg[], int offset) {
		int add = 0;
		if (opt_flag.containsKey(ptn)) {
			opts.put(ptn, new Boolean(true));
			add = 0;
		} else if (ptn.matches(".*=.*")) { // hogehoge=*みたいな形
			int index = ptn.indexOf("=");
			String ptn2 = ptn.substring(0, index);
			if (opt_with_arg.containsKey(ptn2)) {
				opts.put(ptn2, ptn.substring(index + 1));
			} else {
				result = false;
			}
			add = 0;
		} else if (opt_with_arg.containsKey(ptn)) {
			if (arg.length > offset + 1) {
				opts.put(ptn, arg[offset + 1]);
				add = 1;
			} else {
				opts.put(ptn, "");
				result = true;
				add = 0;
			}
		} else {
			result = false;
			add = 0;
		}
		return add;
	}

	public boolean isSuccess() {
		return result;
	}

	/**
	 * オプションが指定されていたかどうかを判定する
	 * 
	 * @param key
	 *            検索するオプション名
	 * @return 指定されていた/いなかった
	 */
	public boolean flag(String key) {
		return opts.containsKey(key);
	}

	/**
	 * オプションで指定されていた値を取得する。
	 * 
	 * @param key
	 *            検索するオプション名
	 * @return 指定されていた値(文字列)
	 * @throws GetOptException
	 *             与えられた文字列のオプションがない場合
	 */
	public String getValue(String key) throws GetOptException {
		Object obj = null;
		if(opts.containsKey(key) == false){
			throw new GetOptException("no such options." + key);
		}
		obj = opts.get(key);
		if (obj instanceof String) {
			return (String)obj;
		} else {
			throw new GetOptException("this option doesn't have any value.");
		}
	}

	/**
	 * すべてのオプションを配列で得る。
	 * 
	 * @return オプションの配列
	 */
	private String[] getAllOpt() {
		String[] o = new String[opts.size()];
		Enumeration keys = opts.keys();
		for(int i = 0; keys.hasMoreElements(); i++){
			o[i] = (String)(keys.nextElement());
		}
		return o;
	}

	/**
	 * すべての引数を配列にして返す
	 * 
	 * @return 引数の配列
	 */
	public String[] getArgs() {
		return args;
	}

	/**
	 * パタンのうちoffset以降を配列に格納して返す
	 * 
	 * @param ptn
	 *            パタン
	 * @param offset
	 *            オフセット
	 * @return 文字列の配列
	 */
	private String[] setArgs(String[] ptn, int offset) {
		int argc = ptn.length - offset;
		String[] args = new String[argc];
		for (int i = 0; i < argc; i++) {
			args[i] = ptn[i + offset];
		}
		return args;
	}

	private void print_opt_flag() {
		Enumeration keys = opt_flag.keys();
		while(keys.hasMoreElements()){
			System.out.println(keys.nextElement());
		}
	}

	private void print_opt_with_arg() {
		Enumeration keys = opt_with_arg.keys();
		while(keys.hasMoreElements()){
			System.out.println(keys.nextElement());
		}
	}

	public static void main(String args[]) throws Exception {
		System.out.println("GetOpt test.");

		String sp = "vh";
		String lp = "version:";

		GetOpt go = new GetOpt(sp, lp, args);

		go.print_opt_flag();
		go.print_opt_with_arg();

		String[] o = go.getAllOpt();
		String[] a = go.getArgs();

		for (int i = 0; i < o.length; i++)
			System.out.println("Option " + o[i]);
		if (a != null) {
			for (int i = 0; i < a.length; i++)
				System.out.println("Argument " + a[i]);
		} else {
			System.out.println("no argument");
		}

		System.out.println(go.flag("v"));
		System.out.println(go.flag("h"));
		System.out.println(go.flag("version"));

		try {
			System.out.println(go.getValue("version"));
		} catch (GetOptException e) {
			System.out.println(e.getMessage());
		}

		System.out.println(go.isSuccess());
	}

}
