/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

public class PathBalanceTreeConstructor {

	public interface TreeElement {
		public boolean isLeaf();
	}
	public interface TreeNode extends TreeElement{
		public int nKids();
		public PathBalanceTreeConstructor.TreeElement kid(int i);
		public void setKid(int i, PathBalanceTreeConstructor.TreeElement node);
		public TreeNode newNode();
	}
	public interface TreeLeaf extends TreeElement{
		public int value();
	}

	class TreeLeafComparatorImpl implements Comparator {
		public int compare(Object obj0, Object obj1) {
			TreeLeaf l0 = (TreeLeaf)obj0;
			TreeLeaf l1 = (TreeLeaf)obj1;
			if (l0.value() > l1.value()) {
				return -1;
			} else if (l0.value() < l1.value()) {
				return 1;
			} else {
				return 0;
			}
		}
	}

	private boolean isPerfectTree(TreeElement n) {
		if (n.isLeaf()){
			return true;
		} else{
			int nKids = ((TreeNode) n).nKids();
			boolean flag = true;
			for(int i = 0; i < nKids; i++){
				// if one of children is not perfect, this node is not perfect either. 
				flag &= isPerfectTree(((TreeNode) n).kid(i));
			}
			if (flag) {
				if (countKidsDepth(((TreeNode) n).kid(0)) == countKidsDepth(((TreeNode) n).kid(1))) {
					return true;
				}else{
					return false;					
				}
			}else{
				return false;
			}
		}
	}

	private int countKidsDepth(TreeElement n) {
		int v0, v1;
		if (n.isLeaf()) {
			return ((TreeLeaf) n).value();
		}
		TreeNode node = (TreeNode) n;
		v0 = (node.kid(0) != null) ? countKidsDepth(node.kid(0)) : 0;
		v1 = (node.kid(1) != null) ? countKidsDepth(node.kid(1)) : 0;
		return 1 + ((v0 > v1) ? v0 : v1);
	}

	private void add(TreeNode node, TreeLeaf leaf) {
		if (node.kid(0) == null) {
			node.setKid(0, leaf);
			// System.out.println("add " + leaf
			// + " to kid(0) (node.kid(0) == null)");
			return;
		}
		if (node.kid(1) == null) {
			// System.out.println("add " + leaf
			// + " to kid(1) (node.kid(1) == null)");
			node.setKid(1, leaf);
			return;
		}
		boolean f0 = isPerfectTree(node.kid(0));
		boolean f1 = isPerfectTree(node.kid(1));
		int depth0 = countKidsDepth(node.kid(0));
		int depth1 = countKidsDepth(node.kid(1));
		// System.out.println(node.kid(0) + " is perfect?: " + f0);
		// System.out.println(node.kid(1) + " is perfect?: " + f1);
		// System.out.println(node.kid(0) + " is depth: " +
		// countKidsDepth(node.kid(0)));
		// System.out.println(node.kid(1) + " is depth: " +
		// countKidsDepth(node.kid(1)));
		if (f0 == f1) {
			if (depth0 > depth1) {
				if(f0){
					TreeNode n = node.newNode();
					n.setKid(0, leaf);
					n.setKid(1, node.kid(1));
					node.setKid(1, n);
				}else{
					add(((TreeNode)node.kid(1)), leaf);
				}
				// System.out.println("add " + leaf + " to kid(1) as " + n);
			} else {
				if(f0){
					TreeNode n = node.newNode();
					n.setKid(0, leaf);
					n.setKid(1, node.kid(0));
					node.setKid(0, n);
				}else{
					add(((TreeNode)node.kid(0)), leaf);
				}
				// System.out.println("add " + leaf + " to kid(1) as " + n);
			}
		} else if (f1 == false) {
			if(depth0 + 1 > depth1){
				add((TreeNode) node.kid(1), leaf);
			}else{
				add((TreeNode) node.kid(0), leaf);
			}
		} else {
			if(depth0 > depth1 + 1){
				add((TreeNode) node.kid(1), leaf);
			}else{
				add((TreeNode) node.kid(0), leaf);
			}
		}
	}

	TreeNode root;

	public TreeLeaf[] makeTestData(int size) {
		ArrayList leaves = new ArrayList();
		Random r = new Random();
		for (int i = 0; i < size; i++) {
			leaves.add(new PathBalanceTreeConstructorTestLeaf((Math.abs(r.nextInt()) % 10) + 1));
		}
		return (TreeLeaf[])(leaves.toArray(new TreeLeaf[leaves.size()]));
	}

	public void run(TreeNode node, TreeLeaf[] leaves) {
		List list = Arrays.asList(leaves); 
		Collections.sort(list, new TreeLeafComparatorImpl());
		root = node;
		Iterator it = list.iterator();
		while (it.hasNext()) {
			add(root, (TreeLeaf)(it.next()));
		}
	}
	
	public static void main(String args[]) {
		PathBalanceTreeConstructor test = new PathBalanceTreeConstructor();
		test.run(new PathBalanceTreeConstructorTestNode(), test.makeTestData(10));
		System.out.println(test.root);
		// ArrayList<Leaf> a = new ArrayList<Leaf>();
		// a.add(new Leaf(10));
		// a.add(new Leaf(10));
		// a.add(new Leaf(10));
		// a.add(new Leaf(10));
		// a.add(new Leaf(10));
		// a.add(new Leaf(10));
		// a.add(new Leaf(10));
		// test.run(new Node(), a);
		//System.out.println(test.root);
	}
}

class PathBalanceTreeConstructorTestLeaf implements PathBalanceTreeConstructor.TreeLeaf {
	int v;

	public int value(){
		return v;
	}
	
	public boolean isLeaf() {
		return true;
	}

	public String toString() {
		return String.valueOf(v);
	}

	public PathBalanceTreeConstructorTestLeaf(int v) {
		this.v = v;
	}
}

class PathBalanceTreeConstructorTestNode implements PathBalanceTreeConstructor.TreeNode {
	PathBalanceTreeConstructor.TreeElement[] kids = new PathBalanceTreeConstructor.TreeElement[2];

	public boolean isLeaf() {
		return false;
	}

	public int nKids(){
		return 2;
	}
	
	public PathBalanceTreeConstructor.TreeElement kid(int i){
		return kids[i];
	}
	
	public void setKid(int i, PathBalanceTreeConstructor.TreeElement node){
		kids[i] = node;
	}
	
	public String toString() {
		return "(" + kid(0).toString() + "." + kid(1).toString() + ")";
	}
	
	public PathBalanceTreeConstructor.TreeNode newNode(){
		return new PathBalanceTreeConstructorTestNode();
	}
}

