/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package net.wasamon.mjlib.xml;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * XMLを簡単に扱うためのユーティリティメソッド
 * 
 * @author Takefumi MIYOSHI
 * 
 */
public class XMLParser {

	public static final String version = "XMLParser 0.3";

	public static final String author = "Takefumi MIYOSHI (miyo@wasamon.net)";

	public static final String copyright = "(c) 2003-2007 All right reserved";

	/**
	 * トップのノードを取得する
	 * 
	 * @param file
	 *            入力ファイル
	 */
	public static Node getTopNode(File file) throws XMLParserException {
		try {
			DocumentBuilderFactory bdf = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = bdf.newDocumentBuilder();
			Node node = builder.parse(file);
			return node.getFirstChild();
		} catch (IOException e) {
			throw new XMLParserException("cannot open the file "
					+ file.getName());
		} catch (org.xml.sax.SAXParseException e) {
			throw new XMLParserException("syntax error in config file.");
		} catch (org.xml.sax.SAXException e) {
			throw new XMLParserException("sax error.");
		} catch (ParserConfigurationException e) {
			throw new XMLParserException("failure initializing of xml parser.");
		}
	}

	/**
	 * 与えられたノードから指定した名前づけられた子ノードを取得
	 * 
	 * @param node
	 *            親ノード
	 * @param name
	 *            名前
	 */
	public static Node getNamedNode(Node node, String name)
			throws XMLParserException {
		NodeList list = node.getChildNodes();
		Node nd = null;
		for (int i = 0; i < list.getLength(); i++) {
			nd = list.item(i);
			if (nd.getNodeType() != Node.TEXT_NODE) {
				if (name.equals(nd.getNodeName())) {
					return nd;
				}
			}
		}
		throw new XMLParserException(name + " is not found under "
				+ node.getNodeValue());
	}

	/**
	 * 指定した名前のノードが、親ノードの子であるかどうかを判定する
	 * 
	 * @param node
	 *            親ノード
	 * @param name
	 *            名前
	 */
	public static boolean isNamedNode(Node node, String name) {
		NodeList list = node.getChildNodes();
		Node nd = null;
		for (int i = 0; i < list.getLength(); i++) {
			nd = list.item(i);
			if (nd.getNodeType() != Node.TEXT_NODE) {
				if (name.equals(nd.getNodeName())) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 与えられたノードから指定した名前づけられた子ノードのリストを取得し，そのリストを返す
	 * 
	 * @param node
	 *            親ノード
	 * @param name
	 *            名前
	 */
	public static Node[] getNamedNodeArray(Node node, String name) {
		NodeList list = node.getChildNodes();
		Node nd = null;
		ArrayList v = new ArrayList();
		for (int i = 0; i < list.getLength(); i++) {
			nd = list.item(i);
			if (nd.getNodeType() != Node.TEXT_NODE) {
				if (name.equals(nd.getNodeName())) {
					v.add(nd);
				}
			}
		}
		return (Node[])(v.toArray(new Node[v.size()]));
	}

	/**
	 * 与えられたノードの子にあたるテキストを返す
	 * 
	 * @param node
	 *            親ノード
	 */
	public static String getNodeText(Node node) throws XMLParserException {
		NodeList list = node.getChildNodes();
		Node nd = null;
		for (int i = 0; i < list.getLength(); i++) {
			nd = list.item(i);
			if (nd.getNodeType() == Node.TEXT_NODE) {
				return nd.getNodeValue();
			}
		}
		throw new XMLParserException("not found text");
	}

	/**
	 * 与えられたノードから、指定した名のアトリビュートを取得
	 * 
	 * @param node
	 *            親ノード
	 * @param name
	 *            名前
	 * @return ノード
	 */
	public static Node getAttribute(Node node, String name)
			throws XMLParserException {
		NamedNodeMap attrs = node.getAttributes(); // Attributeの取得
		Node nd = null;
		try {
			nd = attrs.getNamedItem(name);
		} catch (NullPointerException e) {
			throw new XMLParserException(name + " is not found under "
					+ node.getNodeName());
		}
		if (nd == null) {
			throw new XMLParserException(name + " is not found under "
					+ node.getNodeName());
		}
		return nd;
	}

	/**
	 * 与えられたノードに、指定した名前のアトリビュートがあるかどうか返す
	 * 
	 * @param node
	 *            親ノード
	 * @param name
	 *            名前
	 * @return 真偽値
	 */
	public static boolean isAttribute(Node node, String name) {
		NamedNodeMap attrs = node.getAttributes(); // Attributeの取得
		Node nd = null;
		try {
			nd = attrs.getNamedItem(name);
		} catch (NullPointerException e) {
			return false;
		}
		if (nd == null) {
			return false;
		}
		return true;
	}

	public static void main(String args[]) {
		System.out.println(version);
		System.out.println(author);
		System.out.println("Copyright " + copyright);
	}

}
