package net.wasamon.mics.freehdl.ui.wave;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JViewport;

import net.wasamon.mics.freehdl.data.LogicData;
import net.wasamon.mics.freehdl.data.Signal;
import net.wasamon.mics.freehdl.data.StdLogic;
import net.wasamon.mics.freehdl.data.StdLogicVector;

class WavePanel {
	JSplitPane pane;
	SignalValueCanvas list;
	WavePanelCanvas canvas;
	ArrayList<Signal> data;

	JViewport listView;
	JViewport waveView;

	public WavePanel(FreeHDLWaveFrame frame) {
		pane = new JSplitPane();
		list = new SignalValueCanvas();
		canvas = new WavePanelCanvas();
		pane.setDividerLocation(0.3);
		JScrollPane scr;
		scr = new JScrollPane(list, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
				JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		listView = scr.getViewport();
		listView.addChangeListener(frame);
		pane.setLeftComponent(scr);
		scr = new JScrollPane(canvas, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
				JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		waveView = scr.getViewport();
		waveView.addChangeListener(frame);
		pane.setRightComponent(scr);
	}

	public void update(ArrayList<Signal> data) {
		this.data = data;
		list.update(data);
		canvas.repaint();
	}

	class SignalValueCanvas extends SignalListCanvas implements ActionListener {

		private int index = -1;

		private JPopupMenu list;
		private String[] typeLabel = new String[] { "BINARY", "HEX", "ASCII" };

		public SignalValueCanvas() {
			super();
			list = new JPopupMenu("value");
			//list.setVisible(false);
			for (String s : typeLabel) {
				JMenuItem m = list.add(s);
				m.addActionListener(this);
			}
		}

		public void setMark(int index) {
			this.index = index;
		}

		public void update(ArrayList<Signal> data) {
			super.update(data);
		}

		public String getLabel(Signal s) {
			if (index > 0) {
				return s.getValueAsString(index);
			} else {
				return s.getValueAsString();
			}
		}

		private int selectedIndex = -1;

		public void mousePressed(MouseEvent e) {
			int y = e.getY() - FreeHDLWaveFrame.margin;
			selectedIndex = y / (size + FreeHDLWaveFrame.margin);
			//Point p0 = getLocationOnScreen();
			Point p1 = e.getPoint();
			//list.show(pane, p0.x + p1.x, p0.y + p1.y);
			list.show(pane, p1.x, p1.y);
		}

		private void selected(String value) {
			int type = Signal.BINARY;
			if ("BINARY".equals(value)) {
				type = Signal.BINARY;
			} else if ("HEX".equals(value)) {
				type = Signal.HEX;
			} else if ("ASCII".equals(value)) {
				type = Signal.ASCII;
			}
			if (selectedIndex > 0) {
				data.get(selectedIndex).setStringType(type);
			}
			selectedIndex = -1;
			repaint();
		}

		public void actionPerformed(ActionEvent arg0) {
			JMenuItem item = (JMenuItem)arg0.getSource();
			if(item == null){
				
			}else{
				selected(item.getText());
			}
		}

	}

	class WavePanelCanvas extends JPanel implements MouseListener {
		int size;
		int length = 10;
		int offset = 2;
		int mark = 0;
		Image buf;

		public WavePanelCanvas() {
			super();
			addMouseListener(this);
			setPreferredSize(new Dimension(1000, 1000));
			size = list.getFontMetrics(list.getFont()).getHeight();
		}

		private void drawGrid(Graphics g) {
			int w = getPreferredSize().width;
			int h = getPreferredSize().height;
			int y = FreeHDLWaveFrame.margin;
			while (y < h) {
				g.drawLine(0, y, w, y);
				y += size + FreeHDLWaveFrame.margin;
			}
		}

		private void drawMark(Graphics g) {
			Color c = g.getColor();
			g.setColor(Color.red);
			g.drawLine(mark, 0, mark, getPreferredSize().height);
			g.setColor(c);
		}

		private void check(Graphics g, int x, int y) {
			boolean xFlag, yFlag;
			int w = getPreferredSize().width;
			int h = getPreferredSize().height;
			xFlag = (x < w);
			yFlag = (y < h);
			if (xFlag && yFlag)
				return;
			Dimension d = new Dimension(xFlag ? w : w + 1000, yFlag ? h : y);
			setPreferredSize(d);
			// buf.setSize(d);
			revalidate();
		}

		public void drawGraph(Graphics g, Signal s, int base) {
			if (s instanceof StdLogic) {
				drawGraph(g, (StdLogic) s, base);
			} else if (s instanceof StdLogicVector) {
				drawGraph(g, (StdLogicVector) s, base);
			}
		}

		public void drawGraph(Graphics g, StdLogic s, int base) {
			Iterator<LogicData> i = s.iterator();
			int x0, y0, x1, y1;
			x0 = x1 = 0;
			y0 = -1;
			y1 = base;
			while (i.hasNext()) {
				LogicData v = i.next();
				switch (v.value) {
				case LogicData.POSITIVE:
					y1 = base - (size + FreeHDLWaveFrame.margin) + offset;
					break;
				case LogicData.NEGATIVE:
					y1 = base - offset;
					break;
				default:
					y1 = base - (size + FreeHDLWaveFrame.margin) / 2;
					break;
				}
				x1 = x0 + length;
				if (y0 > 0) {
					g.drawLine(x0, y1, x1, y1);
					if (y0 != y1) {
						g.drawLine(x0, y0, x0, y1);
					} else {
					}
				} else {
				}
				x0 = x1;
				y0 = y1;
				check(g, x0, y0);
			}
		}

		public void drawGraph(Graphics g, StdLogicVector s, int base) {
			StdLogicVector.VectorData prev = null;
			Iterator<StdLogicVector.VectorData> i = s.iterator();
			int trans = 2;
			int x0, x1, x2, y0, y1;
			x0 = x1 = x2 = 0;
			y0 = base - (size + FreeHDLWaveFrame.margin) + offset;
			y1 = base - offset;
			while (i.hasNext()) {
				StdLogicVector.VectorData data = i.next();
				x1 = x0 + length - 2 * trans;
				x2 = x0 + length;
				if (prev != null) {
					g.drawLine(x0, y0, x1, y0);
					g.drawLine(x0, y1, x1, y1);
					if (prev.equals(data)) {
						g.drawLine(x1, y0, x2, y0);
						g.drawLine(x1, y1, x2, y1);
					} else {
						g.drawLine(x1, y0, x2, y1);
						g.drawLine(x1, y1, x2, y0);
					}
				}
				prev = data;
				x0 = x0 + length;
				check(g, x0, y0);
			}
		}

		public void drawGraph(Graphics g) {
			int y0 = FreeHDLWaveFrame.margin;
			for (int i = 0; i < data.size(); i++) {
				y0 += size + FreeHDLWaveFrame.margin;
				drawGraph(g, data.get(i), y0);
			}
			drawGrid(g);
			drawMark(g);
		}

		private int[] cachedSignalSize = new int[0];

		private boolean isDataChanged() {
			if (cachedSignalSize.length != data.size()) {
				return true;
			}
			for (int i = 0; i < cachedSignalSize.length; i++) {
				if (cachedSignalSize[i] != data.get(i).size()) {
					return true;
				}
			}
			return false;
		}

		private void updateCachedData() {
			cachedSignalSize = new int[data.size()];
			for (int i = 0; i < cachedSignalSize.length; i++) {
				cachedSignalSize[i] = data.get(i).size();
			}
		}

		public void paintComponent(Graphics g) {
			super.paintComponent(g);
			/*
			 * if(data == null) return; if(buf == null || isDataChanged()){ buf =
			 * createImage(1000, 1000); drawGraph(buf.getGraphics());
			 * updateCachedData(); } g.drawImage(buf, 0, 0, this);
			 */
			drawGraph(g);
		}

		/**
		 * コンポーネント上でマウスボタンをクリック (押してから離す) したときに呼び出されます。
		 */
		public void mouseClicked(MouseEvent e) {
		}

		/**
		 * コンポーネントにマウスが入ると呼び出されます。
		 */
		public void mouseEntered(MouseEvent e) {
		}

		/**
		 * コンポーネントからマウスが出ると呼び出されます。
		 */
		public void mouseExited(MouseEvent e) {
		}

		/**
		 * コンポーネント上でマウスボタンが押されると呼び出されます。
		 */
		public void mousePressed(MouseEvent e) {
			mark = e.getX();
			repaint();
			list.setMark(mark / length);
			list.repaint();
		}

		public void mouseReleased(MouseEvent e) {
		}

	}

}
