/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * DataBufferやMicsDataPacketなどデータに関するクラスのパッケージ
 */
package net.wasamon.mics.memory;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.DataBufferException;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mjlib.print.FormatException;
import net.wasamon.mjlib.print.PrintFormat;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * ランダムアクセスメモリをシミュレーションするDataBufferの実装
 */
public class RandomAccessMemory extends MicsElement implements DataBuffer {

	private byte[] data;

	public RandomAccessMemory() {
	}

	public RandomAccessMemory(int size) {
		data = new byte[size];
	}

	public String getImagePath() {
		return "pp_memorynote1_1.gif";
	}

	/*
	 * generated automatically by MakeInitialize from
	 * inits/random_access_memory16.init
	 */
	public void initialize(String base, Node n) throws MicsException {
		try {
			{
				Node init_var_node = n;
				int init_var_size;
				init_var_size = DataUtil.parseInt(XMLParser.getAttribute(init_var_node,
						"size").getNodeValue());

				data = new byte[init_var_size];

				{
					Node[] init_var__init_obj = XMLParser.getNamedNodeArray(n, "init");
					for (int i = 0; i < init_var__init_obj.length; i++) {
						Node init_var__init_node = init_var__init_obj[i];
						String init_var__init_file;
						init_var__init_file = XMLParser.getAttribute(init_var__init_node,
								"file").getNodeValue();
						if (init_var__init_file.charAt(0) != '/') {
							init_var__init_file = base + "/" + init_var__init_file;
						}
						int init_var__init_offset;
						init_var__init_offset = DataUtil.parseInt(XMLParser.getAttribute(
								init_var__init_node, "offset").getNodeValue());

						try {
							write(init_var__init_offset, new BufferedInputStream(
									new FileInputStream(init_var__init_file)));
						} catch (FileNotFoundException e) {
							System.out.println("no such file: " + init_var__init_file);
							System.out.println("[W] no data is written as initialize.");
						}

					}
				}
			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.memory.RandomAccessMemory16");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.memory.RandomAccessMemory16");
		}
	}

	public int size() {
		return data.length;
	}

	public byte read(int addr) {
		return data[addr];
	}

	public byte[] read(int addr, int length) {
		byte[] d = new byte[length];
		for (int i = 0; i < length; i++) {
			d[i] = data[i + addr];
		}
		return d;
	}

	public void write(MicsDataPacket data) {
		RandomAccessMemoryDataPacket rd = (RandomAccessMemoryDataPacket) data;
		for (int i = 0; i < rd.length * (rd.width / 8); i++) {
			this.data[rd.addr + i] = rd.data[i];
		}
	}

	public MicsDataPacket read(MicsDataPacket data) {
		RandomAccessMemoryDataPacket rd = (RandomAccessMemoryDataPacket) data;
		RandomAccessMemoryDataPacket d = new RandomAccessMemoryDataPacket(rd.addr,
				rd.length, rd.width, new byte[rd.length * (rd.width / 8)]);
		for (int i = 0; i < d.data.length; i++) {
			d.data[i] = this.data[i + rd.addr];
		}
		// System.out.println("retun:" + d);
		return d;
	}

	public void write(int addr, int length, byte[] d) {
		for (int i = 0; i < length; i++)
			data[i + addr] = d[i];
	}

	public void write(int addr, byte d) {
		data[addr] = d;
	}

	public String toString(int addr, int len) {
		String str = "";
		try {
			for (int i = 0; i < len && addr + i * 16 < data.length; i++) {
				str += PrintFormat.print("%04x: ", addr + i * 16);
				for (int j = 0; j < 16; j++) {
					if (addr + i * 16 + j >= data.length) {
						str += "\n";
						return str;
					}
					str += PrintFormat.print("%02x", (byte) (data[addr + i * 16 + j]));
					if (j == 15) {
						str += "\n";
					} else {
						str += " ";
					}
				}
			}
		} catch (FormatException e) {

		}
		return str;
	}

	public void write(int offset, InputStream reader) throws DataBufferException {
		byte[] buf = new byte[1024];
		int addr = offset;
		int len = 0;
		try {
			while ((len = reader.read(buf, 0, 1024)) != -1) {
				write(addr, len, buf);
				addr += len;
			}
		} catch (IOException e) {
			throw new DataBufferException("file read error.");
		}
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		try {
			writer.write(data, offset, len);
		} catch (IOException e) {
			throw new DataBufferException("file read error.");
		}
	}

	public String getInfo() {
		String str = "";
		str += "RandomAccessMemory16\n";
		str += "  CLASS: " + this.getClass().getName() + "\n";
		str += "  address length: " + data.length + "\n";
		str += "  data width: " + 16 + " bit";
		return str;
	}

	public String[] getConnectedElements() {
		return null;
	}
	
	public void reset(){
		
	}

	public String getDescription(){
		return "TODO";
	}

}
