/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * さまざまなペリフェラルデバイスのシミュレーション要素のパッケージ
 */
package net.wasamon.mics.peripheral;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * DirectMemoryAccessを行うことができるモジュール．srcとdestに指定したチャネル間のデータ転送を行う．<br>
 * 制御は，このモジュールに対するwriteRequestで行う． 制御レジスタは，
 * <dl>
 * <dt>0-3
 * <dd>ソースアドレス
 * <dt>4-7
 * <dd>ディスティネーションアドレス
 * <dt>8-11
 * <dd>長さ
 * <dt>12-14
 * <dd>予約
 * <dt>15
 * <dd>制御コマンド|x|x|x|x|si|di|転送方向|書き込み開始フラグ|-->si=1だとインクリメントしないdi=1だとしない
 * <dd>書き込み開始フラグが1のときに，動作を開始する．転送方向は，0がsrc->destの書き込み．1がdest->srcのよみこみ
 * </dl>
 * 
 * @author Takefumi MIYOSHI
 * 
 */
public class DirectMemoryAccessController extends MicsElement implements
		Channel, ChannelConnectable, ExecutableElement {

	private String srcChannelID;
	private String destChannelID;

	/** 一度に読み書きすることができる幅 */
	private int width = 4;

	public String getImagePath() {
		return "combo_bus3.png";
	}

	public DirectMemoryAccessController() {
	}

	public String getDescription(){
		return "TODO";
	}

	private void initialize(String src, String dest, int width) {
		srcChannelID = src;
		destChannelID = dest;
		this.width = width;
		// this.width = 6;
	}

	public void initialize(String base, Node n) throws MicsException {
		try {
			{
				Node init_var_node = n;
				int init_var_width;
				init_var_width = DataUtil.parseInt(XMLParser.getAttribute(
						init_var_node, "width").getNodeValue());
				String init_var_src;
				init_var_src = XMLParser.getAttribute(init_var_node, "src")
						.getNodeValue();
				String init_var_dest;
				init_var_dest = XMLParser.getAttribute(init_var_node, "dest")
						.getNodeValue();

				initialize(init_var_src, init_var_dest, init_var_width);

			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.peripheral.DirectMemoryAccessController");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.peripheral.DirectMemoryAccessController");
		}
	}

	private int srcAddr;
	private int destAddr;
	private int length;
	private int count;

	Channel srcChannel, destChannel;

	private Channel getSrcChannel() throws MicsException {
		if (srcChannel == null) {
			srcChannel = composite.getChannel(srcChannelID);
		}
		return srcChannel;
	}

	private Channel getDestChannel() throws MicsException {
		if (destChannel == null) {
			destChannel = composite.getChannel(destChannelID);
		}
		return destChannel;
	}

	private Channel getReadChannel() throws MicsException {
		if (transferMode == READ) {
			return getDestChannel();
		} else if (transferMode == WRITE) {
			return getSrcChannel();
		}
		return null;
	}

	private Channel getWriteChannel() throws MicsException {
		if (transferMode == READ) {
			return getSrcChannel();
		} else if (transferMode == WRITE) {
			return getDestChannel();
		}
		return null;
	}

	public ExecInfo exec_first() throws MicsException {
		ExecInfo info = new ExecInfo();
		if (enableFlag) {// enableFlagがwriteの中で定義されている
			MicsDataPacket req = RandomAccessMemoryDataPacket.readPacket(srcAddr,
					width, 8);
			getReadChannel().readRequest(this, req);
			count += width;

			if (sourceIncrement == true)
				srcAddr += width;

			// srcAddr += width;
			if (count < length) {
				enableFlag = true;
			} else {
				enableFlag = false;
			}
			// System.out.println("read from: " +
			// ((MicsElement)getReadChannel()).id());
			// System.out.println(req);
		}
		info.setTerminatableFlag(!enableFlag);
		return info;
	}

	public ExecInfo exec_second() {
		return null;
	}

	public void reset() {
		enableFlag = false;
		transferMode = WRITE;
		count = 0;
	}

	private boolean enableFlag;
	private boolean sourceIncrement, destinationIncrement;// 追加
	private int transferMode;
	private final static int READ = 0;
	private final static int WRITE = 1;

	public void writeRequest(ChannelConnectable dest, MicsDataPacket data)
			throws MicsException {
		// System.out.println(data);
		write(data);
	}

	public void readRequest(ChannelConnectable dest, MicsDataPacket data)
			throws MicsException {
	}

	public void write(MicsDataPacket data) {
		RandomAccessMemoryDataPacket rd = (RandomAccessMemoryDataPacket) data;
		switch (rd.addr / 4) {
		case 0:
			srcAddr = DataUtil.toInteger(rd.data, 0, rd.data.length);
			break;
		case 1:
			destAddr = DataUtil.toInteger(rd.data, 0, rd.data.length);
			break;
		case 2:
			length = DataUtil.toInteger(rd.data, 0, rd.data.length);
			break;
		case 3:
			if ((DataUtil.toInteger(rd.data, 0, rd.data.length) & 0x01) == 1) {
				enableFlag = true;
				count = 0;
			} else {
				enableFlag = false;
				count = 0;
			}
			if ((DataUtil.toInteger(rd.data, 0, rd.data.length) & 0x02) == 0x02) {
				transferMode = READ;
			} else {
				transferMode = WRITE;
			}
			// --------------------------ここから----------------------------------//
			if ((DataUtil.toInteger(rd.data, 0, rd.data.length) & 0x04) == 0x04) {
				destinationIncrement = false;
			} else {
				destinationIncrement = true;
			}
			if ((DataUtil.toInteger(rd.data, 0, rd.data.length) & 0x08) == 0x08) {
				sourceIncrement = false;
			} else {
				sourceIncrement = true;
			}
			// --------------------------ここまで----------------------------------//
			break;
		}
	}

	public int size() {
		return 0;
	}

	public String toString() {
		String s = "";
		s += id() + ":" + getClass().getName() + "\n";
		s += "srcAddr : " + srcAddr + "\n";
		s += "destAddr: " + destAddr + "\n";
		s += "length  : " + length + "\n";
		s += "enable  : " + enableFlag + "\n";
		s += "transfer: " + "\n";
		if (transferMode == READ) {
			s += "READ\n";
		} else if (transferMode == WRITE) {
			s += "WRITE\n";
		} else {
			s += "UNKNOWN\n";
		}
		return s;
	}

	public void writeback(Channel src, MicsDataPacket p) throws MicsException {// ここでdstのインクリメントがされている
		RandomAccessMemoryDataPacket rp = (RandomAccessMemoryDataPacket) p;
		MicsDataPacket req = RandomAccessMemoryDataPacket.writePacket(destAddr,
				width, 8, rp.data);
		// System.out.println("write to: " + ((MicsElement)getWriteChannel()).id());
		// System.out.println(req);
		getWriteChannel().writeRequest(this, req);
		if (destinationIncrement == true)
			destAddr += width;
		// destAddr += width;
	}

	public int getChannelOffset(Channel c) {
		return 0;
	}

	public String[] getConnectedElements() {
		return new String[] { srcChannelID, destChannelID };
	}

}
