/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * プロセッサをシミュレーションする要素のパッケージ
 */
package net.wasamon.mics.processor;

import java.io.InputStream;
import java.io.OutputStream;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.DataBufferException;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.ExecutableElementException;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.memory.RandomAccessMemory;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mics.util.ChannelManager;
import net.wasamon.mjlib.print.FormatException;
import net.wasamon.mjlib.print.PrintFormat;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;
import net.wasamon.wallet.binutils.UnknownInstructionException;

import org.w3c.dom.Node;

/**
 * 単純な機能レベルの16bitRISCプロセッサシミュレータ
 * @author Takefumi MIYOSHI
 * 
 */
public class SimpleProcessor extends MicsElement implements ExecutableElement,
		ChannelConnectable, DataBuffer {

	private static final int INST_HALT = 0x00;

	private static final int INST_NOP = 0x01;

	// �l�����Z
	private static final int INST_ADD = 0x10;

	private static final int INST_SUB = 0x11;

	private static final int INST_MULT = 0x12;

	private static final int INST_DIV = 0x13;

	private static final int INST_ADDI = 0x14;

	private static final int INST_SUBI = 0x15;

	private static final int INST_MOD = 0x16;

	private static final int INST_LSHS = 0x17;
	private static final int INST_RSHS = 0x18;

	// �_�����Z
	private static final int INST_OR = 0x20;

	private static final int INST_AND = 0x21;

	// �W�����v
	private static final int INST_JMP = 0x30;

	private static final int INST_JMPI = 0x31;

	private static final int INST_JCC = 0x32;

	private static final int INST_JCCI = 0x33;

	// ��r
	private static final int INST_EQ = 0x40;

	private static final int INST_NEQ = 0x41;

	private static final int INST_LT = 0x42;

	private static final int INST_GT = 0x43;

	private static final int INST_LEQ = 0x44;

	private static final int INST_GEQ = 0x45;

	// ���W�X�^����
	private static final int INST_MOV = 0x50;

	private static final int INST_MOVI = 0x51;

	// pop/push/return/call
	private static final int INST_POP = 0x60;

	private static final int INST_PUSH = 0x61;

	private static final int INST_CALL = 0x64;

	private static final int INST_CALLI = 0x65;

	private static final int INST_RET = 0x66;

	private static final int INST_LDA = 0x67;

	private static final int INST_STA = 0x68;

	// ����������
	private static final byte INST_LDR = (byte) 0x80;

	private static final byte INST_STR = (byte) 0x81;

	private static final byte INST_LDR8 = (byte) 0x82;

	private static final byte INST_STR8 = (byte) 0x83;

	/** �Ք���W�X�^�̌� */
	private static final int REGISTER_SIZE = 32;

	/** �v���O�����J�E���^�̃��W�X�^�ԍ� */
	private static final int PROGRAM_COUNTER = REGISTER_SIZE - 1;

	/** �t���[���|�C���^�̃��W�X�^�ԍ� */
	private static final int FRAME_POINTER = REGISTER_SIZE - 2;

	/** �X�^�b�N�|�C���^�̃��W�X�^�ԍ� */
	private static final int STACK_POINTER = REGISTER_SIZE - 3;

	/** �Ք���W�X�^ */
	private char[] register = new char[REGISTER_SIZE];

	/** �쏃t���O */
	private boolean condition = false;

	/** �v���O���������� */
	private RandomAccessMemory memory;

	/** �v���O�����������̃A�N�Z�X�I�t�Z�b�g */
	private int programOffset;

	/** �v���O�����R�[�h�̃��[�h�� */
	private int CODE_LENGTH = 4;

	/** �������A�N�Z�X���Ɏw�肷��i�[���W�X�^ID��ۑ����Ă����ϐ� */
	private int readRequestRegID;

	private boolean waitFlag;

	/** MULT��DIV�ɌW��T�C�N���� */
	private final int MULT_DIV_CYCLE = 2;

	private ChannelManager channelManager;

	class OperationCode {
		private byte inst;

		private byte[] args = new byte[3];

		/**
		 * �R���X�g���N�^
		 * 
		 * @param in0
		 *          ��̓��W�X�^0
		 * @param in1
		 *          ��̓��W�X�^1
		 * @param out
		 *          �o�̓��W�X�^
		 */
		OperationCode(byte inst, byte arg0, byte arg1, byte arg2) {
			this.inst = inst;
			this.args[0] = arg0;
			this.args[1] = arg1;
			this.args[2] = arg2;
		}

		public String toString() {
			String str = "";
			switch (inst) {
			case INST_HALT:
				str += "HALT ";
				break;
			case INST_NOP:
				str += "NOP  ";
				break;
			case INST_ADD:
				str += "ADD  ";
				break;
			case INST_ADDI:
				str += "ADDI ";
				break;
			case INST_SUB:
				str += "SUB  ";
				break;
			case INST_SUBI:
				str += "SUBI ";
				break;
			case INST_MULT:
				str += "MULT ";
				break;
			case INST_DIV:
				str += "DIV  ";
				break;
			case INST_MOD:
				str += "MOD  ";
				break;
			case INST_LSHS:
				str += "LSHS ";
				break;
			case INST_RSHS:
				str += "RSHS ";
				break;
			case INST_OR:
				str += "OR   ";
				break;
			case INST_AND:
				str += "AND  ";
				break;
			case INST_JMP:
				str += "JMP  ";
				break;
			case INST_JMPI:
				str += "JMPI ";
				break;
			case INST_JCC:
				str += "JCC  ";
				break;
			case INST_JCCI:
				str += "JCCI ";
				break;
			case INST_EQ:
				str += "EQ   ";
				break;
			case INST_NEQ:
				str += "NEQ  ";
				break;
			case INST_LT:
				str += "LT   ";
				break;
			case INST_GT:
				str += "GT   ";
				break;
			case INST_LEQ:
				str += "LEQ  ";
				break;
			case INST_GEQ:
				str += "GEQ  ";
				break;
			case INST_LDR:
				str += "LDR  ";
				break;
			case INST_STR:
				str += "STR  ";
				break;
			case INST_LDR8:
				str += "LDR8 ";
				break;
			case INST_STR8:
				str += "STR8 ";
				break;
			case INST_MOV:
				str += "MOV  ";
				break;
			case INST_MOVI:
				str += "MOVI ";
				break;
			case INST_POP:
				str += "POP  ";
				break;
			/*
			 * case INST_POPR: str += "POPR "; break;
			 */
			case INST_PUSH:
				str += "PUSH ";
				break;
			case INST_CALL:
				str += "CALL ";
				break;
			case INST_CALLI:
				str += "CALLI";
				break;
			case INST_RET:
				str += "RET  ";
				break;
			case INST_LDA:
				str += "LDA  ";
				break;
			case INST_STA:
				str += "STA  ";
				break;
			default:
				str += "ERROR";
			}
			str += ",";
			try {
				str += PrintFormat.print("%02x", args[0]);
				str += ",";
				str += PrintFormat.print("%02x", args[1]);
				str += ",";
				str += PrintFormat.print("%02x", args[2]);
			} catch (FormatException e) {
				e.printStackTrace();
			}
			return str;
		}
	}

	public SimpleProcessor() {
	}

	/*
	 * generated automatically by MakeInitialize from inits/simple_processor.init
	 */
	public void initialize(String base, Node n) throws MicsException {
		channelManager = new ChannelManager(composite);
		try {
			{
				Node init_var_node = n;
				int init_var_memory;
				init_var_memory = DataUtil.parseInt(XMLParser.getAttribute(
						init_var_node, "memory").getNodeValue());

				memory = new RandomAccessMemory(init_var_memory);

				{
					Node[] init_var__channel_obj = XMLParser.getNamedNodeArray(n,
							"channel");
					for (int i = 0; i < init_var__channel_obj.length; i++) {
						Node init_var__channel_node = init_var__channel_obj[i];
						String init_var__channel_id;
						init_var__channel_id = XMLParser.getAttribute(
								init_var__channel_node, "id").getNodeValue();
						int init_var__channel_offset;
						init_var__channel_offset = DataUtil.parseInt(XMLParser
								.getAttribute(init_var__channel_node, "offset").getNodeValue());

						channelManager.add(init_var__channel_id, init_var__channel_offset);

					}
				}
			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.SimpleProcessor");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.SimpleProcessor");
		}
	}

	public MicsDataPacket read(MicsDataPacket data) {
		return memory.read(data);
	}

	public void write(MicsDataPacket data) {
		memory.write(data);
	}

	public void write(int addr, InputStream reader) throws DataBufferException {
		memory.write(addr, reader);
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		memory.dump(offset, len, writer);
	}

	public int size() {
		return memory.size();
	}

	public String toString(int offset, int length) {
		return memory.toString(offset, length);
	}

	public void execWait() {
		waitFlag = true;
	}

	public void execRestart() {
		waitFlag = false;
	}

	public void reset() {
		this.waitFlag = false;
		for (int i = 0; i < REGISTER_SIZE; i++) {
			register[i] = (char) 0;
		}
	}

	public ExecInfo exec_second() {
		return null;
	}

	/**
	 * �Ք����������v���O�������ǂݍ��ݎ s����
	 * 
	 * @return � s�ɕK�v�ȃT�C�N����
	 */
	public ExecInfo exec_first() throws ExecutableElementException {
		OperationCode op = null;
		ExecInfo info = new ExecInfo();
		info.setCycle(1); // ��{�́C1�T�C�N���Ŗ��߂���������
		info.setTerminatableFlag(false);
		try {
			op = readProgram(register[PROGRAM_COUNTER]);
		} catch (ArrayIndexOutOfBoundsException e) {
			throw new ExecutableElementException("program address error");
		}
		if (waitFlag == true) {
			return info;
		}
		int tmp = 0;
		switch (op.inst) {
		case INST_HALT:
			info.setTerminatableFlag(true);
			break;
		case INST_NOP:
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		// �l�����Z
		case INST_ADD:
			tmp = register[op.args[1]] + register[op.args[2]];
			register[op.args[0]] = DataUtil.toCharL(tmp);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_ADDI: {
			char imm = (char) (((op.args[1] & 0x00ff) << 8) + (op.args[2] & 0x00ff));
			// System.out.println("addi: " + (int)imm);
			// System.out.println("before: " + (int)register[op.args[0]]);
			register[op.args[0]] += imm;
			// System.out.println("after: " + (int)register[op.args[0]]);
		}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_SUB:
			tmp = register[op.args[1]] - register[op.args[2]];
			register[op.args[0]] = DataUtil.toCharL(tmp);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_SUBI: {
			char imm = (char) (((op.args[1] & 0x00ff) << 8) + (op.args[2] & 0x00ff));
			register[op.args[0]] -= imm;
		}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_MULT:
			tmp = register[op.args[1]] * register[op.args[2]];
			// register[op.args[0] + 1] = DataUtil.toCharH(tmp);
			register[op.args[0]] = DataUtil.toCharL(tmp);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			info.setCycle(MULT_DIV_CYCLE);
			break;
		case INST_DIV:
			register[op.args[0]] = DataUtil.toCharL(register[op.args[1]]
					/ register[op.args[2]]);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			info.setCycle(MULT_DIV_CYCLE);
			break;
		case INST_MOD:
			register[op.args[0]] = DataUtil.toCharL(register[op.args[1]]
					% register[op.args[2]]);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			info.setCycle(MULT_DIV_CYCLE);
			break;
		// �_�����Z
		case INST_LSHS:
			register[op.args[0]] = DataUtil
					.toCharL(register[op.args[1]] << register[op.args[2]]);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		// �_�����Z
		case INST_RSHS:
			register[op.args[0]] = DataUtil
					.toCharL(register[op.args[1]] >> register[op.args[2]]);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		// �_�����Z
		case INST_OR:
			tmp = register[op.args[1]] | register[op.args[2]];
			register[op.args[0]] = DataUtil.toCharL(tmp);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_AND:
			tmp = register[op.args[1]] & register[op.args[2]];
			register[op.args[0]] = DataUtil.toCharL(tmp);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		// �W�����v
		case INST_JMP:
			register[PROGRAM_COUNTER] = (char) ((register[op.args[0]]) * CODE_LENGTH);
			break;
		case INST_JMPI:
			register[PROGRAM_COUNTER] = (char) (DataUtil.toChar(op.args[0],
					op.args[1]) * CODE_LENGTH);
			break;
		case INST_JCC:
			if (condition == true) {
				register[PROGRAM_COUNTER] = (char) ((register[op.args[0]]) * CODE_LENGTH);
			} else {
				register[PROGRAM_COUNTER] += CODE_LENGTH;
			}
			break;
		case INST_JCCI:
			if (condition == true) {
				register[PROGRAM_COUNTER] = (char) (DataUtil.toChar(op.args[0],
						op.args[1]) * CODE_LENGTH);
			} else {
				register[PROGRAM_COUNTER] += CODE_LENGTH;
			}
			break;
		// ��r
		case INST_EQ:
			if (register[op.args[1]] == register[op.args[2]]) {
				register[op.args[0]] = 0;
				condition = true;
			} else {
				register[op.args[0]] = 1;
				condition = false;
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_NEQ:
			if (register[op.args[1]] != register[op.args[2]]) {
				register[op.args[0]] = 0;
				condition = true;
			} else {
				register[op.args[0]] = 1;
				condition = false;
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_LT:
			if (DataUtil.toSiginedInteger(register[op.args[1]]) < DataUtil
					.toSiginedInteger(register[op.args[2]])) {
				register[op.args[0]] = 0;
				condition = true;
			} else {
				register[op.args[0]] = 1;
				condition = false;
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_GT:
			if (DataUtil.toSiginedInteger(register[op.args[1]]) > DataUtil
					.toSiginedInteger(register[op.args[2]])) {
				register[op.args[0]] = 0;
				condition = true;
			} else {
				register[op.args[0]] = 1;
				condition = false;
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_LEQ:
			if (DataUtil.toSiginedInteger(register[op.args[1]]) <= DataUtil
					.toSiginedInteger(register[op.args[2]])) {
				register[op.args[0]] = 0;
				condition = true;
			} else {
				register[op.args[0]] = 1;
				condition = false;
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_GEQ:
			if (DataUtil.toSiginedInteger(register[op.args[1]]) >= DataUtil
					.toSiginedInteger(register[op.args[2]])) {
				register[op.args[0]] = 0;
				condition = true;
			} else {
				register[op.args[0]] = 1;
				condition = false;
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		// ����������
		case INST_LDR:
			if ((int) register[op.args[0]] < memory.size()) {
				register[op.args[1]] = DataUtil.toChar(memory.read(
						(int) register[op.args[0]], 2), 0, 2);
			} else {
				readRequestRegID = op.args[1]; // �������f�[�^��i�[���郌�W�X�^ID��ۑ�
				ChannelManager.Element c = channelManager.search(register[op.args[0]]);
				try {
					c.getChannel().readRequest(
							this,
							RandomAccessMemoryDataPacket.readPacket(
									(int) register[op.args[0]] - c.offset, 1, 16));
				} catch (MicsException e) {
					throw new ExecutableElementException(e);
				}
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_LDR8:
			if ((int) register[op.args[0]] < memory.size()) {
				register[op.args[1]] = DataUtil.toChar(memory.read(
						(int) register[op.args[0]], 1), 0, 1);
			} else {
				readRequestRegID = op.args[1]; // �������f�[�^��i�[���郌�W�X�^ID��ۑ�
				ChannelManager.Element c = channelManager.search(register[op.args[0]]);
				try {
					c.getChannel().readRequest(
							this,
							RandomAccessMemoryDataPacket.readPacket(
									((int) register[op.args[0]] - c.offset), 1, 8));
				} catch (MicsException e) {
					throw new ExecutableElementException(e);
				}
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_STR:
			if ((int) register[op.args[0]] < memory.size()) {
				memory.write((int) register[op.args[0]], 2, DataUtil
						.toByteArray(register[op.args[1]]));
			} else {
				ChannelManager.Element c = channelManager
						.search((int) register[op.args[0]]);
				try {
					c.getChannel().writeRequest(
							this,
							RandomAccessMemoryDataPacket.writePacket(
									((int) register[op.args[0]] - c.offset), 2, 8, DataUtil
											.toByteArray(register[op.args[1]])));
				} catch (MicsException e) {
					throw new ExecutableElementException(e);
				}
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_STR8:
			if ((int) register[op.args[0]] < memory.size()) {
				memory.write((int) register[op.args[0]], DataUtil
						.toByteL(register[op.args[1]]));
			} else {
				ChannelManager.Element c = channelManager
						.search((int) register[op.args[0]]);
				try {
					c.getChannel().writeRequest(
							this,
							RandomAccessMemoryDataPacket.writePacket(
									((int) register[op.args[0]] - c.offset), 1, 8,
									new byte[] { DataUtil.toByteL(register[op.args[1]]) }));
				} catch (MicsException e) {
					throw new ExecutableElementException(e);
				}
			}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_MOVI:
			register[op.args[0]] = DataUtil.toChar(op.args[1], op.args[2]);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_MOV:
			register[op.args[0]] = register[op.args[1]];
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_POP:
			register[op.args[0]] = pop();
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_PUSH:
			push(register[op.args[0]]);
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_CALL:
			push((char) (register[PROGRAM_COUNTER] + CODE_LENGTH));
			register[PROGRAM_COUNTER] = (char) ((register[op.args[0]]) * CODE_LENGTH);
			break;
		case INST_CALLI:
			push((char) (register[PROGRAM_COUNTER] + CODE_LENGTH));
			register[PROGRAM_COUNTER] = (char) (DataUtil.toChar(op.args[0],
					op.args[1]) * CODE_LENGTH);
			break;
		case INST_RET:
			register[PROGRAM_COUNTER] = pop();
			break;
		case INST_LDA: {
			int fp = register[FRAME_POINTER];
			char imm = DataUtil.toChar(new byte[] { op.args[1], op.args[2] }, 0, 2);
			int offset = 0;
			if ((imm & 0x8000) == 0x8000) {
				offset = imm - 65536;
			} else {
				offset = imm;
			}
			// System.out.println("LDA: addr = " + (fp+ offset));
			register[op.args[0]] = DataUtil.toChar(memory.read(fp + offset, 2), 0, 2);
		}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		case INST_STA: {
			int fp = register[FRAME_POINTER];
			char imm = DataUtil.toChar(new byte[] { op.args[1], op.args[2] }, 0, 2);
			int offset = 0;
			if ((imm & 0x8000) == 0x8000) {
				offset = (imm - 65536);
			} else {
				offset = imm;
			}
			memory.write(fp + offset, 2, DataUtil.toByteArray(register[op.args[0]]));
		}
			register[PROGRAM_COUNTER] += CODE_LENGTH;
			break;
		default:
			throw new ExecutableElementException("unknow op code "
					+ String.valueOf(op.inst));
		}
		return info;
	}

	private char pop() throws ExecutableElementException {
		if (register[STACK_POINTER] < memory.size() - 1) {
			char data = DataUtil
					.toChar(memory.read(register[STACK_POINTER], 2), 0, 2);
			register[STACK_POINTER] += 2;
			return data;
		} else {
			throw new ExecutableElementException("stack under flow");
		}
	}

	private void push(char v) throws ExecutableElementException {
		if (register[STACK_POINTER] > 2) {
			register[STACK_POINTER] -= 2;
			memory.write(register[STACK_POINTER], 2, DataUtil.toByteArray(v));
		} else {
			throw new ExecutableElementException("stack over flow");
		}
	}

	private OperationCode readProgram(int addr) {
		byte[] data = memory.read(addr + programOffset, 4);
		return new OperationCode(data[0], data[1], data[2], data[3]);
	}

	public void writeback(Channel src, MicsDataPacket data) {
		RandomAccessMemoryDataPacket rd = (RandomAccessMemoryDataPacket) data;
		System.out.println(rd);
		if (rd.length * rd.width / 8 == 2) {
			register[readRequestRegID] = DataUtil.toChar(rd.data, 0, 2);
		} else {
			register[readRequestRegID] = DataUtil.toChar(rd.data, 0, 1);
		}
	}

	public int getChannelOffset(Channel c) {
		return channelManager.search(c);
	}

	public String getInfo() {
		String s = "";
		s += "SimpleProcessor\n";
		s += "  CLASS: " + this.getClass().getName() + "\n";
		s += "  ID: " + id() + "\n";
		s += "  Local Memory ID: " + ((MicsElement) memory).id() + ",";
		s += " offset = " + programOffset + "\n";
		ChannelManager.Element[] channels = channelManager.array();
		for (int i = 0; i < channels.length; i++) {
			ChannelManager.Element element = channels[i];
			s += "  Channel ID: " + element.id + ",";
			s += " offset = " + element.offset + "\n";
		}
		return s;
	}

	public String toString() {
		try {
			String str = "";
			try {
				str += PrintFormat.print("%04d", register[PROGRAM_COUNTER]) + ":";
			} catch (FormatException e) {
				e.printStackTrace();
			}
			str += readProgram(register[PROGRAM_COUNTER]).toString();
			str += "\n";
			for (int i = 0; i < REGISTER_SIZE; i++) {
				try {
					str += PrintFormat.print("%04x", register[i]);
					if (i % 8 == 7) {
						str += "\n";
					} else {
						str += " ";
					}
				} catch (FormatException e) {

				}
			}
			return str;
		} catch (ArrayIndexOutOfBoundsException e) {
			return new String((int) register[PROGRAM_COUNTER]
					+ ": illegal program counter");
		}
	}

	public static int getInstructionCode(String str)
			throws UnknownInstructionException {
		if (str.equals("HALT"))
			return INST_HALT;
		if (str.equals("NOP"))
			return INST_NOP;
		if (str.equals("ADD"))
			return INST_ADD;
		if (str.equals("ADDI"))
			return INST_ADDI;
		if (str.equals("SUB"))
			return INST_SUB;
		if (str.equals("SUBI"))
			return INST_SUBI;
		if (str.equals("MULT"))
			return INST_MULT;
		if (str.equals("DIV"))
			return INST_DIV;
		if (str.equals("MOD"))
			return INST_MOD;
		if (str.equals("LSHS"))
			return INST_LSHS;
		if (str.equals("RSHS"))
			return INST_RSHS;
		if (str.equals("OR"))
			return INST_OR;
		if (str.equals("AND"))
			return INST_AND;
		if (str.equals("JMP"))
			return INST_JMP;
		if (str.equals("JMPI"))
			return INST_JMPI;
		if (str.equals("JCC"))
			return INST_JCC;
		if (str.equals("JCCI"))
			return INST_JCCI;
		if (str.equals("EQ"))
			return INST_EQ;
		if (str.equals("NEQ"))
			return INST_NEQ;
		if (str.equals("LT"))
			return INST_LT;
		if (str.equals("GT"))
			return INST_GT;
		if (str.equals("LEQ"))
			return INST_LEQ;
		if (str.equals("GEQ"))
			return INST_GEQ;
		if (str.equals("MOVI"))
			return INST_MOVI;
		if (str.equals("MOV"))
			return INST_MOV;
		if (str.equals("POP"))
			return INST_POP;
		// if(str.equals("POPR")) return INST_POPR;
		if (str.equals("PUSH"))
			return INST_PUSH;
		if (str.equals("CALL"))
			return INST_CALL;
		if (str.equals("CALLI"))
			return INST_CALLI;
		if (str.equals("RET"))
			return INST_RET;
		if (str.equals("LDA"))
			return INST_LDA;
		if (str.equals("STA"))
			return INST_STA;
		if (str.equals("LDR"))
			return INST_LDR;
		if (str.equals("STR"))
			return INST_STR;
		if (str.equals("LDR8"))
			return INST_LDR8;
		if (str.equals("STR8"))
			return INST_STR8;
		throw new UnknownInstructionException("unknown instruction " + str);
	}

	public String[] getConnectedElements() {
		return channelManager.getConnectedElements();
	}

	public String getImagePath() {
		return "combo_processor.png";
	}


	public String getDescription(){
		return "TODO";
	}
}
