/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * プロセッサをシミュレーションする要素のパッケージ
 */
package net.wasamon.mics.processor;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.DataBufferException;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.ExecutableElementException;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.MicsViewable;
import net.wasamon.mics.memory.InterruptDataPacket;
import net.wasamon.mics.memory.RandomAccessMemory;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mics.util.ChannelManager;
import net.wasamon.mics.util.MicsTableModel;
import net.wasamon.mics.util.MicsTablePanel;
import net.wasamon.mjlib.print.FormatException;
import net.wasamon.mjlib.print.PrintFormat;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * 32bit シンプルプロセッサ
 * @author Takefumi MIYOSHI
 * 
 */
public class SimpleProcessor32 extends MicsElement implements
		ExecutableElement, ChannelConnectable, DataBuffer, MicsViewable {

	private IntRegister[] intRegister = new IntRegister[8];

	private static final int INST_HALT = 0x00;

	private static final int INST_NOP = 0x01;

	// �l�����Z
	private static final int INST_ADD = 0x10;

	private static final int INST_SUB = 0x11;

	private static final int INST_MULT = 0x12;

	private static final int INST_DIV = 0x13;

	private static final int INST_ADDI = 0x14;

	private static final int INST_SUBI = 0x15;

	private static final int INST_MOD = 0x16;

	private static final int INST_LSHS = 0x17;

	private static final int INST_RSHS = 0x18;

	private static final int INST_NEG = 0x19;

	//
	private static final int INST_OR = 0x20;

	private static final int INST_AND = 0x21;

	// �W�����v
	private static final int INST_JMP = 0x30;

	private static final int INST_JMPI = 0x31;

	private static final int INST_JCC = 0x32;

	private static final int INST_JCCI = 0x33;

	// ��r
	private static final int INST_EQ = 0x40;

	private static final int INST_NEQ = 0x41;

	private static final int INST_LT = 0x42;

	private static final int INST_GT = 0x43;

	private static final int INST_LEQ = 0x44;

	private static final int INST_GEQ = 0x45;

	//
	private static final int INST_MOV = 0x50;

	private static final int INST_MOVI = 0x51;

	private static final int INST_MOVIL = 0x52;

	private static final int INST_MOVIH = 0x53;

	// pop/push/return/call
	private static final int INST_POP = 0x60;

	private static final int INST_PUSH = 0x61;

	private static final int INST_CALL = 0x64;

	private static final int INST_CALLI = 0x65;

	private static final int INST_RET = 0x66;

	private static final int INST_LDA = 0x67;

	private static final int INST_STA = 0x68;

	// ����������
	private static final byte INST_LDR = (byte) 0x80;

	private static final byte INST_STR = (byte) 0x81;

	private static final byte INST_LDR8 = (byte) 0x82;

	private static final byte INST_STR8 = (byte) 0x83;

	private static final byte INST_LDR32 = (byte) 0x84;

	private static final byte INST_STR32 = (byte) 0x85;

	private static final byte INST_LDRF = (byte) 0x86;

	private static final byte INST_SIN = (byte) 0x90;

	private static final byte INST_COS = (byte) 0x91;

	private static final byte INST_SINF = (byte) 0x92;

	private static final byte INST_COSF = (byte) 0x93;

	private static final byte INST_SQRT = (byte) 0x94;

	private static final byte INST_SQRTF = (byte) 0x95;

	private static final int REGISTER_SIZE = 32;

	private static final int PROGRAM_COUNTER = REGISTER_SIZE - 1;

	private static final int FRAME_POINTER = REGISTER_SIZE - 2;

	private static final int STACK_POINTER = REGISTER_SIZE - 3;

	private IntRegister[] register = new IntRegister[REGISTER_SIZE];

	private FloatRegister[] fregister = new FloatRegister[REGISTER_SIZE];

	class StateFlagGroup {
		public boolean fNaN = false;

		public boolean fAddressError = false;

		public boolean fCondition = false;

		public boolean fCarry = false;

		public void reset() {
			fNaN = false;
			fAddressError = false;
			fCondition = false;
			fCarry = false;
		}
	}

	private StateFlagGroup status = new StateFlagGroup();

	private RandomAccessMemory memory;

	private int programOffset;

	private int CODE_LENGTH = 4;

	private int readRequestRegID;

	private boolean waitFlag;

	private final int MULT_DIV_CYCLE = 2;

	private final int SIN_COS_CYCLE = 24;

	private final int SQRT_CYCLE = 24;

	private ChannelManager channelManager;

	class OperationCode {
		private byte inst;

		private byte[] args = new byte[3];

		/**
		 * �R���X�g���N�^
		 * 
		 * @param in0
		 *          ��̓��W�X�^0
		 * @param in1
		 *          ��̓��W�X�^1
		 * @param out
		 *          �o�̓��W�X�^
		 */
		OperationCode(byte inst, byte arg0, byte arg1, byte arg2) {
			this.inst = inst;
			this.args[0] = arg0;
			this.args[1] = arg1;
			this.args[2] = arg2;
		}

		public String toString() {
			String str = "";
			switch (inst) {
			case INST_HALT:
				str += "HALT ";
				break;
			case INST_NOP:
				str += "NOP  ";
				break;
			case INST_ADD:
				str += "ADD  ";
				break;
			case INST_ADDI:
				str += "ADDI ";
				break;
			case INST_SUB:
				str += "SUB  ";
				break;
			case INST_NEG:
				str += "NEG  ";
				break;
			case INST_SUBI:
				str += "SUBI ";
				break;
			case INST_MULT:
				str += "MULT ";
				break;
			case INST_DIV:
				str += "DIV  ";
				break;
			case INST_MOD:
				str += "MOD  ";
				break;
			case INST_LSHS:
				str += "LSHS ";
				break;
			case INST_RSHS:
				str += "RSHS ";
				break;
			case INST_OR:
				str += "OR   ";
				break;
			case INST_AND:
				str += "AND  ";
				break;
			case INST_JMP:
				str += "JMP  ";
				break;
			case INST_JMPI:
				str += "JMPI ";
				break;
			case INST_JCC:
				str += "JCC  ";
				break;
			case INST_JCCI:
				str += "JCCI ";
				break;
			case INST_EQ:
				str += "EQ   ";
				break;
			case INST_NEQ:
				str += "NEQ  ";
				break;
			case INST_LT:
				str += "LT   ";
				break;
			case INST_GT:
				str += "GT   ";
				break;
			case INST_LEQ:
				str += "LEQ  ";
				break;
			case INST_GEQ:
				str += "GEQ  ";
				break;
			case INST_LDR:
				str += "LDR  ";
				break;
			case INST_STR:
				str += "STR  ";
				break;
			case INST_LDR8:
				str += "LDR8 ";
				break;
			case INST_STR8:
				str += "STR8 ";
				break;
			case INST_LDR32:
				str += "LDR32";
				break;
			case INST_STR32:
				str += "STR32";
				break;
			case INST_LDRF:
				str += "LDRF ";
				break;
			case INST_MOV:
				str += "MOV  ";
				break;
			case INST_MOVI:
				str += "MOVI ";
				break;
			case INST_MOVIL:
				str += "MOVIL";
				break;
			case INST_MOVIH:
				str += "MOVIH";
				break;
			case INST_POP:
				str += "POP  ";
				break;
			/*
			 * case INST_POPR: str += "POPR "; break;
			 */
			case INST_PUSH:
				str += "PUSH ";
				break;
			case INST_CALL:
				str += "CALL ";
				break;
			case INST_CALLI:
				str += "CALLI";
				break;
			case INST_RET:
				str += "RET  ";
				break;
			case INST_LDA:
				str += "LDA  ";
				break;
			case INST_STA:
				str += "STA  ";
				break;
			case INST_COS:
				str += "COS  ";
				break;
			case INST_SIN:
				str += "SIN  ";
				break;
			case INST_COSF:
				str += "COSF  ";
				break;
			case INST_SINF:
				str += "SINF  ";
				break;
			case INST_SQRT:
				str += "SQRT  ";
				break;
			case INST_SQRTF:
				str += "SQRTF ";
				break;
			default:
				str += "ERROR";
			}
			str += ",";
			try {
				str += PrintFormat.print("%02x", args[0]);
				str += ",";
				str += PrintFormat.print("%02x", args[1]);
				str += ",";
				str += PrintFormat.print("%02x", args[2]);
			} catch (FormatException e) {
				e.printStackTrace();
			}
			return str;
		}
	}

	public SimpleProcessor32() {
		for (int i = 0; i < REGISTER_SIZE; i++) {
			register[i] = new IntRegister();
			fregister[i] = new FloatRegister();
		}
	}

	/**
	 * 
	 * @param index
	 * @return
	 */
	private boolean isFloatRegister(int index) {
		if (index < REGISTER_SIZE) {
			return false;
		} else {
			return true;
		}
	}

	private Register getRegister(int index) {
		if (isFloatRegister(index)) {
			return fregister[index - REGISTER_SIZE];
		} else {
			return register[index];
		}
	}

	/*
	 * generated automatically by MakeInitialize from inits/simple_processor.init
	 */
	public void initialize(String base, Node n) throws MicsException {
		channelManager = new ChannelManager(composite);
		try {
			{
				Node init_var_node = n;
				int init_var_memory;
				init_var_memory = DataUtil.parseInt(XMLParser.getAttribute(
						init_var_node, "memory").getNodeValue());

				memory = new RandomAccessMemory(init_var_memory);

				{
					Node[] init_var__channel_obj = XMLParser.getNamedNodeArray(n,
							"channel");
					for (int i = 0; i < init_var__channel_obj.length; i++) {
						Node init_var__channel_node = init_var__channel_obj[i];
						String init_var__channel_id;
						init_var__channel_id = XMLParser.getAttribute(
								init_var__channel_node, "id").getNodeValue();
						int init_var__channel_offset;
						init_var__channel_offset = DataUtil.parseInt(XMLParser
								.getAttribute(init_var__channel_node, "offset").getNodeValue());

						channelManager.add(init_var__channel_id, init_var__channel_offset);

					}
				}
				{
					Node[] init_var__init_obj = XMLParser.getNamedNodeArray(n, "init");
					for (int i = 0; i < init_var__init_obj.length; i++) {
						Node init_var__init_node = init_var__init_obj[i];
						String init_var__init_file;
						init_var__init_file = XMLParser.getAttribute(init_var__init_node,
								"file").getNodeValue();
						if (init_var__init_file.charAt(0) != '/') {
							init_var__init_file = base + "/" + init_var__init_file;
						}
						int init_var__init_offset;
						init_var__init_offset = DataUtil.parseInt(XMLParser.getAttribute(
								init_var__init_node, "offset").getNodeValue());

						try {
							write(init_var__init_offset, new BufferedInputStream(
									new FileInputStream(init_var__init_file)));
						} catch (FileNotFoundException e) {
							System.out.println("no such file: " + init_var__init_file);
							System.out.println("[W] no data is written as initialize.");
						}

					}
				}

			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.SimpleProcessor");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.SimpleProcessor");
		}
	}

	public MicsDataPacket read(MicsDataPacket data) {
		return memory.read(data);
	}

	public void write(MicsDataPacket data) {
		memory.write(data);
	}

	public void write(int addr, InputStream reader) throws DataBufferException {
		memory.write(addr, reader);
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		memory.dump(offset, len, writer);
	}

	public int size() {
		return memory.size();
	}

	public String toString(int offset, int length) {
		return memory.toString(offset, length);
	}

	public void execWait() {
		waitFlag = true;
	}

	public void execRestart() {
		waitFlag = false;
	}

	public void reset() {
		this.waitFlag = false;
		for (int i = 0; i < REGISTER_SIZE; i++) {
			register[i].reset();
			fregister[i].reset();
		}
	}

	public ExecInfo exec_second() {
		return null;
	}

	/**
	 */
	public ExecInfo exec_first() throws ExecutableElementException {
		OperationCode op = null;
		ExecInfo info = new ExecInfo();
		info.setCycle(1); // ��{�́C1�T�C�N���Ŗ��߂���������
		info.setTerminatableFlag(false);
		Register reg0, reg1;
		try {
			op = readProgram(register[PROGRAM_COUNTER].intValue());
		} catch (ArrayIndexOutOfBoundsException e) {
			throw new ExecutableElementException("program address error");
		}
		// System.out.println(op);
		if (waitFlag == true) {
			return info;
		}
		switch (op.inst) {
		case INST_HALT:
			info.setTerminatableFlag(true);
			break;
		case INST_NOP:
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		// �l�����Z
		case INST_ADD:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).add(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_ADDI:
			getRegister(op.args[0]).add(
					DataUtil.toSiginedInteger(DataUtil.toChar(op.args[1], op.args[2])));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_SUB:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).sub(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_NEG:
			reg0 = getRegister(op.args[0]);
			reg1 = getRegister(op.args[1]);
			reg0.neg(reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_SUBI:
			getRegister(op.args[0]).add(
					-DataUtil.toSiginedInteger(DataUtil.toChar(op.args[1], op.args[2])));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_MULT:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).mult(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(MULT_DIV_CYCLE);
			break;
		case INST_DIV:
			try {
				reg0 = getRegister(op.args[1]);
				reg1 = getRegister(op.args[2]);
				getRegister(op.args[0]).div(reg0, reg1);
			} catch (ArithmeticException e) {
				status.fNaN = true;
			}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(MULT_DIV_CYCLE);
			break;
		case INST_MOD:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).mod(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(MULT_DIV_CYCLE);
			break;
		//
		case INST_LSHS:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).shiftL(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		//
		case INST_RSHS:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).shiftR(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		//
		case INST_OR:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).or(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_AND:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			getRegister(op.args[0]).and(reg0, reg1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		// �W�����v
		case INST_JMP:
			// register[PROGRAM_COUNTER].set(register[op.args[0]].intValue() *
			// CODE_LENGTH);
			register[PROGRAM_COUNTER].set(register[op.args[0]].intValue());
			break;
		case INST_JMPI:
			// register[PROGRAM_COUNTER].set((DataUtil.toChar(op.args[0], op.args[1])
			// * CODE_LENGTH));
			register[PROGRAM_COUNTER].set(DataUtil.toChar(op.args[0], op.args[1]));
			break;
		case INST_JCC:
			if (status.fCondition == true) {
				// register[PROGRAM_COUNTER].set(register[op.args[0]].intValue() *
				// CODE_LENGTH);
				register[PROGRAM_COUNTER].set(register[op.args[0]].intValue());
			} else {
				register[PROGRAM_COUNTER].add(CODE_LENGTH);
			}
			break;
		case INST_JCCI:
			if (status.fCondition == true) {
				// register[PROGRAM_COUNTER].set(DataUtil.toChar(op.args[0], op.args[1])
				// * CODE_LENGTH);
				register[PROGRAM_COUNTER].set(DataUtil.toChar(op.args[0], op.args[1]));
			} else {
				register[PROGRAM_COUNTER].add(CODE_LENGTH);
			}
			break;
		// ��r
		case INST_EQ:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			if (reg0.eqeq(reg1)) {
				getRegister(op.args[0]).set(0);
				status.fCondition = true;
			} else {
				getRegister(op.args[0]).set(1);
				status.fCondition = false;
			}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_NEQ:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			if (!reg0.eqeq(reg1)) {
				getRegister(op.args[0]).set(0);
				status.fCondition = true;
			} else {
				getRegister(op.args[0]).set(1);
				status.fCondition = false;
			}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_LT:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			if (reg0.gt(reg1) == false && reg0.eqeq(reg1) == false) {
				getRegister(op.args[0]).set(0);
				status.fCondition = true;
			} else {
				getRegister(op.args[0]).set(1);
				status.fCondition = false;
			}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_GT:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			if (reg0.gt(reg1)) {
				getRegister(op.args[0]).set(0);
				status.fCondition = true;
			} else {
				getRegister(op.args[0]).set(1);
				status.fCondition = false;
			}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_LEQ:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			if (reg0.gt(reg1) == false) {
				getRegister(op.args[0]).set(0);
				status.fCondition = true;
			} else {
				getRegister(op.args[0]).set(1);
				status.fCondition = false;
			}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_GEQ:
			reg0 = getRegister(op.args[1]);
			reg1 = getRegister(op.args[2]);
			if (reg0.gt(reg1) || reg0.eqeq(reg1)) {
				getRegister(op.args[0]).set(0);
				status.fCondition = true;
			} else {
				getRegister(op.args[0]).set(1);
				status.fCondition = false;
			}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		// ����������
		case INST_LDR:
			loadData(op.args[1], getRegister(op.args[0]).intValue(), 2);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_LDR8:
			loadData(op.args[1], getRegister(op.args[0]).intValue(), 1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_LDR32:
			loadData(op.args[1], getRegister(op.args[0]).intValue(), 4);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_LDRF:
			loadData(op.args[1], getRegister(op.args[0]).intValue(), 4);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_STR:
			storeData(getRegister(op.args[1]).byteArray(2, 2),
					getRegister(op.args[0]).intValue(), 2);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_STR8:
			storeData(getRegister(op.args[1]).byteArray(3, 1),
					getRegister(op.args[0]).intValue(), 1);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_STR32:
			storeData(getRegister(op.args[1]).byteArray(), getRegister(op.args[0])
					.intValue(), 4);
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_MOVI: {
			reg0 = getRegister(op.args[0]);
			reg0.set(DataUtil.subArray(op.args, 1, 2));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		}
		case INST_MOV:
			getRegister(op.args[0]).set(getRegister(op.args[1]));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_POP:
			getRegister(op.args[0]).set(pop());
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_PUSH:
			push(getRegister(op.args[0]));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_CALL:
			push(register[PROGRAM_COUNTER]);
			// register[PROGRAM_COUNTER].set(getRegister(op.args[0]).intValue() *
			// CODE_LENGTH);
			register[PROGRAM_COUNTER].set(getRegister(op.args[0]).intValue());
			break;
		case INST_CALLI:
			push(register[PROGRAM_COUNTER]);
			// register[PROGRAM_COUNTER].set(DataUtil.toChar(op.args[0], op.args[1]) *
			// CODE_LENGTH);
			register[PROGRAM_COUNTER].set(DataUtil.toChar(op.args[0], op.args[1]));
			break;
		case INST_RET:
			register[PROGRAM_COUNTER].set(pop().intValue() + CODE_LENGTH);
			break;
		case INST_LDA: {
			int fp = register[FRAME_POINTER].intValue();
			int offset = DataUtil.toSiginedInteger(DataUtil.toChar(DataUtil.subArray(
					op.args, 1, 2), 0, 2));
			getRegister(op.args[0]).set(memory.read(fp + offset, 4));
		}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_STA: {
			int fp = register[FRAME_POINTER].intValue();
			int offset = DataUtil.toSiginedInteger(DataUtil.toChar(DataUtil.subArray(
					op.args, 1, 2), 0, 2));
			memory.write(fp + offset, 4, getRegister(op.args[0]).byteArray());
		}
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			break;
		case INST_SIN:
			reg0 = getRegister(op.args[0]);
			reg1 = getRegister(op.args[1]);
			getRegister(op.args[0]).set((int) Math.sin(reg1.intValue()));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(SIN_COS_CYCLE);
			break;
		case INST_COS:
			reg0 = getRegister(op.args[0]);
			reg1 = getRegister(op.args[1]);
			getRegister(op.args[0]).set((int) Math.cos(reg1.intValue()));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(SIN_COS_CYCLE);
			break;
		case INST_SINF:
			reg0 = getRegister(op.args[0]);
			reg1 = getRegister(op.args[1]);
			((FloatRegister) getRegister(op.args[0])).set((float) Math.sin(reg1
					.floatValue()));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(SIN_COS_CYCLE);
			break;
		case INST_COSF:
			reg0 = getRegister(op.args[0]);
			reg1 = getRegister(op.args[1]);
			((FloatRegister) getRegister(op.args[0])).set((float) Math.sin(reg1
					.floatValue()));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(SIN_COS_CYCLE);
			break;
		case INST_SQRT:
			reg0 = getRegister(op.args[0]);
			reg1 = getRegister(op.args[1]);
			getRegister(op.args[0]).set((int) (Math.sqrt(reg1.intValue())));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(SQRT_CYCLE);
			break;
		case INST_SQRTF:
			reg0 = getRegister(op.args[0]);
			reg1 = getRegister(op.args[1]);
			((FloatRegister) getRegister(op.args[0])).set((float) Math.sqrt(reg1
					.floatValue()));
			register[PROGRAM_COUNTER].add(CODE_LENGTH);
			info.setCycle(SQRT_CYCLE);
			break;
		default:
			throw new ExecutableElementException("unknow op code "
					+ String.valueOf(op.inst));
		}
		return info;
	}

	private void storeData(byte[] data, int addr, int length)
			throws ExecutableElementException {
		if (addr < memory.size()) {
			memory.write(addr, length, data);
		} else {
			ChannelManager.Element c = channelManager.search(addr);
			try {
				c.getChannel().writeRequest(
						this,
						RandomAccessMemoryDataPacket.writePacket(addr - c.offset, length,
								8, data));
			} catch (MicsException e) {
				throw new ExecutableElementException(e);
			}
		}
	}

	private void loadData(int dest, int addr, int length)
			throws ExecutableElementException {
		if (addr < memory.size()) {
			getRegister(dest).set(memory.read(addr, length));
			// System.out.println(DataUtil.toBigEndianValueString(memory.read(addr,
			// length)));
		} else {
			readRequestRegID = dest;
			ChannelManager.Element c = channelManager.search(addr);
			try {
				c.getChannel()
						.readRequest(
								this,
								RandomAccessMemoryDataPacket.readPacket(addr - c.offset,
										length, 8));
			} catch (MicsException e) {
				throw new ExecutableElementException(e);
			}
		}
	}

	private Register pop() throws ExecutableElementException {
		if (register[STACK_POINTER].intValue() < memory.size() - 1) {
			Register data = new IntRegister();
			data.set(memory.read(register[STACK_POINTER].intValue(), 4));
			register[STACK_POINTER].add(4);
			return data;
		} else {
			throw new ExecutableElementException("stack under flow");
		}
	}

	private void push(Register v) throws ExecutableElementException {
		if (register[STACK_POINTER].intValue() > 4) {
			register[STACK_POINTER].add(-4);
			memory.write(register[STACK_POINTER].intValue(), 4, v.byteArray());
		} else {
			System.out.println("stack pointer " + register[STACK_POINTER]);
			throw new ExecutableElementException("stack over flow");
		}
	}

	private OperationCode readProgram(int addr) {
		byte[] data = memory.read(addr + programOffset, 4);
		return new OperationCode(data[0], data[1], data[2], data[3]);
	}

	public void writeback(Channel src, MicsDataPacket data) {
		if (data instanceof RandomAccessMemoryDataPacket) {
			RandomAccessMemoryDataPacket rd = (RandomAccessMemoryDataPacket) data;
			switch (rd.length * rd.width / 8) {
			case 1:
				register[readRequestRegID].set(rd.data);
				break;
			case 2:
				register[readRequestRegID].set(rd.data);
				break;
			case 4:
				register[readRequestRegID].set(rd.data);
				break;
			}
		} else if (data instanceof InterruptDataPacket) {
			InterruptDataPacket d = (InterruptDataPacket) data;
			// intRegister[d.addr].set(d.data);
			try {
				storeData(d.data, 0x0a000002, d.data.length);
			} catch (ExecutableElementException e) {

			}
		}
	}

	public int getChannelOffset(Channel c) {
		return channelManager.search(c);
	}

	public String getInfo() {
		String s = "";
		s += "SimpleProcessor32\n";
		s += "  CLASS: " + this.getClass().getName() + "\n";
		s += "  ID: " + id() + "\n";
		s += "  Local Memory ID: " + ((MicsElement) memory).id() + ",";
		s += " offset = "
				+ DataUtil.toBigEndianValueString(DataUtil.toByteArray(programOffset))
				+ "\n";
		ChannelManager.Element[] channels = channelManager.array();
		for (int i = 0; i < channels.length; i++) {
			ChannelManager.Element element = channels[i];
			s += "  Channel ID: " + element.id + ",";
			s += " offset = " + element.offset + "\n";
		}
		return s;
	}

	public String toString() {
		try {
			String str = "";
			try {
				str += register[PROGRAM_COUNTER] + ":";
			} catch (FormatException e) {
				e.printStackTrace();
			}
			str += readProgram(register[PROGRAM_COUNTER].intValue()).toString();
			str += "\n";
			for (int i = 0; i < REGISTER_SIZE; i++) {
				try {
					str += register[i];
					if (i % 8 == 7) {
						str += "\n";
					} else {
						str += " ";
					}
				} catch (FormatException e) {

				}
			}
			return str;
		} catch (ArrayIndexOutOfBoundsException e) {
			return new String(register[PROGRAM_COUNTER].intValue()
					+ ": illegal program counter");
		}
	}

	public String[] getConnectedElements() {
		return channelManager.getConnectedElements();
	}

	MicsTablePanel table;

	public void show() {
		if (table == null) {
			String[] label = new String[REGISTER_SIZE];
			String[] flabel = new String[REGISTER_SIZE];
			for (int i = 0; i < REGISTER_SIZE; i++) {
				label[i] = String.valueOf(i);
			}
			for (int i = 0; i < REGISTER_SIZE; i++) {
				flabel[i] = String.valueOf(i + REGISTER_SIZE);
			}
			table = new MicsTablePanel("SimpleProcessor32 Register",
					new MicsTableModel(new String[] { "name", "value", "name", "value" },
							new Object[][] { label, register, flabel, fregister }));
		}
		table.setVisible(true);
	}

	public String getImagePath() {
		return "combo_processor.png";
	}


	public String getDescription(){
		return "TODO";
	}
}
