package net.wasamon.mics.processor.mips;

public class MipsOperation {

    //Opcode

    public static final int OP_SPECIAL = 0x00;

    public static final int OP_BCOND = 0x01;

    public static final int OP_J = 0x02;

    public static final int OP_JAL = 0x03;

    public static final int OP_BEQ = 0x04;

    public static final int OP_BNE = 0x05;

    public static final int OP_BLEZ = 0x06;

    public static final int OP_BGTZ = 0x07;

    public static final int OP_ADDI = 0x08;

    public static final int OP_ADDIU = 0x09;

    public static final int OP_SLTI = 0x0a;

    public static final int OP_SLTIU = 0x0b;

    public static final int OP_ANDI = 0x0c;

    public static final int OP_ORI = 0x0d;

    public static final int OP_XORI = 0x0e;

    public static final int OP_LUI = 0x0f;

    public static final int OP_COP0 = 0x10;

    public static final int OP_COP1 = 0x11;

    public static final int OP_COP2 = 0x12;

    public static final int OP_COP3 = 0x13;

    public static final int OP_LB = 0x20;

    public static final int OP_LH = 0x21;

    public static final int OP_LWL = 0x22;

    public static final int OP_LW = 0x23;

    public static final int OP_LBU = 0x24;

    public static final int OP_LHU = 0x25;

    public static final int OP_LWR = 0x26;

    public static final int OP_SB = 0x28;

    public static final int OP_SH = 0x29;

    public static final int OP_SWL = 0x2a;

    public static final int OP_SW = 0x2b;

    public static final int OP_SWR = 0x2e;

    public static final int OP_LWC0 = 0x30;

    public static final int OP_LWC1 = 0x31;

    public static final int OP_LWC2 = 0x32;

    public static final int OP_LWC3 = 0x33;

    public static final int OP_SWC0 = 0x38;

    public static final int OP_SWC1 = 0x39;

    public static final int OP_SWC2 = 0x3a;

    public static final int OP_SWC3 = 0x3b;

    //Format

    public static final int FMT_S = 0x00;

    public static final int FMT_D = 0x01;

    public static final int FMT_W = 0x04;


    //SPECIAL Function

    public static final int SPECIAL_FUNCT_SLL = 0x00;

    public static final int SPECIAL_FUNCT_SRL = 0x02;

    public static final int SPECIAL_FUNCT_SRA = 0x03;

    public static final int SPECIAL_FUNCT_SLLV = 0x04;

    public static final int SPECIAL_FUNCT_SRLV = 0x06;

    public static final int SPECIAL_FUNCT_SRAV = 0x07;

    public static final int SPECIAL_FUNCT_JR = 0x08;

    public static final int SPECIAL_FUNCT_JALR = 0x09;

    public static final int SPECIAL_FUNCT_SYSCALL = 0x0c;

    public static final int SPECIAL_FUNCT_BREAK = 0x0d;

    public static final int SPECIAL_FUNCT_MFHI = 0x10;

    public static final int SPECIAL_FUNCT_MTHI = 0x11;

    public static final int SPECIAL_FUNCT_MFLO = 0x12;

    public static final int SPECIAL_FUNCT_MTLO = 0x13;

    public static final int SPECIAL_FUNCT_MULT = 0x18;

    public static final int SPECIAL_FUNCT_MULTU = 0x19;

    public static final int SPECIAL_FUNCT_DIV = 0x1a;

    public static final int SPECIAL_FUNCT_DIVU = 0x1b;

    public static final int SPECIAL_FUNCT_ADD = 0x20;

    public static final int SPECIAL_FUNCT_ADDU = 0x21;

    public static final int SPECIAL_FUNCT_SUB = 0x22;

    public static final int SPECIAL_FUNCT_SUBU = 0x23;

    public static final int SPECIAL_FUNCT_AND = 0x24;

    public static final int SPECIAL_FUNCT_OR = 0x25;

    public static final int SPECIAL_FUNCT_XOR = 0x26;

    public static final int SPECIAL_FUNCT_NOR = 0x27;

    public static final int SPECIAL_FUNCT_SLT = 0x2a;

    public static final int SPECIAL_FUNCT_SLTU = 0x2b;

    //BCOND

    public static final int COND_BLTZ = 0x00;

    public static final int COND_BGEZ = 0x01;

    public static final int COND_BLTZAL = 0x10;

    public static final int COND_BGEZAL = 0x11;

    //COPz RS

    public static final int COPZ_RS_MF = 0x00;

    public static final int COPZ_RS_CF = 0x02;

    public static final int COPZ_RS_MT = 0x04;

    public static final int COPZ_RS_CT = 0x06;

    public static final int COPZ_RS_BC = 0x08;

    //COPz RS MSB

    public static final int COPZ_RS_MSB_CO = 0x1;

    //COPz RT

    public static final int COPZ_RT_BCF = 0x00;

    public static final int COPZ_RT_BCT = 0x01;

    //COP0 Function

    public static final int COP0_FUNCT_TLBR = 0x01;

    public static final int COP0_FUNCT_TLBWI = 0x02;

    public static final int COP0_FUNCT_TLBWR = 0x06;

    public static final int COP0_FUNCT_TLBP = 0x08;

    public static final int COP0_FUNCT_RFE = 0x10;

    //COP1 Function

    public static final int COP1_FUNCT_ADD_FMT = 0x00;

    public static final int COP1_FUNCT_SUB_FMT = 0x01;

    public static final int COP1_FUNCT_MUL_FMT = 0x02;

    public static final int COP1_FUNCT_DIV_FMT = 0x03;

    public static final int COP1_FUNCT_ABS_FMT = 0x05;

    public static final int COP1_FUNCT_MOV_FMT = 0x06;

    public static final int COP1_FUNCT_NEG_FMT = 0x07;

    public static final int COP1_FUNCT_CVT_S = 0x20;

    public static final int COP1_FUNCT_CVT_D = 0x21;

    public static final int COP1_FUNCT_CVT_W = 0x24;

    public static final int COP1_FUNCT_C_F = 0x30;

    public static final int COP1_FUNCT_C_UN = 0x31;

    public static final int COP1_FUNCT_C_EQ = 0x32;

    public static final int COP1_FUNCT_C_UEQ = 0x33;

    public static final int COP1_FUNCT_C_OLT = 0x34;

    public static final int COP1_FUNCT_C_ULT = 0x35;

    public static final int COP1_FUNCT_C_OLE = 0x36;

    public static final int COP1_FUNCT_C_ULE = 0x37;

    public static final int COP1_FUNCT_C_SF = 0x38;

    public static final int COP1_FUNCT_C_NGLE = 0x39;

    public static final int COP1_FUNCT_C_SEQ = 0x3a;

    public static final int COP1_FUNCT_C_NGL = 0x3b;

    public static final int COP1_FUNCT_C_LT = 0x3c;

    public static final int COP1_FUNCT_C_NGE = 0x3d;

    public static final int COP1_FUNCT_C_LE = 0x3e;

    public static final int COP1_FUNCT_C_NGT = 0x3f;

    //Instruction

    public static final int INST_ADD = 0;

    public static final int INST_ADDI = 1;

    public static final int INST_ADDIU = 2;

    public static final int INST_ADDU = 3;

    public static final int INST_AND = 4;

    public static final int INST_ANDI = 5;

    public static final int INST_BC0F = 6;

    public static final int INST_BC1F = 7;

    public static final int INST_BC2F = 8;

    public static final int INST_BC3F = 9;

    public static final int INST_BC0T = 10;

    public static final int INST_BC1T = 11;

    public static final int INST_BC2T = 12;

    public static final int INST_BC3T = 13;

    public static final int INST_BEQ = 14;

    public static final int INST_BGEZ = 15;

    public static final int INST_BGEZAL = 16;

    public static final int INST_BGTZ = 17;

    public static final int INST_BLEZ = 18;

    public static final int INST_BLTZ = 19;

    public static final int INST_BLTZAL = 20;

    public static final int INST_BNE = 21;

    public static final int INST_BREAK = 22;

    public static final int INST_CFC0 = 23;

    public static final int INST_CFC1 = 24;

    public static final int INST_CFC2 = 25;

    public static final int INST_CFC3 = 26;

    public static final int INST_CTC0 = 27;

    public static final int INST_CTC1 = 28;

    public static final int INST_CTC2 = 29;

    public static final int INST_CTC3 = 30;

    public static final int INST_DIV = 31;

    public static final int INST_DIVU = 32;

    public static final int INST_J = 33;

    public static final int INST_JAL = 34;

    public static final int INST_JALR = 35;

    public static final int INST_JR = 36;

    public static final int INST_LB = 37;

    public static final int INST_LBU = 38;

    public static final int INST_LH = 39;

    public static final int INST_LHU = 40;

    public static final int INST_LUI = 41;

    public static final int INST_LW = 42;

    public static final int INST_LWC0 = 43;

    public static final int INST_LWC1 = 44;

    public static final int INST_LWC2 = 45;

    public static final int INST_LWC3 = 46;

    public static final int INST_LWL = 47;

    public static final int INST_LWR = 48;

    public static final int INST_MFC0 = 49;

    public static final int INST_MFC1 = 50;

    public static final int INST_MFC2 = 51;

    public static final int INST_MFC3 = 52;

    public static final int INST_MFHI = 53;

    public static final int INST_MFLO = 54;

    public static final int INST_MTC0 = 55;

    public static final int INST_MTC1 = 56;

    public static final int INST_MTC2 = 57;

    public static final int INST_MTC3 = 58;

    public static final int INST_MTHI = 59;

    public static final int INST_MTLO = 60;

    public static final int INST_MULT = 61;

    public static final int INST_MULTU = 62;

    public static final int INST_NOR = 63;

    public static final int INST_OR = 64;

    public static final int INST_ORI = 65;

    public static final int INST_RFE = 66;

    public static final int INST_SB = 67;

    public static final int INST_SH = 68;

    public static final int INST_SLL = 69;

    public static final int INST_SLLV = 70;

    public static final int INST_SLT = 71;

    public static final int INST_SLTI = 72;

    public static final int INST_SLTIU = 73;

    public static final int INST_SLTU = 74;

    public static final int INST_SRA = 75;

    public static final int INST_SRAV = 76;

    public static final int INST_SRL = 77;

    public static final int INST_SRLV = 78;

    public static final int INST_SUB = 79;

    public static final int INST_SUBU = 80;

    public static final int INST_SW = 81;

    public static final int INST_SWC0 = 82;

    public static final int INST_SWC1 = 83;

    public static final int INST_SWC2 = 84;

    public static final int INST_SWC3 = 85;

    public static final int INST_SWL = 86;

    public static final int INST_SWR = 87;

    public static final int INST_SYSCALL = 88;

    public static final int INST_TLBP = 89;

    public static final int INST_TLBR = 90;

    public static final int INST_TLBWI = 91;

    public static final int INST_TLBWR = 92;

    public static final int INST_XOR = 93;

    public static final int INST_XORI = 94;

    //COP1 Instruction

    public static final int INST_ABS_S = 95;

    public static final int INST_ABS_D = 96;

    public static final int INST_ADD_S = 97;

    public static final int INST_ADD_D = 98;

    public static final int INST_C_F_S = 99;

    public static final int INST_C_F_D = 100;

    public static final int INST_C_UN_S = 101;

    public static final int INST_C_UN_D = 102;

    public static final int INST_C_EQ_S = 103;

    public static final int INST_C_EQ_D = 104;

    public static final int INST_C_UEQ_S = 105;

    public static final int INST_C_UEQ_D = 106;

    public static final int INST_C_OLT_S = 107;

    public static final int INST_C_OLT_D = 108;

    public static final int INST_C_ULT_S = 109;

    public static final int INST_C_ULT_D = 110;

    public static final int INST_C_OLE_S = 111;

    public static final int INST_C_OLE_D = 112;

    public static final int INST_C_ULE_S = 113;

    public static final int INST_C_ULE_D = 114;

    public static final int INST_C_SF_S = 115;

    public static final int INST_C_SF_D = 116;

    public static final int INST_C_NGLE_S = 117;

    public static final int INST_C_NGLE_D = 118;

    public static final int INST_C_SEQ_S = 119;

    public static final int INST_C_SEQ_D = 120;

    public static final int INST_C_NGL_S = 121;

    public static final int INST_C_NGL_D = 122;

    public static final int INST_C_LT_S = 123;

    public static final int INST_C_LT_D = 124;

    public static final int INST_C_NGE_S = 125;

    public static final int INST_C_NGE_D = 126;

    public static final int INST_C_LE_S = 127;

    public static final int INST_C_LE_D = 128;

    public static final int INST_C_NGT_S = 129;

    public static final int INST_C_NGT_D = 130;

    public static final int INST_CVT_D_S = 131;

    public static final int INST_CVT_D_W = 132;

    public static final int INST_CVT_S_D = 133;

    public static final int INST_CVT_S_W = 134;

    public static final int INST_CVT_W_S = 135;

    public static final int INST_CVT_W_D = 136;

    public static final int INST_DIV_S = 137;

    public static final int INST_DIV_D = 138;

    public static final int INST_MOV_S = 139;

    public static final int INST_MOV_D = 140;

    public static final int INST_MUL_S = 141;

    public static final int INST_MUL_D = 142;

    public static final int INST_NEG_S = 143;

    public static final int INST_NEG_D = 144;

    public static final int INST_SUB_S = 145;

    public static final int INST_SUB_D = 146;

    private static final String[] instruction = {
	"add", "addi", "addiu", "addu", "and", "andi", "bc0f", "bc1f", "bc2f", "bc3f",
	"bc0t", "bc1t", "bc2t", "bc3t", "beq", "bgez", "bgezal", "bgtz", "blez", "bltz",
	"bltzal", "bne", "break", "cfc0", "cfc1", "cfc2", "cfc3", "ctc0", "ctc1", "ctc2", 
	"ctc3", "div", "divu", "j", "jal", "jalr", "jr", "lb", "lbu", "lh", 
	"lhu", "lui", "lw", "lwc0", "lwc1", "lwc2", "lwc3", "lwl", "lwr", "mfc0",
	"mfc1", "mfc2", "mfc3", "mfhi", "mflo", "mtc0",	"mtc1", "mtc2", "mtc3", "mthi",
	"mtl0", "mult", "multu", "nor", "or", "ori", "rfe", "sb", "sh", "sll",
	"sllv", "slt", "slti", "sltiu", "sltu", "sra", "srav", "srl", "srlv", "sub",
	"subu", "sw", "swc0", "swc1", "swc2", "swc3", "swl", "swr", "syscall", "tlbp",
	"tlbr", "tlbwi", "tlbwr", "xor", "xori", "abs.s", "abs.d", "add.s", "add.d", "c.f.s",
	"c.f.d", "c.un.s", "c.un.d", "c.eq.s", "c.eq.d", "c.ueq.s", "c.ueq.d", "c.olt.s", "c.olt.d", "c.ult.s",
	"c.ult.d", "c.ole.s", "c.ole.d", "c.ule.s", "c.ule.d", "c.sf.s", "c.sf.d", "c.ngle.s", "c.ngle.d", "c.seq.s",
	"c.seq.d", "c.ngl.s", "c.ngl.d", "c.lt.s", "c.lt.d", "c.nge.s", "c.nge.d", "c.le.s", "c.le.d", "c.ngt.s",
	"c.ngt.d", "cvt.d.s", "cvt.d.w", "cvt.s.d", "cvt.s.w", "cvt.w.s", "cvt.w.d", "div.s", "div.d", "mov.s",
	"mov.d", "mul.s", "mul.d", "neg.s", "neg.d", "sub.s", "sub.d"
    };

    private static final int I_TYPE_3000 = 0;

    private static final int J_TYPE_3000 = 1;

    private static final int R_TYPE_3000 = 2;

    private static final int I_TYPE_3010 = 3;

    private static final int M_TYPE_3010 = 4;

    private static final int R_TYPE_3010 = 5;

    private static final int GPR = 0;

    private static final int FGR = 1;

    private static final int FPR = 2;

    private int operation, inst, type;

    MipsOperation(int operation) {
	this.operation = operation;
	decode();
    }

    private void decode() {
	int op = getZeroExtendedOP();
	int funct, rs, rt;

	switch (op) {
	case OP_SPECIAL:
	    funct = getZeroExtendedFunct();
	    type = R_TYPE_3000;
	    switch (funct) {
	    case SPECIAL_FUNCT_SLL:
		inst = INST_SLL;
		break;

	    case SPECIAL_FUNCT_SRL:
		inst = INST_SRL;
		break;

	    case SPECIAL_FUNCT_SRA:
		inst = INST_SRA;
		break;
		    
	    case SPECIAL_FUNCT_SLLV:
		inst = INST_SLLV;
		break;

	    case SPECIAL_FUNCT_SRLV:
		inst = INST_SRLV;
		break;

	    case SPECIAL_FUNCT_SRAV:
		inst = INST_SRAV;
		break;

	    case SPECIAL_FUNCT_JR:
		inst = INST_JR;
		break;

	    case SPECIAL_FUNCT_JALR:
		inst = INST_JALR;
		break;

	    case SPECIAL_FUNCT_SYSCALL:
		inst = INST_SYSCALL;
		break;

	    case SPECIAL_FUNCT_BREAK:
		inst = INST_BREAK;
		break;

	    case SPECIAL_FUNCT_MFHI:
		inst = INST_MFHI;
		break;

	    case SPECIAL_FUNCT_MTHI:
		inst = INST_MTHI;
		break;

	    case SPECIAL_FUNCT_MFLO:
		inst = INST_MFLO;
		break;

	    case SPECIAL_FUNCT_MTLO:
		inst = INST_MTLO;
		break;

	    case SPECIAL_FUNCT_MULT:
		inst = INST_MULT;
		break;

	    case SPECIAL_FUNCT_MULTU:
		inst = INST_MULTU;
		break;

	    case SPECIAL_FUNCT_DIV:
		inst = INST_DIV;
		break;

	    case SPECIAL_FUNCT_DIVU:
		inst = INST_DIVU;
		break;

	    case SPECIAL_FUNCT_ADD:
		inst = INST_ADD;
		break;

	    case SPECIAL_FUNCT_ADDU:
		inst = INST_ADDU;
		break;

	    case SPECIAL_FUNCT_SUB:
		inst = INST_SUB;
		break;

	    case SPECIAL_FUNCT_SUBU:
		inst = INST_SUBU;
		break;

	    case SPECIAL_FUNCT_AND:
		inst = INST_AND;
		break;

	    case SPECIAL_FUNCT_OR:
		inst = INST_OR;
		break;

	    case SPECIAL_FUNCT_XOR:
		inst = INST_XOR;
		break;

	    case SPECIAL_FUNCT_NOR:
		inst = INST_NOR;
		break;

	    case SPECIAL_FUNCT_SLT:
		inst = INST_SLT;
		break;

	    case SPECIAL_FUNCT_SLTU:
		inst = INST_SLTU;
		break;

	    default:
		//throw new ExecutableElementException("unknow special function code " + String.valueOf(funct));
	    }
	    break;

	case OP_BCOND:
	    rt = getZeroExtendedRT();
	    type = I_TYPE_3000;
	    switch (rt) {
	    case COND_BLTZ:
		inst = INST_BLTZ;
		break;
		
	    case COND_BGEZ:
		inst = INST_BGEZ;
		break;
		
	    case COND_BLTZAL:
		inst = INST_BLTZAL;
		break;
		
	    case COND_BGEZAL:
		inst = INST_BGEZAL;
		break;
		
	    default:
		inst = -1;
		//throw new ExecutableElementException("unknow cond rs code " + String.valueOf(rt));
	    }
	    break;

	case OP_COP0:
	    funct = getZeroExtendedFunct();
	    type = R_TYPE_3000;
	    switch (funct) {
	    case COP0_FUNCT_TLBR:
		inst = INST_TLBR;
		break;
		
	    case COP0_FUNCT_TLBWI:
		inst = INST_TLBWI;
		break;
		
	    case COP0_FUNCT_TLBWR:
		inst = INST_TLBWR;
		break;
		
	    case COP0_FUNCT_TLBP:
		inst = INST_TLBP;
		break;
		
	    case COP0_FUNCT_RFE:
		inst = INST_RFE;
		break;

	    default:
		inst = -1;
		//throw new ExecutableElementException("unknow cop0 function code " + String.valueOf(funct));
	    }
	    break;

	case OP_COP1:
	    int msbofrs = getMSBofRS();
	    switch (msbofrs) {
	    case COPZ_RS_MSB_CO:
		funct = getZeroExtendedFunct();
		int format = getZeroExtendedFormat();
		type = R_TYPE_3010;

		switch(funct){
		case COP1_FUNCT_CVT_S:
		    switch(format){
		    case FMT_D:
			inst = INST_CVT_S_D;
			break;
			
		    case FMT_W:
			inst = INST_CVT_S_W;
			break;
			
		    default:
			//bad format exception
			inst = -1;
			break;
		    }
		    break;
		    
		case COP1_FUNCT_CVT_D:
		    switch(format){
		    case FMT_S:
			inst = INST_CVT_D_S;
			break;

		    case FMT_W:
			inst = INST_CVT_D_W;
			break;

		    default:
			//bad format exception
			inst = -1;
			break;
		    }
		    break;
		    
		case COP1_FUNCT_CVT_W:
		    switch(format){
		    case FMT_S:
			inst = INST_CVT_W_S;
			break;

		    case FMT_D:
			inst = INST_CVT_W_D;
			break;

		    default:
			//bad format exception
			inst = -1;
			break;
		    }
		    break;

		default:
		    if(format != FMT_S && format != FMT_D){
			//bad format exception
			inst = -1;
			break;
		    }
		    switch(funct){
		    case COP1_FUNCT_ADD_FMT:
			inst = INST_ADD_S + format;
			break;
			
		    case COP1_FUNCT_SUB_FMT:
			inst = INST_SUB_S + format;
			break;
			
		    case COP1_FUNCT_MUL_FMT:
			inst = INST_MUL_S + format;
			break;
			
		    case COP1_FUNCT_DIV_FMT:
			inst = INST_DIV_S + format;
			break;
			
		    case COP1_FUNCT_ABS_FMT:
			inst = INST_ABS_S + format;
			break;
			
		    case COP1_FUNCT_MOV_FMT:
			inst = INST_MOV_S + format;
			break;
			
		    case COP1_FUNCT_NEG_FMT:
			inst = INST_NEG_S + format;
			break;
			
		    case COP1_FUNCT_C_F:
			inst = INST_C_F_S + format;
			break;
			
		    case COP1_FUNCT_C_UN:
			inst = INST_C_UN_S + format;
			break;
			
		    case COP1_FUNCT_C_EQ:
			inst = INST_C_EQ_S + format;
			break;
			
		    case COP1_FUNCT_C_UEQ:
			inst = INST_C_UEQ_S + format;
			break;
			
		    case COP1_FUNCT_C_OLT:
			inst = INST_C_OLT_S + format;
			break;
			
		    case COP1_FUNCT_C_ULT:
			inst = INST_C_ULT_S + format;
			break;
			
		    case COP1_FUNCT_C_OLE:
			inst = INST_C_OLE_S + format;
			break;
			
		    case COP1_FUNCT_C_ULE:
			inst = INST_C_ULE_S + format;
			break;
			
		    case COP1_FUNCT_C_SF:
			inst = INST_C_SF_S + format;
			break;
			
		    case COP1_FUNCT_C_NGLE:
			inst = INST_C_NGLE_S + format;
			break;
			
		    case COP1_FUNCT_C_SEQ:
			inst = INST_C_SEQ_S + format;
			break;
			
		    case COP1_FUNCT_C_NGL:
			inst = INST_C_NGL_S + format;
			break;
			
		    case COP1_FUNCT_C_LT:
			inst = INST_C_LT_S + format;
			break;
			
		    case COP1_FUNCT_C_NGE:
			inst = INST_C_NGE_S + format;
			break;
			
		    case COP1_FUNCT_C_LE:
			inst = INST_C_LE_S + format;
			break;
			
		    case COP1_FUNCT_C_NGT:
			inst = INST_C_NGT_S + format;
			break;
			
		    default:
			inst = -1;
			//throw new ExecutableElementException("unknow cop1 function code " + String.valueOf(funct));
		    }
		}
		break;

	    default:
		rs = getZeroExtendedRS();
		switch (rs) {
		case COPZ_RS_MF:
		    type = M_TYPE_3010;
		    inst = INST_MFC1;
		    break;
		    
		case COPZ_RS_CF:
		    type = M_TYPE_3010;
		    inst = INST_CFC1;
		    break;
		    
		case COPZ_RS_MT:
		    type = M_TYPE_3010;
		    inst = INST_MTC1;
		    break;
		    
		case COPZ_RS_CT:
		    type = M_TYPE_3010;
		    inst = INST_CTC1;
		    break;
		    
		default:
		    if((rs & 0x00000008) == 0x00000000){
			inst = -1;
			break;
			//throw new ExecutableElementException("unknow cop1 rs code " + String.valueOf(funct));
		    }
		    type = I_TYPE_3010;
		    rt = getZeroExtendedRT();
		    switch (rt) {
		    case COPZ_RT_BCF:
			inst = INST_BC1F;
			break;
			
		    case COPZ_RT_BCT:
			inst = INST_BC1T;
			break;
			
		    default:
			inst = -1;
			//throw new ExecutableElementException("unknow cop1 rt code " + String.valueOf(funct));
		    }
		}
	    }
	    break;

	    ////////////////////////////
	case OP_COP2:
	case OP_COP3:
	    inst = -1;
	    break;
	    ////////////////////////////

	case OP_J:
	    inst = INST_J;
	    type = J_TYPE_3000;
	    break;

	case OP_JAL:
	    inst = INST_JAL;
	    type = J_TYPE_3000;
	    break;

	case OP_BEQ:
	    inst = INST_BEQ;
	    type = I_TYPE_3000;
	    break;

	case OP_BNE:
	    inst = INST_BNE;
	    type = I_TYPE_3000;
	    break;

	case OP_BLEZ:
	    inst = INST_BLEZ;
	    type = I_TYPE_3000;
	    break;

	case OP_BGTZ:
	    inst = INST_BGTZ;
	    type = I_TYPE_3000;
	    break;

	case OP_ADDI:
	    inst = INST_ADDI;
	    type = I_TYPE_3000;
	    break;

	case OP_ADDIU:
	    inst = INST_ADDIU;
	    type = I_TYPE_3000;
	    break;

	case OP_SLTI:
	    inst = INST_SLTI;
	    type = I_TYPE_3000;
	    break;

	case OP_SLTIU:
	    inst = INST_SLTIU;
	    type = I_TYPE_3000;
	    break;

	case OP_ANDI:
	    inst = INST_ANDI;
	    type = I_TYPE_3000;
	    break;

	case OP_ORI:
	    inst = INST_ORI;
	    type = I_TYPE_3000;
	    break;

	case OP_XORI:
	    inst = INST_XORI;
	    type = I_TYPE_3000;
	    break;

	case OP_LUI:
	    inst = INST_LUI;
	    type = I_TYPE_3000;
	    break;

	case OP_LB:
	    inst = INST_LB;
	    type = I_TYPE_3000;
	    break;

	case OP_LH:
	    inst = INST_LH;
	    type = I_TYPE_3000;
	    break;

	case OP_LWL:
	    inst = INST_LWL;
	    type = I_TYPE_3000;
	    break;

	case OP_LW:
	    inst = INST_LW;
	    type = I_TYPE_3000;
	    break;

	case OP_LBU:
	    inst = INST_LBU;
	    type = I_TYPE_3000;
	    break;

	case OP_LHU:
	    inst = INST_LHU;
	    type = I_TYPE_3000;
	    break;

	case OP_LWR:
	    inst = INST_LWR;
	    type = I_TYPE_3000;
	    break;

	case OP_SB:
	    inst = INST_SB;
	    type = I_TYPE_3000;
	    break;

	case OP_SH:
	    inst = INST_SH;
	    type = I_TYPE_3000;
	    break;

	case OP_SWL:
	    inst = INST_SWL;
	    type = I_TYPE_3000;
	    break;

	case OP_SW:
	    inst = INST_SW;
	    type = I_TYPE_3000;
	    break;

	case OP_LWC0:
	    inst = INST_LWC0;
	    type = I_TYPE_3000;
	    break;

	case OP_LWC1:
	    inst = INST_LWC1;
	    type = I_TYPE_3000;
	    break;

	case OP_LWC2:
	    inst = INST_LWC2;
	    type = I_TYPE_3000;
	    break;

	case OP_LWC3:
	    inst = INST_LWC3;
	    type = I_TYPE_3000;
	    break;

	case OP_SWC0:
	    inst = INST_SWC0;
	    type = I_TYPE_3000;
	    break;

	case OP_SWC1:
	    inst = INST_SWC1;
	    type = I_TYPE_3000;
	    break;

	case OP_SWC2:
	    inst = INST_SWC2;
	    type = I_TYPE_3000;
	    break;

	case OP_SWC3:
	    inst = INST_SWC3;
	    type = I_TYPE_3000;
	    break;

	default:
	    inst = -1;
	    //throw new ExecutableElementException("unknow op code " + String.valueOf(op));
	}
    }

    public int getInstruction(){
	return inst;
    }

    public int getType(){
	return type;
    }

    public int getZeroExtendedOP(){
	return (operation & 0xfc000000) >>> 26;
    }

    public int getMSBofRS(){
	return (operation & 0x02000000) >>> 25;
    }

    public int getZeroExtendedFormat(){
	return (operation & 0x01e00000) >>> 21;
    }

    public int getZeroExtendedRS(){
	return (operation & 0x03e00000) >>> 21;
    }

    public int getSignExtendedRS(){
	if((operation & 0x02000000) == 0x02000000){
	    return ((operation & 0x03e00000) | 0xFC000000) >> 21;
	}
	else{
	    return (operation & 0x03e00000) >> 21;
	}
    }

    public int getZeroExtendedRT(){
	return (operation & 0x001f0000) >>> 16;
    }

    public int getSignExtendedRT(){
	if((operation & 0x00100000) == 0x00100000){
	    return ((operation & 0x001f0000) | 0xffe00000) >> 16;
	}
	else{
	    return (operation & 0x001f0000) >> 16;
	}
    }

    public int getZeroExtendedFT(){
	return (operation & 0x001f0000) >>> 16;
    }

    public int getZeroExtendedImmediate(){
	return operation & 0x0000ffff;
    }

    public int getSignExtendedImmediate(){
	if((operation & 0x00008000) == 0x00008000){
	    return (operation & 0x0000ffff) | 0xffff0000;
	}
	else{
	    return operation & 0x0000ffff;
	}
    }

    public int getZeroExtendedTarget(){
	return operation & 0x03ffffff;
    }

    public int getSignExtendedTarget(){
	if((operation & 0x02000000) == 0x02000000){
	    return (operation & 0x03ffffff) | 0xfc000000;
	}
	else{
	    return operation & 0x03ffffff;
	}
    }

    public int getZeroExtendedRD(){
	return (operation & 0x0000f800) >>> 11;
    }

    public int getSignExtendedRD(){
	if((operation & 0x00008000) == 0x00008000){
	    return ((operation & 0x0000f800) | 0xffff0000) >> 11;
	}
	else{
	    return (operation & 0x0000f800) >> 11;
	}
    }

    public int getZeroExtendedFS(){
	return (operation & 0x0000f800) >>> 11;
    }

    public int getZeroExtendedShamt(){
	return (operation & 0x000007c0) >>> 6;
    }

    public int getSignExtendedShamt(){
	if((operation & 0x00000400) == 0x00000400){
	    return ((operation & 0x000007c0) | 0xfffff800) >> 6;
	}
	else{
	    return (operation & 0x000007c0) >> 6;
	}
    }

    public int getZeroExtendedFD(){
	return (operation & 0x000007c0) >>> 6;
    }

    public int getZeroExtendedFunct(){
	return operation & 0x0000003f;
    }

    public String registerName(int number, int registerType){
	String name = "";

	switch(registerType){
	case GPR:
	    switch (number) {
	    case 28:
		name = "$gp";
		break;
		
	    case 29:
		name = "$sp";
		break;
		
	    case 30:
		name = "$fp";
		break;
		
	    default:
		name = String.format("$%d", number);
		break;
	    }
	    break;

	case FGR:
	case FPR:
	    name = String.format("$f%d", number);
	    break;
	}
	
	return name;
    }

    public String toString() {
	if(inst == -1){
	    return "unkown operation" + String.format("(0x%08x", getZeroExtendedOP() + ")");
	}

	String val = instruction[inst];
	String indent = "";
	for(int i = 8 - val.length(); i > 0; --i){
	    indent += " ";
	}
	val += indent;

	switch (type) {
	case I_TYPE_3000:
	    switch (inst) {
	    case INST_ADDI:
	    case INST_ADDIU:
	    case INST_SLTI:
	    case INST_SLTIU:
		val += registerName(getZeroExtendedRT(), GPR);
		val += "," + registerName(getZeroExtendedRS(), GPR);
		val += String.format(",%d", getSignExtendedImmediate());
		break;

	    case INST_BEQ:
	    case INST_BGEZ:
	    case INST_BGEZAL:
	    case INST_BGTZ:
	    case INST_BLEZ:
	    case INST_BLTZ:
	    case INST_BLTZAL:
	    case INST_BNE:
		val += registerName(getZeroExtendedRS(), GPR);
		val += "," + registerName(getZeroExtendedRT(), GPR);
		val += String.format(",%d", getSignExtendedImmediate());
		break;

	    case INST_LB:
	    case INST_LBU:
	    case INST_LH:
	    case INST_LHU:
	    case INST_LW:
	    case INST_SB:
	    case INST_SH:
	    case INST_SW:
		val += registerName(getZeroExtendedRT(), GPR);
		val += String.format(",%d", getSignExtendedImmediate());
		val += "(" + registerName(getZeroExtendedRS(), GPR) + ")";
		break;

	    case INST_LWC0:
	    case INST_LWC1:
	    case INST_LWC2:
	    case INST_LWC3:
	    case INST_SWC0:
	    case INST_SWC1:
	    case INST_SWC2:
	    case INST_SWC3:
		val += registerName(getZeroExtendedRT(), FGR);
		val += String.format(",%d", getSignExtendedImmediate());
		val += "(" + registerName(getZeroExtendedRS(), GPR) + ")";
		break;

	    case INST_ANDI:
	    case INST_LUI:
	    case INST_ORI:
	    case INST_XORI:
		val += registerName(getZeroExtendedRT(), GPR);
		val += "," + registerName(getZeroExtendedRS(), GPR);
		val += String.format(",%d", getZeroExtendedImmediate());
		break;
	    }
	    break;
		
	case J_TYPE_3000:
	    val += String.format("0x%08x", getSignExtendedTarget());
	    break;
		
	case R_TYPE_3000:
	    switch (inst) {
	    case INST_SLL:
		if(operation == 0x00000000){
		    val = "nop";
		    break;
		}
	    case INST_SRA:
	    case INST_SRL:
		val += registerName(getZeroExtendedRD(), GPR);
		val += "," + registerName(getZeroExtendedRT(), GPR);
		val += String.format(",%d", getZeroExtendedShamt());
		break;

	    case INST_MULT:
	    case INST_MULTU:
	    case INST_DIV:
	    case INST_DIVU:
		val += registerName(getZeroExtendedRS(), GPR);
		val += "," + registerName(getZeroExtendedRT(), GPR);
		break;

	    default:
		val += registerName(getZeroExtendedRD(), GPR);
		val += "," + registerName(getZeroExtendedRS(), GPR);
		val += "," + registerName(getZeroExtendedRT(), GPR);
		break;
	    }
	    break;

	case I_TYPE_3010:
	    val += getSignExtendedImmediate();
	    /*
	    int offset = getSignExtendedImmediate() << 2;
	    String sign = (offset >= 0) ? "+" : "-";
	    val += String.format("$pc+4" + sign + "%d", Math.abs(offset));
	    */
	    break;

	case M_TYPE_3010:
	    val += registerName(getZeroExtendedRT(), GPR);
	    val += "," + registerName(getZeroExtendedFS(), FGR);
	    break;

	case R_TYPE_3010:
	    switch(inst){
	    case INST_C_F_S:
	    case INST_C_F_D:
	    case INST_C_UN_S:
	    case INST_C_UN_D:
	    case INST_C_EQ_S:
	    case INST_C_EQ_D:
	    case INST_C_UEQ_S:
	    case INST_C_UEQ_D:
	    case INST_C_OLT_S:
	    case INST_C_OLT_D:
	    case INST_C_ULT_S:
	    case INST_C_ULT_D:
	    case INST_C_OLE_S:
	    case INST_C_OLE_D:
	    case INST_C_ULE_S:
	    case INST_C_ULE_D:
	    case INST_C_SF_S:
	    case INST_C_SF_D:
	    case INST_C_NGLE_S:
	    case INST_C_NGLE_D:
	    case INST_C_SEQ_S:
	    case INST_C_SEQ_D:
	    case INST_C_NGL_S:
	    case INST_C_NGL_D:
	    case INST_C_LT_S:
	    case INST_C_LT_D:
	    case INST_C_NGE_S:
	    case INST_C_NGE_D:
	    case INST_C_LE_S:
	    case INST_C_LE_D:
	    case INST_C_NGT_S:
	    case INST_C_NGT_D:
		val += registerName(getZeroExtendedFS(), FPR);
		val += "," + registerName(getZeroExtendedFT(), FPR);
		break;

	    case INST_CVT_S_D:
	    case INST_CVT_S_W:
	    case INST_CVT_D_S:
	    case INST_CVT_D_W:
	    case INST_CVT_W_S:
	    case INST_CVT_W_D:
	    case INST_ABS_S:
	    case INST_ABS_D:
	    case INST_MOV_S:
	    case INST_MOV_D:
	    case INST_NEG_S:
	    case INST_NEG_D:
		val += registerName(getZeroExtendedFD(), FPR);
		val += "," + registerName(getZeroExtendedFS(), FPR);
		break;

	    default:
		val += registerName(getZeroExtendedFD(), FPR);
		val += "," + registerName(getZeroExtendedFS(), FPR);
		val += "," + registerName(getZeroExtendedFT(), FPR);
	    }
	    break;

	default:
	    break;
	}

	return val;
    }

}
