/* 
 *    Copyright 2007 Takefumi MIYOSHI
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * リコンフィギュラブルプロセッサRLUに関するパッケージ
 */
package net.wasamon.mics.processor.monorlu;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.swing.JFrame;
import javax.swing.JPanel;

import net.wasamon.mics.Channel;
import net.wasamon.mics.ChannelConnectable;
import net.wasamon.mics.DataBuffer;
import net.wasamon.mics.DataBufferException;
import net.wasamon.mics.ExecInfo;
import net.wasamon.mics.ExecutableElement;
import net.wasamon.mics.MicsDataPacket;
import net.wasamon.mics.MicsElement;
import net.wasamon.mics.MicsException;
import net.wasamon.mics.MicsViewable;
import net.wasamon.mics.memory.RandomAccessMemory;
import net.wasamon.mics.memory.RandomAccessMemoryDataPacket;
import net.wasamon.mjlib.print.FormatException;
import net.wasamon.mjlib.print.PrintFormat;
import net.wasamon.mjlib.util.DataUtil;
import net.wasamon.mjlib.xml.XMLParser;
import net.wasamon.mjlib.xml.XMLParserException;

import org.w3c.dom.Node;

/**
 * 回路ユニット全体を表わす
 * @author Takefumi MIYOSHI
 */
public class ReconfigurableUnit extends MicsElement implements
		ExecutableElement, ChannelConnectable, DataBuffer, MicsViewable {

	public static final int NorthWest = 0;

	public static final int North = 1;

	public static final int NorthEast = 2;

	public static final int West = 3;

	public static final int East = 4;

	public static final int SouthWest = 5;

	public static final int South = 6;

	public static final int SouthEast = 7;

	public static final int NonDirection = -1;

	/**
	 * 内部にもっているLogicUnitの配列
	 */
	private LogicUnit[] unit;

	/**
	 * 内部にもっているLogicUnitの一辺のサイズ
	 */
	private int size;

	/** リコンフィギュラブルプロセッサへの入力バッファ */
	private DataProvider[] input;

	/** リコンフィギュラブルプロセッサの出力バッファ */
	private DataProvider[] output;

	private String busID;

	private int ruAddr;

	private int contextAddr;

	private RandomAccessMemory context;

	/**
	 * コンストラクタ
	 * 
	 * @param size
	 *          一辺のRLBの個数．実際にはsize*size個のRLBを持つ．
	 * @param context
	 *          コンフィグ情報の格納されているメモリ
	 * @param offset
	 *          コンフィグ情報の格納されているメモリ上のオフセット
	 */
	public ReconfigurableUnit() {
	}

	private void initialize(int size) {
		this.context = new RandomAccessMemory(4 * size * size);
		this.contextAddr = 0;
		unit = new LogicUnit[size * size];
		for (int i = 0; i < size * size; i++)
			unit[i] = new LogicUnit(this, i);
		input = new DataProvider[size];
		output = new DataProvider[size];
	}

	/*
	 * generated automatically by MakeInitialize from
	 * inits/reconfigurable_unit.init
	 */
	public void initialize(String base, Node n) throws MicsException {
		try {
			{
				Node init_var_node = n;
				int init_var_size;
				init_var_size = Integer.parseInt(XMLParser.getAttribute(init_var_node,
						"size").getNodeValue());

				this.size = init_var_size;
				initialize(init_var_size);

				{
					Node init_var__channel_node = XMLParser.getNamedNode(n, "channel");
					String init_var__channel_id;
					init_var__channel_id = XMLParser.getAttribute(init_var__channel_node,
							"id").getNodeValue();
					int init_var__channel_offset;
					init_var__channel_offset = Integer.parseInt(XMLParser.getAttribute(
							init_var__channel_node, "offset").getNodeValue());

					setDataBus(init_var__channel_id, init_var__channel_offset);

				}

				{
					Node[] init_var__init_obj = XMLParser.getNamedNodeArray(n, "init");
					for (int i = 0; i < init_var__init_obj.length; i++) {
						Node init_var__init_node = init_var__init_obj[i];
						String init_var__init_file;
						init_var__init_file = XMLParser.getAttribute(init_var__init_node,
								"file").getNodeValue();
						if (init_var__init_file.charAt(0) != '/') {
							init_var__init_file = base + "/" + init_var__init_file;
						}
						int init_var__init_offset;
						init_var__init_offset = DataUtil.parseInt(XMLParser.getAttribute(
								init_var__init_node, "offset").getNodeValue());

						try {
							write(init_var__init_offset, new BufferedInputStream(
									new FileInputStream(init_var__init_file)));
						} catch (FileNotFoundException e) {
							System.out.println("no such file: " + init_var__init_file);
							System.out.println("[W] no data is written as initialize.");
						}

					}
				}

			}
		} catch (NumberFormatException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.rlu.ReconfigurableProcessor");
		} catch (XMLParserException e) {
			throw new MicsException(
					"configuration syntax error: net.wasamon.mics.processor.rlu.ReconfigurableProcessor");
		}
	}

	private void setDataBus(String bus, int ruAddr) {
		this.busID = bus;
		this.ruAddr = ruAddr;
		for (int i = 0; i < size; i++) {
			// 一つのRLBの入力は 16bit x 2 なので，アドレスを4バイトずつずらす
			input[i] = new GlobalDataProvider(this, ruAddr + i * 4);
		}
		for (int i = 0; i < size; i++) { // 最後の行のRLBをマッピングする
			output[i] = unit[size * (size - 1) + i];
		}
	}

	public int size() {
		return unit.length;
	}

	public MicsDataPacket read(MicsDataPacket d) {
		return context.read(d);
	}

	public void write(MicsDataPacket d) {
		context.write(d);
	}

	public void write(int addr, InputStream reader) throws DataBufferException {
		context.write(addr, reader);
	}

	public void dump(int offset, int len, OutputStream writer)
			throws DataBufferException {
		context.dump(offset, len, writer);
	}

	public String toString(int addr, int length) {
		return context.toString(addr, length);
	}

	public void reset() {
		for (int i = 0; i < unit.length; i++) {
			unit[i].reset();
		}
	}

	private char busbuffer;

	public void writeback(Channel src, MicsDataPacket value) {
		// System.out.println("writeback: " + value);
		busbuffer = DataUtil.toChar(((RandomAccessMemoryDataPacket) value).data, 0,
				2);
	}

	public int getChannelOffset(Channel c) {
		return ruAddr;
	}

	private Channel busInstance = null;

	private Channel bus() throws MicsException {
		if (busInstance == null) {
			busInstance = composite.getChannel(busID);
		}
		return busInstance;
	}

	public char getInputData(int addr) throws MicsException {
		bus().readRequest(this,
				RandomAccessMemoryDataPacket.readPacket(addr, 1, 16));
		return busbuffer;
	}

	/**
	 * ReconfigurableUnitの1ステップの実行
	 */
	public ExecInfo exec_first() throws MicsException {
		ExecInfo info = new ExecInfo();
		for (int i = 0; i < unit.length; i++) {
			unit[i].load();
		}
		for (int i = 0; i < unit.length; i++) {
			unit[i].exec();
		}
		for (int i = 0; i < size; i++) {
			bus().writeRequest(
					this,
					RandomAccessMemoryDataPacket.writePacket(ruAddr + size * 4 + 4 * i,
							1, 16, DataUtil.toByteArray(output[i].getData(0))));
			bus().writeRequest(
					this,
					RandomAccessMemoryDataPacket.writePacket(ruAddr + size * 4 + 4 * i
							+ 2, 1, 16, DataUtil.toByteArray(output[i].getData(1))));
		}
		info.setTerminatableFlag(true);
		return info;
	}

	public ExecInfo exec_second() {
		return null;
	}

	public String toString() {
		String str = "";
		try {
			str += "==\n";
			try {
				for (int j = 0; j < size; j++) {
					str += PrintFormat.print("%04x", (int) (input[j].getData(0)));
					str += ":";
					str += PrintFormat.print("%04x", (int) (input[j].getData(1)));
					str += ":";
					str += PrintFormat.print("%04x", (int) (input[j].getData(2)));
					str += " ";
				}
			} catch (MicsException e) {
				str += "catch: " + e.getMessage();
			}
			str += "\n";
			str += "--\n";
			for (int i = 0; i < size; i++) {
				for (int j = 0; j < size; j++) {
					str += PrintFormat.print("%04x",
							(int) (unit[i * size + j].getData(0)));
					str += ":";
					str += PrintFormat.print("%04x",
							(int) (unit[i * size + j].getData(1)));
					str += ":";
					str += PrintFormat.print("%04x",
							(int) (unit[i * size + j].getData(2)));
					str += " ";
				}
				str += "\n";
			}
		} catch (FormatException e) {

		}
		return str;
	}

	public int getLogicUnitInst(int id) {
		return context.read(contextAddr + id * 4) & 0x00ff;
	}

	public int getLogicUnitPreShift(int id) {
		return (context.read(contextAddr + id * 4 + 1) >> 2) & 0x03;
	}

	public int getLogicUnitPostShift(int id) {
		return context.read(contextAddr + id * 4 + 1) & 0x03;
	}

	private int getInputSrcID(int id, int src) {
		if (src == 0) {
			return context.read(contextAddr + id * 4 + 2) & 0x00ff;
		} else {
			return context.read(contextAddr + id * 4 + 3) & 0x00ff;
		}
	}

	public String getLogicUnitInstString(int id) {
		return unit[id].getLogicUnitInstString();
	}

	public String getLogicUnitShift(int i) {
		return unit[i].getShiftOpString();
	}

	public DataProvider getInputSrcLogicUnit(int id, int src) {
		int srcid = getInputSrcID(id, src);
		if (id == srcid) {
			if (id / size == 0) {
				return input[id];
			} else {
				return unit[id];
			}
		} else {
			return unit[srcid];
		}
	}

	public int getInputSrcRegister(int id, int src) {
		if (src == 0) {
			return (context.read(contextAddr + id * 4 + 1) >> 7) & 0x01;
		} else {
			return (context.read(contextAddr + id * 4 + 1) >> 6) & 0x01;
		}
	}

	public int getInputSrcDir(int src, int id) {
		int srcid = getInputSrcID(id, src);
		int dir = srcid - id;
		switch (dir) {
		case -9:
			return NorthWest;
		case -8:
			return North;
		case -7:
			return NorthEast;
		case -1:
			return West;
		case 1:
			return East;
		case 7:
			return SouthWest;
		case 8:
			return South;
		case 9:
			return SouthEast;
		default:
			return NonDirection;
		}
	}

	public String[] getConnectedElements() {
		return new String[] { busID };
	}

	public String getInfo() {
		String s = "";
		s += "ReconfigurableUnit\n";
		s += "  CLASS: " + this.getClass().getName() + "\n";
		s += "  ID: " + id() + "\n";
		s += "  Local Memory ID: " + ((MicsElement) context).id() + ",";
		s += " offset = " + contextAddr + "\n";
		s += "  Channel ID: ";
		try {
			s += ((MicsElement) bus()).id() + ",";
		} catch (MicsException e) {
			s += "unconnected,";
		}
		s += " input offset = " + ruAddr + ",";
		s += " output offset = " + (ruAddr + size * 4);
		return s;
	}

	private ConfigDataViewer viewer = null;

	public void show() {
		if (viewer == null) {
			viewer = new ConfigDataViewer();
		}
		viewer.setReconfigurableUnit(this);
		viewer.repaint();
	}

	public String getImagePath() {
		return "cb_rlu.png";
	}


	public String getDescription(){
		return "TODO";
	}

}

class GlobalDataProvider implements DataProvider {

	private ReconfigurableUnit unit;

	private int addr;

	public GlobalDataProvider(ReconfigurableUnit unit, int addr) {
		this.unit = unit;
		this.addr = addr;
	}

	public char getData(int i) throws MicsException {
		char data = 0;
		switch (i) {
		case 0:
			// System.out.println("getData: " + addr);
			data = unit.getInputData(addr);
			break;
		case 1:
			data = unit.getInputData(addr + 2);
			break;
		case 2:
			data = unit.getInputData(addr);
			break;
		default:
		}
		return data;
	}

}

class ConfigDataViewer {

	private ConfigDataCanvas canvas;

	private JFrame frame;

	public ConfigDataViewer() {
		frame = new JFrame("Config Data Viewer");
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				frame.setVisible(false);
			}
		});
		frame.setSize(600, 600);
		frame.getContentPane().add((canvas = new ConfigDataCanvas()));
		frame.setVisible(false);
	}

	public void setReconfigurableUnit(ReconfigurableUnit rp) {
		canvas.setReconfigurableUnit(rp);
		frame.setTitle("Config Data Viewer (ID=" + rp.id() + ")");
		frame.setVisible(true);
	}

	public void repaint() {
		canvas.repaint();
	}

	class ConfigDataCanvas extends JPanel{
		/**
		 * generated by eclipse
		 */
		private static final long serialVersionUID = 1L;

		private ReconfigurableUnit rp;

		private ConfigDataCanvas() {
			this.rp = null;
		}

		private void setReconfigurableUnit(ReconfigurableUnit rp) {
			this.rp = rp;
			repaint();
		}

		private int offset = 60;

		private int luwidth = 60;

		private int luheight = 60;

		private int px(int x) {
			return offset + x * luwidth;
		}

		private int py(int y) {
			return offset + y * luheight;
		}

		private void paintLogicUnit(Graphics g, int n) {
			for (int i = 0; i < n; i++) {
				for (int j = 0; j < n; j++) {
					g.drawRect(px(j), py(i), luwidth, luheight);
					g.drawString(rp.getLogicUnitInstString(i * n + j), px(j) + 5,
							py(i) + 14);
					g.drawString(rp.getLogicUnitShift(i * n + j), px(j) + 5, py(i) + 28);
				}
			}
		}

		private void paintDirection(Graphics g, int n) {
			int dx, dy;
			// System.out.println("0:0 = " + rp.getInputSrcDir(0, 0));
			// System.out.println("0:1 = " + rp.getInputSrcDir(0, 1));
			// System.out.println("1:0 = " + rp.getInputSrcDir(0, 8));
			// System.out.println("1:1 = " + rp.getInputSrcDir(0, 9));
			// System.out.println("0:7 = " + rp.getInputSrcDir(0, 7));
			for (int i = 0; i < n; i++) {
				for (int j = 0; j < n; j++) {
					if (rp.getInputSrcRegister(i * n + j, 0) == 0) {
						g.setColor(Color.blue);
					} else {
						g.setColor(Color.red);
					}
					switch (rp.getInputSrcDir(0, i * n + j)) {
					case ReconfigurableUnit.NorthWest:
						dx = -1;
						dy = -1;
						break;
					case ReconfigurableUnit.North:
						dx = 0;
						dy = -1;
						break;
					case ReconfigurableUnit.NorthEast:
						dx = 1;
						dy = -1;
						break;
					case ReconfigurableUnit.West:
						dx = -1;
						dy = 0;
						break;
					case ReconfigurableUnit.East:
						dx = 1;
						dy = 0;
						break;
					case ReconfigurableUnit.SouthWest:
						dx = -1;
						dy = 1;
						break;
					case ReconfigurableUnit.South:
						dx = 0;
						dy = 1;
						break;
					case ReconfigurableUnit.SouthEast:
						dx = 1;
						dy = 1;
						break;
					default:
						dx = 0;
						dy = 0;
					}
					g.drawLine(px(j) + 28, py(i) + 28, px(j) + 28 + dx * 60, py(i) + 28
							+ dy * 60);
				}
			}
			for (int i = 0; i < n; i++) {
				for (int j = 0; j < n; j++) {
					if (rp.getInputSrcRegister(i * n + j, 1) == 0) {
						g.setColor(Color.blue);
					} else {
						g.setColor(Color.red);
					}
					switch (rp.getInputSrcDir(1, i * n + j)) {
					case ReconfigurableUnit.NorthWest:
						dx = -1;
						dy = -1;
						break;
					case ReconfigurableUnit.North:
						dx = 0;
						dy = -1;
						break;
					case ReconfigurableUnit.NorthEast:
						dx = 1;
						dy = -1;
						break;
					case ReconfigurableUnit.West:
						dx = -1;
						dy = 0;
						break;
					case ReconfigurableUnit.East:
						dx = 1;
						dy = 0;
						break;
					case ReconfigurableUnit.SouthWest:
						dx = -1;
						dy = 1;
						break;
					case ReconfigurableUnit.South:
						dx = 0;
						dy = 1;
						break;
					case ReconfigurableUnit.SouthEast:
						dx = 1;
						dy = 1;
						break;
					default:
						dx = 0;
						dy = 0;
					}
					g.drawLine(px(j) + 32, py(i) + 32, px(j) + 32 + dx * 60, py(i) + 32
							+ dy * 60);
				}
			}
			g.setColor(Color.black);
		}

		public void paintComponent(Graphics g) {
			super.paintComponent(g);
			if (rp == null) {
				return;
			} else {
				int n = (int) (Math.sqrt(rp.size()));
				paintDirection(g, n);
				paintLogicUnit(g, n);
			}
		}

	}
}
