/*
 * ChordDiagram class for MIDI Chord Helper
 *
 *	Copyright (C) 2004-2009 Akiyoshi Kamide
 *	http://www.yk.rim.or.jp/~kamide/music/chordhelper/
 *
 */
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

public class ChordDiagram extends JPanel
{
  public CapoSelecter	capo_selecter;
  public JToggleButton record_text_button;
  //
  private static final String	DEFAULT_TITLE = "Chord Diagram";
  private Insets zero_insets = new Insets(0,0,0,0);
  //
  // Tuning parameters
  //
  private int UKULELE_OPEN_NOTES[] = {9,4,0,7}; // AECG
  private int GUITAR_OPEN_NOTES[] = {4,11,7,2,9,4}; // EBGDAE
  //
  // GUI parts
  //
  private ChordDiagramDisplay diagram_display;
  private JRadioButton ukulele_button, guitar_button;
  private ButtonGroup ukulele_guitar_group = new ButtonGroup();
  public ChordDisplay title_label;
  private JScrollBar
	fret_range_scrollbar, variation_scrollbar;
  private JPanel inner_panel = new JPanel();
  private JPanel main_panel = new JPanel();

  public ChordDiagram( ChordHelperApplet applet ) {
    boolean is_guitar = false;
    // 
    // Chord symbol label
    title_label = new ChordDisplay( DEFAULT_TITLE, null, null );
    title_label.setHorizontalAlignment(title_label.CENTER);
    title_label.setVerticalAlignment(title_label.BOTTOM);
    JPanel title_panel = new JPanel();
    title_panel.add(title_label);
    title_panel.setOpaque(false);
    //
    // Guitar/Ukulele select
    ukulele_guitar_group.add(
      ukulele_button = new JRadioButton("Ukulele", !is_guitar)
    );
    ukulele_guitar_group.add(
      guitar_button = new JRadioButton("Guitar", is_guitar)
    );
    ukulele_button.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        diagram_display.tune( UKULELE_OPEN_NOTES );
        title_label.setText(null); // for resizing
        title_label.setChord(diagram_display.getChord());
      }
    });
    ukulele_button.setOpaque(false);
    guitar_button.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        diagram_display.tune( GUITAR_OPEN_NOTES );
        title_label.setText(null); // for resizing
        title_label.setChord(diagram_display.getChord());
      }
    });
    guitar_button.setOpaque(false);

    JPanel ukulele_guitar_panel = new JPanel();
    ukulele_guitar_panel.setOpaque(false);
    ukulele_guitar_panel.add(ukulele_button);
    ukulele_guitar_panel.add(guitar_button);
    //
    diagram_display = new ChordDiagramDisplay(UKULELE_OPEN_NOTES);
    diagram_display.setOpaque(false);
    diagram_display.setPreferredSize( new Dimension(120,120) );
    //
    variation_scrollbar = new JScrollBar(JScrollBar.VERTICAL);
    variation_scrollbar.setModel(
      diagram_display.chord_variations.model
    );
    variation_scrollbar.addAdjustmentListener(
      new AdjustmentListener() {
        public void adjustmentValueChanged(AdjustmentEvent e) {
          showVariationStatus();
        }
      }
    );
    //
    fret_range_scrollbar = new JScrollBar(JScrollBar.HORIZONTAL);
    fret_range_scrollbar.setModel(
      diagram_display.fret_range_model
    );
    fret_range_scrollbar.setBlockIncrement(
      diagram_display.fret_range_model.getExtent()
    );

    record_text_button = new JToggleButton(
      "REC", new ButtonIcon(ButtonIcon.REC_ICON)
    );
    record_text_button.setMargin(zero_insets);
    record_text_button.setToolTipText("Record to text ON/OFF");

    capo_selecter = new CapoSelecter(applet.chord_matrix.capo_value_model);
    capo_selecter.checkbox.addItemListener(
      new ItemListener() {
        public void itemStateChanged(ItemEvent e) {
          clear();
        }
      }
    );

    JPanel button_panel = new JPanel();
    button_panel.setLayout( new BoxLayout(
      button_panel, BoxLayout.X_AXIS
    ));
    button_panel.setOpaque(false);
    button_panel.add(Box.createHorizontalStrut(2));
    button_panel.add(record_text_button);
    button_panel.add(Box.createHorizontalStrut(2));
    button_panel.add(capo_selecter);
    //
    inner_panel.setLayout(new BoxLayout(
      inner_panel, BoxLayout.Y_AXIS
    ));
    inner_panel.setOpaque(false);
    title_panel.setAlignmentY((float)0);
    inner_panel.add(title_panel);
    inner_panel.add(diagram_display);
    fret_range_scrollbar.setAlignmentY((float)1.0);
    inner_panel.add(fret_range_scrollbar);
    ukulele_guitar_panel.setAlignmentY((float)1.0);
    inner_panel.add(ukulele_guitar_panel);
    //
    main_panel.setLayout(new BoxLayout(
      main_panel, BoxLayout.X_AXIS
    ));
    main_panel.setOpaque(false);
    main_panel.add(inner_panel);
    main_panel.add(variation_scrollbar);
    //
    setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
    add(button_panel);
    add(Box.createHorizontalStrut(5));
    add(main_panel);
  }
  public void setBackground(Color bg_color) {
    super.setBackground(bg_color);
    if( diagram_display == null ) return;
    diagram_display.setBackground(bg_color);
    capo_selecter.setBackground(bg_color);
    capo_selecter.value_selecter.setBackground(bg_color);
    variation_scrollbar.setBackground(bg_color);
    fret_range_scrollbar.setBackground(bg_color);
  }
  public void showVariationStatus() {
    String txt = null;
    if( diagram_display.getChord() != null ) {
      DefaultBoundedRangeModel model = diagram_display.chord_variations.model;
      int val = model.getValue();
      int max = model.getMaximum();
      if( val < 0 ) {
        switch( max ) {
        case 0: txt = "No variation found"; break;
        case 1: txt = "1 variation found"; break;
        default: txt = max + " variations found"; break;
        }
      }
      else txt = "Variation: " + (val+1) + " / " + max ;
    }
    variation_scrollbar.setToolTipText( txt );
  }
  public Music.Chord getChord() {
    return diagram_display.getChord();
  }
  public void setChord(Music.Chord chord) {
    if( !isVisible() ) {
      clear();
      return;
    }
    diagram_display.setChord( chord );
    title_label.setChord( chord );
  }
  public void clear() {
    diagram_display.setChord(null);
    title_label.setChord( null );
    variation_scrollbar.setToolTipText(null);
  }
  public void setGuitar(boolean is_guitar) {
    if( is_guitar ) guitar_button.doClick();
    else ukulele_button.doClick();
  }
  public int getCapo() { return capo_selecter.getCapo(); }
}

class CapoComboBoxModel extends DefaultComboBoxModel {
  public CapoComboBoxModel() {
    for( int i=1; i<=11; i++ ) addElement(i);
  }
}
class CapoSelecter extends JPanel
  implements ItemListener
{
  public JCheckBox checkbox = new JCheckBox("Capo");
  public JComboBox value_selecter = new JComboBox();

  public CapoSelecter( ComboBoxModel cbm ) {
    checkbox.setOpaque(false);
    checkbox.addItemListener(this);
    value_selecter.setModel(cbm);
    value_selecter.setMaximumRowCount(12);
    value_selecter.setVisible(false);
    setLayout( new BoxLayout(this, BoxLayout.X_AXIS) );
    add(checkbox);
    add(value_selecter);
  }
  // ItemListener for JCheckBox
  public void itemStateChanged(ItemEvent e) {
    // Object source = e.getItemSelectable();
    value_selecter.setVisible( checkbox.isSelected() );
  }
  public int getCapo() {
    return checkbox.isSelected() ? value_selecter.getSelectedIndex()+1 : 0;
  }
}


class ChordDiagramDisplay extends JComponent
  implements MouseListener, MouseMotionListener
{
  public static final int VISIBLE_FRETS = 4;
  public static final int MAX_FRETS = 16;
  public static final int MAX_STRINGS = 6; // max of some notes_when_open.length

  public static final int LEFT_MARGIN_WIDTH = 10;
  public static final int RIGHT_MARGIN_WIDTH = 10;
  public static final int UPPER_MARGIN_WIDTH = 5;
  public static final int LOWER_MARGIN_WIDTH = 5;

  int
    string_distance, fret_distance,
    point_size, char_width, char_height;

  DefaultBoundedRangeModel fret_range_model
    = new DefaultBoundedRangeModel(
      0, VISIBLE_FRETS, 0, MAX_FRETS
    );

  int notes_when_open[], default_notes_when_open[];
  Rectangle grid_rect;
  TuningButton tuning_buttons[];
  ChordVariations chord_variations = new ChordVariations();

  class TuningButton extends Rectangle {
    boolean is_entered = false;
    int i_string;
    public TuningButton(int i) {
      super(
        LEFT_MARGIN_WIDTH,
        UPPER_MARGIN_WIDTH + i * string_distance,
        char_width, string_distance
      );
      i_string = i;
    }
  }

  class PressingPoint {
    int i_string;
    int fret_position;
    int index_of_chord_note;
    Rectangle rect = null;
    boolean is_entered = false;
    public PressingPoint(int i_string) {
      this(-1,-1,i_string);
    }
    public PressingPoint(int fp, int iocn) {
      fret_position = fp;
      index_of_chord_note = iocn;
    }
    public PressingPoint(int fp, int iocn, int i_string) {
      this(fp, iocn);
      rect = new Rectangle(
        grid_rect.x + (
          fp<1 ?
            -(point_size + 3) :
            (fp * fret_distance - point_size/2 - fret_distance/2)
        ),
        grid_rect.y
          - point_size/2
          + i_string * string_distance,
        point_size, point_size
      );
      this.i_string = i_string;
    }
  }
  class PressingPoints extends LinkedList<PressingPoint> { }

  class ChordVariations extends LinkedList<PressingPoint[]> {
    public Music.Chord chord = null;
    public DefaultBoundedRangeModel model = new DefaultBoundedRangeModel(0,0,0,0);
    private PressingPoints possible_points[] = null;
    private PressingPoint new_variation[] = null;
    private boolean fifth_omittable = false;
    private boolean root_omittable = false;
    private int check_bits_all_on = 0;
    //
    // Methods
    //
    public PressingPoints[] getPossiblePoints() {
      return possible_points;
    }
    public PressingPoint[] getCurrent() {
      if( size() <= 0 ) return null;
      int i = model.getValue();
      if( i < 0 ) return null;
      return get(i);
    }
    public void setCurrent(int i) {
      model.setValue(i);
    }
    public void setChord(Music.Chord chord) {
      clear();
      if( (this.chord = chord) == null ) {
        possible_points = null;
        model.setRangeProperties(0,0,0,0,false);
        return;
      }
      int n_notes = chord.numberOfNotes();
      root_omittable = ( n_notes == 5 );
      fifth_omittable = ( chord.symbolSuffix().equals("7") || root_omittable );
      check_bits_all_on = (1 << n_notes) - 1;
      int ioc;
      //
      // Collect all fret-points for each note of the chord
      //
      possible_points = new PressingPoints[notes_when_open.length];
      for( int i_string=0; i_string<possible_points.length; i_string++ ) {
        possible_points[i_string] = new PressingPoints();
        for( int i_fret=0;
          i_fret <= fret_range_model.getValue() + fret_range_model.getExtent();
          i_fret++
        ) {
          if( (i_fret == 0 || i_fret > fret_range_model.getValue() )
              && (ioc = chord.indexOf(notes_when_open[i_string] + i_fret)) >= 0 )
            possible_points[i_string].add(new PressingPoint(i_fret,ioc,i_string));
        }
        // 'x'-marking string
        possible_points[i_string].add(new PressingPoint(i_string));
      }
      //
      // Search proper fret-point combinations for the chord
      //
      new_variation = new PressingPoint[notes_when_open.length];
      scanFret(0);
      model.setRangeProperties(-1,1,-1,size(),false);
    }
    private void scanFret( int i_string ) {
      for( PressingPoint pp : possible_points[i_string] ) {
        new_variation[i_string] = pp;
        if( i_string < new_variation.length - 1 ) {
          scanFret( i_string + 1 );
        }
        else if( hasValidNewVariation() ) {
          add(new_variation.clone());
        }
      }
    }
    private boolean hasValidNewVariation() {
      int check_bits = 0;
      int iocn;
      for( PressingPoint pp : new_variation )
        if( (iocn = pp.index_of_chord_note) >= 0 )
          check_bits |= 1 << iocn;
      return ( check_bits == check_bits_all_on
        || check_bits == check_bits_all_on -4 && fifth_omittable
        || (check_bits & (check_bits_all_on -1-4)) == (check_bits_all_on -1-4)
        && (check_bits & (1+4)) != 0 && root_omittable
      );
    }
  }
  //
  // Constructor
  //
  public ChordDiagramDisplay( int[] notes ) {
    addMouseListener(this);
    addMouseMotionListener(this);
    addComponentListener(new ComponentAdapter() {
      public void componentResized(ComponentEvent e) {
        tune();
      }
    });
    chord_variations.model.addChangeListener(
      new ChangeListener() {
        public void stateChanged(ChangeEvent e) {
          repaint();
        }
      }
    );
    fret_range_model.addChangeListener(
      new ChangeListener() {
        public void stateChanged(ChangeEvent e) {
          setChord(); // To reconstruct chord variations
        }
      }
    );
    setMinimumSize( new Dimension(100,70) );
    tune(notes);
  }
  // Callbacks
  //
  public void paint(Graphics g) {
    Graphics2D g2 = (Graphics2D) g;
    Dimension d = getSize();
    Color fret_color = Color.gray; // getBackground().darker();
    FontMetrics fm = g2.getFontMetrics();
    //
    // Copy background color
    g2.setBackground(getBackground());
    g2.clearRect(0, 0, d.width, d.height);
    //
    // Draw frets and its numbers
    //
    for( int i=1; i<=VISIBLE_FRETS; i++ ) {
      g2.setColor(fret_color);
      int fret_x = grid_rect.x
        + (grid_rect.width - 2) * i / VISIBLE_FRETS;
      g2.drawLine(
        fret_x, grid_rect.y,
        fret_x, grid_rect.y + string_distance * (notes_when_open.length - 1)
      );
      g2.setColor(getForeground());
      String s = String.valueOf( i + fret_range_model.getValue() );
      g2.drawString(
        s,
        grid_rect.x
          + fret_distance/2 - fm.stringWidth(s)/2
          + grid_rect.width * (i-1) / VISIBLE_FRETS,
        grid_rect.y
          + string_distance/2 + fm.getHeight()
          + string_distance * (notes_when_open.length - 1) - 1
      );
    }
    //
    // Draw strings and open notes
    for( int i=0; i<notes_when_open.length; i++ ) {
      int string_y = grid_rect.y
        + grid_rect.height * i / (MAX_STRINGS - 1);
      g2.setColor(fret_color);
      g2.drawLine(
        grid_rect.x, string_y,
        grid_rect.x + (grid_rect.width - 2), string_y
      );
      if( notes_when_open[i] != default_notes_when_open[i] ) {
        g2.setColor(Color.yellow);
        g2.fill(tuning_buttons[i]);
      }
      g2.setColor(getForeground());
      g2.drawString(
        Music.noteNoToSymbol( notes_when_open[i], 2 ),
        LEFT_MARGIN_WIDTH,
        string_y + (fm.getHeight() - fm.getDescent())/2
      );
      g2.setColor(fret_color);
      if( tuning_buttons[i].is_entered ) {
        g2.draw(tuning_buttons[i]);
      }
    }
    //
    // Draw left-end of frets
    if( fret_range_model.getValue() == 0 ) {
      g2.setColor(getForeground());
      g2.fillRect(
        grid_rect.x - 1,
        grid_rect.y,
        3,
        string_distance * (notes_when_open.length - 1) + 1
      );
    }
    else {
      g2.setColor(fret_color);
      g2.drawLine(
        grid_rect.x,
        grid_rect.y,
        grid_rect.x,
        grid_rect.y + string_distance * (notes_when_open.length - 1)
      );
    }
    //
    // Draw indicators
    if( chord_variations.chord == null ) {
      return;
    }
    PressingPoint variation[] = chord_variations.getCurrent();
    if( variation != null ) {
      for( PressingPoint pp : variation ) {
        drawIndicator( g2, pp, false );
      }
    }
    PressingPoints possible_points[] = chord_variations.getPossiblePoints();
    if( possible_points != null ) {
      for( PressingPoints pps : possible_points ) {
        for( PressingPoint pp : pps ) {
          if( pp.is_entered ) {
            drawIndicator( g2, pp, false );
            if( variation != null ) {
              return;
            }
          }
          else if( variation == null ) {
            drawIndicator( g2, pp, true );
          }
        } 
      }
    }
  }
  private void drawIndicator(
    Graphics2D g2,
    PressingPoint pp,
    boolean draw_all_points
  ) {
    Rectangle r;
    int fret_point, i_chord;
    g2.setColor( (i_chord = pp.index_of_chord_note) < 0
      ? getForeground() : Music.Chord.index_colors[i_chord]
    );
    if( (r = pp.rect) == null ) {
      return;
    }
    if( (fret_point = pp.fret_position) < 0 ) {
      if( ! draw_all_points ) {
        // Put 'x' mark
        g2.drawLine(
          r.x + 1,
          r.y + 1,
          r.x + r.width - 1,
          r.y + r.height - 1
        );
        g2.drawLine(
          r.x + 1,
          r.y + r.height - 1,
          r.x + r.width - 1,
          r.y + 1
        );
      }
    }
    else if( fret_point == 0 ) {
      // Put 'o' mark
      g2.drawOval( r.x, r.y, r.width, r.height );
    }
    else { // Fret-pressing
      int x = r.x - fret_range_model.getValue() * fret_distance ;
      if( draw_all_points ) {
        g2.drawOval( x, r.y, r.width, r.height );
      }
      else {
        g2.fillOval( x, r.y, r.width, r.height );
      }
    } // End If
  }
  //
  // MouseListener
  public void mousePressed(MouseEvent e) {
    Point p = e.getPoint();
    boolean hit;
    PressingPoints possible_points[] = chord_variations.getPossiblePoints();
    if( possible_points != null ) {
      for( PressingPoints pps : possible_points ) {
        for( PressingPoint pp : pps ) {
          if( pp.fret_position > 0 ) {
            int x_offset = -fret_range_model.getValue()*fret_distance;
            pp.rect.translate( x_offset, 0 );
            hit = pp.rect.contains(p);
            pp.rect.translate( -x_offset, 0 );
          }
          else {
            hit = pp.rect.contains(p);
          }
          if( hit ) {
            int i = 0;
            for( PressingPoint[] variation : chord_variations ) {
              if( variation[pp.i_string].fret_position == pp.fret_position ) {
                chord_variations.setCurrent(i);
                return;
              }
              i++;
            }
          }
        } 
      }
    }
    for( TuningButton button : tuning_buttons ) {
      if( button.contains(p) ) {
        notes_when_open[button.i_string] += (e.getButton()==e.BUTTON3 ? 11 : 1);
        notes_when_open[button.i_string] = Music.mod12(notes_when_open[button.i_string]);
        setChord();
        return;
      }
    }
  }
  public void mouseReleased(MouseEvent e) { }
  public void mouseEntered(MouseEvent e) {
    mouseMoved(e);
  }
  public void mouseExited(MouseEvent e) {
    mouseMoved(e);
  }
  public void mouseClicked(MouseEvent e) { }
  //
  // MouseMotionListener
  public void mouseDragged(MouseEvent e) {
  }
  public void mouseMoved(MouseEvent e) {
    Point p = e.getPoint();
    boolean changed = false;
    boolean hit;
    for( TuningButton button : tuning_buttons ) {
      hit = button.contains(p);
      if ( button.is_entered != hit ) changed = true;
      button.is_entered = hit;
    }
    PressingPoints possible_points[] = chord_variations.getPossiblePoints();
    if( possible_points != null ) {
      for( PressingPoints pps : possible_points ) {
        for( PressingPoint pp : pps ) {
          if( pp.fret_position > 0 ) {
            int x_offset = -fret_range_model.getValue()*fret_distance;
            pp.rect.translate( x_offset, 0 );
            hit = pp.rect.contains(p);
            pp.rect.translate( -x_offset, 0 );
          }
          else {
            hit = pp.rect.contains(p);
          }
          if ( pp.is_entered != hit ) changed = true;
          pp.is_entered = hit;
        } 
      }
    }    
    if( changed ) repaint();
  }
  // Methods
  public void tune( int[] open_notes ) {
    System.arraycopy(
      default_notes_when_open = open_notes, 0,
      notes_when_open = new int[open_notes.length], 0,
      open_notes.length
    );
    tune();
  }
  public void tune() {
    Dimension diagram_size = getSize();

    char_width = 8; // FontMetrics.stringWidth("C#");
    char_height = 16; // FontMetrics.getHeight();

    string_distance = (
      diagram_size.height + 1
      - UPPER_MARGIN_WIDTH - LOWER_MARGIN_WIDTH
      - char_height
    ) / MAX_STRINGS; 

    point_size = string_distance * 4 / 5;

    fret_distance = (
      diagram_size.width + 1 - (
        LEFT_MARGIN_WIDTH + RIGHT_MARGIN_WIDTH
        + char_width + point_size + 8
      )
    ) / VISIBLE_FRETS;

    grid_rect = new Rectangle(
      LEFT_MARGIN_WIDTH + point_size + char_width + 8,
      UPPER_MARGIN_WIDTH + string_distance / 2,
      fret_distance * VISIBLE_FRETS,
      string_distance * (MAX_STRINGS - 1)
    );

    tuning_buttons = new TuningButton[ notes_when_open.length ];
    for( int i=0; i<tuning_buttons.length; i++ ) {
      tuning_buttons[i] = new TuningButton(i);
    }
    setChord();
  }
  public Music.Chord getChord() { return chord_variations.chord; }
  public void setChord() { setChord(getChord()); }
  public void setChord(Music.Chord chord) {
    chord_variations.setChord(chord);
    repaint();
  }
}
