//
// y_EȃASY Java 
//
// Circle-Of-Fifth based music theory functions
//   Copyright (C) 2004-2009 悵 - Akiyoshi Kamide
//
import java.util.*;
import javax.sound.midi.*;
import java.util.regex.*;

// for ComboBoxModel implementation
import javax.swing.*;
import javax.swing.event.*;

import java.awt.*; // Color

public class Music {
  //
  // iIN^[uj\NX
  //
  public static class NoteSymbol
    implements Cloneable
  {
    public static final int NOTE_SYMBOL = 0;	// V{\LiBb, F#j
    public static final int NOTE_NAME = 1;	// p\LiB flat, F sharpj
    public static final int NOTE_IN_JAPANESE = 2; // {\Liσ, dwj
    //
    // `z
    //
    //	OF_utbg
    //	PFtbg
    //	QFωLȂi󗓁j
    //	RFV[v
    //	SF_uV[v
    //	TFωL̂Ȃܓx̂VK
    //
    //	̔zƂɂ FCGDAEB ̌JԂ𓱂oA
    //	V~TRTނ̉o܂B
    //
    private static final String note_strings[][] = {
      { "bb","b","","#","x", "FCGDAEB" },
      { " double flat"," flat",""," sharp"," double sharp", "FCGDAEB" },
      { "d","","","d","dd","wngjCz" },
    };
    private static final boolean pre_sharp_flat[] = {
      false, false, true,
    };
    //
    // ܓxł̈ʒu
    //
    //	𐔒lƂ́AW[L[̒ɂꍇ
    //	AAƂlŎ܂B
    //	Έٖ𖾊mɋʂłAy_IȌvZ
    //	ȒPȐōs悤ɂȂ܂B
    //
    private int major_co5;
    //
    public NoteSymbol(int major_co5) {
      this.major_co5 = major_co5;
    }
    public NoteSymbol(String note_symbol) {
      String tns = note_symbol.trim();
      String note_symbols[] = note_strings[NOTE_SYMBOL];
      if(
        tns.isEmpty() || (
          major_co5 = note_symbols[5].indexOf( tns.charAt(0) )
        ) < 0
      ) {
        // ȂiA`G ͈̔͂ɂȂj
        throw new IllegalArgumentException( "Unknown note name " + tns );
      }
      major_co5--;
      for( int i=0; i<5; i++ ) {
        if( i == 2 ) continue;
        if( tns.startsWith( note_symbols[i], 1 ) ) {
          // ωL𔭌
          // bb ̂ق b Ƀ}b`̂Ō딻̐SzȂ
          major_co5 += (i-2) * 7;
          return;
        }
      }
    }
    protected NoteSymbol clone() {
      return new NoteSymbol(major_co5);
    }
    public boolean equals( NoteSymbol another_one ) {
      return major_co5 == another_one.toCo5();
    }
    public int hashCode() { return major_co5; }
    public int toCo5() { return major_co5; }
    public int toCo5( boolean is_minor ) {
      return is_minor ? major_co5 - 3 : major_co5;
    }
    public String toString() {
      return toStringIn( NOTE_SYMBOL, false );
    }
    public String toStringIn( int language ) {
      return toStringIn( language, false );
    }
    public String toStringIn( int language, boolean is_minor ) {
      int co5_s771 = major_co5 + 15; // Shift 7 + 7 + 1 = 15 steps
      if( is_minor ) {
        // When co5 is for minor (key or chord), shift 3 steps more
        co5_s771 += 3;
      }
      if( co5_s771 < 0 || co5_s771 >= 35 ) {
        //
        // RTނ͈̉̔͂ɓȂ悤ȒlĂ܂ꍇ́A
        // 𒲍ƂČƂ 5b ` 6# ͈̔͂Ɏ܂悤Ȉٖ(enharmonic)ɒuB
        //
        co5_s771 = mod12(co5_s771);  // returns 0(Fbb) ... 7(Fb) 8(Cb) 9(Gb) 10(Db) 11(Ab)
        if( is_minor ) {
          co5_s771 += (co5_s771 == 0 ? 24 : 12);  // 0(Fbbm)+24 = 24(D#m), 1(Cbbm)+12 = 13(Bbm)
        }
        else {
          if( co5_s771 < 10 ) co5_s771 += 12;  // 0(Fbb)+12 = 12(Eb), 9(Gb)+12 = 21(F#)
        }
      }
      int sf_index = co5_s771 / 7;
      int note_index = co5_s771 - sf_index * 7;
      String syms[] = note_strings[language];
      String note = (syms[5]).substring( note_index, note_index+1 );
      String sharp_flat = syms[sf_index];
      return pre_sharp_flat[language] ? sharp_flat + note : note + sharp_flat;
    }
    public int toNoteNo() {
      return mod12(reverseCo5(major_co5)); // range: 0 to 11
    }
  }
  //	MIDI Note No. <-> Co5 converter
  //	MIDIm[gԍƌܓxł̒liW[ŁjƂ̊Ԃ̕ϊ
  //
  public static int reverseCo5(int n) { // Swap C D E <-> Gb Ab Bb
    return (n & 1) == 0 ? n : n+6 ;
  }
  public static int mod12(int n) {
    //
    // m[gԍIN^[u𔲂Bn % 12 ̑ɂ̊֐
    // gČvZA̐łĂ 0`11 ͈̔͂ŕԂ悤ɂB
    // i% Zq̍ӂɕ̐^ƁA̐ɂȂĂ܂j
    // 
    int qn = n % 12;
    return qn < 0 ? qn + 12 : qn ;
  }
  //	MIDI Note No. -> symbol, frequency, etc.
  //
  // MIDIm[gԍ特擾
  //
  //	m[gԍł͕IȉK񂵂Ȃ߁AŁ̂K\
  //	iB#ACb ȂǁjA_uV[vA_utbgg\͕Ԃ܂B
  //
  //	̏ꍇ A ` G ܂ł̕Ȁꍇ́Ɓ̗̕\Ԃ܂B
  //	A̎wɂAƁ̗ԂȂƂ킩ꍇ́A
  //	ܓxŒlOiL[ C / Amj̋AW[A}Ci[̗܂߂
  //	߂ɂق̕\iC# Eb F# Ab Bbĵ݂Ԃ܂B
  //
  public static String noteNoToSymbol(int note_no) {
    return noteNoToSymbol( note_no, 256 );
  }
  public static String noteNoToSymbol(int note_no, int max_chars) {
    int co5 = mod12(reverseCo5(note_no));
    if( co5 == 11 ) {
      return (new NoteSymbol(-1)).toString();
    }
    else if( co5 >= 6 ) {
      if( max_chars >= 7 ) {
        return
          (new NoteSymbol(co5)).toString() + " / " +
          (new NoteSymbol(co5 - 12)).toString();
      }
      else {
        // String capacity not enough
        // Select only one note (sharped or flatted)
        return (new NoteSymbol(
          co5 - ((co5 >= 8) ? 12 : 0))
        ).toString();
      }
    }
    else return (new NoteSymbol(co5)).toString();
  }
  // MIDI m[gԍAA=440Hz Ƃꍇ̉̎g擾
  public static double noteNoToFrequency(int note_no) {
    // Returns frequency when A=440Hz
    return 55 * Math.pow( 2, (double)(note_no - 33)/12 );
  }
  //
  // MIDI m[gԍAkey_co5 Ŏw肳ꂽiܓxł̒lŎwj
  // XP[\ɊYꍇ true AXP[OĂꍇ false Ԃ܂B
  //
  // w肵ȂꍇÂ܂ܔE̔ʂɎg܂
  //itrueFAfalseFjB
  //
  // }Ci[L[̏ꍇAs̃W[L[ɍ킹邽
  // Ƀi`}Ci[XP[ƂȂ܂B
  //
  public static boolean isOnScale(int note_no) {
    return mod12(reverseCo5(note_no) + 1) < 7 ;
  }
  public static boolean isOnScale(int note_no, int key_co5) {
    return mod12(reverseCo5(note_no) - key_co5 + 1) < 7 ;
  }
  //
  // ܓxł̒lŕ\ꂽK𔼉PʂňڒAʂ
  // -5 ` 6 ܂ł͈̔͂ŕԂ܂B
  //
  // Aڒ锼 chromatic_offset ȌꍇA
  // co5 ̒l̂܂ܕԂ܂B
  // Ⴆ co5 == +7 ̏ꍇA-5 ł͂Ȃ +7 Ԃ܂B
  //
  public static int transposeCo5(int co5, int chromatic_offset) {
    if( chromatic_offset == 0 ) return co5;
    int transposed_co5 = mod12( co5 + reverseCo5(chromatic_offset) );
    if( transposed_co5 > 6 ) transposed_co5 -= 12;
    return transposed_co5; // range: -5 to +6
  }
  // ܓx̗ɑlԂ܂B
  //
  public static int oppositeCo5(int co5) {
    return co5 > 0 ? co5 - 6 : co5 + 6;
  }
  //*************************************************************************
  //
  // Range - 
  //
  public static class Range {
    public int min_note = 0;
    public int max_note = MIDISpec.MAX_NOTE_NO;
    public int min_key_offset = 0;
    public boolean is_inversion_mode = true;
    public Range( int min_note, int max_note ) {
      this.min_note = min_note;
      this.max_note = max_note;
    }
    public Range( Integer[] notes ) {
      if( notes == null ) return;
      switch( notes.length ) {
        case 0: return;
        case 1:
          min_note = max_note = notes[0];
          break;
        default:
          if( notes[0] > notes[1] ) {
            min_note = notes[1];
            max_note = notes[0];
          }
          else {
            min_note = notes[0];
            max_note = notes[1];
          }
          break;
      }
    }
    public Range(
      int min_note, int max_note,
      int min_key_offset, boolean inv_mode
    ) {
      this.min_note = min_note;
      this.max_note = max_note;
      this.min_key_offset = min_key_offset;
      this.is_inversion_mode = inv_mode;
    }
    public int invertedNoteOf(int note_no) {
      return invertedNoteOf( note_no, null );
    }
    public int invertedNoteOf(int note_no, Key key) {
      int min_note = this.min_note;
      int max_note = this.max_note;
      int offset = 0;
      if( key != null ) {
        offset = key.relativeDo();
        if( min_key_offset < 0 && offset >= mod12(min_key_offset) ) {
          offset -= 12;
        }
        else if( min_key_offset > 0 && offset < mod12(min_key_offset) ) {
          offset += 12;
        }
        min_note += offset;
        max_note += offset;
      }
      int octave = min_note / 12;
      note_no += 12 * octave;
      while( note_no > max_note ) note_no -= 12;
      while( note_no > MIDISpec.MAX_NOTE_NO ) note_no -= 12;
      while( note_no < min_note ) note_no += 12;
      while( note_no < 0 ) note_no += 12;
      return note_no;
    }
    public void invertNotesOf( int[] notes, Key key ) {
      int i;
      if( is_inversion_mode ) {
        for( i=0; i<notes.length; i++ ) {
          notes[i] = invertedNoteOf( notes[i], key );
        }
      }
      else {
        int n = invertedNoteOf( notes[0], new Key(min_key_offset) );
        int n_diff = n - notes[0];
        notes[0] = n;
        for( i=1; i<notes.length; i++ ) {
          notes[i] += n_diff;
        }
      }
    }
  }
  //
  //*************************************************************************
  //
  //  Key - ̃NX
  //
  public static class Key implements Cloneable {
    public static final int MAJOR_OR_MINOR = 0;
    public static final int MAJOR = 1;
    public static final int MINOR = -1;
    private static final String key_strings[][] = {
      { "", "m", " / " },
      { " major", " minor", " / " },
      { "", "Z", "^" },
    };
    private int co5;
    private int major_minor;
    //
    public Key() { setKey( 0, MAJOR_OR_MINOR ); }
    public Key( int co5 ) {
      setKey( co5, MAJOR_OR_MINOR );
    }
    public Key( int co5, int major_minor ) {
      setKey( co5, major_minor );
    }
    public Key( int co5, boolean is_minor ) {
      setKey( co5, is_minor );
    }
    public Key( byte midi_data[] ) {
      setBytes( midi_data );
    }
    public Key( String key_symbol ) {
      boolean is_minor = key_symbol.matches(".*m");
      setKey(
        (new NoteSymbol(key_symbol)).toCo5(is_minor),
        is_minor
      );
    }
    public Key( Chord chord ) {
      boolean is_minor = chord.isMinor();
      setKey(
        chord.rootNoteSymbol().toCo5(is_minor),
        is_minor
      );
    }
    protected Key clone() {
      return new Key( co5, major_minor );
    }
    public boolean equals(Key another_key) {
      return
        co5 == another_key.toCo5() &&
        major_minor == another_key.majorMinor() ;
    }
    public int hashCode() {
      return (1 - major_minor) * 256 + co5 ;
    }
    private void setKey( int co5, boolean is_minor ) {
      setKey( co5, is_minor ? MINOR : MAJOR );
    }
    private void setKey( int co5, int major_minor ) {
      this.co5 = co5;
      this.major_minor = major_minor;
      normalize();
    }
    public void setBytes( byte[] data ) {
      // MIDI ̃^bZ[W擾f[^i2bytejƂɂĒZbg
      //
      // ̐ivXlj܂́̐i}CiXlj
      byte sharp_flat = data.length > 0 ? data[0] : 0;
      //
      // O@PZ
      byte is_minor = data.length > 1 ? data[1] : 0;
      //
      setKey( (int)sharp_flat, is_minor==1 );      
    }
    public byte[] getBytes() {
      byte data[] = new byte[2];
      data[0] = (byte)(co5 & 0xFF);
      data[1] = (byte)(major_minor == MINOR ? 1 : 0);
      return data;
    }
    public int toCo5() { return co5; }
    public int majorMinor() { return major_minor; }
    public int relativeDo() { return (new NoteSymbol(co5)).toNoteNo(); }
    public int rootNote() {
      int n = relativeDo();
      return major_minor==MINOR ? mod12(n-3) : n;
    }
    public boolean isOnScale( int note_no ) {
      return Music.isOnScale( note_no, co5 );
    }
    public Key transpose(int chromatic_offset) { // ڒ
      co5 = transposeCo5( co5, chromatic_offset );
      return this;
    }
    public void toggleEnharmonically() { // ٖؑ
      if( co5 > 4 )
        co5 -= 12;
      else if( co5 < -4 )
        co5 += 12;
    }
    public void normalize() { // Ki̐Vȉɂj
      if( co5 < -7 || co5 > 7 ) {
        co5 = Music.mod12( co5 );
        if( co5 > 6 ) co5 -= 12;
      }
    }
    public Key relativeKey() { // s
      return new Key( co5, major_minor * (-1) );
    }
    public Key parallelKey() { // 咲
      switch( major_minor ) {
      case MAJOR: return new Key( co5-3, MINOR );
      case MINOR: return new Key( co5+3, MAJOR );
      default: return new Key(co5);
      }
    }
    public Key oppositeKey() { // ܓxŐ^ɂ钲
      return new Key(
        Music.oppositeCo5(co5),
        major_minor
      );
    }
    public String toString() {
      return toStringIn( NoteSymbol.NOTE_SYMBOL );
    }
    public String toStringIn( int language ) {
      NoteSymbol note = new NoteSymbol(co5);
      String major
        = note.toStringIn( language, false ) + key_strings[language][0];
      if( major_minor > 0 ) {
        return major;
      }
      else {
        String minor
          = note.toStringIn( language, true ) + key_strings[language][1];
        return major_minor < 0 ?
          minor :
          major + key_strings[language][2] + minor ;
      }
    }
    public String signature() { // 
      switch(co5) {
      case  0: return "==";
      case  1: return "#";
      case -1: return "b";
      case  2: return "##";
      case -2: return "bb";
      default:
        if( co5 >= 3 && co5 <= 7 ) return co5 + "#" ;
        else if( co5 <= -3 && co5 >= -7 ) return (-co5) + "b" ;
        return "";
      }
    }
    public String signatureDescription() {
      switch(co5) {
      case  0: return "no sharps or flats";
      case  1: return "1 sharp";
      case -1: return "1 flat";
      default: return co5 < 0 ? (-co5) + " flats" : co5 + " sharps" ;
      }
    }
  }


  //*************************************************************************
  //
  // Music chord class implementation
  //
  // aiR[hj̃NX
  //
  public static class Chord implements Cloneable {
    //
    // R[h\̏ɑΉF
    //
    // Standard color for note index of the chord
    public static final Color index_colors[] = {
      Color.red,
      new Color(0x40,0x40,0xFF),
      Color.orange.darker(),
      new Color(0x20,0x99,0x00),
      Color.magenta,
      Color.orange,
      Color.green
    };
    //
    // Root/Bass value in major Circle-of-fifths
    //
    // ܓx̃W[L[ł̒lŕ\[gƃx[X
    // i҂̒lقȂꍇ͕R[hɂȂj
    //
    private NoteSymbol root_note_symbol = new NoteSymbol(0);
    private NoteSymbol bass_note_symbol = new NoteSymbol(0);
    //
    // Root note value in MIDI note No. modulo-12
    //
    // MIDI m[gԍ\iO`PPj
    //
    private int bassnote = 0;
    private int rootnote = 0;
    //
    // [g̔
    //
    public static final int ROOT = 0;	// [giȂ̂Ɠj
    //
    public static final int SUS2 = 2;	// Qx
    public static final int MINOR = 3;	// ZRxA܂͑Qx
    public static final int MAJOR = 4;	// Rx
    public static final int SUS4 = 5;	// SSx
    //
    public static final int FLAT5 = 6;    // TxA܂͑Sx
    public static final int PARFECT5 = 7; // STx
    public static final int SHARP5 = 8;   // TxA܂͒ZUx
    //
    public static final int SIXTH = 9;		// UxA܂͌Vx
    public static final int SEVENTH = 10;	// ZVx
    public static final int MAJOR_SEVENTH = 11;	// Vx
    //
    public static final int FLAT9 = 13;		// ZXxiZQx̂PIN^[uj
    public static final int NINTH = 14;		// XxiQx̂PIN^[uj
    public static final int SHARP9 = 15;	// XxiQx̂PIN^[uj
    //
    public static final int ELEVENTH = 17;	// SPPxiSSx̂PIN^[uj
    public static final int SHARP11 = 18;	// PPxiSx̂PIN^[uj
    //
    public static final int FLAT13 = 20;	// ZPRxiZUx̂PIN^[uj
    public static final int THIRTEENTH = 21;	// PRxiUx̂PIN^[uj
    //
    // index
    public static final int THIRD_OFFSET	= 0;	// Rx
    public static final int FIFTH_OFFSET	= 1;	// Tx
    public static final int SEVENTH_OFFSET	= 2;	// Vx
    public static final int NINTH_OFFSET	= 3;	// Xx
    public static final int ELEVENTH_OFFSET	= 4;	// PPx
    public static final int THIRTEENTH_OFFSET	= 5;	// PRx
    public static int default_offsets[] = {
      MAJOR, PARFECT5, ROOT, ROOT, ROOT, ROOT,
    };
    public int offsets[];
    //
    // Constructer
    //
    public Chord() { // R[h C major VKɍ쐬
      this(0);
    }
    public Chord( int root_co5 ) { // [gR[h𐶐
      offsets = Arrays.copyOf( default_offsets, default_offsets.length );
      setRootAndBassByCo5(root_co5);
    }
    public Chord( NoteSymbol note_symbol ) {
      this( note_symbol.toCo5() );
    }
    public Chord( String chord_symbol ) { // R[hl[̕񂩂R[h𐶐
      offsets = Arrays.copyOf( default_offsets, default_offsets.length );
      setChordSymbol( chord_symbol );
    }
    public Chord( Key key ) { // L[Ɠ̃R[h𐶐
      offsets = Arrays.copyOf( default_offsets, default_offsets.length );
      int kco5 = key.toCo5();
      if( key.majorMinor() == key.MINOR ) {
        // }Ci[L[̏ꍇ̓}Ci[R[h𐶐
        setRootAndBassByCo5( kco5 + 3 );
        setMinorThird();
      }
      else {
        // W[L[A܂̓L[̃W[}Ci[sȏꍇ
        // W[R[h𐶐
        setRootAndBassByCo5( kco5 );
      }
    }
    protected Chord clone() {
      Chord new_chord = new Chord( root_note_symbol );
      new_chord.offsets = Arrays.copyOf( offsets, offsets.length );
      new_chord.setBassByCo5( bass_note_symbol.toCo5() );
      return new_chord;
    }
    //
    // R[h̃[g^x[XAW[Ƃܓxł̈ʒuŎw肵܂B
    //
    // Ⴆ΁AAm 邽߂Ƀ[g A ɐݒ肵Ƃ́A
    // L[ƂĂ Am \ 0 ł͂ȂAKƂĂ A \ 3 w肵܂B
    //
    // ٖʂłȂȂĂ܂Ƃh߁A
    // m[gԍŒڎw肷C^[tF[X͂Đ݂Ă܂B
    //
    // Sets root note by circle-of-fifth major value
    //
    public Chord setRootByCo5(int co5) {
      rootnote = (
        root_note_symbol = new NoteSymbol(co5)
      ).toNoteNo();
      return this;
    }
    public Chord setBassByCo5(int co5) {
      bassnote = (
        bass_note_symbol = new NoteSymbol(co5)
      ).toNoteNo();
      return this;
    }
    public Chord setRootAndBassByCo5(int co5) {
      setRootByCo5(co5);
      setBassByCo5(co5);
      return this;
    }
    // R[h̎ނݒ肵܂B
    //
    public void setMajorThird() { offsets[THIRD_OFFSET] = MAJOR; }
    public void setMinorThird() { offsets[THIRD_OFFSET] = MINOR; }
    public void setSus4() { offsets[THIRD_OFFSET] = SUS4; }
    public void setSus2() { offsets[THIRD_OFFSET] = SUS2; }
    //
    public void setParfectFifth() { offsets[FIFTH_OFFSET] = PARFECT5; }
    public void setFlattedFifth() { offsets[FIFTH_OFFSET] = FLAT5; }
    public void setSharpedFifth() { offsets[FIFTH_OFFSET] = SHARP5; }
    //
    public void clearSeventh() { offsets[SEVENTH_OFFSET] = ROOT; }
    public void setMajorSeventh() { offsets[SEVENTH_OFFSET] = MAJOR_SEVENTH; }
    public void setSeventh() { offsets[SEVENTH_OFFSET] = SEVENTH; }
    public void setSixth() { offsets[SEVENTH_OFFSET] = SIXTH; }
    //
    public void clearNinth() { offsets[NINTH_OFFSET] = ROOT; }
    public void setNinth() { offsets[NINTH_OFFSET] = NINTH; }
    public void setSharpedNinth() { offsets[NINTH_OFFSET] = SHARP9; }
    public void setFlattedNinth() { offsets[NINTH_OFFSET] = FLAT9; }
    //
    public void clearEleventh() { offsets[ELEVENTH_OFFSET] = ROOT; }
    public void setEleventh() { offsets[ELEVENTH_OFFSET] = ELEVENTH; }
    public void setSharpedEleventh() { offsets[ELEVENTH_OFFSET] = SHARP11; }
    //
    public void clearThirteenth() { offsets[THIRTEENTH_OFFSET] = ROOT; }
    public void setThirteenth() { offsets[THIRTEENTH_OFFSET] = THIRTEENTH; }
    public void setFlattedThirteenth() { offsets[THIRTEENTH_OFFSET] = FLAT13; }
    //
    // R[hl[̕񂪎R[hɒu܂B
    public Chord setChordSymbol(String chord_symbol) {
      //
      // R[h̕qƕɕ
      String parts[] = chord_symbol.trim().split("(/|on)");
      if( parts.length == 0 ) {
        return this;
      }
      // [gƃx[Xݒ
      setRootByCo5( (new NoteSymbol(parts[0])).toCo5() );
      setBassByCo5( (new NoteSymbol(parts[ parts.length > 1 ? 1 : 0 ])).toCo5() );
      String suffix = parts[0].replaceFirst("^[A-G][#bx]*","");
      //
      // () ΁A̒go
      String suffix_parts[] = suffix.split("[\\(\\)]");
      if( suffix_parts.length == 0 ) {
        return this;
      }
      String suffix_paren = "";
      if( suffix_parts.length > 1 ) {
        suffix_paren = suffix_parts[1];
        suffix = suffix_parts[0];
      }
      //
      // +5 -5 aug dim ̔
      offsets[FIFTH_OFFSET] = (
        suffix.matches( ".*(\\+5|aug|#5).*" ) ? SHARP5 :
        suffix.matches( ".*(-5|dim|b5).*" ) ? FLAT5 :
        PARFECT5
      );
      // 6 7 M7 ̔
      offsets[SEVENTH_OFFSET] = (
        suffix.matches( ".*(M7|maj7|M9|maj9).*" ) ? MAJOR_SEVENTH :
        suffix.matches( ".*(6|dim[79]).*" ) ? SIXTH :
        suffix.matches( ".*7.*" ) ? SEVENTH :
        ROOT
      );
      // }Ci[̔Bmaj7 ƊԈႦȂ悤ɔr
      offsets[THIRD_OFFSET] = (
        (suffix.matches( ".*m.*" ) && ! suffix.matches(".*ma.*") ) ? MINOR :
        suffix.matches( ".*sus4.*" ) ? SUS4 :
        MAJOR
      );
      // 9th ̔
      if( suffix.matches( ".*9.*" ) ) {
        offsets[NINTH_OFFSET] = NINTH;
        if( ! suffix.matches( ".*(add9|6|M9|maj9|dim9).*") ) {
          offsets[SEVENTH_OFFSET] = SEVENTH;
        }
      }
      else {
        offsets[NINTH_OFFSET] =
        offsets[ELEVENTH_OFFSET] =
        offsets[THIRTEENTH_OFFSET] = ROOT;
        //
        // () ̒ , ŕ
        String parts_in_paren[] = suffix_paren.split(",");
        for( String p : parts_in_paren ) {
          if( p.matches("(\\+9|#9)") ) offsets[NINTH_OFFSET] = SHARP9;
          else if( p.matches("(-9|b9)") ) offsets[NINTH_OFFSET] = FLAT9;
          else if( p.matches("9") ) offsets[NINTH_OFFSET] = NINTH;

          if( p.matches("(\\+11|#11)") ) offsets[ELEVENTH_OFFSET] = SHARP11;
          else if( p.matches("11") ) offsets[ELEVENTH_OFFSET] = ELEVENTH;

          if( p.matches("(-13|b13)") ) offsets[THIRTEENTH_OFFSET] = FLAT13;
          else if( p.matches("13") ) offsets[THIRTEENTH_OFFSET] = THIRTEENTH;

          // -5  +5  () ̒ɂĂ߂ł悤ɂ
          if( p.matches("(-5|b5)") ) offsets[FIFTH_OFFSET] = FLAT5;
          else if( p.matches("(\\+5|#5)") ) offsets[FIFTH_OFFSET] = SHARP5;
        }
      }
      return this;
    }
    // [g^x[X̉Ԃ܂B
    public NoteSymbol rootNoteSymbol() { return root_note_symbol; }
    public NoteSymbol bassNoteSymbol() { return bass_note_symbol; }
    //
    // [g^x[Xm[gԍŕԂ܂ȉٖ͔܂j
    public int rootNote() { return rootnote; }
    public int bassNote() { return bassnote; }
    //
    // R[h̎ނ𒲂ׂ܂B
    public boolean isMajor() { return offsets[THIRD_OFFSET] == MAJOR; }
    public boolean isMinor() { return offsets[THIRD_OFFSET] == MINOR; }
    public boolean isSus4() { return offsets[THIRD_OFFSET] == SUS4; }
    public boolean isSus2() { return offsets[THIRD_OFFSET] == SUS2; }
    //
    public boolean hasParfectFifth() { return offsets[FIFTH_OFFSET] == PARFECT5; }
    public boolean hasFlattedFifth() { return offsets[FIFTH_OFFSET] == FLAT5; }
    public boolean hasSharpedFifth() { return offsets[FIFTH_OFFSET] == SHARP5; }
    //
    public boolean hasNoSeventh() { return offsets[SEVENTH_OFFSET] == ROOT; }
    public boolean hasSeventh() { return offsets[SEVENTH_OFFSET] == SEVENTH; }
    public boolean hasMajorSeventh() { return offsets[SEVENTH_OFFSET] == MAJOR_SEVENTH; }
    public boolean hasSixth() { return offsets[SEVENTH_OFFSET] == SIXTH; }
    //
    public boolean hasNoNinth() { return offsets[NINTH_OFFSET] == ROOT; }
    public boolean hasNinth() { return offsets[NINTH_OFFSET] == NINTH; }
    public boolean hasFlattedNinth() { return offsets[NINTH_OFFSET] == FLAT9; }
    public boolean hasSharpedNinth() { return offsets[NINTH_OFFSET] == SHARP9; }
    //
    public boolean hasNoEleventh() { return offsets[ELEVENTH_OFFSET] == ROOT; }
    public boolean hasEleventh() { return offsets[ELEVENTH_OFFSET] == ELEVENTH; }
    public boolean hasSharpedEleventh() { return offsets[ELEVENTH_OFFSET] == SHARP11; }
    //
    public boolean hasNoThirteenth() { return offsets[THIRTEENTH_OFFSET] == ROOT; }
    public boolean hasThirteenth() { return offsets[THIRTEENTH_OFFSET] == THIRTEENTH; }
    public boolean hasFlattedThirteenth() { return offsets[THIRTEENTH_OFFSET] == FLAT13; }
    //
    // ̃R[hł邩ǂ𔻒肵܂B
    //
    public boolean equals( Chord another_chord ) {
      // ٖ̏ꍇAقȂ̂Ƃ
      return (
        another_chord != null
        && root_note_symbol.toCo5() == another_chord.rootNoteSymbol().toCo5()
        && bass_note_symbol.toCo5() == another_chord.bassNoteSymbol().toCo5()
        && Arrays.equals( offsets, another_chord.offsets )
      );
    }
    public boolean equalsEnharmonically( Chord another_chord ) {
      // ٖ̏ꍇɂƔ
      return (
        another_chord != null
        && rootnote == another_chord.rootNote()
        && bassnote == another_chord.bassNote()
        && Arrays.equals( offsets, another_chord.offsets )
      );
    }
    public int hashCode() { return toString().hashCode(); }
    //
    // R[h\̐Ԃ܂ix[X͊܂܂܂jB
    public int numberOfNotes() {
      int n=1;
      for( int offset : offsets ) if( offset != ROOT ) n++;
      return n;
    }
    // w肳ꂽCfbNXʒuɂm[gԍԂ܂B
    // CfbNXʒúAO[gƂ\̏ŕ\܂B
    //
    public int noteAt(int index) {
      if( index == 0 ) return rootnote;
      int i=0;
      for( int offset : offsets )
        if( offset != ROOT && ++i == index )
          return rootnote + offset;
      return -1;
    }
    // R[h\i[m[gԍ̔zԂ܂B
    //ix[X͊܂܂܂j
    // 悪w肳ꂽꍇẢɍ킹m[gԍԂ܂B
    //
    public int[] toNoteArray() {
      return toNoteArray( (Range)null, (Key)null );
    }
    public int[] toNoteArray(Range range) {
      return toNoteArray( range, (Key)null );
    }
    public int[] toNoteArray(Range range, Key key) {
      int ia[] = new int[numberOfNotes()];
      int i;
      ia[i=0] = rootnote;
      for( int offset : offsets )
        if( offset != ROOT )
          ia[++i] = rootnote + offset;
      if( range != null ) range.invertNotesOf( ia, key );
      return ia;
    }
    // MIDI m[gԍAR[h̍\̉ԖځiO[gj
    // 邩Ԃ܂B\ɊYȂꍇ -1 Ԃ܂B
    // x[X͌܂B
    // 
    public int indexOf(int note_no) {
      int relative_note = note_no - rootnote;
      if( mod12(relative_note) == 0 ) return 0;
      int i=0;
      for( int offset : offsets ) if( offset != ROOT ) {
        i++;
        if( mod12(relative_note - offset) == 0 )
          return i;
      }
      return -1;
    }
    // \̃L[̃XP[OĂȂׂ܂B
    public boolean isOnScaleInKey( Key key ) {
      return isOnScaleInKey( key.toCo5() );
    }
    public boolean isOnScaleInKey( int key_co5 ) {
      if( ! isOnScale( rootnote, key_co5 ) ) return false;
      for( int offset : offsets )
        if( offset != ROOT && ! isOnScale( rootnote + offset, key_co5 ) )
          return false;
      return true;
    }
    //
    // R[hڒ܂B
    // ڒ chromatic_offset ŔPʂɎw肵܂B
    // ڒȌꍇAĝ܂ܕԂ܂B
    //
    public Chord transpose(int chromatic_offset) {
      return transpose(chromatic_offset, 0);
    }
    public Chord transpose(int chromatic_offset, Key original_key) {
      return transpose(chromatic_offset, original_key.toCo5());
    }
    public Chord transpose(int chromatic_offset, int original_key_co5) {
      if( chromatic_offset == 0 ) return this;
      int offset_co5 = mod12(reverseCo5(chromatic_offset));
      if( offset_co5 > 6 ) offset_co5 -= 12;
      int key_co5   = original_key_co5 + offset_co5;
      //
      int new_root_co5 = root_note_symbol.toCo5() + offset_co5;
      int new_bass_co5 = bass_note_symbol.toCo5() + offset_co5;
      if( key_co5 > 6 ) {
        new_root_co5 -= 12;
        new_bass_co5 -= 12;
      }
      else if( key_co5 < -5 ) {
        new_root_co5 += 12;
        new_bass_co5 += 12;
      }
      setRootByCo5(new_root_co5);
      return setBassByCo5(new_bass_co5);
    }
    // ̃IuWFNg̕\Ԃ܂B
    // ia\IuWFNgȂ̂ŃR[hl[Ԃ悤ɂĂ܂j
    //
    public String toString() {
      String chord_symbol = rootNoteSymbol() + symbolSuffix();
      if( ! root_note_symbol.equals(bass_note_symbol) ) {
        chord_symbol += "/" + bassNoteSymbol();
      }
      return chord_symbol;
    }
    // R[hl[ HTML ŕԂ܂B
    // F HTML ̕\iF܂ #RRGGBB `jŎw肵܂B
    //
    // Swing  JLabel.setText()  HTML Ŏwł̂ŁA
    // ̑傫ɕω邱Ƃł܂B
    //
    public String toHtmlString(String color_name) {
      String small_tag = "<span style=\"font-size: 120%\">";
      String end_of_small_tag = "</span>";
      String root = root_note_symbol.toString();
      String formatted_root = (root.length() == 1) ?
        root + small_tag :
        root.replace("#",small_tag+"<sup>#</sup>").
          replace("b",small_tag+"<sup>b</sup>").
          replace("x",small_tag+"<sup>x</sup>");
      String formatted_bass = "";
      if( ! root_note_symbol.equals(bass_note_symbol) ) {
        String bass = bass_note_symbol.toString();
        formatted_bass = (bass.length() == 1) ?
          bass + small_tag :
          bass.replace("#",small_tag+"<sup>#</sup>").
            replace("b",small_tag+"<sup>b</sup>").
            replace("x",small_tag+"<sup>x</sup>");
        formatted_bass = "/" + formatted_bass + end_of_small_tag;
      }
      String suffix = symbolSuffix().
        replace("-5","<sup>-5</sup>").
        replace("+5","<sup>+5</sup>");
      return
        "<html>" +
          "<span style=\"color: " + color_name + "; font-size: 170% ; white-space: nowrap ;\">" +
            formatted_root + suffix + end_of_small_tag + formatted_bass +
          "</span>" +
        "</html>" ;
    }
    //
    // R[h̐ipjԂ܂B
    public String toName() {
      String chord_name = root_note_symbol.toStringIn(NoteSymbol.NOTE_NAME) + nameSuffix() ;
      if( ! root_note_symbol.equals(bass_note_symbol) ) {
        chord_name += " on " + bass_note_symbol.toStringIn(NoteSymbol.NOTE_NAME);
      }
      return chord_name;
    }
    // R[hl[̉gݗĂ܂B
    public String symbolSuffix() {
      String suffix = ( ( offsets[THIRD_OFFSET] == MINOR ) ? "m" : "" );
      switch( offsets[SEVENTH_OFFSET] ) {
      case SIXTH:         suffix += "6";  break;
      case SEVENTH:       suffix += "7";  break;
      case MAJOR_SEVENTH: suffix += "M7"; break;
      }
      switch( offsets[THIRD_OFFSET] ) {
      case SUS4: suffix += "sus4"; break;
      case SUS2: suffix += "sus2"; break;
      default: break;
      }
      switch( offsets[FIFTH_OFFSET] ) {
      case FLAT5:  suffix += "-5"; break;
      case SHARP5: suffix += "+5"; break;
      default: break;
      }
      Vector<String> paren = new Vector<String>();
      switch( offsets[NINTH_OFFSET] ) {
      case NINTH:  paren.add("9"); break;
      case FLAT9:  paren.add("-9"); break;
      case SHARP9: paren.add("+9"); break;
      }
      switch( offsets[ELEVENTH_OFFSET] ) {
      case ELEVENTH: paren.add("11"); break;
      case SHARP11:  paren.add("+11"); break;
      }
      switch( offsets[THIRTEENTH_OFFSET] ) {
      case THIRTEENTH: paren.add("13"); break;
      case FLAT13:     paren.add("-13"); break;
      }
      if( paren.size() > 0 ) {
        boolean is_first = true;
        suffix += "(";
        for( String p : paren ) {
          if( is_first )
            is_first = false;
          else
            suffix += ",";
          suffix += p;
        }
        suffix += ")";
      }
      if( suffix.equals("m-5") ) return "dim";
      else if( suffix.equals("+5") ) return "aug";
      else if( suffix.equals("m6-5") ) return "dim7";
      else if( suffix.equals("(9)") ) return "add9";
      else if( suffix.equals("7(9)") ) return "9";
      else if( suffix.equals("M7(9)") ) return "M9";
      else if( suffix.equals("7+5") ) return "aug7";
      else if( suffix.equals("m6-5(9)") ) return "dim9";
      else return suffix ;
    }
    // R[h̐̂AgݗĂ܂B
    public String nameSuffix() {
      String suffix = "";
      if( offsets[THIRD_OFFSET] == MINOR ) suffix += " minor";
      switch( offsets[SEVENTH_OFFSET] ) {
      case SIXTH:         suffix += " 6th"; break;
      case SEVENTH:       suffix += " 7th"; break;
      case MAJOR_SEVENTH: suffix += " major 7th"; break;
      }
      switch( offsets[THIRD_OFFSET] ) {
      case SUS4: suffix += " suspended 4th"; break;
      case SUS2: suffix += " suspended 2nd"; break;
      default: break;
      }
      switch( offsets[FIFTH_OFFSET] ) {
      case FLAT5 : suffix += " flatted 5th"; break;
      case SHARP5: suffix += " sharped 5th"; break;
      default: break;
      }
      Vector<String> paren = new Vector<String>();
      switch( offsets[NINTH_OFFSET] ) {
      case NINTH:  paren.add("9th"); break;
      case FLAT9:  paren.add("flatted 9th"); break;
      case SHARP9: paren.add("sharped 9th"); break;
      }
      switch( offsets[ELEVENTH_OFFSET] ) {
      case ELEVENTH: paren.add("11th"); break;
      case SHARP11:  paren.add("sharped 11th"); break;
      }
      switch( offsets[THIRTEENTH_OFFSET] ) {
      case THIRTEENTH: paren.add("13th"); break;
      case FLAT13:     paren.add("flatted 13th"); break;
      }
      if( paren.size() > 0 ) {
        boolean is_first = true;
        suffix += "(additional ";
        for( String p : paren ) {
          if( is_first )
            is_first = false;
          else
            suffix += ",";
          suffix += p;
        }
        suffix += ")";
      }
      if( suffix.equals(" minor flatted 5th") ) return " diminished (triad)";
      else if( suffix.equals(" sharped 5th") ) return " augumented";
      else if( suffix.equals(" minor 6th flatted 5th") ) return " diminished 7th";
      else if( suffix.equals(" 7th(additional 9th)") ) return " 9th";
      else if( suffix.equals(" major 7th(additional 9th)") ) return " major 9th";
      else if( suffix.equals(" 7th sharped 5th") ) return " augumented 7th";
      else if( suffix.equals(" minor 6th flatted 5th(additional 9th)") ) return " diminished 9th";
      else if( suffix.isEmpty() ) return " major";
      else return suffix ;
    }
  }


  //******************************************************************
  //
  // Chord Progression - R[his̃NX
  //
  //******************************************************************
  public static class ChordProgression {

    private class TickRange implements Cloneable {
      long start_tick_pos = 0, end_tick_pos = 0;
      public TickRange() { }
      public TickRange( long tick_pos ) {
        end_tick_pos = start_tick_pos = tick_pos;
      }
      public TickRange( long start_tick_pos, long end_tick_pos ) {
        this.start_tick_pos = start_tick_pos;
        this.end_tick_pos = end_tick_pos;
      }
      protected TickRange clone() {
        return new TickRange( start_tick_pos, end_tick_pos );
      }
      public void moveForward() {
        start_tick_pos = end_tick_pos;
      }
      public void moveForward( long duration ) {
        start_tick_pos = end_tick_pos;
        end_tick_pos += duration;
      }
      public long duration() {
        return end_tick_pos - start_tick_pos;
      }
      public boolean contains( long tick ) {
        return ( tick >= start_tick_pos && tick < end_tick_pos );
      }
    }

    private class ChordStroke {
      Chord chord; int beat_length; TickRange tick_range = null;
      public ChordStroke(Chord chord) { this( chord, 1 ); }
      public ChordStroke(Chord chord, int beat_length) {
        this.chord = chord;
        this.beat_length = beat_length;
      }
      public String toString() {
        String str = chord.toString();
        for( int i=2; i <= beat_length; i++ ) str += " %";
        return str;
      }
    }

    // Ԉʒut̎
    private class Lyrics {
      String text = null;
      Long start_tick_pos = null;
      public Lyrics(String text) { this.text = text; }
      public Lyrics(String text, long tick_pos) {
        this.text = text; start_tick_pos = tick_pos;
      }
      public String toString() { return text; }
    }

    private class Measure extends Vector<Object> {
      Long ticks_per_beat = null;
      public int numberOfBeats() {
        int n = 0;
        for( Object obj : this ) {
          if( obj instanceof ChordStroke ) {
            n += ((ChordStroke)obj).beat_length;
          }
        }
        return n;
      }
      // ߓ̃R[hXg[NԓIɓԊuǂׂB
      // ԊȕꍇAeLXgo͎ % KvȂȂB
      public boolean isEquallyDivided() {
        int l, l_prev = 0;
        for( Object obj : this ) {
          if( obj instanceof ChordStroke ) {
            l = ((ChordStroke)obj).beat_length;
            if( l_prev > 0 && l_prev != l ) {
              return false;
            }
            l_prev = l;
          }
        }
        return true;
      }
      public int addBeat() { return addBeat(1); }
      public int addBeat(int num_beats) {
        ChordStroke last_chord_stroke = null;
        for( Object obj : this ) {
          if( obj instanceof ChordStroke ) {
            last_chord_stroke = (ChordStroke)obj;
          }
        }
        if( last_chord_stroke == null ) {
          return 0;
        }
        return last_chord_stroke.beat_length += num_beats;
      }
      public String toString() {
        String str = "";
        boolean is_eq_dev = isEquallyDivided();
        for( Object element : this ) {
          str += " ";
          if( element instanceof ChordStroke ) {
            ChordStroke cs = (ChordStroke)element;
            str += is_eq_dev ? cs.chord : cs;
          }
          else if( element instanceof Lyrics ) {
            str += (String)element;
          }
        }
        return str;
      }
      public TickRange getRange() {
        long start_tick_pos = -1;
        long end_tick_pos = -1;
        for( Object element : this ) {
          if( ! (element instanceof ChordProgression.ChordStroke) )
            continue;
          ChordProgression.ChordStroke chord_stroke
            = (ChordProgression.ChordStroke)element;
          Chord chord = chord_stroke.chord;
          // ߂̐擪Ɩ tick ߂
          if( start_tick_pos < 0 ) {
            start_tick_pos = chord_stroke.tick_range.start_tick_pos;
          }
          end_tick_pos = chord_stroke.tick_range.end_tick_pos;
        }
        if( start_tick_pos < 0 || end_tick_pos < 0 ) {
          return null;
        }
        return new TickRange( start_tick_pos, end_tick_pos );
      }
      public ChordStroke chordStrokeAt( long tick ) {
        for( Object element : this ) {
          if( ! (element instanceof ChordProgression.ChordStroke) )
            continue;
          ChordProgression.ChordStroke chord_stroke
            = (ChordProgression.ChordStroke)element;
          if( chord_stroke.tick_range.contains(tick) ) {
            return chord_stroke;
          }
        }
        return null;
      }
    }
    private class Line extends Vector<Measure> {
      public String toString() {
        String str = "";
        for( Measure measure : this ) str += measure + "|";
        return str;
      }
    }

    // ϐ
    private Vector<Line> lines = null;
    private Key key = null;
    private Long ticks_per_measure = null;

    public Key getKey() { return key; }
    public void setKey(Key key) { this.key = key; }

    public String toString() {
      String str = "";
      if( key != null ) str += "Key: " + key + "\n";
      for( Line line : lines ) str += line + "\n";
      return str;
    }

    // w肳ꂽߐAL[Aqɍ킹R[his_Ɏ
    //
    public ChordProgression() { }
    public ChordProgression( int measure_length, int timesig_upper ) {
      int key_co5 = (int)(Math.random() * 12) - 5;
      key = new Key( key_co5, Key.MAJOR );
      lines = new Vector<Line>();
      Line line = new Line();
      boolean is_end;
      Chord chord, prev_chord = new Chord( key_co5 );
      int co5_offset, prev_co5_offset;
      double r;
      for( int mp=0; mp<measure_length; mp++ ) {
        is_end = (mp == 0 || mp == measure_length - 1); // ŏ܂͍Ō̏߂oĂ
        Measure measure = new Measure();
        ChordStroke last_chord_stroke = null;
        for( int i=0; i<timesig_upper; i++ ) {
          if(
            i % 4 == 2 && Math.random() < 0.8
            ||
            i % 2 == 1 && Math.random() < 0.9
          ){
            // ꔏ
            last_chord_stroke.beat_length++;
            continue;
          }
          chord = new Chord( key_co5 ); co5_offset = 0;
          prev_co5_offset = prev_chord.rootNoteSymbol().toCo5() - key_co5;
          if( ! is_end ) {
            //
            // ŏ܂͍Ō̏߂͏ɃgjbNɂB
            // Sܓxis{ƂAXłȂiso悤ɂB
            // Tuh~ig𒴂ƃXP[ÔŁAɂȂ烉_ɌߒB
            //
            r = Math.random();
            co5_offset = prev_co5_offset - 1;
            if( co5_offset < -1 || (prev_co5_offset < 5 && r < 0.5) ) {
              //
              // Vx[gƂȂR[h̏om𔼌ȂR[h߂
              // i]肪ÛƂVxj
              // ȂAOƓR[h͎gȂ悤ɂB
              do {
                co5_offset = (int)(Math.random() * 13) % 7 - 1;
              } while( co5_offset == prev_co5_offset );
            }
            chord.setRootAndBassByCo5(key_co5 + co5_offset);
          }
          switch( co5_offset ) {
            // [gƂɁA7th Ȃǂ̕tAW[}Ci[]sm߂
            case 5: // VII
              if( Math.random() < 0.5 ) {
                // m7-5
                chord.setMinorThird();
                chord.setFlattedFifth();
              }
              if( Math.random() < 0.8 ) chord.setSeventh();
              break;
            case 4: // Secondary dominant (III)
              if( prev_co5_offset == 5 ) {
                // [gVxRx̐iŝƂA]mグB
                // inł Bm7-5 ̎ E7 o₷j
                if( Math.random() < 0.2 ) chord.setMinorThird();
              }
              else {
                if( Math.random() < 0.8 ) chord.setMinorThird();
              }
              if( Math.random() < 0.7 ) chord.setSeventh();
              break;
            case 3: // VI
              if( Math.random() < 0.8 ) chord.setMinorThird();
              if( Math.random() < 0.7 ) chord.setSeventh();
              break;
            case 2: // II
              if( Math.random() < 0.8 ) chord.setMinorThird();
              if( Math.random() < 0.7 ) chord.setSeventh();
              break;
            case 1: // Dominant (V)
              if( Math.random() < 0.1 ) chord.setMinorThird();
              if( Math.random() < 0.3 ) chord.setSeventh();
              if( Math.random() < 0.2 ) chord.setNinth();
              break;
            case 0: // ToniciŃ}Ci[ŏIƂ݂̂ setMinorThird() ͂Ȃj
              if( Math.random() < 0.2 ) chord.setMajorSeventh();
              if( Math.random() < 0.2 ) chord.setNinth();
              break;
            case -1: // Sub-dominant (IV)
              if( Math.random() < 0.1 ) {
                chord.setMinorThird();
                if( Math.random() < 0.3 ) chord.setSeventh();
              }
              else
                if( Math.random() < 0.2 ) chord.setMajorSeventh();
              if( Math.random() < 0.2 ) chord.setNinth();
              break;
          }
          measure.add( last_chord_stroke = new ChordStroke(chord) );
          prev_chord = chord;
        }
        line.add(measure);
        if( (mp+1) % 8 == 0 ) { // W߂ɉs
          lines.add(line);
          line = new Line();
        }
      }
      if( line.size() > 0 ) lines.add(line);
    }
    // eLXgR[his𐶐
    public ChordProgression( String source_text ) {
      if( source_text == null ) return;
      Measure measure;
      Line line;
      String[] lines_src, measures_src, elements_src;
      Chord chord, last_chord = null;
      String key_regex = "^Key(\\s*):(\\s*)";
      //
      // L[ł邩ǂ邽߂̃p^[
      Pattern key_match_pattern = Pattern.compile(
        key_regex+".*$", Pattern.CASE_INSENSITIVE
      );
      // L[̃wb_[菜߂̃p^[
      Pattern key_repl_pattern = Pattern.compile(
        key_regex, Pattern.CASE_INSENSITIVE
      );
      //
      lines_src = source_text.split("[\r\n]+");
      lines = new Vector<Line>();
      for( String line_src : lines_src ) {
        measures_src = line_src.split("\\|");
        if( measures_src.length > 0 ) {
          String key_string = measures_src[0].trim();
          if( key_match_pattern.matcher(key_string).matches() ) {
            key = new Key(
              key_repl_pattern.matcher(key_string).replaceFirst("")
            );
            // System.out.println("Key = " + key);
            continue;
          }
        }
        line = new Line();
        for( String measure_src : measures_src ) {
          elements_src = measure_src.split("[ \t]+");
          measure = new Measure();
          for( String element_src : elements_src ) {
            if( element_src.isEmpty() ) continue;
            if( element_src.equals("%") ) {
              if( measure.addBeat() == 0 ) {
                measure.add( new ChordStroke(last_chord) );
              }
              continue;
            }
            try {
              measure.add( new ChordStroke(
                last_chord = new Music.Chord(element_src)
              ));
            } catch( IllegalArgumentException ex ) {
              measure.add( new Lyrics(element_src) );
            }
          }
          line.add(measure);
        }
        lines.add(line);
      }
    }

    // Major/minor ؂ւ
    public void toggleKeyMajorMinor() {
      key = key.relativeKey();
    }

    // R[his̈ڒ
    public void transpose(int chromatic_offset) {
      for( Line line : lines ) {
        for( Measure measure : line ) {
          for( int i=0; i<measure.size(); i++ ) {
            Object element = measure.get(i);
            if( element instanceof ChordStroke ) {
              ChordStroke cs = (ChordStroke)element;
              Chord new_chord = cs.chord.clone();
              //
              // L[ݒ̂Ƃ́Aŏ̃R[h琄Đݒ
              if( key == null ) key = new Key( new_chord );
              //
              new_chord.transpose( chromatic_offset, key );
              measure.set( i, new ChordStroke( new_chord, cs.beat_length ) );
            }
          }
        }
      }
      key.transpose(chromatic_offset);
    }
    // ٖ́Ɓ؂ւ
    public void toggleEnharmonically() {
      if( key == null ) return;
      int original_key_co5 = key.toCo5();
      int co5_offset = 0;
      if( original_key_co5 > 4 ) {
        co5_offset = -12;
      }
      else if( original_key_co5 < -4 ) {
        co5_offset = 12;
      }
      else {
        return;
      }
      key.toggleEnharmonically();
      for( Line line : lines ) {
        for( Measure measure : line ) {
          for( int i=0; i<measure.size(); i++ ) {
            Object element = measure.get(i);
            if( element instanceof ChordStroke ) {
              ChordStroke cs = (ChordStroke)element;
              Chord new_chord = cs.chord.clone();
              new_chord.setRootByCo5( new_chord.rootNoteSymbol().toCo5() + co5_offset );
              new_chord.setBassByCo5( new_chord.bassNoteSymbol().toCo5() + co5_offset );
              measure.set( i, new ChordStroke( new_chord, cs.beat_length ) );
            }
          }
        }
      }
    }
    // R[his̒ɎԎiMIDI tickj
    //
    public void setTickPositions( FirstTrackSpec first_track ) {
      ticks_per_measure = first_track.ticks_per_measure;
      TickRange tick_range = new TickRange(
        first_track.pre_measures * ticks_per_measure
      );
      for( Line line : lines ) { // sPʂ̏
        for( Measure measure : line ) { // ߒPʂ̏
          int n_beats = measure.numberOfBeats();
          if( n_beats == 0 ) continue;
          long tpb = measure.ticks_per_beat = ticks_per_measure / n_beats ;
          for( Object element : measure ) {
            if( element instanceof Lyrics ) {
              ((Lyrics)element).start_tick_pos = tick_range.start_tick_pos;
              continue;
            }
            else if( element instanceof ChordStroke ) {
              ChordStroke chord_stroke = (ChordStroke)element;
              tick_range.moveForward( tpb * chord_stroke.beat_length );
              chord_stroke.tick_range = tick_range.clone();
            }
          }
        }
      }
    }
    // R[h̏
    public void setChordSymbolTextTo( AbstractTrackSpec ts ) {
      for( Line line : lines ) {
        for( Measure measure : line ) {
          if( measure.ticks_per_beat == null ) continue;
          for( Object element : measure ) {
            if( element instanceof ChordStroke ) {
              ts.addStringTo( 0x01, (ChordStroke)element );
            }
          }
        }
      }
    }
    // ̎̏
    public void setLyricsTo( AbstractTrackSpec ts ) {
      for( Line line : lines ) {
        for( Measure measure : line ) {
          if( measure.ticks_per_beat == null ) continue;
          for( Object element : measure ) {
            if( element instanceof Lyrics ) {
              ts.addStringTo( 0x05, (Lyrics)element );
            }
          }
        }
      }
    }
    // R[hisƂ MIDI V[PX𐶐
    //
    public Sequence toMidiSequence() {
      return toMidiSequence(48);
    }
    public Sequence toMidiSequence( int ppq ) {
      //
      // PPQ = Pulse Per Quarter (TPQN = Tick Per Quearter Note)
      //
      return toMidiSequence( ppq, 0, 0, null, null );
    }
    public Sequence toMidiSequence(
      int ppq, int start_measure_pos, int end_measure_pos,
      FirstTrackSpec first_track,
      Vector<AbstractNoteTrackSpec> track_specs
    ) {
      // MIDIV[PX̐
      Sequence seq;
      try {
        seq = new Sequence(Sequence.PPQ, ppq);
      } catch ( InvalidMidiDataException e ) {
        e.printStackTrace();
        return null;
      }
      // }X^[gbN̐
      if( first_track == null ) {
        first_track = new FirstTrackSpec();
      }
      first_track.key = this.key;
      first_track.createTrack( seq, start_measure_pos, end_measure_pos );
      //
      // gȂ΂ŏI
      if( lines == null || track_specs == null ) return seq;
      //
      // R[his̒ɎԎiMIDI tickj
      setTickPositions( first_track );
      //
      // R[h̃eLXgƉ̎
      setChordSymbolTextTo( first_track );
      setLyricsTo( first_track );
      //
      // c̃gbN𐶐
      for( AbstractNoteTrackSpec ts : track_specs ) {
        ts.createTrack( seq, first_track );
        if( ts instanceof DrumTrackSpec ) {
          ((DrumTrackSpec)ts).addDrums(this);
        }
        else {
          ((MelodyTrackSpec)ts).addChords(this);
        }
      }
      return seq;
    }
  }

  /////////////////////////////////////////////////
  //
  // MIDIgbN̎dl\NX
  //
  public static abstract class AbstractTrackSpec {
    public static final int BEAT_RESOLUTION = 2;
      // ŒZ̉̒il񔼕ɂ邩j
    String name = null;
    Track track = null;
    FirstTrackSpec first_track_spec = null;
    Sequence sequence = null;
    long min_note_ticks = 0;
    int pre_measures = 2;
    public AbstractTrackSpec() { }
    public AbstractTrackSpec(String name) {
      this.name = name;
    }
    public String toString() {
      return name;
    }
    public Track createTrack( Sequence seq, FirstTrackSpec first_track_spec ) {
      this.first_track_spec = first_track_spec;
      track = (sequence = seq).createTrack();
      if( name != null ) addStringTo( 0x03, name, 0 );
      min_note_ticks = (long)( seq.getResolution() >> 2 );
      return track;
    }
    public boolean addMetaEventTo( int type, byte data[], long tick_pos  ) {
      MetaMessage meta_msg = new MetaMessage();
      try {
        meta_msg.setMessage( type, data, data.length );
      } catch( InvalidMidiDataException ex ) {
        ex.printStackTrace();
        return false;
      }
      return track.add(new MidiEvent( (MidiMessage)meta_msg, tick_pos ));
    }
    public boolean addStringTo( int type, String str, long tick_pos ) {
      if( str == null ) str = "";
      return addMetaEventTo( type, str.getBytes(), tick_pos );
    }
    public boolean addStringTo( int type, ChordProgression.ChordStroke cs ) {
      return addStringTo(
        type,
        cs.chord.toString(),
        cs.tick_range.start_tick_pos
      );
    }
    public boolean addStringTo( int type, ChordProgression.Lyrics lyrics ) {
      return addStringTo(
        type, lyrics.text, lyrics.start_tick_pos
      );
    }
    public boolean addEOT( long tick_pos ) {
      return addMetaEventTo( 0x2F, new byte[0], tick_pos );
    }
    public void setChordSymbolText( ChordProgression cp ) {
      cp.setChordSymbolTextTo( this );
    }
  }

  // ŏ̃gbNp
  //
  public static class FirstTrackSpec extends AbstractTrackSpec {
    static byte default_tempo_data[] = { 0x07, (byte)0xA1, 0x20 };	// 120[QPM]
    static byte default_timesig_data[] = { 0x04, 0x02, 0x18, 0x08 };	// 4/4
    byte tempo_data[] = default_tempo_data;
    byte timesig_data[] = default_timesig_data;
    Key key = null;
    long ticks_per_measure;
    public FirstTrackSpec() { }
    public FirstTrackSpec(String name) {
      this.name = name;
    }
    public FirstTrackSpec(
      String name, byte[] tempo_data, byte[] timesig_data
    ) {
      this.name = name;
      if( tempo_data != null ) this.tempo_data = tempo_data;
      if( timesig_data != null ) this.timesig_data = timesig_data;
    }
    public FirstTrackSpec(
      String name, byte[] tempo_data, byte[] timesig_data, Key key
    ) {
      this(name,tempo_data,timesig_data);
      this.key = key;
    }
    public Track createTrack(Sequence seq) {
      return createTrack( seq, 0, 0 );
    }
    public Track createTrack(
      Sequence seq,
      int start_measure_pos, int end_measure_pos
    ) {
      this.pre_measures = start_measure_pos - 1;
      Track track = super.createTrack( seq, this );
      addTempo(
        this.tempo_data = (
          tempo_data == null ? default_tempo_data : tempo_data
        ), 0
      );
      addTimeSignature(
        this.timesig_data = (
          timesig_data == null ? default_timesig_data : timesig_data
        ), 0
      );
      if( key != null ) addKeySignature( key, 0 );
      ticks_per_measure = (long)(
        ( 4 * seq.getResolution() * this.timesig_data[0] )
        >> this.timesig_data[1]
      );
      addEOT( end_measure_pos * ticks_per_measure );
      return track;
    }
    public boolean addKeySignature( Key key, long tick_pos ) {
      return addMetaEventTo( 0x59, key.getBytes(), tick_pos );
    }
    public boolean addTempo( byte data[], long tick_pos ) {
      return addMetaEventTo( 0x51, data, tick_pos );
    }
    public boolean addTimeSignature( byte data[], long tick_pos ) {
      return addMetaEventTo( 0x58, data, tick_pos );
    }
  }

  // ʂ̃gbNifBAhʁj
  //
  public static abstract class AbstractNoteTrackSpec extends AbstractTrackSpec {
    int midi_channel = -1;
    int program_no = -1;
    int velocity = 64;

    public AbstractNoteTrackSpec() {}
    public AbstractNoteTrackSpec(int ch) {
      midi_channel = ch;
    }
    public AbstractNoteTrackSpec(int ch, String name) {
      midi_channel = ch;
      this.name = name;
    }
    public AbstractNoteTrackSpec(int ch, String name, int program_no) {
      this(ch,name);
      this.program_no = program_no;
    }
    public AbstractNoteTrackSpec(int ch, String name, int program_no, int velocity) {
      this(ch,name,program_no);
      this.velocity = velocity;
    }
    public Track createTrack( Sequence seq, FirstTrackSpec first_track_spec ) {
      Track track = super.createTrack( seq, first_track_spec );
      if( program_no >= 0 ) addProgram( program_no, 0 );
      return track;
    }
    public boolean addProgram( int program_no, long tick_pos ) {
      ShortMessage short_msg;
      try {
        (short_msg = new ShortMessage()).setMessage(
          ShortMessage.PROGRAM_CHANGE, midi_channel, program_no, 0
        );
      } catch( InvalidMidiDataException ex ) {
        ex.printStackTrace();
        return false;
      }
      return track.add(
        new MidiEvent( (MidiMessage)short_msg, tick_pos )
      );
    }
    public boolean addNote(
      long start_tick_pos, long end_tick_pos,
      int note_no
    ) {
      return addNote(
        start_tick_pos, end_tick_pos, note_no, velocity
      );
    }
    public boolean addNote(
      long start_tick_pos, long end_tick_pos,
      int note_no, int velocity
    ) {
      ShortMessage short_msg;
      //
      try {
        (short_msg = new ShortMessage()).setMessage(
          ShortMessage.NOTE_ON, midi_channel, note_no, velocity
        );
      } catch( InvalidMidiDataException ex ) {
        ex.printStackTrace();
        return false;
      }
      if( ! track.add(
          new MidiEvent( (MidiMessage)short_msg, start_tick_pos )
      ) ) return false;
      //
      try {
        (short_msg = new ShortMessage()).setMessage(
          ShortMessage.NOTE_OFF, midi_channel, note_no, velocity
        );
      } catch( InvalidMidiDataException ex ) {
        ex.printStackTrace();
        return false;
      }
      return track.add( new MidiEvent( (MidiMessage)short_msg, end_tick_pos ) );
    }
  }

  public static class DrumTrackSpec
    extends AbstractNoteTrackSpec
  {
    static int default_percussions[] = { // h̉FXg
      36, // Bass Drum 1
      44, // Pedal Hi-Hat
      39, // Hand Clap
      48, // Hi Mid Tom
      50, // High Tom
      38, // Acoustic Snare
      62, // Mute Hi Conga
      63, // Open Hi Conga
    };
    PercussionComboBoxModel models[]
      = new PercussionComboBoxModel[
        default_percussions.length
      ];
    int[] beat_patterns = {
      0x8888, 0x2222, 0x0008, 0x0800,
      0, 0, 0, 0
    };

    public class PercussionComboBoxModel 
      implements ComboBoxModel
    {
      private int note_no;
      //
      // Constructor
      public PercussionComboBoxModel(int default_note_no) {
        note_no = default_note_no;
      }
      // ComboBoxModel
      public Object getSelectedItem() {
        return note_no + ": " + MIDISpec.percussion_names[
          note_no - MIDISpec.MIN_PERCUSSION_NO
        ];
      }
      public void setSelectedItem(Object anItem) {
        String name = (String)anItem;
        int i = MIDISpec.MIN_PERCUSSION_NO;
        for( String pname : MIDISpec.percussion_names ) {
          if( name.equals(i + ": " + pname) ) {
            note_no = i; return;
          }
          i++;
        }
      }
      // ListModel
      public Object getElementAt(int index) {
        return (index + MIDISpec.MIN_PERCUSSION_NO) + ": "
          + MIDISpec.percussion_names[index];
      }
      public int getSize() {
        return MIDISpec.percussion_names.length;
      }
      public void addListDataListener(ListDataListener l) { }
      public void removeListDataListener(ListDataListener l) { }
      // Methods
      public int getSelectedNoteNo() {
        return note_no;
      }
      public void setSelectedNoteNo(int note_no) {
        this.note_no = note_no;
      }
    }

    public DrumTrackSpec(int ch, String name) {
      super(ch,name);
      for( int i=0; i<default_percussions.length; i++ ) {
        models[i] = new PercussionComboBoxModel(
          default_percussions[i]
        );
      }
    }

    public void addDrums( ChordProgression cp ) {
      int i;
      long tick;
      for( ChordProgression.Line line : cp.lines ) { // sPʂ̏
        for( ChordProgression.Measure measure : line ) { // ߒPʂ̏
          if( measure.ticks_per_beat == null )
            continue;
          ChordProgression.TickRange range = measure.getRange();
          int beat, mask;
          for(
            tick = range.start_tick_pos, beat = 0, mask = 0x8000;
            tick < range.end_tick_pos;
            tick += min_note_ticks, beat++, mask >>>= 1
          ) {
            for( i=0; i<beat_patterns.length; i++ ) {
              if( (beat_patterns[i] & mask) == 0 )
                continue;
              addNote(
                tick, tick+10,
                models[i].getSelectedNoteNo(),
                velocity
              );
            }
          }
        }
      }
    }
  }

  public static class MelodyTrackSpec extends AbstractNoteTrackSpec {

    Range range; // 

    // oǂ\rbg
    int beat_pattern = 0xFFFF;

    // Ƃŉo邩ǂ\rbg
    int continuous_beat_pattern = 0xEEEE;

    // x[Xgꍇ true
    // ȊÕR[h\gꍇ false
    boolean is_bass = false;

    // fB邩ǂ
    boolean random_melody = false;

    // ̎邩ǂ
    boolean random_lyric = false;

    public MelodyTrackSpec(int ch, String name) {
      super(ch,name);
      range = new Range( 60, 72 );
    }
    public MelodyTrackSpec(int ch, String name, Range range) {
      super(ch,name);
      this.range = range;
    }

    public void addChords( ChordProgression cp ) {
      int beat, mask;
      long tick;
      long start_tick_pos;

      // KƂ̐Nm߂d݃Xgirandom_melody ̏ꍇj
      int i, note_no, prev_note_no = 1;
      int note_weights[] = new int[range.max_note - range.min_note];
      //
      Key key = cp.key;
      if( key == null ) key = new Key("C");

      for( ChordProgression.Line line : cp.lines ) { // sPʂ̏
        for( ChordProgression.Measure measure : line ) { // ߒPʂ̏
          if( measure.ticks_per_beat == null )
            continue;
          ChordProgression.TickRange tick_range = measure.getRange();
          boolean is_note_on = false;
          //
          // er[gƂɌJԂ
          for(
            tick = start_tick_pos = tick_range.start_tick_pos,
              beat = 0, mask = 0x8000;
            tick < tick_range.end_tick_pos;
            tick += min_note_ticks,
              beat++, mask >>>= 1
          ) {
            // tickn_̃R[h𒲂ׂ
            Chord chord = measure.chordStrokeAt(tick).chord;
            int notes[] = chord.toNoteArray(range);
            //
            // eKƂɌJԂ
            if( Math.random() < 0.9 ) {
              if( (beat_pattern & mask) == 0 ) {
                // oȂ
                continue;
              }
            }
            else {
              // _ɋtp^[
              if( (beat_pattern & mask) != 0 ) {
                continue;
              }
            }
            if( ! is_note_on ) {
              // Õr[gŌpĂȂ̂ŁA
              // ̒n_ŉon߂邱ƂɂB
              start_tick_pos = tick;
              is_note_on = true;
            }
            if( Math.random() < 0.9 ) {
              if( (continuous_beat_pattern & mask) != 0 ) {
                // p
                continue;
              }
            }
            else {
              // _ɋtp^[
              if( (continuous_beat_pattern & mask) == 0 ) {
                continue;
              }
            }
            // ̃r[g̏ItickŉoI邱ƂɂB
            if( random_melody ) {
              // KƂɏom߂
              int total_weight = 0;
              for( i=0; i<note_weights.length; i++ ) {
                note_no = range.min_note + i;
                int m12 = mod12(note_no - chord.rootNote());
                int w;
                if( chord.indexOf(note_no) >= 0 ) {
                  // R[h\͊mグ
                  w = 255;
                }
                else {
                  switch( m12 ) {
                    case 2: // Qx
                    case 9: // Ux
                      w = 63; break;
                    case 5: // SSx
                    case 11: // Vx
                      w = 47; break;
                    default:
                      w = 0; break;
                  }
                  if( ! key.isOnScale( note_no ) ) {
                    // XP[OĂ鉹͍̗pȂ
                    w = 0;
                  }
                }
                // ŷ߁AOƂ̍ɂĊm𒲐
                int diff = note_no - prev_note_no;
                if( diff < 0 ) diff = -diff;
                if( diff == 0 ) w /= 8;
                else if( diff > 7 ) w = 0;
                else if( diff > 4 ) w /= 8;
                total_weight += (note_weights[i] = w);
              }
              // UĉK
              note_no = range.invertedNoteOf(key.rootNote());
              double r = Math.random();
              total_weight *= r;
              for( i=0; i<note_weights.length; i++ ) {
                if( (total_weight -= note_weights[i]) < 0 ) {
                  note_no = range.min_note + i;
                  break;
                }
              }
              // 肳ꂽǉ
              addNote(
                start_tick_pos, tick + min_note_ticks,
                note_no, velocity
              );
              if( random_lyric ) {
                // _ȉ̎ǉ
                addStringTo(
                  0x05,
                  VocaloidLyricGenerator.getRandomLyric(),
                  start_tick_pos
                );
              }
              prev_note_no = note_no;
            }
            else if( is_bass ) {
              // x[Xǉ
              int note = range.invertedNoteOf(chord.bassNote());
              addNote(
                start_tick_pos, tick + min_note_ticks,
                note, velocity
              );
            }
            else {
              // R[h{̂̉ǉ
              for( int note : notes ) {
                addNote(
                  start_tick_pos, tick + min_note_ticks,
                  note, velocity
                );
              }
            }
            is_note_on = false;
          }
        }
      }

    }

  }

  //
  // VOCALOID݊̉fPʂŉ̎𐶐NX
  //
  public static class VocaloidLyricGenerator {
    private static String lyric_elements[] = {
/*
      "","","",
      "","","",
      "","","",
      "ɂ","ɂ","ɂ",
      "Ђ","Ђ","Ђ",
      "݂","݂","݂",
      "","","",
      "","","",
      "","","",
      "","","",
      "т","т","т",
      "҂","҂","҂",
*/
      "","","","","",
      "","","","","",
      "","","","","",
      "","","","","",
      "","","","","",
      "","","","","",
      "","","","","",
      "","","",
      "","","","","",
      "","","",
      "","","","","",
      "","","","","",
      "","","","","",
      "","","","","",
      "","","","","",
    };
    //
    // _ɉfԂ
    public static String getRandomLyric() {
      return lyric_elements[
        (int)(Math.random() * lyric_elements.length)
      ];
    }
/*
    // eLXgfɕ
    public static Vector<String> split(String text) {
      Vector<String> sv = new Vector<String>();
      String s, prev_s;
      int i;
      for( i=0; i < text.length(); i++ ) {
        s = text.substring(i,i+1);
        if( "".indexOf(s) < 0 ) {
          sv.add(s);
        }
        else {
          prev_s = sv.remove(sv.size()-1);
          sv.add( prev_s + s );
        }
      }
      return sv;
    }
*/
  }

}

