/*
 * Copyright (c) 2003, jMonkeyEngine - Mojo Monkey Coding
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this 
 * list of conditions and the following disclaimer. 
 * 
 * Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 
 * Neither the name of the Mojo Monkey Coding, jME, jMonkey Engine, nor the 
 * names of its contributors may be used to endorse or promote products derived 
 * from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */
package jme.physics;

import jme.math.Vector;


/**
 * <code>PhysicsModule</code> is an interface that defines the method for 
 * moving an entity through the environment. Implementation will define how
 * the position and angle of the entity is affected per round. Calling 
 * <code>update</code> will set the physics attributes of the entity, such as
 * the newest angle and velocity. Where <code>updatePosition</code> will set 
 * the new position of the entity.
 * 
 * @author Mark Powell
 */
public interface PhysicsModule {

    /**
     * <code>update</code> should update any physical attributes of the entity.
     * @param time the time between updates.
     */
    public void update(float time);
    /**
     * <code>updatePosition</code> sets the position (supplied as a parameter)
     * to the new position defined by the module's attributes.
     * @param position the position to set.
     */
    public void updatePosition(Vector position);
    /**
     * <code>turn</code> turns the entity in the way defined by the
     * implementation of <code>PhysicsModule</code>. 
     * @param turnScalar any scalar to the turning properties of the entity.
     */
    public void turn(float turnScalar);
    /**
     * <code>move</code> moves the entity in the way defined by the 
     * implementation of <code>PhysicsModule</code>.
     * @param moveScalar any scalar to the movement properties of the entity.
     */
    public void move(float moveScalar);
    /**
     * <code>strafe</code> strafes the entity in the way defined by the 
     * implementation of <code>PhysicsModule</code>.
     * @param strafeScalar any scalar to the strafe properties of the entity.
     */
    public void strafe(float strafeScalar);
    /**
     * 
     * <code>getCurrentAngle</code> returns the current angle of the entity as
     * defined by the physics module.
     * @return the current angle of the entity.
     */
    public float getCurrentAngle();
    /**
     * 
     * <code>getCurrentVelocity</code> returns the current velocity of the 
     * entity as defined by the physics module.
     * @return the current velocity of the entity.
     */
    public float getCurrentVelocity();
    /**
     * 
     * <code>getCurrentTurningVel</code> returns the current turning velocity
     * of the entity as defined by the physics module.
     * @return the current turning velocity of the entity.
     */
    public float getCurrentTurningVel();

}
