# -*- coding: nil -*-

module Mint::Generator

  #
  # 四則演算を行う式のジェネレータの基底クラス
  #
  class Arithmetic < Base

    # 使用する演算子
    DEFAULT_OPERATORS = %w[+ - * div]

    private

    include Utilities

    option :term_number, 2
    option :operators, DEFAULT_OPERATORS

    def generate_problem
      expression
    end

    #
    # 後処理 (継承した時は super を呼ぶこと)
    #
    def teardown(problem)
      String(problem).gsub(%r! (-[0-9./]+)!, ' (\\1)')
    end

    #
    # １つの項を返すメソッド
    # 各ジェネレータでオーバライドする必要あり
    #
    def operand
      raise 'need to overwrite'
    end

    #
    # 最後に生成したオペランドを返す。
    #
    def last_operand(operand = nil)
      @last_operand = operand if operand
      @last_operand
    end

    #
    # 最後に生成したオペレータを返す
    #
    def last_operator(operator = nil)
      @last_operator = operator if operator
      @last_operator
    end

    #
    # 式を生成するメソッド
    #
    def expression
      result = []
      options[:term_number].times do
        result << last_operand(operand)
        result << last_operator(options[:operators].sample)
      end
      # except last operator
      result[0..-2].join(" ")
    end

    #
    # 整数を返すメソッド
    #
    def integer
      min = options[:min]
      min = 1 if last_operator == 'div'
      create_integer(min, options[:max], options[:minus]).to_s
    end

    #
    # 小数を返すメソッド
    #
    def decimal
      use_integer = rand(3) == 0
      integer_part = 0
      unless use_integer
        integer_part = create_integer(options[:min].to_i, options[:max].to_i, false)
      end
      decimal_part = rand(10**options[:digits])
      ("#{integer_part}.#{decimal_part}".to_f * sign(options[:minus])).to_s
    end

    #
    # 分数を返すメソッド
    #
    def fraction
      begin
        numerator   = numerator_part.to_i
        denominator = denominator_part.to_i
      end until numerator != denominator
      until is_lowest_term?(numerator, denominator)
        gcd = numerator.gcd(denominator)
        numerator  /= gcd
        denominator /= gcd
      end
      "#{numerator}/#{denominator}"
    end

    def is_lowest_term?(numerator, denominator)
      numerator.gcd(denominator) == 1
    end

    def numerator_part
      create_integer(options[:numerator_min], options[:numerator_max], options[:minus])
    end

    def denominator_part
      create_integer(options[:denominator_min], options[:denominator_max], false)
    end

    #
    # 最大値、最小値から項数をランダムに生成するメソッド
    #
    def term_number(position)
      min, max = options.values_at(:"#{position}_term_min", :"#{position}_term_max")
      create_integer(min, max, false)
    end
  end
end

