//
// Stage.cpp
//

#include "Stage.hpp"
#include <string>

Stage::Stage(const std::string& model_path, float model_scale) : map_scale_(model_scale)
{
    map_handle_ = MV1LoadModel(model_path.c_str());
    MV1SetScale(map_handle_, VGet(map_scale_, map_scale_, map_scale_));
    MV1SetupCollInfo(map_handle_, -1, 256, 256, 256);
}

void Stage::Draw()
{
    MV1DrawModel(map_handle_);
}

float Stage::GetFloorY(const VECTOR& v1, const VECTOR& v2) const
{
    auto coll_info = MV1CollCheck_Line(map_handle_, -1, v1, v2);
    return coll_info.HitFlag ? coll_info.HitPosition.y : 0;
}

std::pair<bool, VECTOR> Stage::FloorExists(const VECTOR& foot_pos, float model_height, float collision_depth_limit) const
{
    // 床検出
    auto coll_info = MV1CollCheck_Line(map_handle_, -1,
            foot_pos + VGet(0, 0.5 * model_height * map_scale_, 0),
            foot_pos - VGet(0, collision_depth_limit * map_scale_, 0));
    if (coll_info.HitFlag)
    {
        return std::make_pair(true, coll_info.HitPosition);
    }

    return std::make_pair(false, VGet(0, 0, 0));
}

bool Stage::IsFlatFloor(const VECTOR& foot_pos, const VECTOR& direction) const
{
    const float interval = 0.01 * map_scale_;
    const auto direction_ = VNorm(VGet(direction.x, 0, direction.z));
    const float threshold = 0.02 * map_scale_;
    const auto threshold_vector = VGet(0, threshold, 0);

    int flat_max = 0, flat_min = 0;
    for (int z = 1; z <= 20; ++z)
    {
        const auto check_pos1 = foot_pos + z * direction_ * interval;
        const auto check_pos2 = foot_pos - z * direction_ * interval;
        const auto coll_info1 = MV1CollCheck_Line(map_handle_, -1,
                check_pos1 + threshold_vector,
                check_pos1 - threshold_vector);
        const auto coll_info2 = MV1CollCheck_Line(map_handle_, -1,
                check_pos2 + threshold_vector,
                check_pos2 - threshold_vector);
        if (coll_info1.HitFlag && flat_max == z - 1)
        {
            ++flat_max;
        }
        if (coll_info2.HitFlag && flat_min == -(z - 1))
        {
            // 初めて衝突しなかった
            --flat_min;
        }
    }

    return flat_max - flat_min >= 10;
}

// 移動方向に障害物があるかどうか（当たり判定）
bool Stage::FrontCollides(float collision_length, const VECTOR& current_pos, const VECTOR& prev_pos,
        float height_begin, float height_end, size_t num_division) const
{
    const auto height_diff = (height_end - height_begin) / num_division;

    auto direction = current_pos - prev_pos;
    direction.y = 0; // 水平な進行方向を求める
    const auto collision_vector =
        //VNorm(direction) * (collision_length * map_scale_); // 体中心から15cm前方までの線分で当たり判定
        VAdjustLength(direction, collision_length * map_scale_);

    for (size_t i = 0; i <= num_division; ++i)
    {
        auto collision_vector_pos = prev_pos + VGet(0, height_begin + height_diff * i, 0);
        auto coll_info = MV1CollCheck_Line(map_handle_, -1,
                collision_vector_pos,
                collision_vector_pos + collision_vector);
        if (coll_info.HitFlag)
        {
            return true;
        }
    }

    return false;
}

int Stage::map_handle() const
{
    return map_handle_;
}

float Stage::map_scale() const
{
    return map_scale_;
}
