//
// InputManager.cpp
//

#include "InputManager.hpp"
#include <DxLib.h>

int InputManager::mouse_x_ = 0, InputManager::mouse_y_ = 0;
int InputManager::static_mouse_right_count_ = 0,
        InputManager::static_mouse_left_count_ = 0,
        InputManager::static_mouse_middle_count_ = 0;

bool InputManager::static_mouse_right_ = false,
        InputManager::static_mouse_left_ = false,
        InputManager::static_mouse_middle_ = false;

bool InputManager::prev_mouse_right_ = false, InputManager::prev_mouse_left_ =
        false, InputManager::prev_mouse_middle_ = false;

int InputManager::static_mouse_wheel_ = 0;

std::array<int, 256> InputManager::static_key_count_ = {{0}};
std::array<int, 256> InputManager::static_key_count_tmp_ = {{0}};

// キーバインドの初期設定
int
InputManager::KEYBIND_FORWARD =    KEY_INPUT_W,
InputManager::KEYBIND_BACK =       KEY_INPUT_S,
InputManager::KEYBIND_RIGHT_TRUN = KEY_INPUT_D,
InputManager::KEYBIND_LEFT_TURN =  KEY_INPUT_A,
InputManager::KEYBIND_JUMP =       KEY_INPUT_SPACE,
InputManager::KEYBIND_CHANGE_SPEED =      KEY_INPUT_LSHIFT,
InputManager::KEYBIND_CHANGE_SPEED2 =     KEY_INPUT_RSHIFT,
InputManager::KEYBIND_TAB =     KEY_INPUT_TAB,
InputManager::KEYBIND_SHIFT =   KEY_INPUT_LSHIFT,
InputManager::KEYBIND_RETURN =   KEY_INPUT_RETURN,
InputManager::KEYBIND_EXIT =   KEY_INPUT_ESCAPE,
InputManager::KEYBIND_REFRESH =   KEY_INPUT_F12,
InputManager::KEYBIND_SCRIPT_MODE =   KEY_INPUT_F8;

InputManager::InputManager() :
        mouse_right_(static_mouse_right_),
        mouse_left_(static_mouse_left_),
        mouse_middle_(static_mouse_middle_),
        mouse_right_count_(static_mouse_right_count_),
        mouse_left_count_(static_mouse_left_count_),
        mouse_middle_count_(static_mouse_middle_count_),
        mouse_wheel_(static_mouse_wheel_),
        key_count_(static_key_count_tmp_)
{

}

void InputManager::Update()
{
    GetMousePoint(&mouse_x_, &mouse_y_);

    prev_mouse_right_ = static_mouse_right_;
    prev_mouse_left_ = static_mouse_left_;
    prev_mouse_middle_ = static_mouse_middle_;

    int mouse_button = GetMouseInput();
    static_mouse_right_ = mouse_button & MOUSE_INPUT_RIGHT;
    static_mouse_left_ = mouse_button & MOUSE_INPUT_LEFT;
    static_mouse_middle_ = mouse_button & MOUSE_INPUT_MIDDLE;

    if (static_mouse_right_) {
        static_mouse_right_count_++;
    } else {
        static_mouse_right_count_ = 0;
    }

    if (static_mouse_left_) {
        static_mouse_left_count_++;
    } else {
        static_mouse_left_count_ = 0;
    }

    if (static_mouse_middle_) {
        static_mouse_middle_count_++;
    } else {
        static_mouse_middle_count_ = 0;
    }

    static_mouse_wheel_ = GetMouseWheelRotVol();

    char KeyBuf[256];
    GetHitKeyStateAll(KeyBuf);

    for (int i = 0; i < 256; i++) {
        if (KeyBuf[i] == 1) {
            static_key_count_[i]++;
        } else {
            static_key_count_[i] = 0;
        }
        static_key_count_tmp_[i] = static_key_count_[i];
    }
}

void InputManager::operator&=(const InputManager& input)
{
    mouse_right_ &= input.mouse_right_;
    mouse_left_ &= input.mouse_left_;
    mouse_middle_ &= input.mouse_middle_;

    mouse_right_count_ = std::min(mouse_right_count_, input.mouse_right_count_);
    mouse_left_count_ = std::min(mouse_left_count_, input.mouse_left_count_);
    mouse_middle_count_ = std::min(mouse_middle_count_, input.mouse_middle_count_);

    mouse_wheel_ = std::min(mouse_wheel_, input.mouse_wheel_);

    auto it = key_count_.begin();
    auto it2 = input.key_count_.begin();
    for (;it != key_count_.end() && it2 != input.key_count_.end(); ++it, ++it2) {
        *it = std::min(*it, *it2);
    }
}

int InputManager::GetMouseX() const
{
    return mouse_x_;
}

int InputManager::GetMouseY() const
{
    return mouse_y_;
}

std::pair<int, int> InputManager::GetMousePos() const
{
    return std::pair<int, int>(mouse_x_, mouse_y_);
}

int InputManager::GetMouseWheel() const
{
    return mouse_wheel_;
}

bool InputManager::GetMouseRight() const
{
    return mouse_right_;
}

bool InputManager::GetMouseLeft() const
{
    return mouse_left_;
}

bool InputManager::GetMouseMiddle() const
{
    return mouse_middle_;
}

int InputManager::GetMouseRightCount() const
{
    return mouse_right_count_;
}

int InputManager::GetMouseLeftCount() const
{
    return mouse_left_count_;
}

int InputManager::GetMouseMiddleCount() const
{
    return mouse_middle_count_;
}


bool InputManager::GetPrevMouseRight() const
{
    return prev_mouse_right_;
}

bool InputManager::GetPrevMouseLeft() const
{
    return prev_mouse_left_;
}

bool InputManager::GetPrevMouseMiddle() const
{
    return prev_mouse_middle_;
}

int InputManager::GetKeyCount(int key) const
{
    return key_count_[key];
}



void InputManager::CancelMouseWheel()
{
    mouse_wheel_ = 0;
}

void InputManager::CancelMouseRight()
{
    mouse_right_ = false;
    mouse_right_count_ = 0;
}

void InputManager::CancelMouseLeft()
{
    mouse_left_ = false;
    mouse_left_count_ = 0;
}

void InputManager::CancelMouseMiddle()
{
    mouse_middle_ = false;
    mouse_middle_count_ = 0;
}

void InputManager::CancelKeyCount(int key)
{
    if (key >= 0 && key < 256) {
        key_count_[key] = 0;
    }
}

void InputManager::CancelKeyCountAll()
{
    for (int i = 0; i < 255; i++) {
        if (i != KEYBIND_TAB && i != KEYBIND_SHIFT) {
            key_count_[i] = 0;
        }
    }
}
