//
// Account.cpp
//

#include "Account.hpp"
#include "../network/Encrypter.hpp"
#include "../network/Utils.hpp"
#include <iostream>
#include <string.h>
#include <boost/date_time/posix_time/posix_time.hpp>
#include "../Logger.hpp"

Account::Account(const std::string& logfile) :
db_(nullptr),
stmt_get_current_revision_(nullptr),
stmt_get_revision_patch_(nullptr),
stmt_apply_revision_patch_(nullptr),
stmt_clean_logout_(nullptr),
stmt_get_user_id_(nullptr),
stmt_get_public_key_(nullptr),
stmt_register_public_key_(nullptr),
stmt_get_json_string_(nullptr),
stmt_login_(nullptr),
stmt_logout_(nullptr),
stmt_logout_all_(nullptr),
stmt_clean_expired_(nullptr),
stmt_get_user_name_(nullptr),
stmt_set_user_name_(nullptr),
stmt_get_user_trip_(nullptr),
stmt_set_user_trip_(nullptr),
stmt_get_user_ip_address_(nullptr),
stmt_set_user_ip_address_(nullptr),
stmt_get_user_udp_port_(nullptr),
stmt_set_user_udp_port_(nullptr)
{

    int result = sqlite3_open(logfile.c_str(), &db_);
    if (result != SQLITE_OK){
        Logger::Error("Cannot open account data.");
        sqlite3_close(db_);
        db_ = nullptr;
    } else {
        const char query[] =
        u8R"(
                CREATE TABLE IF NOT EXISTS "account" (
                    "user_id" INTEGER PRIMARY KEY,
                    "name" TEXT,
                    "trip" TEXT,
                    "finger_print" TEXT,
                    "public_key" TEXT,
                    "ip_address" TEXT,
                    "udp_port" INT DEFAULT 0,
                    "login" INT DEFAULT 0,
                    "login_date" REAL,
                    "revision" INT DEFAULT 0,
                    "authority" INT DEFAULT 0
                );
                CREATE UNIQUE INDEX IF NOT EXISTS finger_print_index ON account(finger_print);
                VACUUM;
                REINDEX;
        )"; // ヒアドキュメント

        // テーブルを作成
        char *error = nullptr;
        int result = sqlite3_exec(db_, query, nullptr, nullptr, &error);
        if (result != SQLITE_OK) {
            Logger::Error(error);
        }

        // stmt_get_current_revision_
        {
            const char query[] =
            u8R"(
                SELECT revision FROM `account`
                ORDER BY revision DESC
                LIMIT 1
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_current_revision_, nullptr);
        }

        // stmt_get_revision_patch_
        {
            const char query[] =
            u8R"(
                SELECT user_id, name, trip, ip_address, udp_port, login, revision FROM `account`
                WHERE revision > ? AND (login = 1 OR revision=(SELECT MAX(revision) FROM account))
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_revision_patch_, nullptr);
        }

        // stmt_apply_revision_patch_
        {
            const char query[] =
            u8R"(
                INSERT OR REPLACE INTO `account`
                (`user_id`, `name`, `trip`, `ip_address`, `udp_port`, `login`, `revision`)
                VALUES(?,?,?,?,?,?,?);
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_apply_revision_patch_, nullptr);
        }

        // stmt_clean_logout_
        {
            const char query[] =
            u8R"(
                DELETE FROM `account`
                WHERE login = 0
                AND revision != (SELECT MAX(revision) FROM account)
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_clean_logout_, nullptr);
        }

        // stmt_get_user_id_
        {
            const char query[] =
            u8R"(
                SELECT user_id FROM `account`
                WHERE finger_print = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_user_id_, nullptr);
        }

        // stmt_get_public_key_
        {
            const char query[] =
            u8R"(
                SELECT public_key FROM `account`
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_public_key_, nullptr);
        }

        // stmt_register_public_key_
        {
            const char query[] =
            u8R"(
                INSERT INTO `account`
                (`name`, `finger_print`, `public_key`, `revision`)
                VALUES('Anonymous',?,?,ifnull((SELECT MAX(revision) FROM account) + 1,1));
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_register_public_key_, nullptr);
        }

        // stmt_get_json_string_
        {
            const char query[] =
            u8R"(
                SELECT user_id, name FROM `account`
                WHERE login = 1
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_json_string_, nullptr);
        }

        // stmt_login_
        {
            const char query[] =
            u8R"(
                UPDATE `account`
                SET login = 1,
                login_date = julianday('now'),
                revision = (SELECT MAX(revision) FROM account) + 1
                WHERE user_id = ?
                AND login = 0
            )"; // ヒアドキュメント

            // login_date = (julianday('now') as julianday),
            sqlite3_prepare_v2(db_, query, -1, &stmt_login_, nullptr);
        }

        // stmt_logout_
        {
            const char query[] =
            u8R"(
                UPDATE `account`
                SET login = 0,
                revision = (SELECT MAX(revision) FROM account) + 1
                WHERE user_id = ?
                AND login = 1
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_logout_, nullptr);
        }

        // stmt_logout_all_
        {
            const char query[] =
            u8R"(
                UPDATE `account`
                SET login = 0
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_logout_all_, nullptr);
        }

        // stmt_clean_expired_
        {
            const char query[] =
            u8R"(
                DELETE FROM `account`
                WHERE login_date < julianday('now') - 30 * 6
                AND revision != (SELECT MAX(revision) FROM account)
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_clean_logout_, nullptr);
        }

        // stmt_get_user_name_
        {
            const char query[] =
            u8R"(
                SELECT name FROM `account`
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_user_name_, nullptr);
        }

        // stmt_set_user_name_
        {
            const char query[] =
            u8R"(
                UPDATE `account`
                SET name = ?,
                revision = (SELECT MAX(revision) FROM account) + 1
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_set_user_name_, nullptr);
        }

        // stmt_get_user_trip_
        {
            const char query[] =
            u8R"(
                SELECT trip FROM `account`
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_user_trip_, nullptr);
        }

        // stmt_set_user_trip_
        {
            const char query[] =
            u8R"(
                UPDATE `account`
                SET trip = ?,
                revision = (SELECT MAX(revision) FROM account) + 1
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_set_user_trip_, nullptr);
        }

        // stmt_get_user_ip_address_
        {
            const char query[] =
            u8R"(
                SELECT ip_address FROM `account`
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_user_ip_address_, nullptr);
        }

        // stmt_set_user_ip_address_
        {
            const char query[] =
            u8R"(
                UPDATE `account`
                SET ip_address = ?,
                revision = (SELECT MAX(revision) FROM account) + 1
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_set_user_ip_address_, nullptr);
        }

        // stmt_get_user_udp_port_
        {
            const char query[] =
            u8R"(
                SELECT udp_port FROM `account`
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_get_user_udp_port_, nullptr);
        }

        // stmt_set_user_udp_port_
        {
            const char query[] =
            u8R"(
                UPDATE `account`
                SET udp_port = ?,
                revision = (SELECT MAX(revision) FROM account) + 1
                WHERE user_id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, -1, &stmt_set_user_udp_port_, nullptr);
        }

        CleanExpired();
        LogOutAll();
    }
}

Account::~Account()
{
    if (db_) {
        sqlite3_finalize(stmt_get_current_revision_);
        sqlite3_finalize(stmt_get_revision_patch_);
        sqlite3_finalize(stmt_apply_revision_patch_);
        sqlite3_finalize(stmt_clean_logout_);
        sqlite3_finalize(stmt_get_user_id_);
        sqlite3_finalize(stmt_get_public_key_);
        sqlite3_finalize(stmt_register_public_key_);
        sqlite3_finalize(stmt_get_json_string_);
        sqlite3_finalize(stmt_login_);
        sqlite3_finalize(stmt_logout_);
        sqlite3_finalize(stmt_logout_all_);
        sqlite3_finalize(stmt_clean_expired_);
        sqlite3_finalize(stmt_get_user_name_);
        sqlite3_finalize(stmt_set_user_name_);
        sqlite3_finalize(stmt_get_user_trip_);
        sqlite3_finalize(stmt_set_user_trip_);
        sqlite3_finalize(stmt_get_user_ip_address_);
        sqlite3_finalize(stmt_set_user_ip_address_);
        sqlite3_finalize(stmt_get_user_udp_port_);
        sqlite3_finalize(stmt_set_user_udp_port_);

        sqlite3_close(db_);
        db_ = nullptr;
    }
}

int Account::GetCurrentRevision()
{
    int revision = 0;
    {
        sqlite3_reset(stmt_get_current_revision_);

        int result;
        while (SQLITE_ROW == (result = sqlite3_step(stmt_get_current_revision_))){
            revision = sqlite3_column_int(stmt_get_current_revision_, 0);
        }
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
    return revision;
}

std::string Account::GetRevisionPatch(int revision)
{
    std::string patch;

    sqlite3_reset(stmt_get_revision_patch_);
    sqlite3_bind_int(stmt_get_revision_patch_, 1, revision);

    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_get_revision_patch_))){
        unsigned int user_id = sqlite3_column_int(stmt_get_revision_patch_, 0);

        std::string name(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_revision_patch_, 1)),
                sqlite3_column_bytes(stmt_get_revision_patch_, 1));

        std::string trip(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_revision_patch_, 2)),
                sqlite3_column_bytes(stmt_get_revision_patch_, 2));

        std::string ip_address(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_revision_patch_, 3)),
                sqlite3_column_bytes(stmt_get_revision_patch_, 3));
        uint16_t udp_port = sqlite3_column_int(stmt_get_revision_patch_, 4);

        char login = sqlite3_column_int(stmt_get_revision_patch_, 5);
        int revision = sqlite3_column_int(stmt_get_revision_patch_, 6);

        patch += network::Utils::Serialize(user_id, name, trip, ip_address, udp_port, login, revision);
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }

    return patch;
}

void Account::ApplyRevisionPatch(const std::string& patch)
{
    std::string buffer(patch);

    while (buffer.size() > 0) {

        std::tuple<unsigned int, std::string, std::string, std::string, uint16_t, char, int> result_tuple;
        size_t readed = network::Utils::Deserialize(buffer, &result_tuple);
        buffer.erase(0, readed);

        unsigned int user_id;
        std::string name, trip;
        std::string ip_address;
        uint16_t udp_port;
        char login;
        int revision;

        user_id = std::get<0>(result_tuple);
        name = std::get<1>(result_tuple);
        trip = std::get<2>(result_tuple);
        ip_address = std::get<3>(result_tuple);
        udp_port = std::get<4>(result_tuple);
        login = std::get<5>(result_tuple);
        revision = std::get<6>(result_tuple);

        sqlite3_reset(stmt_apply_revision_patch_);
        sqlite3_bind_int(stmt_apply_revision_patch_, 1, user_id);
        sqlite3_bind_text(stmt_apply_revision_patch_, 2, name.data(), name.size(), SQLITE_TRANSIENT);
        sqlite3_bind_text(stmt_apply_revision_patch_, 3, trip.data(), trip.size(), SQLITE_TRANSIENT);
        sqlite3_bind_text(stmt_apply_revision_patch_, 4, ip_address.data(), ip_address.size(), SQLITE_TRANSIENT);
        sqlite3_bind_int(stmt_apply_revision_patch_, 5, udp_port);
        sqlite3_bind_int(stmt_apply_revision_patch_, 6, login);
        sqlite3_bind_int(stmt_apply_revision_patch_, 7, revision);

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_apply_revision_patch_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }

    // 削除フラグの付いたデータを消去
    sqlite3_reset(stmt_clean_logout_);

    int result, loop=0;
    while (SQLITE_DONE != (result = sqlite3_step(stmt_clean_logout_)) && loop++<1000){}
    if (result != SQLITE_OK) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

UserID Account::GetUserId(const std::string& finger_print)
{
    sqlite3_reset(stmt_get_user_id_);

    auto finger_print_base64 = network::Utils::Base64Encode(finger_print);
    sqlite3_bind_text(stmt_get_user_id_, 1, finger_print_base64.c_str(), finger_print_base64.size(), SQLITE_STATIC);

    UserID id = 0;
    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_get_user_id_))){
       id = (UserID)sqlite3_column_int(stmt_get_user_id_, 0);
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }

    return id;
}

std::string Account::GetPublicKey(UserID user_id)
{
    sqlite3_reset(stmt_get_public_key_);
    sqlite3_bind_int(stmt_get_public_key_, 1, static_cast<long>(user_id));

    const char* buffer;
    std::string key;
    int result;

    while (SQLITE_ROW == (result = sqlite3_step(stmt_get_public_key_))){
        int size = sqlite3_column_bytes(stmt_get_public_key_, 0);
        buffer = reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_public_key_, 0));
        key = std::string(buffer,size);
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }

    return network::Utils::Base64Decode(key);
}

UserID Account::RegisterPublicKey(const std::string& key)
{
    auto finger_print = network::Encrypter::GetHash(key);

    {
        sqlite3_reset(stmt_register_public_key_);

        auto public_key_base64 = network::Utils::Base64Encode(key);
        auto finger_print_base64 = network::Utils::Base64Encode(finger_print);

        sqlite3_bind_text(stmt_register_public_key_, 1, finger_print_base64.c_str(),  finger_print_base64.size(), SQLITE_STATIC);
        sqlite3_bind_text(stmt_register_public_key_, 2, public_key_base64.c_str(), public_key_base64.size(), SQLITE_STATIC);

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_register_public_key_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }

    auto user_id = GetUserId(finger_print);
    return user_id;

}

std::string Account::GetJSONString()
{
    std::string json;
    json += "([";

    sqlite3_reset(stmt_get_json_string_);
    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_get_json_string_))){
        int user_id = sqlite3_column_int(stmt_get_json_string_, 0);

        std::string name(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_json_string_, 1)),
                sqlite3_column_bytes(stmt_get_json_string_, 1));

        json += (boost::format("{id:%d,name:'%s'},") % user_id % name.c_str()).str();
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }

    json += "])";
    return json;
}

void Account::LogIn(UserID user_id)
{
    using namespace boost::posix_time;
    ptime now = second_clock::local_time();
    auto time_string = to_iso_extended_string(now);

    sqlite3_reset(stmt_login_);
    sqlite3_bind_int(stmt_login_, 1, static_cast<long>(user_id));

    int result, loop=0;
    while (SQLITE_DONE != (result = sqlite3_step(stmt_login_)) && loop++<1000){}
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

void Account::LogOut(UserID user_id)
{
    sqlite3_reset(stmt_logout_);
    sqlite3_bind_int(stmt_logout_, 1, static_cast<long>(user_id));

    int result, loop=0;
    while (SQLITE_DONE != (result = sqlite3_step(stmt_logout_)) && loop++<1000){}
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

void Account::LogOutAll()
{
    sqlite3_reset(stmt_logout_all_);

    int result, loop=0;
    while (SQLITE_DONE != (result = sqlite3_step(stmt_logout_all_)) && loop++<1000){}
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

void Account::CleanExpired()
{
    sqlite3_reset(stmt_clean_expired_);

    int result, loop=0;
    while (SQLITE_DONE != (result = sqlite3_step(stmt_clean_expired_)) && loop++<1000){}
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

std::string Account::GetUserName(UserID user_id)
{
    std::string name;
    {
        sqlite3_reset(stmt_get_user_name_);
        sqlite3_bind_int(stmt_get_user_name_, 1, static_cast<long>(user_id));

        const char* buffer;
        int result;

        while (SQLITE_ROW == (result = sqlite3_step(stmt_get_user_name_))){
            int size = sqlite3_column_bytes(stmt_get_user_name_, 0);
            buffer = reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_user_name_, 0));
            name = std::string(buffer,size);
        }
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
    return name;
}

void Account::SetUserName(UserID user_id, const std::string& name)
{
    if (GetUserName(user_id) != name) {
        sqlite3_reset(stmt_set_user_name_);

        sqlite3_bind_text(stmt_set_user_name_, 1, name.c_str(), name.size(), SQLITE_STATIC);
        sqlite3_bind_int(stmt_set_user_name_, 2, static_cast<long>(user_id));

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_set_user_name_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
}

std::string Account::GetUserTrip(UserID user_id)
{
    std::string trip;
    {
        sqlite3_reset(stmt_get_user_trip_);
        sqlite3_bind_int(stmt_get_user_trip_, 1, static_cast<long>(user_id));

        const char* buffer;
        int result;

        while (SQLITE_ROW == (result = sqlite3_step(stmt_get_user_trip_))){
            int size = sqlite3_column_bytes(stmt_get_user_trip_, 0);
            buffer = reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_user_trip_, 0));
            trip = std::string(buffer,size);
        }
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
    return trip;
}

void Account::SetUserTrip(UserID user_id, const std::string& trip)
{
    std::string crypted_trip = network::Encrypter::GetTrip(trip);

    if (GetUserTrip(user_id) != crypted_trip) {
        sqlite3_reset(stmt_set_user_trip_);

        sqlite3_bind_text(stmt_set_user_trip_, 1, crypted_trip.c_str(), crypted_trip.size(), SQLITE_STATIC);
        sqlite3_bind_int(stmt_set_user_trip_, 2, static_cast<long>(user_id));

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_set_user_trip_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
}

std::string Account::GetUserIPAddress(UserID user_id)
{
    std::string ip_address;
    {
        sqlite3_reset(stmt_get_user_ip_address_);
        sqlite3_bind_int(stmt_get_user_ip_address_, 1, static_cast<long>(user_id));

        const char* buffer;
        int result;

        while (SQLITE_ROW == (result = sqlite3_step(stmt_get_user_ip_address_))){
            int size = sqlite3_column_bytes(stmt_get_user_ip_address_, 0);
            buffer = reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_user_ip_address_, 0));
            ip_address = std::string(buffer,size);
        }
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
    return ip_address;
}
void Account::SetUserIPAddress(UserID user_id, const std::string& ip_address)
{

    if (GetUserIPAddress(user_id) != ip_address) {
        sqlite3_reset(stmt_set_user_ip_address_);

        sqlite3_bind_text(stmt_set_user_ip_address_, 1, ip_address.c_str(), ip_address.size(), SQLITE_STATIC);
        sqlite3_bind_int(stmt_set_user_ip_address_, 2, static_cast<long>(user_id));

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_set_user_ip_address_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
}

uint16_t Account::GetUserUDPPort(UserID user_id)
{
    uint16_t udp_port = 0;
    {
        sqlite3_reset(stmt_get_user_udp_port_);
        sqlite3_bind_int(stmt_get_user_udp_port_, 1, static_cast<long>(user_id));

        int result;
        while (SQLITE_ROW == (result = sqlite3_step(stmt_get_user_udp_port_))){
            udp_port = static_cast<uint16_t>(sqlite3_column_int(stmt_get_user_udp_port_, 0));
        }
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
    return udp_port;
}

void Account::SetUserUDPPort(UserID user_id, uint16_t udp_port)
{
    if (GetUserUDPPort(user_id) != udp_port) {
        sqlite3_reset(stmt_set_user_udp_port_);

        sqlite3_bind_int(stmt_set_user_udp_port_, 1, (int)udp_port);
        sqlite3_bind_int(stmt_set_user_udp_port_, 2, static_cast<long>(user_id));

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_set_user_udp_port_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
}
