//
// Client.hpp
//

#pragma once

#include <string>
#include <queue>
#include "Session.hpp"
#include "Signature.hpp"

namespace network {

    class Client {
        private:
            class ClientSession : public Session {
                public:
                ClientSession(boost::asio::io_service& io_service,
                        boost::asio::io_service& io_service_udp,
                        tcp::resolver::iterator endpoint_iterator,
                        uint16_t udp_port) :
                                Session(io_service),
                                endpoint_iterator_(endpoint_iterator),
                                io_service_udp_(io_service_udp),
                                socket_udp_(io_service_udp, udp::endpoint(udp::v4(), udp_port))
                {
                }
                ;
                ~ClientSession();

                    void Start();
                    void Connect(const boost::system::error_code& error);
                    void SendUDP(const std::string& data, const std::string& dist);

                private:
                    tcp::resolver::iterator endpoint_iterator_;
                    boost::asio::io_service& io_service_udp_;
                    udp::socket socket_udp_;
            };

            typedef boost::shared_ptr<ClientSession> ClientSessionPtr;

        public:
           Client(const std::string& host,
                   uint16_t remote_tcp_port,
                   uint16_t local_udp_port,
                   const std::string& public_key,
                   const std::string& private_key,
                   const std::string& server_public_key_filename,
                   bool upnp);
           ~Client();

           void Write(const Command&);
           void WriteUDP(const Command&);
           void Close();

           Command PopCommand();
           bool command_empty();

           unsigned int id() const;

           double GetReadByteAverage() const;
           double GetWriteByteAverage() const;

           bool online() const;

        private:
           boost::asio::io_service io_service_;
           tcp::resolver resolver_;
           tcp::resolver::iterator iterator_;

           boost::asio::io_service io_service_udp_;
           udp::resolver::iterator iterator_udp_;

           boost::thread thread_;
           ClientSessionPtr session_;

           Signature signature_;

           int write_average_limit_;

           std::queue<Command> command_queue_;
           boost::mutex mutex_;

           uint16_t udp_port_;
           std::string global_ip_;

        private:
           constexpr static int DEFAULT_WRITE_AVERAGE_LIMIT = 200;

    };

}
