//
// Server.cpp
//

#include "Server.hpp"
#include "Utils.hpp"
#include <algorithm>
#include <boost/make_shared.hpp>
#include "../Logger.hpp"

namespace network {

    const int Server::UDP_TEST_PACKET_TIME = 5;

    Server::Server(int port) :
            endpoint_(tcp::v4(), port),
            acceptor_(io_service_, endpoint_),
            udp_socket_(io_service_udp_, udp::endpoint(udp::v4(), 39391)),
            max_total_read_average_(5000),
            max_session_read_average_(600),
            min_session_read_average_(100),
            session_read_average_(200)
    {

    }

    void Server::Start(CallbackFuncPtr callback)
    {
        callback_ = std::make_shared<CallbackFunc>(
                [&](network::Command c){

            // ログアウト
            if (c.header() == network::header::FatalConnectionError) {
                // 受信制限量を更新
                /*
                auto new_average = GetSessionReadAverageLimit();
                if (session_read_average_ != new_average) {
                    session_read_average_ = new_average;
                    SendAll(network::ClientReceiveWriteAverageLimitUpdate(session_read_average_));
                }
                */
            }

            // 通信量制限を越えていた場合、強制的に切断
            else if (auto session = c.session().lock()) {
                if (session->GetReadByteAverage() > session_read_average_) {
                    if (auto session = c.session().lock()) {
                        if (callback) {
                            (*callback)(network::PlayerLogoutNotify(session->id()));
                        }
                        return;
                    }
                    session->Close();
                }
            }

            if (callback) {
                (*callback)(c);
            }
        });

        {
        auto new_session = boost::make_shared<ServerSession>(io_service_, io_service_udp_);
        acceptor_.async_accept(new_session->socket(),
                              boost::bind(&Server::ReceiveSession, this, new_session, boost::asio::placeholders::error));
        }

        boost::asio::io_service::work work(io_service_);
        io_service_.run();
        io_service_udp_.run();
    }

    void Server::ReceiveSession(const SessionPtr& session, const boost::system::error_code& error)
    {
        if (session_read_average_ > min_session_read_average_) {
            session->set_on_receive(callback_);
            session->Start();
            sessions_.push_back(SessionWeakPtr(session));

            session->Send(ClientRequestedPublicKeyFingerPrint());

            // 受信制限量を更新
            auto new_average = GetSessionReadAverageLimit();
            session->Send(network::ClientReceiveWriteAverageLimitUpdate(session_read_average_));

            if (session_read_average_ != new_average) {
                session_read_average_ = new_average;
                SendOthers(network::ClientReceiveWriteAverageLimitUpdate(session_read_average_),
                        session);
            }

        } else {
            Logger::Info("Refuse Session");
            session->SyncSend(ClientReceiveServerCrowdedError());
            session->Close();
        }

        auto new_session = boost::make_shared<ServerSession>(io_service_, io_service_udp_);
         acceptor_.async_accept(new_session->socket(),
                 boost::bind(&Server::ReceiveSession, this, new_session, boost::asio::placeholders::error));

        // 使用済のセッションのポインタを破棄
        auto it = std::remove_if(sessions_.begin(), sessions_.end(),
                [](const SessionWeakPtr& ptr){
            return ptr.expired();
        });
        sessions_.erase(it, sessions_.end());

    }

    void Server::SendAll(const Command& command)
    {
        for (SessionWeakPtr& ptr : sessions_) {
            if (auto session = ptr.lock()) {
                session->Send(command);
            }
        }
    }

    void Server::SendOthers(const Command& command, SessionWeakPtr self_ptr)
    {
        for (SessionWeakPtr& ptr : sessions_) {
            if (auto session = ptr.lock()) {
                if (auto self = self_ptr.lock()) {
                    if (*session != *self) {
                        session->Send(command);
                    }
                }
            }
        }
    }

    void Server::SendUDPTestPacket(const std::string& ip_address, uint16_t port)
    {
        using boost::asio::ip::udp;

        std::stringstream port_str;
        port_str << (int)port;

        udp::resolver resolver(io_service_udp_);
        udp::resolver::query query(udp::v4(), ip_address.c_str(), port_str.str().c_str());
        udp::resolver::iterator iterator = resolver.resolve(query);

        static char request[] = "MMO UDP Test Packet";
        for (int i = 0; i < UDP_TEST_PACKET_TIME; i++) {
            udp_socket_.send_to(boost::asio::buffer(request, sizeof(request)), *iterator);
        }
    }

    void Server::ServerSession::Start()
    {
        online_ = true;

        // Nagleアルゴリズムを無効化
        socket_.set_option(boost::asio::ip::tcp::no_delay(true));

        // IPアドレスを取得
        global_ip_ = socket_.remote_endpoint().address().to_string();

        boost::asio::async_read_until(socket_,
            receive_buf_, Utils::DELIMITOR,
            boost::bind(
              &ServerSession::Receive, shared_from_this(),
              boost::asio::placeholders::error));

    }

    void Server::UpdateUserList() {
        boost::mutex::scoped_lock(mutex_);

        unsigned int channel = 0;
        std::string channel_str(reinterpret_cast<const char*>(&channel), sizeof(int));
        std::string id_str;

        std::remove_if(sessions_.begin(), sessions_.end(), [](const SessionWeakPtr& ptr){
            return ptr.expired();
        });

        for (SessionWeakPtr& ptr : sessions_) {
            if (auto session = ptr.lock()) {
                if (session->login()) {
                    unsigned int id = session->id();
                    id_str += std::string(reinterpret_cast<const char*>(&id), sizeof(int));
                }
            }
        }
        Command command(network::header::ClientUpdateChannelUserList, channel_str + id_str);
        SendAll(command);
    }

    int Server::GetSessionReadAverageLimit()
    {
        int byte = max_total_read_average_ / (sessions_.size() + 1);
        byte = std::min(byte, max_session_read_average_);
        return byte;
    }

    int Server::max_total_read_average() const
    {
        return max_total_read_average_;
    }

    int Server::max_session_read_average() const
    {
        return max_session_read_average_;
    }

    int Server::min_session_read_average() const
    {
        return min_session_read_average_;
    }

    void Server::set_max_total_read_average(int byte)
    {
        max_total_read_average_ = byte;
    }

    void Server::set_max_session_read_average(int byte)
    {
        max_session_read_average_ = byte;
    }

    void Server::set_min_session_read_average(int byte)
    {
        min_session_read_average_ = byte;
    }

}
