//
// MMO Server
//

//
// Version
const int VERSION[] = {
        0, 1, 0, 0,             // Major
        0, 0, 0, 0              // Minor
};

#include <iostream>
#include <sstream>
#include <ctime>
#include <boost/format.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include "../common/network/Server.hpp"
#include "../common/network/Encrypter.hpp"
#include "../common/network/Signature.hpp"
#include "../common/Logger.hpp"
#include "Config.hpp"
#include "ChatLog.hpp"
#include "../common/database/Account.hpp"
#include "../common/database/CardLibrary.hpp"

using namespace boost::posix_time;

int main(int argc, char* argv[])
{

  try {

    // 署名
    network::Signature sign("server_key");

    // 設定を読み込み
    Config config("config.json");

    // チャットログ
    ChatLog chat_log("chat.db");

    // アカウント
    Account account("account.db");

    // カード
    CardLibrary card("card.db");

    // バージョン情報
    auto version = (boost::format("Start MMO Server  version: %d.%d.%d.%d.%d.%d.%d.%d")
        % VERSION[0] % VERSION[1] % VERSION[2] % VERSION[3]
        % VERSION[4] % VERSION[5] % VERSION[6] % VERSION[7]
    ).str();
    Logger::Info(version);

    int port = config.port();
    network::Server server(port);

    // サーバー設定
    server.set_max_total_read_average(config.max_total_read_average());
    server.set_max_session_read_average(config.max_session_read_average());
    server.set_min_session_read_average(config.min_session_read_average());

    auto callback = std::make_shared<std::function<void(network::Command)>>(
            [&server, &chat_log, &account, &card, &sign](network::Command c){

        // ログを出力
        auto msg = (boost::format("Receive: 0x%08x %dbyte") % c.header() % c.body().size()).str();
        if (auto session = c.session().lock()) {
            msg += " from " + session->global_ip();
        }

        // if (auto session = c.session().lock()) {
        //     std::cout << "Write Average: " << session->GetReadByteAverage() << "bytes" << std::endl;
        // }

        switch (c.header()) {

//        // チャットメッセージ受信
//        case network::header::ServerReceiveChatMessage:
//        {
//            if (auto session = c.session().lock()) {
//                unsigned int id = static_cast<unsigned int>(session->id());
//                ptime now = second_clock::local_time();
//                auto time_string = to_iso_extended_string(now);
//
//                server.SendAll(network::ClientReceiveChatMessage(id, time_string, c.body()));
//                ChatLog::Message chat = {session->id(), time_string, c.body()};
//                chat_log.Push(chat);
//                Logger::Info(msg);
//            }
//        }
//            break;

        // JSONメッセージ受信
        case network::header::ServerReceiveJSON:
        {
            if (auto session = c.session().lock()) {
                std::tuple<std::string> result_tuple;
                network::Utils::Deserialize(c.body(), &result_tuple);

                unsigned int id = static_cast<unsigned int>(session->id());
                ptime now = second_clock::universal_time();
                auto time_string = to_iso_extended_string(now);

                std::string info_json;
                info_json += "{";
                info_json += FormatString::New("\"id\":\"%d\",", id);
                info_json += FormatString::New("\"time\":\"%s\"", time_string);
                info_json += "}";

                std::string message_json = std::get<0>(result_tuple);

                server.SendAll(network::ClientReceiveJSON(info_json, message_json));
                Logger::Info("Receive JSON: %s", message_json);
                Logger::Info(msg);
            }
        }
            break;

        // 位置情報受信
        case network::header::ServerUpdatePlayerPosition:
        {
            if (auto session = c.session().lock()) {
                std::tuple<float, float, float, float> result_tuple;
                network::Utils::Deserialize(c.body(), &result_tuple);
                server.SendOthers(network::ClientUpdatePlayerPosition(
                        session->id(),
                        std::get<0>(result_tuple),
                        std::get<1>(result_tuple),
                        std::get<2>(result_tuple),
                        std::get<3>(result_tuple)), c.session());
            }
        }
            break;

        // 公開鍵フィンガープリント受信
        case network::header::ServerReceiveClientInfo:
        {
            if (auto session = c.session().lock()) {

                std::tuple<std::string, uint16_t> result_tuple;
                network::Utils::Deserialize(c.body(), &result_tuple);

                // UDPパケットの宛先を設定
                //session->set_global_ip(std::get<1>(result_tuple));
                session->set_udp_port(std::get<1>(result_tuple));

                Logger::Info("UDP destination is %s:%d", session->global_ip(), session->udp_port());

                // テスト送信
                server.SendUDPTestPacket(session->global_ip(), session->udp_port());

                long id = account.GetUserId(std::get<0>(result_tuple));
                if (id < 1) {
                    // 未登録の場合、公開鍵を要求
                    session->Send(network::ClientRequestedPublicKey());
                } else {
                    unsigned int user_id = static_cast<unsigned int>(id);
                    // ログイン
                    account.LogIn(user_id);
                    session->LogIn(user_id);
                    session->encrypter().SetPublicKey(account.GetPublicKey(user_id));

                    account.SetUserIPAddress(session->id(), session->global_ip());
                    account.SetUserUDPPort(session->id(), session->udp_port());

                    // 共通鍵を送り返す
                    auto key = session->encrypter().GetCryptedCommonKey();
                    session->Send(network::ClientReceiveCommonKey(key, sign.Sign(key), user_id));

                    server.SendAll(
                            network::ClientReceiveAccountRevisionUpdateNotify(account.GetCurrentRevision()));

                    Logger::Info(msg);
                }
            }
        }
            break;

        // 公開鍵受信
        case network::header::ServerReceivePublicKey:
        {
            if (auto session = c.session().lock()) {
                long id = account.RegisterPublicKey(c.body());
                unsigned int user_id = static_cast<unsigned int>(id);
                // ログイン
                account.LogIn(user_id);
                session->LogIn(user_id);
                session->encrypter().SetPublicKey(account.GetPublicKey(user_id));

                account.SetUserIPAddress(session->id(), session->global_ip());
                account.SetUserUDPPort(session->id(), session->udp_port());

                // 共通鍵を送り返す
                auto key = session->encrypter().GetCryptedCommonKey();
                session->Send(network::ClientReceiveCommonKey(key, sign.Sign(key), user_id));

                server.SendAll(
                        network::ClientReceiveAccountRevisionUpdateNotify(account.GetCurrentRevision()));

                Logger::Info(msg);
            }
        }
            break;

        // 暗号化通信開始
        case network::header::ServerStartEncryptedSession:
        {
            if (auto session = c.session().lock()) {
                session->Send(network::ClientStartEncryptedSession());
                session->EnableEncryption();

                // ユーザーリスト更新
                server.UpdateUserList();

                Logger::Info(msg);
            }
        }
            break;

        // チャットログ要求受信
        case network::header::ServerRequestedChatLog:
        {
            if (auto session = c.session().lock()) {
                for (const ChatLog::Message& chat : chat_log.Load()) {
                    network::Command res(network::header::ClientReceiveChatLog, chat.body);
                    session->Send(res);
                }
                Logger::Info(msg);
            }
        }
            break;

        // カードの申請
        case network::header::ServerReceiveNewCard:
        {
            if (auto session = c.session().lock()) {
                std::tuple<std::string, std::string, std::string> result_tuple;
                network::Utils::Deserialize(c.body(), &result_tuple);

                card.Register(
                        std::get<0>(result_tuple),
                        std::get<1>(result_tuple),
                        std::get<2>(result_tuple),
                        session->id());

                Logger::Info(msg);
            }
        }
            break;

        // カード更新情報の要求
        case network::header::ServerRequestedCardRevisionPatch:
        {
            if (auto session = c.session().lock()) {
                int client_revision;
                network::Utils::Deserialize(c.body(), &client_revision);
                if (client_revision < card.GetCurrentRevision()) {
                    session->Send(network::ClientReceiveCardRevisionPatch(card.GetRevisionPatch(client_revision)));
                }
                Logger::Info(msg);
            }
        }
            break;

        // アカウント更新情報の要求
        case network::header::ServerRequestedAccountRevisionPatch:
        {
            if (auto session = c.session().lock()) {
                int client_revision;
                network::Utils::Deserialize(c.body(), &client_revision);
                session->set_account_revision(client_revision);

                if (client_revision < account.GetCurrentRevision()) {
                    session->Send(network::ClientReceiveAccountRevisionPatch(account.GetRevisionPatch(client_revision)));
                }
                Logger::Info(msg);
            }
        }
        break;

        // ユーザー名の更新
        case network::header::ServerUpdatePlayerName:
        {
            if (auto session = c.session().lock()) {
                std::string name;
                network::Utils::Deserialize(c.body(), &name);
                if (name.size() > 0 && name.size() <= 64) {
                    account.SetUserName(session->id(), name);
                    server.SendAll(
                            network::ClientReceiveAccountRevisionUpdateNotify(account.GetCurrentRevision()));
                }
                Logger::Info(msg);
            }
        }
        break;

        // トリップの更新
        case network::header::ServerUpdatePlayerTrip:
        {
            if (auto session = c.session().lock()) {
                std::string trip;
                network::Utils::Deserialize(c.body(), &trip);
                if (trip.size() > 0 && trip.size() <= 64) {
                    account.SetUserTrip(session->id(), trip);
                    server.SendAll(
                            network::ClientReceiveAccountRevisionUpdateNotify(account.GetCurrentRevision()));
                }
                Logger::Info(msg);
            }
        }
        break;

        // カード存在確認
        //case network::header::ServerRequestedCardExistence:
        //{
        //    if (auto session = c.session().lock()) {
        //
        //    }
        //}
        //    break;

       // ログアウト
       case network::header::PlayerLogoutNotify:
       {
           Logger::Info(msg);
       }
       break;

       // エラー
       case network::header::FatalConnectionError:
       {
           if (c.body().size() > 0) {
               std::cout << "Logout" << std::endl;
               int user_id;
               network::Utils::Deserialize(c.body(), &user_id);
               account.LogOut(user_id);

               server.SendAll(
                       network::ClientReceiveAccountRevisionUpdateNotify(account.GetCurrentRevision()));
           }
       }
       Logger::Info(msg);
       break;

        default:
            break;
        }

    });

    server.Start(callback);

    Logger::Info("UUU");

  } catch (std::exception& e) {
      Logger::Error(e.what());
      Logger::Info(u8"Stop Server");
  }

  return 0;

}
