//
// Command.hpp
//

#pragma once

#include <string>
#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>
#include "CommandHeader.hpp"
#include "Utils.hpp"

namespace network {

class Session;

typedef boost::weak_ptr<Session> SessionWeakPtr;

    struct Command {
        public:
            Command(header::CommandHeader header, const std::string body) :
                header_(header), body_(body) {}

            Command(header::CommandHeader header, const std::string body, const SessionWeakPtr& session) :
                header_(header), body_(body), session_(session) {}

            header::CommandHeader header() const;
            const std::string& body() const;
            SessionWeakPtr session();

        private:
            header::CommandHeader header_;

        protected:
            std::string body_;
            SessionWeakPtr session_;
    };

    // コネクションの確立に成功
    struct ConnectionSucceeded : public Command {
        ConnectionSucceeded();
    };

    // コネクションの確立に失敗
    struct ConnectionFailed : public Command {
        ConnectionFailed();
    };

    // コネクションの切断
    struct FatalConnectionError : public Command {
        FatalConnectionError();
        FatalConnectionError(unsigned int user_id);
    };

    // チャットメッセージを受信した
    struct ClientReceiveChatMessage : public Command {
        ClientReceiveChatMessage(unsigned int user_id, const std::string& time_string, const std::string& msg);
    };

    // チャットメッセージを受信した
    struct ServerReceiveChatMessage : public Command {
        ServerReceiveChatMessage(const std::string& msg);
    };

    // 暗号化通信を開始
    struct ServerStartEncryptedSession : public Command {
        ServerStartEncryptedSession();
    };

    // 暗号化通信を開始
    struct ClientStartEncryptedSession : public Command {
        ClientStartEncryptedSession();
    };

    // クライアントからの公開鍵を受信
    struct ServerReceivePublicKey : public Command {
        ServerReceivePublicKey(const std::string& key);
    };

    // サーバーからの共通鍵を受信
    struct ClientReceiveCommonKey : public Command {
        ClientReceiveCommonKey(const std::string& key,
                const std::string& sign,
                unsigned int user_id);
    };

    // チャットログを受信した
    struct ClientReceiveChatLog : public Command {
        ClientReceiveChatLog();
    };

    // チャットログを要求された
    struct ServerRequestedChatLog : public Command {
        ServerRequestedChatLog();
    };

    // ユーザーが入室した
    struct ClientJoinPlayer : public Command {
        ClientJoinPlayer();
    };

    // ユーザーが退室した
    struct ClientLeavePlayer : public Command {
        ClientLeavePlayer();
    };

    // 他のプレイヤーの位置情報が更新された
    struct ClientUpdatePlayerPosition : public Command {
        ClientUpdatePlayerPosition(unsigned int id, float x, float y, float z, float theta);
    };

    // プレイヤーの位置情報が更新された
    struct ServerUpdatePlayerPosition : public Command {
        ServerUpdatePlayerPosition(float x, float y, float z, float theta);
    };

    // クライアントの情報を受信した　（公開鍵のフィンガープリント, UDPポート）
    struct ServerReceiveClientInfo : public Command {
        ServerReceiveClientInfo(const std::string& key, uint32_t version, uint16_t udp_port);
    };

    // 公開鍵を要求された
    struct ClientRequestedPublicKey : public Command {
        ClientRequestedPublicKey();
    };

    // クライアントの情報を要求された
    struct ClientRequestedClientInfo : public Command {
        ClientRequestedClientInfo();
    };

    // チャンネルのプレイヤーリストが更新された
    struct ClientUpdateChannelUserList : public Command {
        ClientUpdateChannelUserList();
    };

    // セッションを切断された
    struct ServerCloseSession : public Command {
        ServerCloseSession();
    };

    // カードの登録要求
    struct ServerReceiveNewCard : public Command {
        ServerReceiveNewCard(const std::string& name,
                             const std::string& note,
                             const std::string& url);
    };

    //
    struct ServerRequestedCardRevisionPatch : public Command {
        ServerRequestedCardRevisionPatch(int revision);
    };

    struct ServerRequestedAccountRevisionPatch : public Command {
        ServerRequestedAccountRevisionPatch(unsigned int user_id, int revision);
    };

    // カードデータベースの更新データ
    struct ClientReceiveCardRevisionPatch : public Command {
        ClientReceiveCardRevisionPatch(const std::string& patch);
    };

    // カードデータベースの更新通知
    struct ClientReceiveCardRevisionUpdateNotify : public Command {
        ClientReceiveCardRevisionUpdateNotify(int revision);
    };

    // アカウントデータベースの更新データ
    struct ClientReceiveAccountRevisionPatch : public Command {
        ClientReceiveAccountRevisionPatch(const std::string& patch);
    };

    // アカウントデータベースの更新通知
    struct ClientReceiveAccountRevisionUpdateNotify : public Command {
        ClientReceiveAccountRevisionUpdateNotify(unsigned int user_id, int revision);
    };

    struct ClientReceiveWriteAverageLimitUpdate : public Command {
        ClientReceiveWriteAverageLimitUpdate(unsigned short byte);
    };

    struct ClientReceiveServerCrowdedError : public Command {
        ClientReceiveServerCrowdedError();
    };

    struct PlayerLogoutNotify : public Command {
        PlayerLogoutNotify(unsigned int user_id);
    };

    struct ServerUpdatePlayerName : public Command {
        ServerUpdatePlayerName(const std::string& name);
    };

    struct ServerUpdatePlayerTrip : public Command {
        ServerUpdatePlayerTrip(const std::string& trip);
    };

    struct ClientReceiveUnsupportVersionError : public Command {
        ClientReceiveUnsupportVersionError(uint32_t require_version);
    };

    struct ServerReceiveAccountInitializeData : public Command {
        ServerReceiveAccountInitializeData(const std::string& data);
    };

    struct ServerUpdatePlayerModelName : public Command {
        ServerUpdatePlayerModelName(const std::string& name);
    };

    struct ServerReceiveJSON : public Command {
        ServerReceiveJSON(const std::string& json);
    };

    struct ClientReceiveJSON : public Command {
        ClientReceiveJSON(const std::string& info_json, const std::string& message_json);
    };


}
