/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.beans.dataset;

import java.util.*;
import java.io.*;

import jp.ossc.nimbus.core.*;

/**
 * R[hXL[}B<p>
 * {@link PropertySchema vpeBXL[}}̏WŁÃvpeBBeañXL[}\B<br>
 * R[hXL[}́A{@link PropertySchema vpeBXL[}}̏WłA<br>
 * <pre>
 *   vpeBXL[}̎NX:vpeBXL[}`
 *   vpeBXL[}̎NX:vpeBXL[}`
 *                   :
 * </pre>
 * Ƃ悤ɁAvpeB̐s؂Œ`B<br>
 * ܂AvpeBXL[}̎NX͏ȗ\ŁAȗꍇ́A{@link DefaultPropertySchema}KpB<br>
 * ܂AvpeBXL[}̎NXɁA{@link RecordListPropertySchema}w肵ꍇ́AGCAXg"LIST:...."ƒ`łB<br>
 * ܂AR[hXL[}AvpeBXL[}̃CX^XǗAXL[}`̃CX^X͐Ȃ悤ɂĂB<br>
 * 
 * @author M.Takata
 */
public class RecordSchema{
    
    /**
     * vpeBXL[}̎NX̃GCAX {@link RecordListPropertySchema}̃GCAXB<p>
     */
    public static final String PROPERTY_SCHEMA_ALIAS_NAME_LIST = "LIST";
    
    /**
     * vpeBXL[}̎NX̃GCAX {@link RecordPropertySchema}̃GCAXB<p>
     */
    public static final String PROPERTY_SCHEMA_ALIAS_NAME_RECORD = "RECORD";
    
    /**
     * vpeBXL[}̎NX̃GCAX {@link XpathPropertySchema}̃GCAXB<p>
     */
    public static final String PROPERTY_SCHEMA_ALIAS_NAME_XPATH = "XPATH";
    
    /**
     * vpeBXL[}̎NX̃GCAX {@link RowVersionPropertySchema}̃GCAXB<p>
     */
    public static final String PROPERTY_SCHEMA_ALIAS_NAME_ROW_VERSION = "ROWVERSION";
    
    private static final String PROP_SCHEMA_CLASS_DELIMETER = ":";
    
    protected static final Map<String, RecordSchema> recordSchemaManager
         = Collections.synchronizedMap(new HashMap<String, RecordSchema>());
    
    protected static final Map<String, PropertySchema> propertySchemaManager
         = Collections.synchronizedMap(new HashMap<String, PropertySchema>());
    
    protected static final Map<String, Class<?>> propertySchemaAliasMap
         = Collections.synchronizedMap(new HashMap<String, Class<?>>());
    
    protected Map<String, PropertySchema> propertySchemaMap = new LinkedHashMap<String, PropertySchema>();
    protected List<String> propertyNames = new ArrayList<String>();
    protected List<PropertySchema> primaryKeyProperties;
    
    static{
        propertySchemaAliasMap.put(
            PROPERTY_SCHEMA_ALIAS_NAME_LIST,
            RecordListPropertySchema.class
        );
        propertySchemaAliasMap.put(
            PROPERTY_SCHEMA_ALIAS_NAME_RECORD,
            RecordPropertySchema.class
        );
        propertySchemaAliasMap.put(
            PROPERTY_SCHEMA_ALIAS_NAME_ROW_VERSION,
            RowVersionPropertySchema.class
        );
        propertySchemaAliasMap.put(
            PROPERTY_SCHEMA_ALIAS_NAME_XPATH,
            XpathPropertySchema.class
        );
    }
    
    /**
     * XL[}B<p>
     */
    protected String schema;
    
    /**
     * ̃R[hXL[}𐶐B<p>
     */
    public RecordSchema(){
    }
    
    /**
     * R[hXL[}擾B<p>
     * XL[}`̃R[hXL[}AyуvpeBXL[}̃CX^XVȂ悤ɁAŊǗĂB<br>
     *
     * @param schema R[hXL[}
     * @return R[hXL[}
     */
    public static RecordSchema getInstance(String schema)
     throws PropertySchemaDefineException{
        RecordSchema recordSchema = recordSchemaManager.get(schema);
        if(recordSchema == null){
            recordSchema = new RecordSchema();
            recordSchema.setSchema(schema);
            recordSchemaManager.put(schema, recordSchema);
        }
        return recordSchema;
    }
    
    /**
     * R[hXL[}擾B<p>
     * XL[}`̃R[hXL[}AyуvpeBXL[}̃CX^XVȂ悤ɁAŊǗĂB<br>
     *
     * @param schemata R[h̃XL[}`\vpeBXL[}z
     * @return R[hXL[}
     */
    public static RecordSchema getInstance(PropertySchema[] schemata)
     throws PropertySchemaDefineException{
        final StringBuilder buf = new StringBuilder();
        final String lineSep = System.getProperty("line.separator");
        for(int i = 0; i < schemata.length; i++){
            PropertySchema propertySchema = schemata[i];
            buf.append(propertySchema.getSchema());
            if(i != schemata.length - 1){
                buf.append(lineSep);
            }
        }
        final String schema = buf.toString();
        RecordSchema recordSchema = recordSchemaManager.get(schema);
        if(recordSchema == null){
            recordSchema = new RecordSchema();
            recordSchema.setPropertySchemata(schemata);
            recordSchemaManager.put(schema, recordSchema);
        }
        return recordSchema;
    }
    
    /**
     * XL[}ǉR[hXL[}擾B<p>
     * XL[}`̃R[hXL[}AyуvpeBXL[}̃CX^XVȂ悤ɁAŊǗĂB<br>
     *
     * @param schema R[hXL[}
     * @return R[hXL[}
     */
    public RecordSchema appendSchema(String schema)
     throws PropertySchemaDefineException{
        final StringBuilder buf = new StringBuilder();
        if(this.schema != null){
            buf.append(this.schema);
            buf.append(System.getProperty("line.separator"));
        }
        buf.append(schema);
        final String newSchema = buf.toString();
        RecordSchema recordSchema = recordSchemaManager.get(newSchema);
        if(recordSchema == null){
            recordSchema = new RecordSchema();
            recordSchema.setSchema(newSchema);
            recordSchemaManager.put(newSchema, recordSchema);
        }
        return recordSchema;
    }
    
    /**
     * R[h̃XL[}`ݒ肷B<p>
     *
     * @param schema R[h̃XL[}`
     * @exception PropertySchemaDefineException R[h̃XL[}`Ɏsꍇ
     */
    public void setSchema(String schema) throws PropertySchemaDefineException{
        propertySchemaMap.clear();
        propertyNames.clear();
        if(primaryKeyProperties != null){
            primaryKeyProperties = null;
        }
        BufferedReader reader = new BufferedReader(new StringReader(schema));
        String propertySchemaStr = null;
        try{
            while((propertySchemaStr = reader.readLine()) != null){
                PropertySchema propertySchema
                    = createPropertySchema(propertySchemaStr);
                if(propertySchema == null){
                    continue;
                }
                if(propertySchemaMap.containsKey(propertySchema.getName())){
                    throw new PropertySchemaDefineException(
                        propertySchemaStr,
                        "Property name is duplicated."
                    );
                }
                propertySchemaMap.put(propertySchema.getName(), propertySchema);
                propertyNames.add(propertySchema.getName());
                if(propertySchema.isPrimaryKey()){
                    if(primaryKeyProperties == null){
                        primaryKeyProperties = new ArrayList<PropertySchema>();
                    }
                    primaryKeyProperties.add(propertySchema);
                }
            }
        }catch(IOException e){
            // NȂ͂
            throw new PropertySchemaDefineException(schema, e);
        }
        this.schema = schema;
    }
    
    /**
     * vpeB̃XL[}`𐶐B<p>
     *
     * @param schema vpeB̃XL[}`
     * @return vpeB̃XL[}`
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    protected PropertySchema createPropertySchema(String schema)
     throws PropertySchemaDefineException{
        if(schema == null || schema.length() == 0){
            return null;
        }
        Class<?> propertySchemaClass = DefaultPropertySchema.class;
        final int index = schema.indexOf(PROP_SCHEMA_CLASS_DELIMETER);
        if(index == -1 || index == schema.length() - 1){
            throw new PropertySchemaDefineException(
                schema,
                "The class name of PropertySchema is not specified."
            );
        }else if(index != 0){
            String propertySchemaClassName
                 = schema.substring(0, index);
            if(propertySchemaAliasMap.containsKey(propertySchemaClassName)){
                propertySchemaClass = propertySchemaAliasMap.get(propertySchemaClassName);
            }else{
                try{
                    propertySchemaClass = Class.forName(
                        propertySchemaClassName,
                        true,
                        NimbusClassLoader.getInstance()
                    );
                }catch(ClassNotFoundException e){
                    throw new PropertySchemaDefineException(
                        schema,
                        "The class name of PropertySchema is illegal.",
                        e
                    );
                }
            }
        }
        schema = schema.substring(index + 1);
        final String propertySchemaKey
             = propertySchemaClass.getName() + schema;
        PropertySchema propertySchema
             = propertySchemaManager.get(propertySchemaKey);
        if(propertySchema == null){
            try{
                propertySchema = (PropertySchema)propertySchemaClass.newInstance();
            }catch(InstantiationException e){
                throw new PropertySchemaDefineException(
                    schema,
                    e
                );
            }catch(IllegalAccessException e){
                throw new PropertySchemaDefineException(
                    schema,
                    e
                );
            }
            propertySchema.setSchema(schema);
            propertySchemaManager.put(propertySchemaKey, propertySchema);
        }
        return propertySchema;
    }
    
    /**
     * R[h̃XL[}擾B<p>
     *
     * @return R[h̃XL[}
     */
    public String getSchema(){
        return schema;
    }
    
    /**
     * R[h̃XL[}`ݒ肷B<p>
     *
     * @param schemata R[h̃XL[}`\vpeBXL[}z
     */
    public void setPropertySchemata(PropertySchema[] schemata){
        propertySchemaMap.clear();
        propertyNames.clear();
        if(primaryKeyProperties != null){
            primaryKeyProperties = null;
        }
        final StringBuilder buf = new StringBuilder();
        final String lineSep = System.getProperty("line.separator");
        for(int i = 0; i < schemata.length; i++){
            PropertySchema propertySchema = schemata[i];
            buf.append(propertySchema.getSchema());
            if(i != schemata.length - 1){
                buf.append(lineSep);
            }
            final String propertySchemaKey
                 = propertySchema.getClass().getName() + propertySchema.getSchema();
            propertySchemaManager.put(propertySchemaKey, propertySchema);
            
            propertySchemaMap.put(propertySchema.getName(), propertySchema);
            propertyNames.add(propertySchema.getName());
            if(propertySchema.isPrimaryKey()){
                if(primaryKeyProperties == null){
                    primaryKeyProperties = new ArrayList<PropertySchema>();
                }
                primaryKeyProperties.add(propertySchema);
            }
        }
        
        schema = buf.toString();
    }
    
    /**
     * vpeBXL[}z擾B<p>
     *
     * @return vpeBXL[}z
     */
    public PropertySchema[] getPropertySchemata(){
        return propertySchemaMap.values().toArray(
            new PropertySchema[propertySchemaMap.size()]
        );
    }
    
    /**
     * vC}L[ƂȂvpeBXL[}z擾B<p>
     *
     * @return vpeBXL[}z
     */
    public PropertySchema[] getPrimaryKeyPropertySchemata(){
        return (primaryKeyProperties == null
            || primaryKeyProperties.size() == 0) ? null
                : primaryKeyProperties.toArray(
                    new PropertySchema[primaryKeyProperties.size()]);
    }
    
    /**
     * w肳ꂽCfbNX̃vpeBX擾B<p>
     *
     * @param index CfbNX
     * @return vpeBX
     */
    public String getPropertyName(int index){
        if(index < 0 || index >= propertyNames.size()){
            return null;
        }
        return propertyNames.get(index);
    }
    
    /**
     * w肳ꂽvpeBX̃CfbNX擾B<p>
     *
     * @param name CfbNX
     * @return CfbNX
     */
    public int getPropertyIndex(String name){
        return propertyNames.indexOf(name);
    }
    
    /**
     * w肳ꂽCfbNX̃vpeBXL[}擾B<p>
     *
     * @param index CfbNX
     * @return vpeBXL[}
     */
    public PropertySchema getPropertySchema(int index){
        if(index < 0 || index >= propertyNames.size()){
            return null;
        }
        return propertySchemaMap.get(propertyNames.get(index));
    }
    
    /**
     * w肳ꂽvpeB̃vpeBXL[}擾B<p>
     *
     * @param name vpeB
     * @return vpeBXL[}
     */
    public PropertySchema getPropertySchema(String name){
        if(name == null){
            return null;
        }
        return propertySchemaMap.get(name);
    }
    
    /**
     * vpeB̐擾B<p>
     *
     * @return vpeB̐
     */
    public int getPropertySize(){
        return propertyNames.size();
    }
    
    /**
     * ̃R[hXL[}̕\擾B<p>
     *
     * @return \
     */
    @Override
    public String toString(){
        final StringBuilder buf = new StringBuilder();
        buf.append('{');
        if(propertySchemaMap != null){
            final Iterator<PropertySchema> schemata = propertySchemaMap.values().iterator();
            while(schemata.hasNext()){
                buf.append(schemata.next());
                if(schemata.hasNext()){
                    buf.append(';');
                }
            }
        }
        buf.append('}');
        return buf.toString();
    }
}