/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import java.util.Locale;
import java.io.Serializable;

import jp.ossc.nimbus.service.log.Logger;

/**
 * {@link Logger}bp[B<p>
 *
 * @author M.Takata
 */
class LoggerWrapper implements Logger, ServiceStateListener, Serializable{
    
    private static final long serialVersionUID = 6446367171255245828L;
    
    private Logger defaultLog;
    private Logger currentLog;
    
    public LoggerWrapper(Logger defaultLog){
        this(null, null, defaultLog);
    }
    
    public LoggerWrapper(
        Logger log,
        Service logService,
        Logger defaultLog
    ){
        setDefaultLogger(defaultLog);
        setLogger(log, logService);
    }
    
    public void setDefaultLogger(Logger log){
        if(log == this){
            return;
        }
        this.defaultLog = log;
    }
    
    public Logger getDefaultLogger(){
        return defaultLog;
    }
    
    public void setLogger(Logger log){
        setLogger(log, null);
    }
    
    public void setLogger(
        Logger log,
        Service logService
    ){
        if(log == this){
            return;
        }
        if(logService != null){
            if(logService.getState() == Service.State.STARTED){
                currentLog = log;
            }else{
                currentLog = defaultLog;
            }
            try{
                final ServiceStateListenable listenable
                    = ServiceManagerFactory.getServiceStateListenable(
                        logService.getServiceManagerName(),
                        logService.getServiceName()
                    );
                if(listenable != null){
                    listenable.addServiceStateListener(this);
                }
            }catch(ServiceNotFoundException e){
            }
        }else{
            if(log != null){
                currentLog = log;
            }else{
                currentLog = defaultLog;
            }
        }
    }
    
    public Logger getLogger(){
        return currentLog;
    }
    
    @Override
    public void write(String logCode, Object... embeds){
        if(currentLog != null){
            currentLog.write(logCode, embeds);
        }
    }
    
    @Override
    public void write(String logCode, Throwable oException, Object... embeds){
        if(currentLog != null){
            currentLog.write(logCode, oException, embeds);
        }
    }
    
    @Override
    public void write(String logCode, Locale lo, Object... embeds){
        if(currentLog != null){
            currentLog.write(logCode, lo, embeds);
        }
    }
    
    @Override
    public void write(String logCode, Locale lo, Throwable oException, Object... embeds){
        if(currentLog != null){
            currentLog.write(logCode, lo, oException, embeds);
        }
    }
    
    @Override
    public boolean isWrite(String logCode){
        if(currentLog != null){
            return currentLog.isWrite(logCode);
        }
        return false;
    }
    
    @Override
    public void stateChanged(ServiceStateChangeEvent e) throws Exception{
        final Service service = e.getService();
        final Service.State state = service.getState();
        final String managerName = service.getServiceManagerName();
        final String serviceName = service.getServiceName();
        switch(state){
        case STARTED:
            currentLog = (Logger)ServiceManagerFactory.getServiceObject(
                managerName,
                serviceName
            );
            break;
        case STOPPING:
            currentLog = defaultLog;
            break;
        default:
        }
        
    }
    
    @Override
    public boolean isEnabledState(Service.State state){
        switch(state){
        case STARTED:
        case STOPPING:
            return true;
        default:
            return false;
        }
    }
    
    @Override
    public String toString(){
        return super.toString() + '{' + "currentLog=" + currentLog + "defaultLog=" + defaultLog  + '}';
    }
}
