/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import java.io.*;
import java.util.*;
import java.net.*;
import org.w3c.dom.*;

/**
 * T[rX`&lt;nimbus&gt;vf^f[^B<p>
 * T[rX`t@C&lt;nimbus&gt;vfɋLqꂽei[郁^f[^ReiłB<p>
 *
 * @author M.Takata
 * @see <a href="nimbus-service_2_0.xsd">T[rX`t@CXL[}`</a>
 */
public class NimbusMetaData extends MetaData implements Serializable{
    
    private static final long serialVersionUID = -5309966428718081110L;
    
    /**
     * &lt;nimbus&gt;vf̗vfB<p>
     */
    public static final String NIMBUS_TAG_NAME = "nimbus";
    
    /**
     * &lt;nimbus&gt;vf̎qvf&lt;manager-repository&gt;vf̗vfB<p>
     */
    private static final String REPOSITORY_TAG_NAME = "manager-repository";
    
    /**
     * &lt;nimbus&gt;vf̎qvf&lt;log&gt;vf̗vfB<p>
     */
    private static final String LOG_TAG_NAME = "log";
    
    /**
     * &lt;nimbus&gt;vf̎qvf&lt;message&gt;vf̗vfB<p>
     */
    private static final String MESSAGE_TAG_NAME = "message";
    
    /**
     * &lt;nimbus&gt;vf̎qvfƂĒ`ꂽ&lt;manager&gt;vf̃^f[^i[}bvB<p>
     *
     * @see #getManager(String)
     * @see #getManagers()
     * @see #addManager(ManagerMetaData)
     */
    private final Map<String, ManagerMetaData> managers = new LinkedHashMap<String, ManagerMetaData>();
    
    /**
     * &lt;nimbus&gt;vf̎qvfƂĒ`ꂽ&lt;property-editors&gt;vf̃^f[^B<p>
     *
     * @see #getPropertyEditors()
     * @see #addPropertyEditor(String, String)
     */
    private PropertyEditorsMetaData propertyEditors;
    
    /**
     * ̃^f[^`ĂT[rX`t@CURLB<p>
     */
    private final URL myUrl;
    
    /**
     * &lt;manager-repository&gt;vf̃^f[^B<p>
     *
     * @see #getRepository()
     */
    private ServiceNameMetaData repository;
    
    /**
     * &lt;log&gt;vfŎw肳ꂽLogger̃^f[^B<p>
     *
     * @see #getLog()
     */
    private ServiceNameMetaData log;
    
    /**
     * &lt;message&gt;vfŎw肳ꂽMessageRecordFactorỹ^f[^B<p>
     *
     * @see #getMessage()
     */
    private ServiceNameMetaData message;
    
    /**
     * &lt;default-log&gt;vfŎw肳ꂽLogger̃^f[^B<p>
     *
     * @see #getDefaultLog()
     */
    private DefaultLogMetaData defaultLog;
    
    /**
     * [hServiceLoader
     */
    private ServiceLoader myLoader;
    
    /**
     * &lt;server-property&gt;vfŎw肳ꂽvpeBB<p>
     */
    private Map<String, ServerPropertyMetaData> properties = new LinkedHashMap<String, ServerPropertyMetaData>();
    
    private String encoding;
    private String docType;
    
    private List<IfDefMetaData> ifDefMetaDataList;
    
    /**
     * ̃^f[^`ĂT[rX`t@CURLA[g^f[^𐶐B<p>
     *
     * @param loader [hServiceLoader
     * @param url ̃^f[^`ĂT[rX`t@CURL
     */
    public NimbusMetaData(ServiceLoader loader, URL url){
        super();
        myUrl = url;
        myLoader = loader;
    }
    
    /**
     * ̃^f[^`ĂT[rX`t@CURL擾B<p>
     * 
     * @return ̃^f[^`ĂT[rX`t@CURL
     */
    public URL getURL(){
        return myUrl;
    }
    
    /**
     * [h{@link ServiceLoader}擾B<p>
     *
     * @return [hServiceLoader 
     */
    public ServiceLoader getServiceLoader(){
        return myLoader;
    }
    
    /**
     * [h{@link ServiceLoader}ݒ肷B<p>
     *
     * @param loader [hServiceLoader 
     */
    public void setServiceLoader(ServiceLoader loader){
        myLoader = loader;
    }
    
    /**
     * &lt;nimbus&gt;vf̎qvfAw肳ꂽT[rX&lt;manager&gt;vf̃^f[^擾B<p>
     * w肳ꂽT[rX&lt;manager&gt;vfA&lt;nimbus&gt;vf̎qvfɒ`ĂȂꍇ́AnullԂB<br>
     *
     * @param name T[rX
     * @return &lt;manager&gt;vf̃^f[^
     */
    public ManagerMetaData getManager(String name){
        return managers.get(name);
    }
    
    /**
     * &lt;nimbus&gt;vf̎qvfƂĒ`Ă&lt;manager&gt;vfnamel̏W擾B<p>
     * &lt;nimbus&gt;vf̎qvf&lt;manager&gt;vfP`ĂȂꍇ́ȀWԂB<br>
     * 
     * @return &lt;manager&gt;vfnamel̏W
     */
    public Set<String> getManagerNameSet(){
        return managers.keySet();
    }
    
    /**
     * &lt;nimbus&gt;vf̎qvfƂĒ`Ă&lt;manager&gt;vf̏W擾B<p>
     * &lt;nimbus&gt;vf̎qvf&lt;manager&gt;vfP`ĂȂꍇ́ȀWԂB<br>
     * 
     * @return &lt;manager&gt;vf̏W
     */
    public Collection<ManagerMetaData> getManagers(){
        return managers.values();
    }
    
    /**
     * &lt;nimbus&gt;vf̎qvfƂĒ`Ă&lt;manager&gt;vf̃^f[^o^B<p>
     *
     * @param manager &lt;manager&gt;vf̃^f[^
     */
    public void addManager(ManagerMetaData manager){
        if(manager == null){
            return;
        }
        manager.setParent(this);
        manager.setServiceLoader(getServiceLoader());
        ManagerMetaData removed = managers.put(manager.getName(), manager);
        if(removed != null && removed.getParent() == this){
            removed.setParent(null);
        }
    }
    
    /**
     * &lt;nimbus&gt;vf̎qvfƂĒ`Ă&lt;manager&gt;vf̃^f[^폜B<p>
     *
     * @param name &lt;manager&gt;vfname̒l
     */
    public void removeManager(String name){
        ManagerMetaData removed = managers.remove(name);
        if(removed != null && removed.getParent() == this){
            removed.setParent(null);
        }
    }
    
    /**
     * &lt;nimbus&gt;vf̎qvfƂĒ`Ă&lt;manager&gt;vf̃^f[^Sč폜B<p>
     */
    public void clearManagers(){
        for(Object name : managers.keySet().toArray()){
            removeManager((String)name);
        }
    }
    
    /**
     * &lt;property-editors&gt;vfŎw肳ꂽ^java.beans.PropertyEditor̃}bsO擾B<p>
     *
     * @return &lt;property-editors&gt;vfŎw肳ꂽ^java.beans.PropertyEditor̃}bsO
     */
    public Map<String, String> getPropertyEditors(){
        Map<String, String> result = new HashMap<String, String>();
        if(propertyEditors == null){
            return result;
        }
        for(String type : propertyEditors.getPropertyEditorTypes()){
            result.put(type, propertyEditors.getPropertyEditor(type));
        }
        return result;
    }
    
    /**
     * &lt;property-editors&gt;vfŎw肳ꂽ^java.beans.PropertyEditor̃}bsOǉB<p>
     *
     * @param type java.beans.PropertyEditorҏWNX
     * @param editor java.beans.PropertyEditorNX
     */
    public void addPropertyEditor(String type, String editor){
        if(propertyEditors == null){
            propertyEditors = new PropertyEditorsMetaData(this);
        }
        propertyEditors.setPropertyEditor(type, editor);
    }
    
    /**
     * &lt;property-editors&gt;vfŎw肳ꂽ^java.beans.PropertyEditor폜B<p>
     *
     * @param type java.beans.PropertyEditorҏWNX
     */
    public void removePropertyEditor(String type){
        if(propertyEditors == null){
            return;
        }
        propertyEditors.removePropertyEditor(type);
    }
    
    /**
     * &lt;property-editors&gt;vfŒ`ꂽjava.beans.PropertyEditor̃}bsOSč폜B<p>
     */
    public void clearPropertyEditors(){
        propertyEditors.clearPropertyEditors();
    }
    
    /**
     * &lt;manager-repository&gt;vfŎw肳ꂽ|Wg̃^f[^擾B<p>
     * &lt;manager-repository&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @return &lt;manager-repository&gt;vfŎw肳ꂽ|Wg̃^f[^
     */
    public ServiceNameMetaData getRepository(){
        return repository;
    }
    
    /**
     * &lt;manager-repository&gt;vfŎw肳ꂽ|Wg̃^f[^ݒ肷B<p>
     *
     * @param data &lt;manager-repository&gt;vfŎw肳ꂽ|Wg̃^f[^
     */
    public void setRepository(ServiceNameMetaData data){
        repository = data;
    }
    
    /**
     * &lt;log&gt;vfŎw肳ꂽLogger̃^f[^擾B<p>
     * &lt;log&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @return &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public ServiceNameMetaData getLog(){
        return log;
    }
    
    /**
     * &lt;log&gt;vfŎw肳ꂽLogger̃^f[^ݒ肷B<p>
     *
     * @param data &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public void setLog(ServiceNameMetaData data){
        log = data;
    }
    
    /**
     * &lt;message&gt;vfŎw肳ꂽMessageRecordFactorỹ^f[^擾B<p>
     * &lt;message&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @return &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public ServiceNameMetaData getMessage(){
        return message;
    }
    
    /**
     * &lt;message&gt;vfŎw肳ꂽMessageRecordFactorỹ^f[^ݒ肷B<p>
     *
     * @param data &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public void setMessage(ServiceNameMetaData data){
        message = data;
    }
    
    /**
     * &lt;default-log&gt;vfŎw肳ꂽLogger̃^f[^擾B<p>
     * &lt;default-log&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @return &lt;default-log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public DefaultLogMetaData getDefaultLog(){
        return defaultLog;
    }
    
    /**
     * &lt;default-log&gt;vfŎw肳ꂽLogger̃^f[^ݒ肷B<p>
     *
     * @param data &lt;default-log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public void setDefaultLog(DefaultLogMetaData data){
        defaultLog = data;
    }
    
    /**
     * vpeB̏W擾B<p>
     *
     * @return &lt;manager-property&gt;vfŎw肳ꂽvpeB̏W
     */
    public Set<String> getPropertyNameSet(){
        return properties.keySet();
    }
    
    /**
     * w肳ꂽvpeB&lt;server-property&gt;vfŎw肳ꂽvpeBl擾B<p>
     * YvpeB&lt;server-property&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @param property vpeB
     * @return &lt;server-property&gt;vfŎw肳ꂽvpeBl
     */
    public String getProperty(String property){
        ServerPropertyMetaData propData = properties.get(property);
        return propData == null ? null : propData.getValue();
    }
    
    /**
     * &lt;server-property&gt;vfŎw肳ꂽvpeB擾B<p>
     * &lt;server-property&gt;vfw肳ĂȂꍇ́APropertiesԂB<br>
     *
     * @return &lt;server-property&gt;vfŎw肳ꂽvpeB
     */
    public Map<String, String> getProperties(){
        final Map<String, String> props = new HashMap<String, String>();
        for(ServerPropertyMetaData propData : properties.values()){
            props.put(propData.getName(), propData.getValue() == null ? "" : propData.getValue());
        }
        return props;
    }
    
    /**
     * w肳ꂽvpeB&lt;server-property&gt;vfŎw肳ꂽvpeBlݒ肷B<p>
     *
     * @param property vpeB
     * @param value &lt;server-property&gt;vfŎw肳ꂽvpeBl
     */
    public void setProperty(String property, String value){
        ServerPropertyMetaData propData = properties.get(property);
        if(propData == null){
            propData = new ServerPropertyMetaData(this);
        }
        propData.setName(property);
        propData.setValue(value);
        properties.put(property, propData);
    }
    
    /**
     * w肳ꂽvpeB&lt;server-property&gt;vfŎw肳ꂽvpeB폜B<p>
     *
     * @param property vpeB
     */
    public void removeProperty(String property){
        properties.remove(property);
    }
    
    /**
     * &lt;server-property&gt;vfŎw肳ꂽvpeBSč폜B<p>
     */
    public void clearProperties(){
        properties.clear();
    }
    
    /**
     * &lt;manager-repository&gt;vf𐶐B<p>
     *
     * @param managerName |WgT[rXo^Ă}l[W
     * @param serviceName |WgT[rX̃T[rX
     * @return &lt;manager-repository&gt;vf̃^f[^
     */
    public ServiceNameMetaData createRepositoryMetaData(
        String managerName,
        String serviceName
    ){
        return new ServiceNameMetaData(
            this,
            REPOSITORY_TAG_NAME,
            managerName,
            serviceName
        );
    }
    
    /**
     * &lt;log&gt;vf𐶐B<p>
     *
     * @param managerName OT[rXo^Ă}l[W
     * @param serviceName OT[rX̃T[rX
     * @return &lt;log&gt;vf̃^f[^
     */
    public ServiceNameMetaData createLogMetaData(
        String managerName,
        String serviceName
    ){
        return new ServiceNameMetaData(
            this,
            LOG_TAG_NAME,
            managerName,
            serviceName
        );
    }
    
    /**
     * &lt;message&gt;vf𐶐B<p>
     *
     * @param managerName bZ[WT[rXo^Ă}l[W
     * @param serviceName bZ[WT[rX̃T[rX
     * @return &lt;message&gt;vf̃^f[^
     */
    public ServiceNameMetaData createMessageMetaData(
        String managerName,
        String serviceName
    ){
        return new ServiceNameMetaData(
            this,
            MESSAGE_TAG_NAME,
            managerName,
            serviceName
        );
    }
    
    public void setDocType(String str){
        docType = str;
    }
    
    public String getDocType(){
        return docType;
    }
    
    public void setEncoding(String encoding){
        this.encoding = encoding;
    }
    
    public String getEncoding(){
        return encoding;
    }
    
    /**
     * &lt;nimbus&gt;vfElementp[XāAg̏Ayюqvf̃^f[^̐sB<p>
     *
     * @param element &lt;nimbus&gt;vfElement
     * @exception DeploymentException &lt;nimbus&gt;vf̉́Ǎʂɂ郁^f[^̐Ɏsꍇ
     */
    @Override
    public void importXML(Element element) throws DeploymentException{
        super.importXML(element);
        
        if(!element.getTagName().equals(NIMBUS_TAG_NAME)){
            throw new DeploymentException(
                "Root tag must be " + NIMBUS_TAG_NAME + " : "
                 + element.getTagName()
            );
        }
        
        importXMLInner(element, null);
        
        final Iterator<Element> ifDefElements = getChildrenByTagName(
            element,
            IfDefMetaData.IFDEF_TAG_NAME
        );
        while(ifDefElements.hasNext()){
            if(ifDefMetaDataList == null){
                ifDefMetaDataList = new ArrayList<IfDefMetaData>();
            }
            final IfDefMetaData ifdefData
                 = new IfDefMetaData(this);
            ifdefData.importXML((Element)ifDefElements.next());
            ifDefMetaDataList.add(ifdefData);
        }
        
        if(ifDefMetaDataList == null || ifDefMetaDataList.size() == 0){
            return;
        }
        
        for(int i = 0, imax = ifDefMetaDataList.size(); i < imax; i++){
            IfDefMetaData ifdefData = ifDefMetaDataList.get(i);
            Element ifDefElement = ifdefData.getElement();
            if(ifDefElement == null){
                continue;
            }
            
            importXMLInner(ifDefElement, ifdefData);
            
            ifdefData.setElement(null);
        }
    }
    
    protected void importXMLInner(Element element, IfDefMetaData ifdefData) throws DeploymentException{
        
        final boolean ifdefMatch
            = ifdefData == null ? true : ifdefData.isMatch();
        
        final Iterator<Element> propElements = getChildrenByTagName(
            element,
            ServerPropertyMetaData.SERVER_PROPERTY_TAG_NAME
        );
        while(propElements.hasNext()){
            ServerPropertyMetaData propData = new ServerPropertyMetaData(this);
            if(ifdefData != null){
                propData.setIfDefMetaData(ifdefData);
                ifdefData.addChild(propData);
            }
            propData.importXML((Element)propElements.next());
            if(ifdefMatch && properties.containsKey(propData.getName())){
                throw new DeploymentException("Name of " + ServerPropertyMetaData.SERVER_PROPERTY_TAG_NAME + " is duplicated. name=" + propData.getName());
            }
            if(ifdefMatch){
                properties.put(
                    propData.getName(),
                    propData
                );
                String prop = propData.getValue();
                // VXevpeB̒u
                prop = Utility.replaceSystemProperty(prop, false);
                // T[rX[_\vpeB̒u
                prop = Utility.replaceServiceLoderConfig(
                    myLoader == null ? null : myLoader.getConfig(),
                    prop,
                    false
                );
                // T[ovpeB̒u
                prop = Utility.replaceServerProperty(prop, true);
                propData.setValue(prop);
            }
        }
        
        final Iterator<Element> managerElements = getChildrenByTagName(
            element,
            ManagerMetaData.MANAGER_TAG_NAME
        );
        while(managerElements.hasNext()){
            final ManagerMetaData managerData
                 = new ManagerMetaData(myLoader, this);
            if(ifdefData != null){
                managerData.setIfDefMetaData(ifdefData);
                ifdefData.addChild(managerData);
            }
            managerData.importXML((Element)managerElements.next());
            if(ifdefMatch && getManager(managerData.getName()) != null){
                throw new DeploymentException("Name of manager is duplicated. name=" + managerData.getName());
            }
            if(ifdefMatch){
                addManager(managerData);
            }
        }
        
        final Element repositoryElement
             = getOptionalChild(element, REPOSITORY_TAG_NAME);
        if(repositoryElement != null){
            if(ifdefMatch && repository != null){
                throw new DeploymentException("Element of " + REPOSITORY_TAG_NAME + " is duplicated.");
            }
            ServiceNameMetaData tmp = new ServiceNameMetaData(this);
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(repositoryElement);
            if(ifdefMatch){
                repository = tmp;
            }
        }
        
        final Element logElement = getOptionalChild(
            element,
            LOG_TAG_NAME
        );
        if(logElement != null){
            if(ifdefMatch && log != null){
                throw new DeploymentException("Element of " + LOG_TAG_NAME + " is duplicated.");
            }
            ServiceNameMetaData tmp = new ServiceNameMetaData(this);
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(logElement);
            if(ifdefMatch){
                log = tmp;
            }
        }
        
        final Element messageElement = getOptionalChild(
            element,
            MESSAGE_TAG_NAME
        );
        if(messageElement != null){
            if(ifdefMatch && message != null){
                throw new DeploymentException("Element of " + MESSAGE_TAG_NAME + " is duplicated.");
            }
            ServiceNameMetaData tmp = new ServiceNameMetaData(this);
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(messageElement);
            if(ifdefMatch){
                message = tmp;
            }
        }
        
        final Element defaultLogElement = getOptionalChild(
            element,
            DefaultLogMetaData.DEFAULT_LOG_TAG_NAME
        );
        if(defaultLogElement != null){
            if(ifdefMatch && defaultLog != null){
                throw new DeploymentException("Element of " + DefaultLogMetaData.DEFAULT_LOG_TAG_NAME + " is duplicated.");
            }
            DefaultLogMetaData tmp = new DefaultLogMetaData(this);
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(defaultLogElement);
            if(ifdefMatch){
                defaultLog = tmp;
            }
        }
        
        final Element editorsElement
             = getOptionalChild(element, PropertyEditorsMetaData.PROPERTY_EDITORS_TAG_NAME);
        if(editorsElement != null){
            if(ifdefMatch && propertyEditors != null){
                throw new DeploymentException("Element of " + PropertyEditorsMetaData.PROPERTY_EDITORS_TAG_NAME + " is duplicated.");
            }
            PropertyEditorsMetaData tmp = null;
            if(ifdefMatch){
                if(propertyEditors == null){
                    tmp = new PropertyEditorsMetaData(this);
                }else{
                    tmp = propertyEditors;
                }
            }else{
                tmp = new PropertyEditorsMetaData(this);
            }
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(editorsElement);
            if(ifdefMatch){
                propertyEditors = tmp;
            }
        }
    }
    
    @Override
    public StringBuilder toXML(StringBuilder buf){
        buf.append("<?xml version=\"1.0\"");
        if(encoding != null){
            buf.append(" encoding=\"");
            buf.append(encoding);
            buf.append("\"");
        }
        buf.append("?>");
        buf.append(LINE_SEPARATOR);
        if(docType != null){
            buf.append(docType);
            buf.append(LINE_SEPARATOR);
            buf.append(LINE_SEPARATOR);
        }
        appendComment(buf);
        buf.append('<').append(NIMBUS_TAG_NAME).append('>');
        if(properties.size() != 0){
            buf.append(LINE_SEPARATOR);
            final StringBuilder subBuf = new StringBuilder();
            Iterator<ServerPropertyMetaData> props = properties.values().iterator();
            while(props.hasNext()){
                ServerPropertyMetaData propData = props.next();
                if(propData.getIfDefMetaData() != null){
                    continue;
                }
                propData.toXML(subBuf);
                if(props.hasNext()){
                    subBuf.append(LINE_SEPARATOR);
                }
            }
            buf.append(addIndent(subBuf));
        }
        if(propertyEditors != null
            && propertyEditors.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            final StringBuilder subBuf = new StringBuilder();
            propertyEditors.toXML(subBuf);
            buf.append(addIndent(subBuf));
        }
        if(defaultLog != null && defaultLog.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            buf.append(
                addIndent(defaultLog.toXML(new StringBuilder()))
            );
        }
        if(repository != null && repository.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            buf.append(
                addIndent(repository.toXML(new StringBuilder()))
            );
        }
        if(log != null && log.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            buf.append(
                addIndent(log.toXML(new StringBuilder()))
            );
        }
        if(message != null && message.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            buf.append(
                addIndent(message.toXML(new StringBuilder()))
            );
        }
        if(managers.size() != 0){
            buf.append(LINE_SEPARATOR);
            Iterator<ManagerMetaData> datas = managers.values().iterator();
            while(datas.hasNext()){
                ManagerMetaData managerData = datas.next();
                if(managerData.getIfDefMetaData() != null){
                    continue;
                }
                buf.append(
                    addIndent(managerData.toXML(new StringBuilder()))
                );
                if(datas.hasNext()){
                    buf.append(LINE_SEPARATOR);
                }
            }
        }
        buf.append(LINE_SEPARATOR);
        if(ifDefMetaDataList != null && ifDefMetaDataList.size() != 0){
            for(int i = 0, imax = ifDefMetaDataList.size(); i < imax; i++){
                IfDefMetaData ifdefData = ifDefMetaDataList.get(i);
                buf.append(
                    addIndent(ifdefData.toXML(new StringBuilder()))
                );
                buf.append(LINE_SEPARATOR);
            }
        }
        buf.append("</").append(NIMBUS_TAG_NAME).append('>');
        return buf;
    }
}
