/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.aop.interceptor;

import java.util.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.aop.*;
import jp.ossc.nimbus.util.SynchronizeMonitor;
import jp.ossc.nimbus.util.WaitSynchronizeMonitor;

/**
 * uCN|CgC^[Zv^B<p>
 * \bȟĂяoɑ΂āACӂ̊ԁAXbh𒆒fC^[Zv^łB<br>
 * ȉɁAuCN|CgC^[Zv^̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="BreakPointInterceptor"
 *                  code="jp.ossc.nimbus.service.aop.interceptor.BreakPointInterceptorService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class BreakPointInterceptorService extends ServiceBase
 implements Interceptor, BreakPointInterceptorServiceMBean{
    
    private static final long serialVersionUID = -2667830848395155759L;
    
    private SynchronizeMonitor monitor = new WaitSynchronizeMonitor();
    private SynchronizeMonitor listenerMonitor = new WaitSynchronizeMonitor();
    private boolean enabled = true;
    private int breakPoint = BREAK_POINT_IN;
    private long timeout;
    private List<String> threads = new ArrayList<String>();
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void setEnabled(boolean enabled){
        this.enabled = enabled;
    }
    // BreakPointInterceptorServiceMBeanJavaDoc
    public boolean isEnabled(){
        return enabled;
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void setMonitor(SynchronizeMonitor monitor){
        this.monitor = monitor;
    }
    // BreakPointInterceptorServiceMBeanJavaDoc
    public SynchronizeMonitor getMonitor(){
        return monitor;
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void setBreakPoint(int breakPoint){
        this.breakPoint = breakPoint;
    }
    // BreakPointInterceptorServiceMBeanJavaDoc
    public int getBreakPoint(){
        return breakPoint;
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void setTimeout(long timeout){
        this.timeout = timeout;
    }
    // BreakPointInterceptorServiceMBeanJavaDoc
    public long getTimeout(){
        return timeout;
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void resume(){
        synchronized(monitor){
            monitor.notifyMonitor();
        }
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void resumeAll(){
        synchronized(monitor){
            monitor.notifyAllMonitor();
        }
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public List<String> suspendThreads(){
        synchronized(threads){
            return new ArrayList<String>(threads);
        }
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void waitSuspend() throws InterruptedException{
        if(getState() != State.STARTED){
            return;
        }
        synchronized(listenerMonitor){
            synchronized(threads){
                if(threads.size() != 0){
                    return;
                }
            }
            listenerMonitor.initMonitor();
            listenerMonitor.waitMonitor();
        }
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public boolean waitSuspend(long timeout) throws InterruptedException{
        if(getState() != State.STARTED){
            return false;
        }
        synchronized(listenerMonitor){
            synchronized(threads){
                if(threads.size() != 0){
                    return true;
                }
            }
            listenerMonitor.initMonitor();
            listenerMonitor.waitMonitor(timeout);
            synchronized(threads){
                if(threads.size() == 0){
                    return false;
                }else{
                    return true;
                }
            }
        }
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public void waitSuspend(String threadName) throws InterruptedException{
        if(getState() != State.STARTED){
            return;
        }
        synchronized(listenerMonitor){
            synchronized(threads){
                if(threads.contains(threadName)){
                    return;
                }
            }
            listenerMonitor.initMonitor();
            listenerMonitor.waitMonitor();
            synchronized(threads){
                if(threads.contains(threadName)){
                    return;
                }
            }
            waitSuspend(threadName);
        }
    }
    
    // BreakPointInterceptorServiceMBeanJavaDoc
    public boolean waitSuspend(String threadName, long timeout) throws InterruptedException{
        if(getState() != State.STARTED){
            return false;
        }
        synchronized(listenerMonitor){
            synchronized(threads){
                if(threads.contains(threadName)){
                    return true;
                }
            }
            final long startTime = System.currentTimeMillis();
            listenerMonitor.initMonitor();
            listenerMonitor.waitMonitor(timeout);
            final long waitTime = System.currentTimeMillis() - startTime;
            synchronized(threads){
                if(threads.contains(threadName)){
                    return true;
                }else if(waitTime >= timeout){
                    return false;
                }
            }
            return waitSuspend(threadName, timeout - waitTime);
        }
    }
    
    public void stopService() throws Exception{
        synchronized(listenerMonitor){
            listenerMonitor.notifyAllMonitor();
        }
        synchronized(monitor){
            monitor.notifyAllMonitor();
        }
    }
    
    /**
     * w肳ꂽuCN|Cg{@link #BREAK_POINT_IN}̏ꍇA{@link #resume()}{@link #resumeAll()}Ăяo܂őҋ@āÃC^[Zv^ĂяoBuCN|Cg{@link #BREAK_POINT_OUT}̏ꍇÃC^[Zv^ĂяoA{@link #resume()}{@link #resumeAll()}Ăяo܂őҋ@B<p>
     * T[rXJnĂȂꍇ́AɎ̃C^[Zv^ĂяoB<br>
     *
     * @param context ĂяõReLXg
     * @param chain ̃C^[Zv^Ăяo߂̃`F[
     * @return Ăяoʂ̖߂l
     * @exception Throwable ĂяoŗOꍇA܂͂̃C^[Zv^ŔCӂ̗OꍇBAA{Ăяo鏈throwȂRuntimeExceptionȊO̗OthrowĂAĂяoɂ͓`dȂB
     */
    public Object invoke(
        InvocationContext context,
        InterceptorChain chain
    ) throws Throwable{
        if(getState() == State.STARTED && enabled && breakPoint == BREAK_POINT_IN){
            breakpoint();
        }
        try{
            return chain.invokeNext(context);
        }finally{
            if(getState() == State.STARTED && enabled && breakPoint == BREAK_POINT_OUT){
                breakpoint();
            }
        }
    }
    
    private void breakpoint(){
        Thread thread = Thread.currentThread();
        try{
            synchronized(listenerMonitor){
                synchronized(threads){
                    threads.add(thread.getName());
                }
                listenerMonitor.notifyAllMonitor();
            }
            synchronized(monitor){
                try{
                    monitor.initMonitor();
                    if(timeout > 0){
                        monitor.waitMonitor(timeout);
                    }else{
                        monitor.waitMonitor();
                    }
                }catch(InterruptedException e){
                }
            }
        }finally{
            synchronized(listenerMonitor){
                synchronized(threads){
                    threads.remove(thread.getName());
                }
            }
        }
    }
}
