/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.io.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.io.Externalizer;

/**
 * L[tt@CLbVQƁB<p>
 *
 * @author M.Takata
 */
public class FileKeyCachedReference<K, V> extends DefaultKeyCachedReference<K, V>
 implements Serializable{
    
    private static final long serialVersionUID = 1386986712271917526L;
    
    private transient Externalizer<Object> externalizer;
    
    /**
     * ̃LbVt@CL[tt@CLbVQƂ𐶐B<p>
     * w肳ꂽLbVt@CǂݍŃLbVL[擾B<br>
     * 
     * @param file LbVt@C
     * @exception IOException LbVt@C̕Ɏsꍇ
     * @exception ClassNotFoundException LbVt@C̕ʂ̃NX݂Ȃꍇ
     */
    @SuppressWarnings("unchecked")
    public FileKeyCachedReference(File file)
     throws IOException, ClassNotFoundException{
        super(null, file);
        cacheKey = ((MapEntry<K, V>)deserializeObject(file)).getKey();
    }
    
    /**
     * ̃LbVt@CL[tt@CLbVQƂ𐶐B<p>
     * w肳ꂽLbVt@CǂݍŃLbVL[擾B<br>
     * 
     * @param file LbVt@C
     * @param ext 񉻂sExternalizer
     * @exception IOException LbVt@C̕Ɏsꍇ
     * @exception ClassNotFoundException LbVt@C̕ʂ̃NX݂Ȃꍇ
     */
    @SuppressWarnings("unchecked")
    public FileKeyCachedReference(File file, Externalizer<Object> ext)
     throws IOException, ClassNotFoundException{
        super(null, file);
        externalizer = ext;
        cacheKey = ((MapEntry<K, V>)deserializeObject(file)).getKey();
    }
    
    /**
     * w肳ꂽLbVL[ALbVt@CALbVIuWFNgێVL[tt@CLbVQƂ𐶐B<p>
     * LbVIuWFNǵA񉻂ăLbVt@CɕۑB<br>
     * 
     * @param key LbVL[
     * @param file LbVt@C
     * @param obj LbVIuWFNg
     * @exception IOException LbVIuWFNg̒񉻂Ɏsꍇ
     */
    public FileKeyCachedReference(K key, File file, V obj)
     throws IOException{
        super(key, file);
        serializeObject(file, new MapEntry<K, V>(key, obj));
    }
    
    /**
     * w肳ꂽLbVL[ALbVt@CALbVIuWFNgێVL[tt@CLbVQƂ𐶐B<p>
     * LbVIuWFNǵA񉻂ăLbVt@CɕۑB<br>
     * 
     * @param key LbVL[
     * @param file LbVt@C
     * @param obj LbVIuWFNg
     * @exception IOException LbVIuWFNg̒񉻂Ɏsꍇ
     */
    public FileKeyCachedReference(K key, File file, V obj, Externalizer<Object> ext)
     throws IOException{
        super(key, file);
        externalizer = ext;
        serializeObject(file, new MapEntry<K, V>(key, obj));
    }
    
    /**
     * IuWFNgwt@CɃVACYĊi[B<p>
     * 
     * @param file LbVt@C
     * @param obj LbVIuWFNg
     * @exception IOException LbVIuWFNg̒񉻂Ɏsꍇ
     */
    protected void serializeObject(File file, Object obj)
     throws IOException{
        OutputStream os = new FileOutputStream(file);
        try{
            if(externalizer == null){
                ObjectOutputStream oos = new ObjectOutputStream(os);
                try{
                    oos.writeObject(obj);
                    oos.flush();
                }finally{
                    try{
                        oos.close();
                    }catch(IOException e){}
                }
            }else{
                externalizer.writeExternal(obj, os);
            }
        }finally{
            if(os!=null){
                try{
                    os.close();
                }catch(IOException e){}
            }
        }
    }
    
    /**
     * w肳ꂽt@CIuWFNg𕜌B<p>
     * 
     * @param file LbVt@C
     * @return LbVIuWFNg
     * @exception IOException LbVt@C̕Ɏsꍇ
     * @exception ClassNotFoundException LbVt@C̕ʂ̃NX ݂Ȃꍇ
     */
    protected Object deserializeObject(File file)
     throws IOException, ClassNotFoundException{
        Object entry = null;
        InputStream is = new FileInputStream(file);
        try{
            if(externalizer == null){
                ObjectInputStream ois = new ObjectInputStream(is);
                try{
                    entry = ois.readObject();
                }finally{
                    try{
                        ois.close();
                    }catch(IOException e){}
                }
            }else{
                entry = externalizer.readExternal(is);
            }
        }finally{
            if(is!=null){
                try{
                    is.close() ;
                }catch(IOException ex){
                }
            }
        }
        return entry;
    }
    
    /**
     * LbVt@C擾B<p>
     *
     * @param source 擾̎Q
     * @return LbVt@C
     */
    public File getFile(Object source){
        return (File)super.get(source, false);
    }
    
    /**
     * LbVꂽIuWFNgLbVt@C畜Ď擾B<p>
     * truȅꍇ́A{@link #addCacheAccessListener(CacheAccessListener)}œo^ꂽ{@link CacheAccessListener}ɒʒmBAAœnꂽĂяoIuWFNgʒmCacheAccessListener̃CX^XƓꍇ́AʒmȂB<br>
     * gێLbVIuWFNgnullłȂꍇ́AԂBnull̏ꍇ́A{@link #addLinkedReference(LinkedReference)}œo^ꂽ{@link LinkedReference}擾݂B<br>
     *
     * @param source LbV擾邱̃\bȟĂяoIuWFNg
     * @param notify LbVANZXXiɒʒmꍇtrue
     * @return LbVIuWFNg
     */
    @SuppressWarnings("unchecked")
    @Override
    public V get(Object source, boolean notify){
        final V obj = super.get(source, notify);
        if(obj instanceof File){
            File file = (File)obj;
            if(file.exists() && file.canRead()){
                MapEntry<K, V> entry = null;
                try{
                    entry = ((MapEntry<K, V>)deserializeObject(file));
                }catch(IOException e){
                    return null;
                }catch(ClassNotFoundException e){
                    return null;
                }
                return (V)entry.getValue();
            }else{
                return null;
            }
        }else{
            return obj;
        }
    }
    
    /**
     * LbVIuWFNg𒼗񉻂ăt@CɕۑB<p>
     * {@link #addCacheChangeListener(CacheChangeListener)}œo^ꂽ{@link CacheChangeListener}ɒʒmBAAœnꂽĂяoIuWFNgʒmCacheChangeListener̃CX^XƓꍇ́AʒmȂB<br>
     *
     * @param source LbVIuWFNgύX邱̃\bȟĂяoIuWFNg
     * @param obj ݒ肷LbVIuWFNg
     * @exception IllegalCachedReferenceException LbVQƂ̏ԂsȈ׃LbVIuWFNg̐ݒɎsꍇ
     */
    @Override
    public void set(Object source, V obj)
     throws IllegalCachedReferenceException{
        notifyChange(source, obj);
        if(obj != null){
            try{
                serializeObject((File)cacheObj, obj);
            }catch(IOException e){
                throw new IllegalCachedReferenceException(e);
            }
        }else{
            ((File)cacheObj).delete();
        }
    }
    
    /**
     * LbVIuWFNgۑLbVt@C폜B<p>
     * {@link #addCacheRemoveListener(CacheRemoveListener)}œo^ꂽ{@link CacheRemoveListener}ɒʒmBAAœnꂽĂяoIuWFNgʒmCacheChangeListener̃CX^XƓꍇ́AʒmȂB<br>
     *
     * @param source LbVIuWFNg폜邱̃\bȟĂяoIuWFNg
     */
    @Override
    public void remove(Object source){
        if(cacheObj != null){
            notifyRemoved(source);
            File file = (File)cacheObj;
            if(file.exists() && file.canRead()){
                ((File)cacheObj).delete();
            }
            cacheObj = null;
            if(linkedReferences != null){
                linkedReferences.clear();
            }
        }
    }
    
    /**
     * LbVL[ƃLbVIuWFNgێ钼񉻗pNXB<p>
     *
     * @author M.Takata
     */
    public static class MapEntry<K, V> implements Serializable{
        
        private static final long serialVersionUID = 4635469653838112700L;
        
        /**
         * LbVL[B<p>
         */
        protected K key;
        
        /**
         * LbVIuWFNgB<p>
         */
        protected V value;
        
        /**
         * LbVL[ƃLbVIuWFNgێLbVGg𐶐B<p>
         *
         * @param k LbVL[
         * @param val LbVIuWFNg
         */
        public MapEntry(K k, V val){
            key = k;
            value = val;
        }
        
        /**
         * LbVL[擾B<p>
         * 
         * @return LbVL[
         */
        public K getKey(){
            return key;
        }
        
        /**
         * LbVIuWFNg擾B<p>
         * 
         * @return LbVIuWFNg
         */
        public V getValue(){
            return value;
        }
    }
    
    private void writeObject(ObjectOutputStream out) throws IOException{
        out.defaultWriteObject();
        writeSet(out, linkedReferences);
        writeSet(out, removeListeners);
        writeSet(out, accessListeners);
        writeSet(out, changeListeners);
        ServiceName name = null;
        if(externalizer != null){
            name = getServiceName(externalizer);
        }
        if(name != null){
            out.writeObject(name);
        }else{
            out.writeObject(externalizer);
        }
    }
    
    @SuppressWarnings("unchecked")
    private void readObject(ObjectInputStream in)
     throws IOException, ClassNotFoundException{
        in.defaultReadObject();
        linkedReferences = readSet(in);
        removeListeners = readSet(in);
        accessListeners = readSet(in);
        changeListeners = readSet(in);
        Object obj = in.readObject();
        if(obj != null){
            if(obj instanceof ServiceName){
                externalizer = ServiceManagerFactory.getServiceObject(
                    (ServiceName)obj
                );
            }else{
                externalizer = (Externalizer<Object>)obj;
            }
        }
    }
}