/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.scheduler;

import java.util.*;
import javax.naming.*;
import javax.transaction.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.service.context.Context;
import jp.ossc.nimbus.service.sequence.Sequence;
import jp.ossc.nimbus.service.aop.interceptor.ThreadContextKey;
import jp.ossc.nimbus.service.transaction.TransactionManagerFactory;
import jp.ossc.nimbus.service.distribute.ClusterService;

/**
 * ۃXPW[B<p>
 * XPW[ꂽ^XNsӔC𕉂B<br>
 * {@link ScheduleManager}sׂ{@link Schedule}擾āA{@link ScheduleExecutor}Ɏs˗B<br>
 *
 * @author M.Takata
 */
public abstract class AbstractSchedulerService extends ServiceBase
 implements AbstractSchedulerServiceMBean, Scheduler{
    
    private static final long serialVersionUID = 6938915052580428501L;
    
    /**
     * TransactionManagerJNDIB<p>
     * J2EE̎dlŁA\񂳂ĂJNDIłB
     */
    protected static final String TRANSACTION_MANAGER_JNDI_NAME
         = "java:/TransactionManager";
    
    protected long scheduleTickerInterval = 1000l;
    
    protected ServiceName scheduleManagerServiceName;
    protected ScheduleManager scheduleManager;
    
    protected ServiceName[] scheduleExecutorServiceNames;
    protected Map<String, ScheduleExecutor> scheduleExecutors;
    
    protected Daemon scheduleTicker;
    
    protected boolean isTransactionControl;
    protected TransactionManager transactionManager;
    protected ServiceName transactionManagerFactoryServiceName;
    
    protected String executorKey;
    
    protected ServiceName threadContextServiceName;
    protected Context<Object, Object> threadContext;
    
    protected ServiceName sequenceServiceName;
    protected Sequence sequence;
    
    protected ServiceName clusterServiceName;
    protected ClusterService cluster;
    protected ClusterListener clusterListener;
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setScheduleTickerInterval(long interval){
        scheduleTickerInterval = interval;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public long getScheduleTickerInterval(){
        return scheduleTickerInterval;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setScheduleManagerServiceName(ServiceName name){
        scheduleManagerServiceName = name;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public ServiceName getScheduleManagerServiceName(){
        return scheduleManagerServiceName;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setScheduleExecutorServiceName(ServiceName name){
        scheduleExecutorServiceNames = name == null ? null : new ServiceName[]{name};
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public ServiceName getScheduleExecutorServiceName(){
        return scheduleExecutorServiceNames == null || scheduleExecutorServiceNames.length != 1 ? null : scheduleExecutorServiceNames[0];
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setScheduleExecutorServiceNames(ServiceName[] names){
        scheduleExecutorServiceNames = names;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public ServiceName[] getScheduleExecutorServiceNames(){
        return scheduleExecutorServiceNames;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setThreadContextServiceName(ServiceName name){
        threadContextServiceName = name;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public ServiceName getThreadContextServiceName(){
        return threadContextServiceName;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setSequenceServiceName(ServiceName name){
        sequenceServiceName = name;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public ServiceName getSequenceServiceName(){
        return sequenceServiceName;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setTransactionControl(boolean isControl){
        isTransactionControl = isControl;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public boolean isTransactionControl(){
        return isTransactionControl;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setExecutorKey(String key){
        executorKey = key;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public String getExecutorKey(){
        return executorKey;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setTransactionManagerFactoryServiceName(ServiceName name){
        transactionManagerFactoryServiceName = name;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public ServiceName getTransactionManagerFactoryServiceName(){
        return transactionManagerFactoryServiceName;
    }
    
    // AbstractSchedulerServiceMBeanJavaDoc
    public void setClusterServiceName(ServiceName name){
        clusterServiceName = name;
    }
    // AbstractSchedulerServiceMBeanJavaDoc
    public ServiceName getClusterServiceName(){
        return clusterServiceName;
    }
    
    /**
     * T[rX̐OsB<p>
     *
     * @exception Exception T[rX̐OɎsꍇ
     */
    public void preCreateService() throws Exception{
        super.preCreateService();
    }
    
    /**
     * T[rX̊JnOsB<p>
     *
     * @exception Exception T[rX̊JnOɎsꍇ
     */
    public void preStartService() throws Exception{
        super.preStartService();
        
        if(isTransactionControl){
            if(transactionManagerFactoryServiceName == null){
                final InitialContext context = new InitialContext();
                transactionManager = (TransactionManager)context.lookup(
                    TRANSACTION_MANAGER_JNDI_NAME
                );
            }else{
                TransactionManagerFactory transactionManagerFactory = ServiceManagerFactory
                    .getServiceObject(transactionManagerFactoryServiceName);
                transactionManager = transactionManagerFactory.getTransactionManager();
            }
        }
        
        if(scheduleExecutorServiceNames != null){
            scheduleExecutors = new HashMap<String, ScheduleExecutor>();
            for(int i = 0; i < scheduleExecutorServiceNames.length; i++){
                ScheduleExecutor executor = ServiceManagerFactory
                    .getServiceObject(scheduleExecutorServiceNames[i]);
                scheduleExecutors.put(executor.getType(), executor);
            }
        }
        if(scheduleExecutors == null || scheduleExecutors.size() == 0){
            throw new IllegalArgumentException("ScheduleExecutor is null.");
        }
        
        if(scheduleManagerServiceName != null){
            scheduleManager = (ScheduleManager)ServiceManagerFactory
                .getServiceObject(scheduleManagerServiceName);
        }
        if(scheduleManager == null){
            throw new IllegalArgumentException("ScheduleManager is null.");
        }
        
        if(threadContextServiceName != null){
            threadContext = ServiceManagerFactory
                .getServiceObject(threadContextServiceName);
        }
        
        if(sequenceServiceName != null){
            sequence = (Sequence)ServiceManagerFactory
                .getServiceObject(sequenceServiceName);
        }
    }
    
    /**
     * T[rX̊Jn㏈sB<p>
     *
     * @exception Exception T[rX̊Jn㏈Ɏsꍇ
     */
    public void postStartService() throws Exception{
        scheduleManager.addScheduleControlListener(this);
        
        scheduleTicker = new Daemon(new ScheduleTicker());
        scheduleTicker.setName(getServiceNameObject() + " ScheduleTicker");
        scheduleTicker.suspend();
        scheduleTicker.start();
        
        if(clusterServiceName != null){
            cluster = ServiceManagerFactory.getServiceObject(clusterServiceName);
            if(cluster.isJoin()){
                throw new IllegalArgumentException("ClusterService already join.");
            }
            clusterListener = new ClusterListener();
            cluster.addClusterListener(clusterListener);
            cluster.join();
        }else{
            scheduleTicker.resume();
        }
        
        super.postStartService();
    }
    
    /**
     * T[rX̒~OsB<p>
     *
     * @exception Exception T[rX̒~OɎsꍇ
     */
    public void preStopService() throws Exception{
        
        if(scheduleTicker != null){
            scheduleTicker.stop();
        }
        
        scheduleManager.removeScheduleControlListener(this);
        
        if(cluster != null){
            cluster.removeClusterListener(clusterListener);
            clusterListener = null;
            cluster.leave();
            cluster = null;
        }
        
        super.preStopService();
    }
    
    /**
     * T[rX̒~㏈sB<p>
     *
     * @exception Exception T[rX̒~㏈Ɏsꍇ
     */
    public void postStopService() throws Exception{
        
        super.postStopService();
    }
    
    /**
     * T[rX̔j㏈sB<p>
     *
     * @exception Exception T[rX̔j㏈Ɏsꍇ
     */
    public void postDestroyService() throws Exception{
        
        scheduleTicker = null;
        
        super.postDestroyService();
    }
    
    /**
     * NGXgʔԂݒ肷{@link Context}T[rXݒ肷B<p>
     *
     * @param context ContextT[rX
     */
    public void setThreadContext(Context<Object, Object> context){
        threadContext = context;
    }
    
    /**
     * NGXgʔԂݒ肷{@link Context}T[rX擾B<p>
     *
     * @return ContextT[rX
     */
    public Context<Object, Object> getThreadContext(){
        return threadContext;
    }
    
    /**
     * NGXgʔԂ𔭍s{@link Sequence}T[rXݒ肷B<p>
     *
     * @param seq SequenceT[rX
     */
    public void setSequence(Sequence seq){
        sequence = seq;
    }
    
    /**
     * NGXgʔԂ𔭍s{@link Sequence}T[rX擾B<p>
     *
     * @return SequenceT[rX
     */
    public Sequence getSequence(){
        return sequence;
    }
    
    // SchedulerJavaDoc
    public ScheduleManager getScheduleManager(){
        return scheduleManager;
    }
    
    // SchedulerJavaDoc
    public void setScheduleManager(ScheduleManager manager){
        scheduleManager = manager;
    }
    
    // SchedulerJavaDoc
    public ScheduleExecutor getScheduleExecutor(String type){
        if(scheduleExecutors.size() == 1){
            ScheduleExecutor executor = scheduleExecutors.values().iterator().next();
            return type == null || type.equals(executor.getType()) ? executor : null;
        }else{
            return scheduleExecutors.get(type);
        }
    }
    
    // SchedulerJavaDoc
    public void setScheduleExecutor(ScheduleExecutor executor){
        scheduleExecutors.put(executor.getType(), executor);
    }
    
    // SchedulerJavaDoc
    public Map<String, ScheduleExecutor> getScheduleExecutors(){
        return scheduleExecutors;
    }
    
    // SchedulerJavaDoc
    public void startEntry(){
        if(scheduleTicker != null){
            scheduleTicker.resume();
        }
    }
    
    // SchedulerJavaDoc
    public boolean isStartEntry(){
        return scheduleTicker == null ? false : scheduleTicker.isRunning() && !scheduleTicker.isSusupend();
    }
    
    // SchedulerJavaDoc
    public void stopEntry(){
        if(scheduleTicker != null){
            scheduleTicker.suspend();
        }
    }
    
    /**
     * XPW[ԂύXꂽɒʒmB<p>
     * s̃XPW[̐Ԃ𐧌䂷B<br>
     *
     * @param id XPW[ID
     * @param state ύXꂽ
     * @exception ScheduleStateControlException sXPW[̐Ԃ̕ύXɎsꍇ
     */
    public void changedControlState(String id, int state)
     throws ScheduleStateControlException{
        ScheduleExecutor[] executors = scheduleExecutors.values().toArray(
            new ScheduleExecutor[scheduleExecutors.size()]
        );
        for(int i = 0; i < executors.length; i++){
            if(executors[i].controlState(id, state)){
                break;
            }
        }
    }
    
    /**
     * XPW[𓊓L[JTAT|[g邩ǂ𔻒肷B<p>
     *
     * @return JTAT|[gꍇ́Atrue
     */
    protected abstract boolean isTransactableQueue();
    
    /**
     * XPW[L[ɓB<p>
     *
     * @param request XPW[NGXg
     */
    protected abstract void entrySchedule(ScheduleRequest request)
     throws Throwable;
    
    /**
     * XPW[{@link ScheduleExecutor}Ɏs˗B<p>
     * ܂A{@link ScheduleManager}gāAXPW[̏ԂύXB<br>
     * 
     * @param request L[oXPW[NGXg
     */
    protected void dispatchSchedule(ScheduleRequest request){
        if(threadContext != null){
            threadContext.clear();
        }
        if(threadContext != null && request.getRequestId() != null){
            threadContext.put(
                ThreadContextKey.REQUEST_ID,
                request.getRequestId()
            );
        }
        Schedule schedule = request.getSchedule();
        ScheduleExecutor scheduleExecutor = getScheduleExecutor(schedule.getExecutorType());
        if(scheduleExecutor == null){
            getLogger().write(
                MSG_ID_NOT_FOUND_EXECUTOR_ERROR,
                schedule.getId(),
                schedule.getTaskName()
            );
            try{
                scheduleManager.changeState(
                    schedule.getId(),
                    Schedule.STATE_FAILED
                );
            }catch(ScheduleStateControlException e){
                getLogger().write(
                    MSG_ID_STATE_CHANGE_ERROR,
                    e,
                    schedule.getId(),
                    schedule.getTaskName(),
                    new Integer(Schedule.STATE_FAILED)
                );
            }
            return;
        }
        schedule.setRetry(false);
        schedule.setOutput(null);
        try{
            schedule = scheduleExecutor.execute(schedule);
        }catch(Throwable th){
            getLogger().write(
                MSG_ID_EXECUTE_ERROR,
                th,
                schedule.getId(),
                schedule.getTaskName()
            );
            try{
                scheduleManager.changeState(
                    schedule.getId(),
                    Schedule.STATE_FAILED
                );
            }catch(ScheduleStateControlException e){
                getLogger().write(
                    MSG_ID_STATE_CHANGE_ERROR,
                    e,
                    schedule.getId(),
                    schedule.getTaskName(),
                    new Integer(Schedule.STATE_FAILED)
                );
            }
            return;
        }
    }
    
    protected class ClusterListener implements jp.ossc.nimbus.service.distribute.ClusterListener{
        
        public void memberInit(Object myId, List<? extends Object> members){}
        
        public void memberChange(List<? extends Object> oldMembers, List<? extends Object> newMembers){}
        
        public void changeMain() throws Exception{
            startEntry();
        }
        
        public void changeSub(){
            stopEntry();
        }
    }
    
    /**
     * XPW[eBbJ[B<p>
     * XPW[IɎ擾āAsL[ɓB<br>
     *
     * @author M.Takata
     */
    protected class ScheduleTicker implements DaemonRunnable<Object>{
        
        /**
         * f[JnɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStart() {
            return true;
        }
        
        /**
         * f[~ɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStop() {
            return true;
        }
        
        /**
         * f[fɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onSuspend() {
            return true;
        }
        
        /**
         * f[ĊJɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onResume() {
            return true;
        }
        
        /**
         * 莞ԋ󂯂B<p>
         * 
         * @param ctrl DaemonControlIuWFNg
         * @return XPW[̔z
         */
        public Object provide(DaemonControl ctrl) throws Throwable{
            Thread.sleep(getScheduleTickerInterval());
            return null;
        }
        
        /**
         * dequeuedœnꂽIuWFNgQueueHandlerĂяoB<p>
         *
         * @param schedules L[oꂽIuWFNg
         * @param ctrl DaemonControlIuWFNg
         */
        public void consume(Object schedules, DaemonControl ctrl)
         throws Throwable{
            boolean rollbackMark = false;
            List<Schedule> scheduleList = null;
            List<ScheduleRequest> scheduleRequests = null;
            try{
                if(isTransactionControl){
                    transactionManager.begin();
                }
                try{
                    final String[] executorTypes = scheduleExecutors.keySet().toArray(
                        new String[scheduleExecutors.size()]
                    );
                    if(executorKey == null){
                        scheduleList = scheduleManager.findExecutableSchedules(
                            new Date(),
                            executorTypes
                        );
                    }else{
                        scheduleList = scheduleManager.findExecutableSchedules(
                            new Date(),
                            executorTypes,
                            executorKey
                        );
                    }
                }catch(ScheduleManageException e){
                    getLogger().write(MSG_ID_SCHEDULE_GET_ERROR, e);
                    rollbackMark = true;
                    return;
                }
                if(scheduleList == null || scheduleList.size() == 0){
                    return;
                }
                scheduleRequests = new ArrayList<ScheduleRequest>();
                final Iterator<Schedule> itr = scheduleList.iterator();
                while(itr.hasNext()){
                    Schedule schedule = (Schedule)itr.next();
                    final ScheduleRequest request
                        = new ScheduleRequest(schedule);
                    if(sequence != null){
                        request.setRequestId(sequence.increment());
                        if(threadContext != null){
                            threadContext.put(
                                ThreadContextKey.REQUEST_ID,
                                request.getRequestId()
                            );
                        }
                    }
                    scheduleRequests.add(request);
                    try{
                        final int nowState = scheduleManager.getState(schedule.getId());
                        switch(nowState){
                        case Schedule.STATE_INITIAL:
                        case Schedule.STATE_RETRY:
                            break;
                        default:
                            getLogger().write(
                                MSG_ID_STATE_TRANS_ERROR,
                                schedule.getId(),
                                schedule.getTaskName(),
                                new Integer(nowState),
                                new Integer(Schedule.STATE_ENTRY)
                            );
                            return;
                        }
                        final boolean isChanged = scheduleManager.changeState(
                            schedule.getId(),
                            nowState,
                            Schedule.STATE_ENTRY
                        );
                        if(!isChanged){
                            itr.remove();
                            getLogger().write(
                                MSG_ID_STATE_TRANS_ERROR,
                                schedule.getId(),
                                schedule.getTaskName(),
                                new Integer(nowState),
                                new Integer(Schedule.STATE_ENTRY)
                            );
                            continue;
                        }
                    }catch(ScheduleStateControlException e){
                        getLogger().write(
                            MSG_ID_STATE_CHANGE_ERROR,
                            e,
                            schedule.getId(),
                            schedule.getTaskName(),
                            new Integer(Schedule.STATE_ENTRY)
                        );
                        rollbackMark = true;
                        break;
                    }
                    if(!isTransactionControl || isTransactableQueue()){
                        getLogger().write(
                            MSG_ID_ENTRY,
                            schedule.getId(),
                            schedule.getTaskName(),
                            schedule.getInput()
                        );
                        try{
                            entrySchedule(request);
                        }catch(Throwable th){
                            getLogger().write(
                                MSG_ID_ENTRY_ERROR,
                                th,
                                schedule.getId(),
                                schedule.getTaskName(),
                                schedule.getInput()
                            );
                            rollbackMark = true;
                            break;
                        }
                    }
                }
            }catch(Throwable th){
                getLogger().write(MSG_ID_UNEXPEXTED_ERROR, th);
                if(isTransactionControl){
                    transactionManager.rollback();
                }
                throw th;
            }finally{
                if(isTransactionControl){
                    if(rollbackMark){
                        transactionManager.rollback();
                    }else{
                        transactionManager.commit();
                    }
                }
            }
            if(scheduleRequests != null && scheduleRequests.size() != 0
                 && isTransactionControl && !isTransactableQueue()
                 && !rollbackMark){
                for(int i = 0, imax = scheduleRequests.size(); i < imax; i++){
                    final ScheduleRequest request
                        = (ScheduleRequest)scheduleRequests.get(i);
                    if(threadContext != null && request.getRequestId() != null){
                        threadContext.put(
                            ThreadContextKey.REQUEST_ID,
                            request.getRequestId()
                        );
                    }
                    final Schedule schedule = request.getSchedule();
                    getLogger().write(
                        MSG_ID_ENTRY,
                        schedule.getId(),
                        schedule.getTaskName(),
                        schedule.getInput()
                    );
                    try{
                        entrySchedule(request);
                    }catch(Throwable th){
                        getLogger().write(
                            MSG_ID_ENTRY_ERROR,
                            th,
                            schedule.getId(),
                            schedule.getTaskName(),
                            schedule.getInput()
                        );
                    }
                }
            }
        }
        
        /**
         * ȂB<p>
         */
        public void garbage(){
        }
    }
    
    /**
     * XPW[NGXgB<p>
     *
     * @author M.Takata
     */
    protected static class ScheduleRequest implements java.io.Serializable{
        
        private static final long serialVersionUID = 8405850740460011444L;
        protected Schedule schedule;
        protected String requestId;
        
        public ScheduleRequest(Schedule schedule){
            this.schedule = schedule;
        }
        
        public Schedule getSchedule(){
            return schedule;
        }
        public void setSchedule(Schedule schedule){
            this.schedule = schedule;
        }
        
        public String getRequestId(){
            return requestId;
        }
        public void setRequestId(String id){
            requestId = id;
        }
    }
}