/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.scheduler;

import java.util.*;

import jp.ossc.nimbus.core.*;


/**
 * ftHgXPW[쐬T[rXB<p>
 *
 * @author M.Takata
 */
public class DefaultScheduleMakerService extends ServiceBase
 implements ScheduleMaker, DefaultScheduleMakerServiceMBean{
    
    private static final long serialVersionUID = -8603198587673383956L;
    
    /**
     * w肳ꂽXPW[}X^XPW[쐬B<p>
     * XPW[}X^̐`FbNAXPW[}X^Ɏw肳ꂽtKpAXPW[쐬B<br>
     * XPW[̍쐬ł́AJԂ̃XPW[}X^̏ꍇ́Aw肳ꂽJԂԊuŃXPW[炵XPW[Aw肳ꂽI܂ō쐬B<br>
     * JԂ̃XPW[}X^łȂꍇ́APXPW[쐬B<br>
     * {@link Schedule}̎NXƂ{@link DefaultSchedule}gpB<br>
     *
     * @param date 쐬
     * @param master XPW[}X^
     * @return XPW[̔z
     * @exception ScheduleMakeException XPW[̍쐬Ɏsꍇ
     */
    public Schedule[] makeSchedule(Date date, ScheduleMaster master)
     throws ScheduleMakeException{
        
        if(!master.isEnabled()){
            return new Schedule[0];
        }
        
        if(!isNecessaryMake(date, master)){
            return new Schedule[0];
        }
        
        master.applyDate(date);
        
        checkScheduleMaster(master);
        
        if(master.getEndTime() == null
             || master.getStartTime().equals(master.getEndTime())
             || (master.getRepeatInterval() <= 0
                  && master.getRetryInterval() > 0)
        ){
            return new Schedule[]{
                makeSingleSchedule(master)
            };
        }else{
            return makeRepeatSchedule(master);
        }
    }
    
    public boolean isMakeSchedule(Date date, ScheduleMaster master)
     throws ScheduleMakeException{
        return isNecessaryMake(date, master);
    }
    
    /**
     * XPW[}X^̐`FbNB<p>
     * <ul>
     *   <li>}X^ID̕K{`FbNB</li>
     *   <li>^XN̕K{`FbNB</li>
     *   <li>Jn̕K{`FbNB</li>
     *   <li>Iw肳ĂꍇɁAJn&lt;=IƂȂĂ邩̃`FbNB</li>
     *   <li>Iw肳ĂāAJn&lt;IƂȂĂꍇɁAJԂԊu&gt;0ƂȂĂ邩̃`FbNB</li>
     *   <li>gCIw肳ĂꍇɁAJn&lt;=gCIƂȂĂ邩̃`FbNB</li>
     *   <li>gCԊu&gt;0̏ꍇɁAgCIw肳Ă邩̃`FbNB</li>
     *   <li>gCIw肳ĂꍇɁAgCԊu&lt;=0ł邩̃`FbNB</li>
     *   <li>gCIw肳ĂăgCԊu&gt;0̏ꍇɁAŏ̃gC&lt;=gCIƂȂĂ邩̃`FbNB</li>
     * </ul>
     *
     * @param master XPW[}X^
     * @exception IllegalScheduleMasterException XPW[}X^Ȃꍇ
     */
    protected void checkScheduleMaster(ScheduleMaster master)
     throws IllegalScheduleMasterException{
        
        if(master.getId() == null){
            throw new IllegalScheduleMasterException("Id is null.");
        }
        if(master.getTaskName() == null){
            throw new IllegalScheduleMasterException("TaskName is null. id=" + master.getId());
        }
        if(master.getStartTime() == null){
            throw new IllegalScheduleMasterException("StartTime is null. id=" + master.getId());
        }
        if(master.getEndTime() != null
            && master.getEndTime().before(master.getStartTime())){
            throw new IllegalScheduleMasterException("EndTime is before StartTime. id=" + master.getId());
        }
        if((master.getEndTime() != null
            && master.getEndTime().after(master.getStartTime()))
            && master.getRepeatInterval() <= 0){
            throw new IllegalScheduleMasterException("RepeatInterval <= 0. id=" + master.getId());
        }
        if(master.getRetryEndTime() != null
            && !master.getRetryEndTime().after(master.getStartTime())){
            throw new IllegalScheduleMasterException("RetryEndTime is before StartTime. id=" + master.getId());
        }
        if(master.getRetryInterval() <= 0 && master.getRetryEndTime() != null){
            throw new IllegalScheduleMasterException("RetryInterval <= 0. id=" + master.getId());
        }
        if(master.getRetryInterval() > 0 && master.getRetryEndTime() != null){
            final Calendar offset = Calendar.getInstance();
            offset.setTime(master.getStartTime());
            final Calendar end = Calendar.getInstance();
            end.setTime(master.getRetryEndTime());
            final Date firstRetryTime = calculateNextDate(
                offset,
                master.getRetryInterval(),
                end
            );
            if(firstRetryTime == null){
                throw new IllegalScheduleMasterException("First RetryTime is after RetryEndTime. id=" + master.getId());
            }
        }
    }
    
    /**
     * ̓tŁAXPW[쐬Kv邩ǂ𔻒肷B<p>
     * ̎ł́AKtrueԂB<br>
     * tɑ΂āAXPW[̍쐬L𔻒fKvꍇ́ATuNXŃI[o[Ch鎖B<br>
     *
     * @param date 쐬
     * @param master XPW[}X^
     * @return truȅꍇAKv
     * @exception ScheduleMakeException Ɏsꍇ
     */
    protected boolean isNecessaryMake(Date date, ScheduleMaster master)
     throws ScheduleMakeException{
        return true;
    }
    
    /**
     * w肳ꂽXPW[}X^AJԂȂXPW[쐬B<p>
     *
     * @param master XPW[}X^
     * @return XPW[
     */
    protected Schedule makeSingleSchedule(ScheduleMaster master)
     throws ScheduleMakeException{
        return new DefaultSchedule(
            master.getId(),
            master.getStartTime(),
            master.getTaskName(),
            master.getInput(),
            master.getDepends(),
            master.getExecutorKey(),
            master.getExecutorType(),
            master.getRetryInterval(),
            master.getRetryEndTime(),
            master.getMaxDelayTime()
        );
    }
    
    /**
     * w肳ꂽXPW[}X^AJԂXPW[쐬B<p>
     *
     * @param master XPW[}X^
     * @return XPW[z
     */
    protected Schedule[] makeRepeatSchedule(ScheduleMaster master)
     throws ScheduleMakeException{
        final List<Schedule> result = new ArrayList<Schedule>();
        Date time = master.getStartTime();
        final Calendar offset = Calendar.getInstance();
        offset.setTime(master.getStartTime());
        final Calendar end = Calendar.getInstance();
        end.setTime(master.getEndTime());
        do{
            result.add(
                new DefaultSchedule(
                    master.getId(),
                    time,
                    master.getTaskName(),
                    master.getInput(),
                    master.getDepends(),
                    master.getExecutorKey(),
                    master.getExecutorType(),
                    master.getRetryInterval(),
                    master.getRetryEndTime(),
                    master.getMaxDelayTime()
                )
            );
        }while((time = calculateNextDate(offset, master.getRepeatInterval(), end)) != null);
        return (Schedule[])result.toArray(new Schedule[result.size()]);
    }
    
    /**
     * JԂXPW[̎̃XPW[vZB<p>
     * ̃XPW[I߂ꍇɂ́AnullԂB<br>
     *
     * @param offset ݂̃XPW[
     * @param interval JԂԊu[ms]
     * @param end XPW[̏I
     * @return ̃XPW[
     */
    protected Date calculateNextDate(
        Calendar offset,
        long interval,
        Calendar end
    ){
        if(interval > Integer.MAX_VALUE){
            long offsetInterval = interval;
            int tmpInterval = 0;
            do{
                if(offsetInterval >= Integer.MAX_VALUE){
                    tmpInterval = Integer.MAX_VALUE;
                }else{
                    tmpInterval = (int)offsetInterval;
                }
                offset.add(Calendar.MILLISECOND, tmpInterval);
                offsetInterval -= Integer.MAX_VALUE;
            }while(offsetInterval > 0);
        }else{
            offset.add(Calendar.MILLISECOND, (int)interval);
        }
        if(offset.after(end)){
            return null;
        }else{
            return offset.getTime();
        }
    }
}