/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.aop.interceptor.servlet;

import java.util.*;
import java.io.*;
import java.lang.reflect.InvocationTargetException;

import javax.servlet.*;

import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.util.*;
import jp.ossc.nimbus.service.aop.*;

import org.apache.commons.jexl2.Expression;
import org.apache.commons.jexl2.JexlEngine;
import org.apache.commons.jexl2.JexlContext;
import org.apache.commons.jexl2.MapContext;

/**
 * OC^[Zv^B<p>
 *
 * @author M.Takata
 */
public class ExceptionHandlingInterceptorService
 extends ServletFilterInterceptorService
 implements ExceptionHandlingInterceptorServiceMBean{
    
    private static final long serialVersionUID = -5888057404214069278L;
    
    private static final String JMS_EXCEPTION_NAME = "javax.jms.JMSException";
    private static final String GET_LINKED_EXCEPTION_METHOD = "getLinkedException";
    
    private static final String EHI__00001 = "EHI__00001";
    
    protected Map<String, String> exceptionAndHandlerMapping;
    protected ClassMappingTree<Object> exceptionMapForHandler;
    protected ServiceName defaultExceptionHandlerServiceName;
    protected ExceptionHandler defaultExceptionHandler;
    
    // ExceptionHandlingInterceptorServiceMBeanJavaDoc
    public void setExceptionAndHandlerMapping(Map<String, String> map){
        exceptionAndHandlerMapping = map;
    }
    
    // ExceptionHandlingInterceptorServiceMBeanJavaDoc
    public Map<String, String> getExceptionAndHandlerMapping(){
        return exceptionAndHandlerMapping;
    }
    
    // ExceptionHandlingInterceptorServiceMBeanJavaDoc
    public void setDefaultExceptionHandlerServiceName(ServiceName name){
        defaultExceptionHandlerServiceName = name;
    }
    
    // ExceptionHandlingInterceptorServiceMBeanJavaDoc
    public ServiceName getDefaultExceptionHandlerServiceName(){
        return defaultExceptionHandlerServiceName;
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    public void startService() throws Exception{
        if(exceptionAndHandlerMapping != null){
            exceptionMapForHandler = new ClassMappingTree<Object>();
            final ClassLoader loader = NimbusClassLoader.getInstance();
            final ServiceNameEditor editor = new ServiceNameEditor();
            editor.setServiceManagerName(getServiceManagerName());
            for(Map.Entry<String, String> entry : exceptionAndHandlerMapping.entrySet()){
                String exName = entry.getKey();
                final String name = entry.getValue();
                String condition = null;
                final int index = exName.indexOf('(');
                if(index != -1 && exName.charAt(exName.length() - 1) == ')'){
                    condition = exName.substring(
                        index + 1,
                        exName.length() - 1
                    );
                    exName = exName.substring(0, index);
                }
                final Class<?> clazz = Class.forName(exName, true, loader);
                editor.setAsText(name);
                final ServiceName serviceName = (ServiceName)editor.getValue();
                final ExceptionHandler handler = ServiceManagerFactory
                        .getServiceObject(serviceName);
                if(condition == null){
                    exceptionMapForHandler.add(
                        clazz,
                        handler
                    );
                }else{
                    Condition cond = new Condition(condition, handler);
                    try{
                        cond.evaluate(
                            (Throwable)clazz.newInstance(),
                            null,
                            null,
                            true
                        );
                    }catch(InstantiationException e){
                    }catch(IllegalAccessException e){
                    }
                    exceptionMapForHandler.add(clazz, cond);
                }
            }
        }
        if(defaultExceptionHandlerServiceName != null){
            defaultExceptionHandler = ServiceManagerFactory
                .getServiceObject(defaultExceptionHandlerServiceName);
        }
    }
    
    /**
     * ̃C^[Zv^ĂяoAnhOΏۂ̗OƗOsJSPɃtH[hB<p>
     * T[rXJnĂȂꍇ́AɎ̃C^[Zv^ĂяoB<br>
     *
     * @param context ĂяõReLXg
     * @param chain ̃C^[Zv^Ăяo߂̃`F[
     * @return Ăяoʂ̖߂l
     * @exception Throwable ĂяoŗOꍇA܂͂̃C^[Zv^ŔCӂ̗OꍇBAA{Ăяo鏈throwȂRuntimeExceptionȊO̗OthrowĂAĂяoɂ͓`dȂB
     */
    public Object invokeFilter(
        ServletFilterInvocationContext context,
        InterceptorChain chain
    ) throws Throwable{
        if(getState() == State.STARTED){
            Object ret = null;
            try{
                ret = chain.invokeNext(context);
            }catch(Throwable th){
                final ServletRequest request = context.getServletRequest();
                final ServletResponse response = context.getServletResponse();
                boolean isHandled = false;
                ExceptionHandler handler = getTargetExceptionHandlerCause(
                    th,
                    request,
                    response
                );
                if(handler != null){
                    // nhOΏۂ̗O
                    Throwable targetTh = getTargetException(
                        exceptionMapForHandler,
                        th,
                        request,
                        response
                    );
                    if(targetTh == null){
                        targetTh = th;
                    }
                    handler.handleException(targetTh, request, response);
                    isHandled = true;
                }
                if(!isHandled){
                    throw th;
                }
            }
            return ret;
        }else{
            return chain.invokeNext(context);
        }
    }
    
    /**
     * w肳ꂽÕnhNX}bv猩ĕԂB<p>
     *
     * @param th O(nhOΏۂ̗O܂ł)
     * @param request NGXg
     * @param response X|X
     * @return Onh
     */
    protected ExceptionHandler getTargetExceptionHandlerCause(
        Throwable th,
        ServletRequest request,
        ServletResponse response
    ){
        ExceptionHandler handler = (ExceptionHandler)getTargetHandlerCause(
            exceptionMapForHandler,
            th,
            request,
            response
        );
        return handler == null ? defaultExceptionHandler : handler;
    }
    
    /**
     * w肳ꂽÕnh}bv猩ĕԂB<p>
     *
     * @param handlerMap nh̃}bv
     * @param th O(nhOΏۂ̗O܂ł)
     * @param request NGXg
     * @param response X|X
     * @return Onh
     */
    protected ExceptionHandler getTargetHandlerCause(
        ClassMappingTree<Object> handlerMap,
        Throwable th,
        ServletRequest request,
        ServletResponse response
    ){
        if(handlerMap == null){
            return null;
        }
        // ONXɊ֘AtĂOnh擾
        List<Object> handlers = handlerMap.getValueList(th.getClass());
        if(handlers != null){
            for(int i = 0, imax = handlers.size(); i < imax; i++){
                Object handler = handlers.get(i);
                if(handler instanceof Condition){
                    Condition condition = (Condition)handler;
                    try{
                        if(condition.evaluate(th, request, response)){
                            return condition.handler;
                        }
                    }catch(Exception e){
                        getLogger().write(EHI__00001, e);
                    }
                }else{
                    return (ExceptionHandler)handler;
                }
            }
        }
        Throwable cause = getCause(th);
        return cause == null ? null : getTargetHandlerCause(handlerMap, cause, request, response);
    }
    
    /**
     * w肳ꂽOAnhOΏۂ̗OoB<p>
     * 
     * @param handlerMap nh̃}bv
     * @param th O
     * @param request NGXg
     * @param response X|X
     * @return nhOΏۂ̗O
     */
    protected Throwable getTargetException(
        ClassMappingTree<Object> handlerMap,
        Throwable th,
        ServletRequest request,
        ServletResponse response
    ) {
        if(handlerMap == null){
            return th;
        }
        // ONXɊ֘AtĂOnh擾
        List<Object> handlers = handlerMap.getValueList(th.getClass());
        if(handlers != null){
            for(int i = 0, imax = handlers.size(); i < imax; i++){
                Object handler = handlers.get(i);
                if(handler instanceof Condition){
                    Condition condition = (Condition)handler;
                    try{
                        if(condition.evaluate(th, request, response)){
                            return th;
                        }
                    }catch(Exception e){
                        getLogger().write(EHI__00001, e);
                    }
                }else{
                    return th;
                }
            }
        }
        
        Throwable cause = getCause(th);
        return cause == null ? null : getTargetException(
            handlerMap,
            cause,
            request,
            response
        );
    }
    
    /**
     * w肳ꂽO猴擾B<p>
     *
     * @param th O
     * @return 
     */
    protected Throwable getCause(Throwable th){
        Throwable cause = null;
        if(th instanceof ServletException){
            // OServletException̏ꍇ́A[ǧ擾
            cause = ((ServletException)th).getRootCause();
        }else if(th.getClass().getName().equals(JMS_EXCEPTION_NAME)){
            // OJMSException̏ꍇ́ANO擾
            try{
                cause = (Exception)th.getClass()
                    .getMethod(GET_LINKED_EXCEPTION_METHOD).invoke(th);
            }catch(NoSuchMethodException e){
            }catch(IllegalAccessException e){
            }catch(InvocationTargetException e){
            }
        }else{
            cause = th.getCause();
        }
        return cause == th ? null : cause;
    }
    
    protected static class Condition implements Serializable{
        
        private static final long serialVersionUID = 1115982642791231865L;
        
        public static final String EXCEPTION_KEY = "exception";
        public static final String REQUEST_KEY = "request";
        public static final String RESPONSE_KEY = "response";
        protected static final String PROP_FUNCTION_NAME = "prop";
        
        public ExceptionHandler handler;
        
        protected transient Expression expression;
        protected String condition;
        
        public Condition(String cond, ExceptionHandler handler){
            initCondition(cond);
            condition = cond;
            this.handler = handler;
        }
        
        protected void initCondition(String cond){
            JexlEngine jexl = new JexlEngine();
            jexl.setSilent(true);
            Map<String, Object> funcs = new HashMap<String, Object>();
            PropertyAccess propAccess = new PropertyAccess();
            propAccess.setIgnoreNullProperty(true);
            funcs.put(PROP_FUNCTION_NAME, propAccess);
            jexl.setFunctions(funcs);
            expression = jexl.createExpression(cond);
        }
        
        public boolean evaluate(
            Throwable th,
            ServletRequest request,
            ServletResponse response
        ){
            return evaluate(th, request, response, false);
        }
        
        public boolean evaluate(
            Throwable th,
            ServletRequest request,
            ServletResponse response,
            boolean isTest
        ){
            JexlContext jexlContext = new MapContext();
            jexlContext.set(EXCEPTION_KEY, th);
            jexlContext.set(REQUEST_KEY, request);
            jexlContext.set(RESPONSE_KEY, response);
            
            Object exp = expression.evaluate(jexlContext);
            if(exp instanceof Boolean){
                return ((Boolean)exp).booleanValue();
            }else{
                if(exp == null && isTest){
                    return true;
                }
                throw new IllegalArgumentException(expression.getExpression());
            }
        }
        
        private void readObject(ObjectInputStream in)
         throws IOException, ClassNotFoundException{
            in.defaultReadObject();
            initCondition(condition);
        }
    }
}