/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.context;

import java.util.*;
import java.lang.reflect.InvocationTargetException;

import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.publish.*;
import jp.ossc.nimbus.service.cache.CacheMap;
import jp.ossc.nimbus.service.cache.CacheRemoveListener;
import jp.ossc.nimbus.service.cache.CachedReference;
import jp.ossc.nimbus.service.cache.IllegalCachedReferenceException;
import jp.ossc.nimbus.service.cache.KeyCachedReference;
import jp.ossc.nimbus.service.distribute.ClusterListener;
import jp.ossc.nimbus.service.distribute.ClusterService;
import jp.ossc.nimbus.service.interpreter.Interpreter;
import jp.ossc.nimbus.service.interpreter.EvaluateException;
import jp.ossc.nimbus.service.queue.QueueHandlerContainerService;
import jp.ossc.nimbus.service.queue.QueueHandler;
import jp.ossc.nimbus.service.queue.AsynchContext;
import jp.ossc.nimbus.util.SynchronizeMonitor;
import jp.ossc.nimbus.util.WaitSynchronizeMonitor;

/**
 * LReLXgB<p>
 * T[oԂŃReLXgLB<br>
 * ȉɁAT[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="Context"
 *                  code="jp.ossc.nimbus.service.context.SharedContextService"&gt;
 *             &lt;attribute name="RequestConnectionFactoryServiceName"&gt;#RequestConnectionFactory&lt;/attribute&gt;
 *             &lt;attribute name="ClusterServiceName"&gt;#Cluster&lt;/attribute&gt;
 *             &lt;depends&gt;RequestConnectionFactory&lt;/depends&gt;
 *             &lt;depends&gt;Cluster&lt;/depends&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="RequestConnectionFactory"
 *                  code="jp.ossc.nimbus.service.publish.RequestConnectionFactoryService"&gt;
 *             &lt;attribute name="ServerConnectionFactoryServiceName"&gt;#ServerConnectionFactory&lt;/attribute&gt;
 *             &lt;attribute name="MessageReceiverServiceName"&gt;#MessageReceiver&lt;/attribute&gt;
 *             &lt;depends&gt;
 *                 &lt;service name="MessageReceiver"
 *                          code="jp.ossc.nimbus.service.publish.MessageReceiverService"&gt;
 *                     &lt;attribute name="ClientConnectionFactoryServiceName"&gt;#ClientConnectionFactory&lt;/attribute&gt;
 *                     &lt;attribute name="StartReceiveOnStart"&gt;true&lt;/attribute&gt;
 *                     &lt;depends&gt;ClientConnectionFactory&lt;/depends&gt;
 *                 &lt;/service&gt;
 *             &lt;/depends&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="ClientConnectionFactory"
 *                  code="jp.ossc.nimbus.service.publish.ClusterConnectionFactoryService"&gt;
 *             &lt;attribute name="JndiRepositoryServiceName"&gt;#JNDIRepository&lt;/attribute&gt;
 *             &lt;attribute name="ClusterServiceName"&gt;#Cluster&lt;/attribute&gt;
 *             &lt;attribute name="ClientConnectionFactoryServiceName"&gt;#ServerConnectionFactory&lt;/attribute&gt;
 *             &lt;attribute name="Multiple"&gt;true&lt;/attribute&gt;
 *             &lt;attribute name="FlexibleConnect"&gt;true&lt;/attribute&gt;
 *             &lt;depends&gt;
 *                 &lt;service name="JNDIRepository"
 *                          code="jp.ossc.nimbus.service.repository.JNDIRepositoryService" /&gt;
 *             &lt;/depends&gt;
 *             &lt;depends&gt;
 *                 &lt;service name="Cluster"
 *                          code="jp.ossc.nimbus.service.keepalive.ClusterService"&gt;
 *                     &lt;attribute name="LocalAddress"&gt;0.0.0.0&lt;/attribute&gt;
 *                     &lt;attribute name="MulticastGroupAddress"&gt;224.1.1.1&lt;/attribute&gt;
 *                     &lt;attribute name="HeartBeatRetryCount"&gt;2&lt;/attribute&gt;
 *                     &lt;attribute name="JoinOnStart"&gt;false&lt;/attribute&gt;
 *                 &lt;/service&gt;
 *             &lt;/depends&gt;
 *         &lt;/service&gt;
 *         
 *         &lt;service name="ServerConnectionFactory"
 *                  code="jp.ossc.nimbus.service.publish.tcp.ConnectionFactoryService" /&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class SharedContextService<K> extends DefaultContextService<K, Object>
 implements SharedContext<K>, RequestMessageListener, CacheRemoveListener<Object>, ClusterListener, SharedContextServiceMBean<K, Object>, java.io.Serializable{
    
    private static final long serialVersionUID = -7616415086512838961L;
    
    private ServiceName requestConnectionFactoryServiceName;
    private RequestServerConnection serverConnection;
    private MessageReceiver messageReceiver;
    
    private ServiceName clusterServiceName;
    private ClusterService cluster;
    private ServiceName clientCacheMapServiceName;
    private ServiceName serverCacheMapServiceName;
    private CacheMap<K, Object> clientCacheMap;
    private CacheMap<K, Object> serverCacheMap;
    private CacheMap<K, Object> cacheMap;
    private boolean isClient;
    private ServiceName[] sharedContextUpdateListenerServiceNames;
    private ServiceName interpreterServiceName;
    private Interpreter interpreter;
    private int executeThreadSize;
    private QueueHandlerContainerService<AsynchContext<Object[], Object>> executeQueueHandlerContainer;
    private ServiceName executeQueueServiceName;
    
    private long synchronizeTimeout = 5000l;
    
    private long defaultTimeout = 1000l;
    
    private String subject = DEFAULT_SUBJECT;
    private String clientSubject;
    
    private boolean isSynchronizeOnStart = true;
    private boolean isSaveOnlyMain;
    
    private Map<Object, Lock> keyLockMap;
    private Map<Object, Set<K>> idLocksMap;
    private Map<Object, ClientCacheLock> clientCacheLockMap;
    private Message targetMessage;
    private List<SharedContextUpdateListener<K>> updateListeners;
    private Map<String,Property[]> keyIndexMap;
    private Map<String,Map<IndexKey,Set<K>>> keyIndexResultMap;
    private Timer lockTimeoutTimer;
    
    @Override
    public void setRequestConnectionFactoryServiceName(ServiceName name){
        requestConnectionFactoryServiceName = name;
    }
    @Override
    public ServiceName getRequestConnectionFactoryServiceName(){
        return requestConnectionFactoryServiceName;
    }
    
    @Override
    public void setClusterServiceName(ServiceName name){
        clusterServiceName = name;
    }
    @Override
    public ServiceName getClusterServiceName(){
        return clusterServiceName;
    }
    
    @Override
    public void setClientCacheMapServiceName(ServiceName name){
        clientCacheMapServiceName = name;
    }
    @Override
    public ServiceName getClientCacheMapServiceName(){
        return clientCacheMapServiceName;
    }
    
    @Override
    public void setServerCacheMapServiceName(ServiceName name){
        serverCacheMapServiceName = name;
    }
    @Override
    public ServiceName getServerCacheMapServiceName(){
        return serverCacheMapServiceName;
    }
    
    @Override
    public void setInterpreterServiceName(ServiceName name){
        interpreterServiceName = name;
    }
    @Override
    public ServiceName getInterpreterServiceName(){
        return interpreterServiceName;
    }
    
    @Override
    public void setExecuteThreadSize(int size){
        executeThreadSize = size;
    }
    @Override
    public int getExecuteThreadSize(){
        return executeThreadSize;
    }
    
    @Override
    public void setExecuteQueueServiceName(ServiceName name){
        executeQueueServiceName = name;
    }
    @Override
    public ServiceName getExecuteQueueServiceName(){
        return executeQueueServiceName;
    }
    
    @Override
    public void setSubject(String subject){
        this.subject = subject;
    }
    @Override
    public String getSubject(){
        return subject;
    }
    
    @Override
    public synchronized void setClient(boolean isClient) throws SharedContextSendException, SharedContextTimeoutException{
        if(this.isClient == isClient){
            return;
        }
        if(getState() == State.STARTED){
            if(isClient){
                if(clientCacheMapServiceName != null){
                    cacheMap = ServiceManagerFactory.getServiceObject(clientCacheMapServiceName);
                }else if(clientCacheMap != null){
                    cacheMap = clientCacheMap;
                }
            }else{
                if(serverCacheMapServiceName != null){
                    cacheMap = ServiceManagerFactory.getServiceObject(serverCacheMapServiceName);
                }else if(serverCacheMap != null){
                    cacheMap = serverCacheMap;
                }
            }
            if(keyIndexResultMap != null){
                keyIndexResultMap.clear();
            }
            try{
                if(isClient){
                    synchronizeForClient(synchronizeTimeout);
                }else{
                    synchronizeWithMain(synchronizeTimeout);
                }
            }catch(NoConnectServerException e){
            }
            try{
                messageReceiver.removeMessageListener(this);
                messageReceiver.addSubject(this, isClient ? clientSubject :  subject);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
            if(cacheMap != null){
                Object[] keys = null;
                synchronized(context){
                    keys = (Object[])super.keySet().toArray();
                }
                for(int i = 0; i < keys.length; i++){
                    cacheMap.remove(keys[i]);
                }
            }
        }
        this.isClient = isClient;
    }
    
    @Override
    public boolean isClient(){
        return isClient;
    }
    
    @Override
    public void setSynchronizeOnStart(boolean isSynch){
        isSynchronizeOnStart = isSynch;
    }
    @Override
    public boolean isSynchronizeOnStart(){
        return isSynchronizeOnStart;
    }
    
    @Override
    public void setSaveOnlyMain(boolean isSave){
        isSaveOnlyMain = isSave;
    }
    @Override
    public boolean isSaveOnlyMain(){
        return isSaveOnlyMain;
    }
    
    @Override
    public void setSynchronizeTimeout(long timeout){
        synchronizeTimeout = timeout;
    }
    @Override
    public long getSynchronizeTimeout(){
        return synchronizeTimeout;
    }
    
    @Override
    public void setDefaultTimeout(long timeout){
        defaultTimeout = timeout;
    }
    @Override
    public long getDefaultTimeout(){
        return defaultTimeout;
    }
    
    public void setClientCacheMap(CacheMap<K, Object> map){
        clientCacheMap = map;
    }
    public void setServerCacheMap(CacheMap<K, Object> map){
        serverCacheMap = map;
    }
    
    public void setSharedContextUpdateListenerServiceNames(ServiceName[] names){
        sharedContextUpdateListenerServiceNames = names;
    }
    public ServiceName[] getSharedContextUpdateListenerServiceNames(){
        return sharedContextUpdateListenerServiceNames;
    }
    
    public void setKeyIndex(String name, String[] keyProps) throws SharedContextIllegalIndexException{
        Property[] props = new Property[keyProps.length];
        for(int i = 0; i < keyProps.length; i++){
            try{
                props[i] = PropertyFactory.createProperty(keyProps[i]);
            }catch(IllegalArgumentException e){
                throw new SharedContextIllegalIndexException(e);
            }
            props[i].setIgnoreNullProperty(true);
        }
        if(keyIndexMap == null){
            keyIndexMap = Collections.synchronizedMap(new HashMap<String,Property[]>());
            keyIndexResultMap = Collections.synchronizedMap(new HashMap<String,Map<IndexKey,Set<K>>>());
        }
        keyIndexMap.put(name, props);
    }
    
    public void analyzeKeyIndex(String name) throws SharedContextIllegalIndexException, SharedContextSendException, SharedContextTimeoutException{
        if(!isClient){
            if(keyIndexMap == null || !keyIndexMap.containsKey(name)){
                return;
            }
            Property[] keyProps = keyIndexMap.get(name);
            final Map<IndexKey,Set<K>> indexMap = Collections.synchronizedMap(new HashMap<IndexKey,Set<K>>());
            synchronized(context){
                for(Iterator<Map.Entry<K,Object>> entries = context.entrySet().iterator(); entries.hasNext();){
                    Map.Entry<K,Object> entry = entries.next();
                    Object[] keys = new Object[keyProps.length];
                    for(int i = 0; i < keyProps.length; i++){
                        try{
                            keys[i] = keyProps[i].getProperty(entry.getValue());
                        }catch(NoSuchPropertyException e){
                            throw new SharedContextIllegalIndexException(e);
                        }catch(InvocationTargetException e){
                            throw new SharedContextIllegalIndexException(e);
                        }
                    }
                    IndexKey key = new IndexKey(keys);
                    Set<K> keySet = indexMap.get(key);
                    if(keySet == null){
                        keySet = Collections.synchronizedSet(new HashSet<K>());
                        indexMap.put(key, keySet);
                    }
                    keySet.add(entry.getKey());
                }
            }
            keyIndexResultMap.put(name, indexMap);
        }
        try{
            Message message = serverConnection.createMessage(subject, null);
            message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_ANALYZE_KEY_INDEX, name));
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() == 0){
                if(isClient){
                    throw new NoConnectServerException();
                }
            }else{
                Message[] responses = serverConnection.request(
                    message,
                    clientSubject,
                    null,
                    0,
                    defaultTimeout
                );
                if(responses != null && responses.length > 0){
                    for(int i = 0; i < responses.length; i++){
                        Object ret = responses[i].getObject();
                        if(ret instanceof Throwable){
                            throw new SharedContextSendException((Throwable)ret);
                        }
                    }
                }else{
                    throw new SharedContextTimeoutException();
                }
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
    }
    
    protected void addAllKeyIndex(K key, Object value){
        if(isClient || keyIndexMap == null || keyIndexMap.size() == 0){
            return;
        }
        Object[] names = keyIndexMap.keySet().toArray();
        for(int i = 0; i < names.length; i++){
            addKeyIndex((String)names[i], key, value);
        }
    }
    
    protected void addKeyIndex(String name, K key, Object value){
        if(isClient || keyIndexMap == null || !keyIndexMap.containsKey(name)){
            return;
        }
        Map<IndexKey,Set<K>> indexMap = keyIndexResultMap.get(name);
        if(indexMap == null){
            synchronized(keyIndexResultMap){
                indexMap = keyIndexResultMap.get(name);
                if(indexMap == null){
                    indexMap = Collections.synchronizedMap(new HashMap<IndexKey,Set<K>>());
                    keyIndexResultMap.put(name, indexMap);
                }
            }
        }
        Property[] keyProps = keyIndexMap.get(name);
        Object[] keys = new Object[keyProps.length];
        for(int i = 0; i < keyProps.length; i++){
            try{
                keys[i] = keyProps[i].getProperty(value);
            }catch(NoSuchPropertyException e){
                throw new SharedContextIllegalIndexException(e);
            }catch(InvocationTargetException e){
                throw new SharedContextIllegalIndexException(e);
            }
        }
        IndexKey indexKey = new IndexKey(keys);
        Set<K> keySet = indexMap.get(indexKey);
        if(keySet == null){
            synchronized(indexMap){
                keySet = indexMap.get(indexKey);
                if(keySet == null){
                    keySet = Collections.synchronizedSet(new HashSet<K>());
                    indexMap.put(indexKey, keySet);
                }
            }
        }
        keySet.add(key);
    }
    
    protected void removeAllKeyIndex(Object key, Object value){
        if(isClient || keyIndexMap == null || keyIndexMap.size() == 0){
            return;
        }
        Object[] names = keyIndexMap.keySet().toArray();
        for(int i = 0; i < names.length; i++){
            removeKeyIndex((String)names[i], key, value);
        }
    }
    
    protected void removeKeyIndex(String name, Object key, Object value){
        if(isClient || keyIndexMap == null || !keyIndexMap.containsKey(name)){
            return;
        }
        Map<IndexKey,Set<K>> indexMap = keyIndexResultMap.get(name);
        if(indexMap == null){
            return;
        }
        Property[] keyProps = keyIndexMap.get(name);
        Object[] keys = new Object[keyProps.length];
        for(int i = 0; i < keyProps.length; i++){
            try{
                keys[i] = keyProps[i].getProperty(value);
            }catch(NoSuchPropertyException e){
                throw new SharedContextIllegalIndexException(e);
            }catch(InvocationTargetException e){
                throw new SharedContextIllegalIndexException(e);
            }
        }
        IndexKey indexKey = new IndexKey(keys);
        Set<K> keySet = indexMap.get(indexKey);
        if(keySet == null){
            return;
        }
        keySet.remove(key);
        if(keySet.size() == 0){
            indexMap.remove(indexKey);
        }
    }
    
    /**
     * T[rX̐sB<p>
     *
     * @exception Exception T[rX̐Ɏsꍇ
     */
    @Override
    public void createService() throws Exception{
        super.createService();
        keyLockMap = Collections.synchronizedMap(new HashMap<Object, Lock>());
        idLocksMap = Collections.synchronizedMap(new HashMap<Object, Set<K>>());
        clientCacheLockMap = Collections.synchronizedMap(new HashMap<Object, ClientCacheLock>());
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    @SuppressWarnings("unchecked")
    @Override
    public void startService() throws Exception{
        if(requestConnectionFactoryServiceName == null){
            throw new IllegalArgumentException("RequestConnectionFactoryServiceName must be specified.");
        }
        if(isClient){
            if(clientCacheMapServiceName != null){
                cacheMap = ServiceManagerFactory.getServiceObject(clientCacheMapServiceName);
            }else if(clientCacheMap != null){
                cacheMap = clientCacheMap;
            }
        }else{
            if(serverCacheMapServiceName != null){
                cacheMap = ServiceManagerFactory.getServiceObject(serverCacheMapServiceName);
            }else if(serverCacheMap != null){
                cacheMap = serverCacheMap;
            }
        }
        
        if(interpreterServiceName != null){
            interpreter = ServiceManagerFactory.getServiceObject(interpreterServiceName);
        }
        
        executeQueueHandlerContainer = new QueueHandlerContainerService<AsynchContext<Object[], Object>>();
        executeQueueHandlerContainer.create();
        executeQueueHandlerContainer.setQueueHandlerSize(executeThreadSize);
        executeQueueHandlerContainer.setQueueServiceName(executeQueueServiceName);
        executeQueueHandlerContainer.setQueueHandler(new ExecuteQueueHandler());
        executeQueueHandlerContainer.start();
        
        if(sharedContextUpdateListenerServiceNames != null){
            for(int i = 0; i < sharedContextUpdateListenerServiceNames.length; i++){
                addSharedContextUpdateListener(
                    (SharedContextUpdateListener<K>)ServiceManagerFactory.getServiceObject(sharedContextUpdateListenerServiceNames[i])
                );
            }
        }
        
        ServerConnectionFactory factory = ServiceManagerFactory.getServiceObject(requestConnectionFactoryServiceName);
        serverConnection = (RequestServerConnection)factory.getServerConnection();
        targetMessage = serverConnection.createMessage(subject, null);
        messageReceiver = ServiceManagerFactory.getServiceObject(requestConnectionFactoryServiceName);
        clientSubject = subject + CLIENT_SUBJECT_SUFFIX;
        messageReceiver.addSubject(this, isClient ? clientSubject :  subject);
        if(clusterServiceName == null){
            throw new IllegalArgumentException("ClusterServiceName must be specified.");
        }
        cluster = ServiceManagerFactory.getServiceObject(clusterServiceName);
        cluster.addClusterListener(this);
        
        lockTimeoutTimer = new Timer("SharedContext LockTimeoutTimerThread of " + getServiceNameObject(), true);
        
        super.startService();
        
        if(isSynchronizeOnStart && !isMain()){
            try{
                synchronize();
            }catch(NoConnectServerException e){
                if(!isClient){
                    throw e;
                }
            }
        }
    }
    
    /**
     * T[rX̒~sB<p>
     *
     * @exception Exception T[rX̒~Ɏsꍇ
     */
    @Override
    public void stopService() throws Exception{
        unlockAll();
        if(messageReceiver != null){
            messageReceiver.removeMessageListener(this);
        }
        if(cluster != null){
            cluster.removeClusterListener(this);
        }
        super.stopService();
    }
    
    /**
     * T[rX̔jsB<p>
     * CX^XϐjB<br>
     *
     * @exception Exception T[rX̔jɎsꍇ
     */
    @Override
    public void destroyService() throws Exception{
        keyLockMap = null;
        idLocksMap = null;
        clientCacheLockMap = null;
        super.destroyService();
    }
    
    @Override
    public synchronized void load() throws Exception{
        load(-1l);
    }
    
    @Override
    public synchronized void load(long timeout) throws Exception{
        if(isMain()){
            super.load();
        }else{
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_LOAD));
                    Message[] responses = serverConnection.request(
                        message,
                        isClient ? clientSubject : subject,
                        null,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        Object ret = responses[0].getObject();
                        if(ret instanceof Throwable){
                            throw new SharedContextSendException((Throwable)ret);
                        }
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException("Main server is not found.");
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    
    @Override
    public synchronized void loadKey() throws Exception{
        loadKey(-1l);
    }
    
    @Override
    public synchronized void loadKey(long timeout) throws Exception{
        if(isMain()){
            super.loadKey();
        }else{
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_LOAD_KEY));
                    Message[] responses = serverConnection.request(
                        message,
                        isClient ? clientSubject : subject,
                        null,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        Object ret = responses[0].getObject();
                        if(ret instanceof Throwable){
                            throw new SharedContextSendException((Throwable)ret);
                        }
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException("Main server is not found.");
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    public void load(Object key) throws Exception{
        load(key, -1l);
    }
    
    public void load(Object key, long timeout) throws Exception{
        if(isMain()){
            super.load(key);
        }else{
            try{
                Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_LOAD, key));
                    Message[] responses = serverConnection.request(
                        message,
                        isClient ? clientSubject : subject,
                        null,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        Object ret = responses[0].getObject();
                        if(ret instanceof Throwable){
                            throw new SharedContextSendException((Throwable)ret);
                        }
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException("Main server is not found.");
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    
    @Override
    public synchronized void save() throws Exception{
        save(-1l);
    }
    
    @Override
    public synchronized void save(long timeout) throws Exception{
        if(!isMain()){
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_SAVE));
                    Message[] responses = serverConnection.request(
                        message,
                        isClient ? clientSubject : subject,
                        null,
                        isSaveOnlyMain ? 1 : 0,
                        timeout
                    );
                    if(responses != null
                        && (isSaveOnlyMain ? responses.length > 0 : responses.length >= receiveClients.size())
                    ){
                        for(int i = 0; i < responses.length; i++){
                            Object ret = responses[i].getObject();
                            if(ret instanceof Throwable){
                                throw new SharedContextSendException((Throwable)ret);
                            }
                        }
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException("Main server is not found.");
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
        if(!isClient && (isMain() || !isSaveOnlyMain)){
            super.save();
        }
    }
    
    @Override
    public void synchronize() throws SharedContextSendException, SharedContextTimeoutException{
        synchronize(synchronizeTimeout);
    }
    
    @Override
    public synchronized void synchronize(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(isClient){
            synchronizeForClient(timeout);
        }else if(isMain()){
            try{
                Message message = serverConnection.createMessage(subject, null);
                message.setSubject(clientSubject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_SYNCH_ALL, null, new Long(timeout)));
                    Message[] responses = serverConnection.request(message, 0, timeout);
                    if(responses != null && responses.length >= receiveClients.size()){
                        for(int i = 0; i < responses.length; i++){
                            if(responses[i].getObject() == null || !((Boolean)responses[i].getObject()).booleanValue()){
                                throw new SharedContextSendException("It faild to synchronize.");
                            }
                        }
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }else{
            synchronizeWithMain(timeout);
        }
    }
    
    private synchronized void synchronizeForClient(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(cacheMap != null){
            Object[] keys = null;
            synchronized(context){
                keys = (Object[])super.keySet().toArray();
            }
            for(int i = 0; i < keys.length; i++){
                cacheMap.remove(keys[i]);
            }
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onClearSynchronize(this);
            }
        }
        super.clear();
        if(updateListeners != null){
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_GET_ALL));
                    Message[] responses = serverConnection.request(message, 1, timeout);
                    if(responses != null && responses.length > 0){
                        @SuppressWarnings("unchecked")
                        Map<K, Object> result = (Map<K, Object>)responses[0].getObject();
                        if(result != null){
                            Iterator<Map.Entry<K, Object>> entries = result.entrySet().iterator();
                            while(entries.hasNext()){
                                Map.Entry<K, Object> entry = entries.next();
                                if(updateListeners != null){
                                    for(int i = 0; i < updateListeners.size(); i++){
                                        if(!updateListeners.get(i).onPutSynchronize(this, entry.getKey(), entry.getValue())){
                                            break;
                                        }
                                    }
                                }
                            }
                        }
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException();
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    
    private synchronized void synchronizeWithMain(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        try{
            Message message = serverConnection.createMessage(subject, null);
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_GET_ALL));
                Message[] responses = serverConnection.request(
                    message,
                    isClient ? clientSubject : subject,
                    null,
                    1,
                    timeout
                );
                if(responses != null && responses.length > 0){
                    @SuppressWarnings("unchecked")
                    Map<K, Object> result = (Map<K, Object>)responses[0].getObject();
                    if(updateListeners != null){
                        for(int i = 0; i < updateListeners.size(); i++){
                            updateListeners.get(i).onClearSynchronize(this);
                        }
                    }
                    super.clear();
                    if(result != null){
                        Iterator<Map.Entry<K, Object>> entries = result.entrySet().iterator();
                        while(entries.hasNext()){
                            Map.Entry<K, Object> entry = entries.next();
                            boolean isPut = true;
                            if(updateListeners != null){
                                for(int i = 0; i < updateListeners.size(); i++){
                                    if(!updateListeners.get(i).onPutSynchronize(this, entry.getKey(), entry.getValue())){
                                        isPut = false;
                                        break;
                                    }
                                }
                            }
                            if(isPut){
                                super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                                addAllKeyIndex(entry.getKey(), entry.getValue());
                            }
                        }
                    }
                }else{
                    throw new SharedContextTimeoutException();
                }
            }else{
                throw new NoConnectServerException("Main server is not found.");
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
    }
    
    protected Object wrapCachedReference(K key, Object value){
        if(value == null || cacheMap == null){
            return value;
        }else{
            cacheMap.put(key, value);
            CachedReference<Object> ref = cacheMap.getCachedReference(key);
            if(ref != null){
                ref.addCacheRemoveListener(this);
            }
            return ref;
        }
    }
    
    protected Object unwrapCachedReference(Object value, boolean notify, boolean remove){
        if(value == null){
            return null;
        }
        if(cacheMap == null){
            return value;
        }else{
            @SuppressWarnings("unchecked")
            CachedReference<Object> ref = (CachedReference<Object>)value;
            if(remove){
                ref.remove(this);
            }
            return ref.get(this, notify);
        }
    }
    
    @Override
    public void lock(K key) throws SharedContextSendException, SharedContextTimeoutException{
        lock(key, defaultTimeout);
    }
    
    @Override
    public void lock(K key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        final Object id = cluster.getUID();
        Lock lock = null;
        synchronized(keyLockMap){
            lock = keyLockMap.get(key);
            if(lock == null){
                lock = new Lock(key);
                keyLockMap.put(key, lock);
            }
        }
        Object lockedOwner = lock.getOwner();
        if(id.equals(lockedOwner)){
            return;
        }
        if(isMain()){
            final long start = System.currentTimeMillis();
            if(lock.acquire(id, timeout)){
                final boolean isNoTimeout = timeout <= 0;
                timeout = isNoTimeout ? timeout : timeout - (System.currentTimeMillis() - start);
                if(!isNoTimeout && timeout <= 0){
                    lock.release(id, false);
                    throw new SharedContextTimeoutException();
                }else{
                    try{
                        Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
                        message.setSubject(clientSubject, key == null ? null : key.toString());
                        Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                        if(receiveClients.size() != 0){
                            message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_GOT_LOCK, key, new Object[]{id, new Long(timeout)}));
                            Message[] responses = serverConnection.request(
                                message,
                                isClient ? clientSubject : subject,
                                key == null ? null : key.toString(),
                                0,
                                timeout
                            );
                            if(responses != null && responses.length >= receiveClients.size()){
                                for(int i = 0; i < responses.length; i++){
                                    Object ret = responses[i].getObject();
                                    if(ret instanceof Throwable){
                                        unlock(key);
                                        throw new SharedContextSendException((Throwable)ret);
                                    }else if(ret == null || !((Boolean)ret).booleanValue()){
                                        unlock(key);
                                        throw new SharedContextTimeoutException();
                                    }
                                }
                            }else{
                                unlock(key);
                                throw new SharedContextTimeoutException();
                            }
                        }
                    }catch(MessageException e){
                        unlock(key);
                        throw new SharedContextSendException(e);
                    }catch(MessageSendException e){
                        unlock(key);
                        throw new SharedContextSendException(e);
                    }catch(RuntimeException e){
                        unlock(key);
                        throw e;
                    }catch(Error e){
                        unlock(key);
                        throw e;
                    }catch(Throwable th){
                        unlock(key);
                        throw new SharedContextSendException(th);
                    }
                }
            }else{
                throw new SharedContextTimeoutException();
            }
        }else{
            try{
                Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_GET_LOCK, key, new Long(timeout)));
                    Message[] responses = serverConnection.request(
                        message,
                        isClient ? clientSubject : subject,
                        key == null ? null : key.toString(),
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        Object ret = responses[0].getObject();
                        if(ret instanceof Throwable){
                            unlock(key);
                            throw new SharedContextSendException((Throwable)ret);
                        }else if(ret == null || !((Boolean)ret).booleanValue()){
                            unlock(key);
                            throw new SharedContextTimeoutException();
                        }
                    }else{
                        unlock(key);
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException("Main server is not found.");
                }
            }catch(MessageException e){
                unlock(key);
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                unlock(key);
                throw new SharedContextSendException(e);
            }
            if(!lock.acquire(id, timeout)){
                unlock(key);
                throw new SharedContextTimeoutException();
            }
        }
    }
    
    @Override
    public boolean unlock(K key) throws SharedContextSendException{
        return unlock(key, false);
    }
    
    @Override
    public boolean unlock(K key, boolean force) throws SharedContextSendException{
        Lock lock = null;
        synchronized(keyLockMap){
            lock = keyLockMap.get(key);
        }
        Object id = cluster.getUID();
        if(force && lock != null && lock.getOwner() != null){
            id = lock.getOwner();
        }
        try{
            Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
            message.setSubject(clientSubject, key == null ? null : key.toString());
            message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_RELEASE_LOCK, key, id));
            serverConnection.sendAsynch(message);
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        boolean result = lock == null;
        if(lock != null){
            result = lock.release(id, force);
        }
        return result;
    }
    
    @SuppressWarnings("unchecked")
    protected void unlockAll(){
        synchronized(keyLockMap){
            Object myId = cluster.getUID();
            final Iterator<Map.Entry<Object, Lock>> entries = keyLockMap.entrySet().iterator();
            while(entries.hasNext()){
                Map.Entry<Object, Lock> entry = entries.next();
                Object key = entry.getKey();
                Lock lock = entry.getValue();
                Object owner = lock.getOwner();
                if(owner == null || !owner.equals(myId)){
                    continue;
                }
                try{
                    unlock((K)key, true);
                }catch(SharedContextSendException e){
                    lock.release(myId, true);
                }
            }
        }
        synchronized(clientCacheLockMap){
            final Iterator<ClientCacheLock> locks = clientCacheLockMap.values().iterator();
            while(locks.hasNext()){
                ClientCacheLock lock = locks.next();
                lock.notifyAllLock();
            }
        }
        lockTimeoutTimer.cancel();
    }
    
    @Override
    public Object getLockOwner(K key){
        Lock lock = null;
        synchronized(keyLockMap){
            lock = keyLockMap.get(key);
        }
        return lock == null ? null : lock.getOwner();
    }
    
    @Override
    public Object put(K key, Object value) throws SharedContextSendException{
        return put(key, value, defaultTimeout);
    }
    
    @Override
    public Object put(K key, Object value, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onPutBefore(this, true, key, value)){
                    return null;
                }
            }
        }
        Object result = null;
        try{
            Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
            message.setSubject(clientSubject, key == null ? null : key.toString());
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_PUT, key, value));
                Message[] responses = serverConnection.request(
                    message,
                    isClient ? clientSubject : subject,
                    key == null ? null : key.toString(),
                    0,
                    timeout
                );
                if(responses != null && responses.length >= receiveClients.size()){
                    for(int i = 0; i < responses.length; i++){
                        if(responses[i].getObject() != null){
                            result = responses[i].getObject();
                            break;
                        }
                    }
                }else{
                    throw new SharedContextTimeoutException();
                }
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        if(isClient){
            if(super.containsKey(key)){
                result = super.put(key, wrapCachedReference(key, value));
                result = unwrapCachedReference(result, false, true);
            }
        }else{
            final boolean isContainsKey = super.containsKey(key);
            result = super.put(key, wrapCachedReference(key, value));
            result = unwrapCachedReference(result, false, true);
            if(isContainsKey){
                removeAllKeyIndex(key, result);
            }
            addAllKeyIndex(key, value);
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onPutAfter(this, true, key, value, result);
            }
        }
        return result;
    }
    
    @Override
    public Object putLocal(K key, Object value){
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onPutBefore(this, true, key, value)){
                    return null;
                }
            }
        }
        Object result = null;
        if(isClient){
            if(super.containsKey(key)){
                result = super.put(key, wrapCachedReference(key, value));
                result = unwrapCachedReference(result, false, true);
            }
        }else{
            final boolean isContainsKey = super.containsKey(key);
            result = super.put(key, wrapCachedReference(key, value));
            result = unwrapCachedReference(result, false, true);
            if(isContainsKey){
                removeAllKeyIndex(key, result);
            }
            addAllKeyIndex(key, value);
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onPutAfter(this, true, key, value, result);
            }
        }
        return result;
    }
    
    @Override
    public void putAsynch(K key, Object value) throws SharedContextSendException{
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onPutBefore(this, true, key, value)){
                    return;
                }
            }
        }
        try{
            Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
            message.setSubject(clientSubject, key == null ? null : key.toString());
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_PUT, key, value));
                serverConnection.sendAsynch(message);
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        Object removed = null;
        if(isClient){
            if(super.containsKey(key)){
                removed = super.put(key, wrapCachedReference(key, value));
                removed = unwrapCachedReference(removed, false, true);
            }
        }else{
            final boolean isContainsKey = super.containsKey(key);
            removed = super.put(key, wrapCachedReference(key, value));
            removed = unwrapCachedReference(removed, false, true);
            if(isContainsKey){
                removeAllKeyIndex(key, removed);
            }
            addAllKeyIndex(key, value);
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onPutAfter(this, true, key, value, removed);
            }
        }
    }
    
    @Override
    public void update(K key, SharedContextValueDifference diff) throws SharedContextSendException{
        update(key, diff, defaultTimeout);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public void update(K key, SharedContextValueDifference diff, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(diff == null){
            return;
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onUpdateBefore(this, true, key, diff)){
                    return;
                }
            }
        }
        final long start = System.currentTimeMillis();
        try{
            Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_UPDATE, key, diff));
                Message[] responses = serverConnection.request(
                    message,
                    isClient ? clientSubject : subject,
                    key == null ? null : key.toString(),
                    0,
                    timeout
                );
                if(responses != null && responses.length >= receiveClients.size()){
                    for(int i = 0; i < responses.length; i++){
                        Object ret = responses[i].getObject();
                        if(ret != null){
                            if(ret instanceof Throwable){
                                throw new SharedContextSendException((Throwable)ret);
                            }
                        }
                    }
                }else{
                    throw new SharedContextTimeoutException();
                }
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        if(!isClient || super.containsKey(key)){
            Object current = getRawLocal(key);
            Object currentValue = unwrapCachedReference(current, false, false);
            if(currentValue == null){
                throw new SharedContextUpdateException("Current value is null. key=" + key);
            }else if(currentValue instanceof SharedContextValueDifferenceSupport){
                if(!((SharedContextValueDifferenceSupport)currentValue).update(diff)){
                    throw new SharedContextUpdateException(
                        "An update version is mismatching. currentVersion="
                            + ((SharedContextValueDifferenceSupport)currentValue).getUpdateVersion()
                            + ", updateVersion=" + diff.getUpdateVersion()
                    );
                }
            }else{
                throw new SharedContextUpdateException("Not support SharedContextValueDifference. key=" + key + ", value=" + currentValue);
            }
            if(current instanceof CachedReference){
                try{
                    ((CachedReference<Object>)current).set(this, currentValue);
                }catch(IllegalCachedReferenceException e){
                    throw new SharedContextUpdateException(e);
                }
            }
        }
        try{
            Message message = serverConnection.createMessage(clientSubject, key == null ? null : key.toString());
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_UPDATE, key, diff));
                final boolean isNoTimeout = timeout <= 0;
                timeout = isNoTimeout ? timeout : timeout - (System.currentTimeMillis() - start);
                if(!isNoTimeout && timeout <= 0){
                    throw new SharedContextTimeoutException();
                }
                Message[] responses = serverConnection.request(
                    message,
                    isClient ? clientSubject : subject,
                    key == null ? null : key.toString(),
                    0,
                    timeout
                );
                if(responses != null && responses.length >= receiveClients.size()){
                    for(int i = 0; i < responses.length; i++){
                        Object ret = responses[i].getObject();
                        if(ret != null){
                            if(ret instanceof Throwable){
                                throw new SharedContextSendException((Throwable)ret);
                            }
                        }
                    }
                }else{
                    throw new SharedContextTimeoutException();
                }
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onUpdateAfter(this, true, key, diff);
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public void updateLocal(K key, SharedContextValueDifference diff){
        if(diff == null){
            return;
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onUpdateBefore(this, true, key, diff)){
                    return;
                }
            }
        }
        if(!isClient || super.containsKey(key)){
            Object current = getRawLocal(key);
            Object currentValue = unwrapCachedReference(current, false, false);
            if(currentValue == null){
                throw new SharedContextUpdateException("Current value is null. key=" + key);
            }else if(currentValue instanceof SharedContextValueDifferenceSupport){
                if(!((SharedContextValueDifferenceSupport)currentValue).update(diff)){
                    throw new SharedContextUpdateException(
                        "An update version is mismatching. currentVersion="
                            + ((SharedContextValueDifferenceSupport)currentValue).getUpdateVersion()
                            + ", updateVersion=" + diff.getUpdateVersion()
                    );
                }
            }else{
                throw new SharedContextUpdateException("Not support SharedContextValueDifference. key=" + key + ", value=" + currentValue);
            }
            if(current instanceof CachedReference){
                try{
                    ((CachedReference<Object>)current).set(this, currentValue);
                }catch(IllegalCachedReferenceException e){
                    throw new SharedContextUpdateException(e);
                }
            }
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onUpdateAfter(this, true, key, diff);
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public void updateAsynch(K key, SharedContextValueDifference diff) throws SharedContextSendException{
        if(diff == null){
            return;
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onUpdateBefore(this, true, key, diff)){
                    return;
                }
            }
        }
        try{
            Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
            message.setSubject(clientSubject, key == null ? null : key.toString());
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_UPDATE, key, diff));
                serverConnection.sendAsynch(message);
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        if(!isClient || super.containsKey(key)){
            Object current = getRawLocal(key);
            Object currentValue = unwrapCachedReference(current, false, false);
            if(currentValue == null){
                throw new SharedContextUpdateException("Current value is null. key=" + key);
            }else if(currentValue instanceof SharedContextValueDifferenceSupport){
                if(!((SharedContextValueDifferenceSupport)currentValue).update(diff)){
                    throw new SharedContextUpdateException(
                        "An update version is mismatching. currentVersion="
                            + ((SharedContextValueDifferenceSupport)currentValue).getUpdateVersion()
                            + ", updateVersion=" + diff.getUpdateVersion()
                    );
                }
            }else{
                throw new SharedContextUpdateException("Not support SharedContextValueDifference. key=" + key + ", value=" + currentValue);
            }
            if(current instanceof CachedReference){
                try{
                    ((CachedReference<Object>)current).set(this, currentValue);
                }catch(IllegalCachedReferenceException e){
                    throw new SharedContextUpdateException(e);
                }
            }
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onUpdateAfter(this, true, key, diff);
            }
        }
    }
    
    @Override
    public Object remove(Object key) throws SharedContextSendException{
        return remove(key, defaultTimeout);
    }
    
    @Override
    public Object remove(Object key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(isMain() && !super.containsKey(key)){
            return null;
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onRemoveBefore(this, true, key)){
                    return null;
                }
            }
        }
        Object result = null;
        try{
            Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
            message.setSubject(clientSubject, key == null ? null : key.toString());
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_REMOVE, key));
                Message[] responses = serverConnection.request(
                    message,
                    isClient ? clientSubject : subject,
                    key == null ? null : key.toString(),
                    0,
                    timeout
                );
                if(responses != null && responses.length >= receiveClients.size()){
                    if(isMain()){
                        result = super.remove(key);
                        result = unwrapCachedReference(result, false, true);
                    }else{
                        Object removed = super.remove(key);
                        unwrapCachedReference(removed, false, true);
                        for(int i = 0; i < responses.length; i++){
                            if(responses[i].getObject() != null){
                                result = responses[i].getObject();
                                break;
                            }
                        }
                    }
                }else{
                    throw new SharedContextTimeoutException();
                }
            }else if(isClient){
                throw new NoConnectServerException();
            }else{
                result = super.remove(key);
                result = unwrapCachedReference(result, false, true);
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        if(!isClient){
            removeAllKeyIndex(key, result);
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onRemoveAfter(this, true, key, result);
            }
        }
        return result;
    }
    
    @Override
    public Object removeLocal(Object key){
        if(isMain() && !super.containsKey(key)){
            return null;
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onRemoveBefore(this, true, key)){
                    return null;
                }
            }
        }
        Object result = super.remove(key);
        result = unwrapCachedReference(result, false, true);
        if(!isClient){
            removeAllKeyIndex(key, result);
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onRemoveAfter(this, true, key, result);
            }
        }
        return result;
    }
    
    @Override
    public void removeAsynch(Object key) throws SharedContextSendException{
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onRemoveBefore(this, true, key)){
                    return;
                }
            }
        }
        try{
            Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
            message.setSubject(clientSubject, key == null ? null : key.toString());
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_REMOVE, key));
                serverConnection.sendAsynch(message);
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        Object removed = super.remove(key);
        removed = unwrapCachedReference(removed, false, true);
        if(!isClient){
            removeAllKeyIndex(key, removed);
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onRemoveAfter(this, true, key, removed);
            }
        }
    }
    
    @Override
    public void putAll(Map<? extends K,? extends Object> t){
        putAll(t, defaultTimeout);
    }
    
    @Override
    public synchronized void putAll(Map<? extends K,? extends Object> t, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(t.size() == 0){
            return;
        }
        if(updateListeners != null){
            Map<K,Object> tmpMap = new LinkedHashMap<K,Object>();
            Iterator<?> entries = t.entrySet().iterator();
            while(entries.hasNext()){
                @SuppressWarnings("unchecked")
                Map.Entry<? extends K,? extends Object> entry = (Map.Entry<? extends K, ? extends Object>)entries.next();
                for(int i = 0; i < updateListeners.size(); i++){
                    if(!isClient || super.containsKey(entry.getKey())){
                        if(updateListeners.get(i).onPutBefore(this, true, entry.getKey(), entry.getValue())){
                            tmpMap.put((K)entry.getKey(), entry.getValue());
                        }
                    }
                }
            }
            if(tmpMap.size() == 0){
                return;
            }
            t = tmpMap;
        }
        try{
            Message message = serverConnection.createMessage(subject, null);
            message.setSubject(clientSubject, null);
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_PUT_ALL, null, t));
                Message[] responses = serverConnection.request(
                    message,
                    isClient ? clientSubject : subject,
                    null,
                    0,
                    timeout
                );
                if(responses == null || responses.length < receiveClients.size()){
                    throw new SharedContextTimeoutException();
                }
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        Iterator<?> entries = t.entrySet().iterator();
        while(entries.hasNext()){
            @SuppressWarnings("unchecked")
            Map.Entry<K,Object> entry = (Map.Entry<K,Object>)entries.next();
            if(isClient){
                if(super.containsKey(entry.getKey())){
                    Object old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                    old = unwrapCachedReference(old, false, true);
                    if(updateListeners != null){
                        for(int i = 0; i < updateListeners.size(); i++){
                            updateListeners.get(i).onPutAfter(this, true, entry.getKey(), entry.getValue(), old);
                        }
                    }
                }
            }else{
                final boolean isContainsKey = super.containsKey(entry.getKey());
                Object old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                old = unwrapCachedReference(old, false, true);
                if(isContainsKey){
                    removeAllKeyIndex(entry.getKey(), old);
                }
                addAllKeyIndex(entry.getKey(), entry.getValue());
                if(updateListeners != null){
                    for(int i = 0; i < updateListeners.size(); i++){
                        updateListeners.get(i).onPutAfter(this, true, entry.getKey(), entry.getValue(), old);
                    }
                }
            }
        }
    }
    
    @Override
    public synchronized void putAllLocal(Map<? extends K,? extends Object> t){
        if(t.size() == 0){
            return;
        }
        if(updateListeners != null){
            Map<K,Object> tmpMap = new LinkedHashMap<K,Object>();
            Iterator<?> entries = t.entrySet().iterator();
            while(entries.hasNext()){
                @SuppressWarnings("unchecked")
                Map.Entry<? extends K,? extends Object> entry = (Map.Entry<? extends K, ? extends Object>)entries.next();
                for(int i = 0; i < updateListeners.size(); i++){
                    if(!isClient || super.containsKey(entry.getKey())){
                        if(updateListeners.get(i).onPutBefore(this, true, entry.getKey(), entry.getValue())){
                            tmpMap.put((K)entry.getKey(), entry.getValue());
                        }
                    }
                }
            }
            if(tmpMap.size() == 0){
                return;
            }
            t = tmpMap;
        }
        Iterator<?> entries = t.entrySet().iterator();
        while(entries.hasNext()){
            @SuppressWarnings("unchecked")
            Map.Entry<K,Object> entry = (Map.Entry<K,Object>)entries.next();
            if(isClient){
                if(super.containsKey(entry.getKey())){
                    Object old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                    old = unwrapCachedReference(old, false, true);
                    if(updateListeners != null){
                        for(int i = 0; i < updateListeners.size(); i++){
                            updateListeners.get(i).onPutAfter(this, true, entry.getKey(), entry.getValue(), old);
                        }
                    }
                }
            }else{
                final boolean isContainsKey = super.containsKey(entry.getKey());
                Object old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                old = unwrapCachedReference(old, false, true);
                if(isContainsKey){
                    removeAllKeyIndex(entry.getKey(), old);
                }
                addAllKeyIndex(entry.getKey(), entry.getValue());
                if(updateListeners != null){
                    for(int i = 0; i < updateListeners.size(); i++){
                        updateListeners.get(i).onPutAfter(this, true, entry.getKey(), entry.getValue(), old);
                    }
                }
            }
        }
    }
    
    @Override
    public synchronized void putAllAsynch(Map<? extends K,? extends Object> t) throws SharedContextSendException{
        if(t.size() == 0){
            return;
        }
        if(updateListeners != null){
            Map<K,Object> tmpMap = new LinkedHashMap<K,Object>();
            Iterator<?> entries = t.entrySet().iterator();
            while(entries.hasNext()){
                @SuppressWarnings("unchecked")
                Map.Entry<K,Object> entry = (Map.Entry<K,Object>)entries.next();
                for(int i = 0; i < updateListeners.size(); i++){
                    if(!isClient || super.containsKey(entry.getKey())){
                        if(updateListeners.get(i).onPutBefore(this, true, entry.getKey(), entry.getValue())){
                            tmpMap.put(entry.getKey(), entry.getValue());
                        }
                    }
                }
            }
            if(tmpMap.size() == 0){
                return;
            }
            t = tmpMap;
        }
        try{
            Message message = serverConnection.createMessage(subject, null);
            message.setSubject(clientSubject, null);
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_PUT_ALL, null, t));
                serverConnection.sendAsynch(message);
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        Iterator<?> entries = t.entrySet().iterator();
        while(entries.hasNext()){
            @SuppressWarnings("unchecked")
            Map.Entry<? extends K,? extends Object> entry = (java.util.Map.Entry<? extends K, ? extends Object>)entries.next();
            if(isClient){
                if(super.containsKey(entry.getKey())){
                    Object old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                    old = unwrapCachedReference(old, false, true);
                    if(updateListeners != null){
                        for(int i = 0; i < updateListeners.size(); i++){
                            updateListeners.get(i).onPutAfter(this, true, entry.getKey(), entry.getValue(), old);
                        }
                    }
                }
            }else{
                final boolean isContainsKey = super.containsKey(entry.getKey());
                Object old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                old = unwrapCachedReference(old, false, true);
                if(isContainsKey){
                    removeAllKeyIndex(entry.getKey(), old);
                }
                addAllKeyIndex(entry.getKey(), entry.getValue());
                if(updateListeners != null){
                    for(int i = 0; i < updateListeners.size(); i++){
                        updateListeners.get(i).onPutAfter(this, true, entry.getKey(), entry.getValue(), old);
                    }
                }
            }
        }
    }
    
    @Override
    public void clear() throws SharedContextSendException{
        clear(defaultTimeout);
    }
    
    @Override
    public synchronized void clear(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(isMain() && size() == 0){
            return;
        }
        try{
            Message message = serverConnection.createMessage(subject, null);
            message.setSubject(clientSubject, null);
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_CLEAR));
                Message[] responses = serverConnection.request(
                    message,
                    isClient ? clientSubject : subject,
                    null,
                    0,
                    timeout
                );
                if(responses == null || responses.length < receiveClients.size()){
                    throw new SharedContextTimeoutException();
                }
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        Object[] keys = null;
        synchronized(context){
            keys = super.keySet().toArray();
        }
        for(int i = 0; i < keys.length; i++){
            if(updateListeners != null){
                boolean isRemove = true;
                for(int j = 0; j < updateListeners.size(); j++){
                    if(!updateListeners.get(j).onRemoveBefore(this, true, keys[i])){
                        isRemove = false;
                        break;
                    }
                }
                if(!isRemove){
                    continue;
                }
            }
            Object removed = super.remove(keys[i]);
            removed = unwrapCachedReference(removed, false, true);
            if(!isClient){
                removeAllKeyIndex(keys[i], removed);
            }
            if(updateListeners != null){
                for(int j = 0; j < updateListeners.size(); j++){
                    updateListeners.get(j).onRemoveAfter(this, true, keys[i], removed);
                }
            }
        }
    }
    
    @Override
    public synchronized void clearLocal(){
        if(size() == 0){
            return;
        }
        Object[] keys = null;
        synchronized(context){
            keys = super.keySet().toArray();
        }
        for(int i = 0; i < keys.length; i++){
            if(updateListeners != null){
                boolean isRemove = true;
                for(int j = 0; j < updateListeners.size(); j++){
                    if(!updateListeners.get(j).onRemoveBefore(this, true, keys[i])){
                        isRemove = false;
                        break;
                    }
                }
                if(!isRemove){
                    continue;
                }
            }
            Object removed = super.remove(keys[i]);
            removed = unwrapCachedReference(removed, false, true);
            if(!isClient){
                removeAllKeyIndex(keys[i], removed);
            }
            if(updateListeners != null){
                for(int j = 0; j < updateListeners.size(); j++){
                    updateListeners.get(j).onRemoveAfter(this, true, keys[i], removed);
                }
            }
        }
    }
    
    @Override
    public synchronized void clearAsynch() throws SharedContextSendException{
        if(isMain() && size() == 0){
            return;
        }
        try{
            Message message = serverConnection.createMessage(subject, null);
            message.setSubject(clientSubject, null);
            Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
            if(receiveClients.size() != 0){
                message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_CLEAR));
                serverConnection.sendAsynch(message);
            }else if(isClient){
                throw new NoConnectServerException();
            }
        }catch(MessageException e){
            throw new SharedContextSendException(e);
        }catch(MessageSendException e){
            throw new SharedContextSendException(e);
        }
        Object[] keys = null;
        synchronized(context){
            keys = super.keySet().toArray();
        }
        for(int i = 0; i < keys.length; i++){
            if(updateListeners != null){
                boolean isRemove = true;
                for(int j = 0; j < updateListeners.size(); j++){
                    if(!updateListeners.get(j).onRemoveBefore(this, true, keys[i])){
                        isRemove = false;
                        break;
                    }
                }
                if(!isRemove){
                    continue;
                }
            }
            Object removed = super.remove(keys[i]);
            removed = unwrapCachedReference(removed, false, true);
            if(!isClient){
                removeAllKeyIndex(keys[i], removed);
            }
            if(updateListeners != null){
                for(int j = 0; j < updateListeners.size(); j++){
                    updateListeners.get(j).onRemoveAfter(this, true, keys[i], removed);
                }
            }
        }
    }
    
    @Override
    public Object get(Object key) throws SharedContextSendException{
        return get(key, defaultTimeout);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Object get(Object key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        Object result = null;
        if(isClient){
            if(super.containsKey(key)){
                result = getLocal(key);
                if(result == null && !super.containsKey(key)){
                    result = get(key, timeout);
                }
            }else{
                ClientCacheLock lock = null;
                ClientCacheLock newLock = null;
                synchronized(clientCacheLockMap){
                    lock = clientCacheLockMap.get(key);
                    if(lock == null){
                        newLock = new ClientCacheLock(key);
                        clientCacheLockMap.put(key, newLock);
                    }else{
                        lock.init();
                    }
                }
                if(lock != null){
                    final long start = System.currentTimeMillis();
                    if(!lock.waitLock(timeout)){
                        throw new SharedContextTimeoutException();
                    }
                    final boolean isNoTimeout = timeout <= 0;
                    timeout = isNoTimeout ? timeout : timeout - (System.currentTimeMillis() - start);
                    if(!isNoTimeout && timeout <= 0){
                        throw new SharedContextTimeoutException();
                    }
                    result = get(key, timeout);
                }else{
                    try{
                        Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
                        Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                        if(receiveClients.size() != 0){
                            message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_GET, key));
                            Message[] responses = serverConnection.request(
                                message,
                                clientSubject,
                                key == null ? null : key.toString(),
                                1,
                                timeout
                            );
                            if(responses != null && responses.length > 0){
                                result = responses[0].getObject();
                                synchronized(newLock){
                                    if(!newLock.isRemove()){
                                        result = newLock.updateValue(result);
                                        super.put((K)key, wrapCachedReference((K)key, result));
                                    }
                                }
                            }else{
                                throw new SharedContextTimeoutException();
                            }
                        }else{
                            throw new NoConnectServerException();
                        }
                    }catch(MessageException e){
                        throw new SharedContextSendException(e);
                    }catch(MessageSendException e){
                        throw new SharedContextSendException(e);
                    }finally{
                        synchronized(clientCacheLockMap){
                            clientCacheLockMap.remove(key);
                            newLock.notifyAllLock();
                        }
                    }
                }
            }
        }else{
            result = getLocal(key);
        }
        return result;
    }
    
    @Override
    public Object getLocal(Object key){
        Object result = getRawLocal(key);
        result = unwrapCachedReference(result, true, false);
        return result;
    }
    
    private Object getRawLocal(Object key){
        Object result = super.get(key);
        if(result == null && !isClient && contextStore != null && super.containsKey(key)){
            try{
                contextStore.load(this, key);
            }catch(Exception e){
                getLogger().write("SCS__00001", e, key, subject);
            }
            result = super.get(key);
        }
        return result;
    }
    
    @Override
    public Set<K> searchByKeyIndex(String name, Object key) throws SharedContextIllegalIndexException, SharedContextSendException, SharedContextTimeoutException{
        return searchByKeyIndex(name, key, defaultTimeout);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Set<K> searchByKeyIndex(String name, Object key, long timeout) throws SharedContextIllegalIndexException, SharedContextSendException, SharedContextTimeoutException{
        Set<K> result = null;
        if(isClient){
            try{
                Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_SEARCH_KEY_INDEX, name, key));
                    Message[] responses = serverConnection.request(
                        message,
                        clientSubject,
                        name,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        result = (Set<K>)responses[0].getObject();
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException();
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
            
        }else{
            if(keyIndexMap == null || !keyIndexMap.containsKey(name)){
                throw new SharedContextIllegalIndexException("No such keyIndex." + name);
            }
            result = new HashSet<K>();
            Map<IndexKey,Set<K>> indexMap = keyIndexResultMap.get(name);
            if(indexMap == null){
                return result;
            }
            Property[] keyProps = keyIndexMap.get(name);
            Object[] keys = new Object[keyProps.length];
            for(int i = 0; i < keyProps.length; i++){
                try{
                    keys[i] = keyProps[i].getProperty(key);
                }catch(NoSuchPropertyException e){
                    throw new SharedContextIllegalIndexException(e);
                }catch(InvocationTargetException e){
                    throw new SharedContextIllegalIndexException(e);
                }
            }
            IndexKey indexKey = new IndexKey(keys);
            Set<K> keySet = indexMap.get(indexKey);
            if(keySet == null){
                return result;
            }
            result.addAll(keySet);
        }
        return result;
    }
    
    @Override
    public Set<K> searchByKeyIndexLocal(String name, Object key) throws SharedContextIllegalIndexException{
        if(keyIndexMap == null || !keyIndexMap.containsKey(name)){
            throw new SharedContextIllegalIndexException("No such keyIndex." + name);
        }
        Set<K> result = new HashSet<K>();
        Map<IndexKey,Set<K>> indexMap = keyIndexResultMap.get(name);
        if(indexMap == null){
            return result;
        }
        Property[] keyProps = keyIndexMap.get(name);
        Object[] keys = new Object[keyProps.length];
        for(int i = 0; i < keyProps.length; i++){
            try{
                keys[i] = keyProps[i].getProperty(key);
            }catch(NoSuchPropertyException e){
                throw new SharedContextIllegalIndexException(e);
            }catch(InvocationTargetException e){
                throw new SharedContextIllegalIndexException(e);
            }
        }
        IndexKey indexKey = new IndexKey(keys);
        Set<K> keySet = indexMap.get(indexKey);
        if(keySet == null){
            return result;
        }
        result.addAll(keySet);
        result.retainAll(super.keySet());
        return result;
    }
    
    @Override
    public Set<K> keySet() throws SharedContextSendException{
        return keySet(defaultTimeout);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public synchronized Set<K> keySet(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(!isClient){
            return keySetLocal();
        }else{
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_KEY_SET));
                    Message[] responses = serverConnection.request(
                        message,
                        clientSubject,
                        null,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        return (Set<K>)responses[0].getObject();
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException();
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    
    @Override
    public synchronized Set<K> keySetLocal(){
        return super.keySet();
    }
    
    @Override
    public int size() throws SharedContextSendException{
        return size(defaultTimeout);
    }
    
    @Override
    public synchronized int size(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(!isClient){
            return sizeLocal();
        }else{
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_SIZE));
                    Message[] responses = serverConnection.request(
                        message,
                        clientSubject,
                        null,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        return ((Integer)responses[0].getObject()).intValue();
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    return sizeLocal();
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    
    @Override
    public synchronized int sizeLocal(){
        return super.size();
    }
    
    @Override
    public boolean isEmpty() throws SharedContextSendException{
        return size() == 0 ? true : false;
    }
    
    @Override
    public boolean isEmptyLocal(){
        return super.isEmpty();
    }
    
    @Override
    public boolean containsKey(Object key) throws SharedContextSendException{
        return containsKey(key, defaultTimeout);
    }
    
    @Override
    public boolean containsKey(Object key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(!isClient){
            return containsKeyLocal(key);
        }else{
            try{
                Message message = serverConnection.createMessage(subject, key == null ? null : key.toString());
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_CONTAINS_KEY, key));
                    Message[] responses = serverConnection.request(
                        message,
                        clientSubject,
                        key == null ? null : key.toString(),
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        return ((Boolean)responses[0].getObject()).booleanValue();
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException();
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    
    @Override
    public boolean containsKeyLocal(Object key){
        return super.containsKey(key);
    }
    
    @Override
    public boolean containsValue(Object value) throws SharedContextSendException{
        return containsValue(value, defaultTimeout);
    }
    
    @Override
    public boolean containsValue(Object value, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
        if(!isClient){
            return containsValueLocal(value);
        }else{
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_CONTAINS_VALUE, null, value));
                    Message[] responses = serverConnection.request(
                        message,
                        clientSubject,
                        null,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length > 0){
                        return ((Boolean)responses[0].getObject()).booleanValue();
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException();
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
    }
    
    @Override
    public boolean containsValueLocal(Object value){
        if(cacheMap == null){
            return super.containsValue(value);
        }else{
            Object[] keys = null;
            synchronized(context){
                keys = super.keySet().toArray();
            }
            for(int i = 0; i < keys.length; i++){
                Object val = cacheMap.get(keys[i]);
                if(val == null){
                    if(value == null){
                        return true;
                    }
                }else if(val.equals(value)){
                    return true;
                }
            }
            return false;
        }
    }
    
    @Override
    public Collection<Object> values(){
        if(!isClient){
            return valuesLocal();
        }else{
            throw new UnsupportedOperationException();
        }
    }
    
    @Override
    public Collection<Object> valuesLocal(){
        if(cacheMap == null){
            return super.values();
        }else{
            List<Object> result = new ArrayList<Object>();
            Object[] keys = null;
            synchronized(context){
                keys = super.keySet().toArray();
            }
            for(int i = 0; i < keys.length; i++){
                result.add(cacheMap.get(keys[i]));
            }
            return result;
        }
    }
    
    @Override
    public Map<K, Object> all(){
        if(!isClient){
            return allLocal();
        }else{
            throw new UnsupportedOperationException();
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Map<K, Object> allLocal(){
        if(cacheMap == null){
            return super.all();
        }else{
            Map<K, Object> result = new HashMap<K, Object>();
            Object[] keys = null;
            synchronized(context){
                keys = super.keySet().toArray();
            }
            for(int i = 0; i < keys.length; i++){
                result.put((K)keys[i], cacheMap.get(keys[i]));
            }
            return result;
        }
    }
    
    @Override
    public Set<Map.Entry<K, Object>> entrySet(){
        if(!isClient){
            return entrySetLocal();
        }else{
            throw new UnsupportedOperationException();
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Set<Map.Entry<K, Object>> entrySetLocal(){
        if(cacheMap == null){
            return super.entrySet();
        }else{
            Set<Map.Entry<K, Object>> reuslt = new HashSet<Map.Entry<K, Object>>();
            Map.Entry<K, Object>[] entries = null;
            synchronized(context){
                entries = cacheMap.entrySet().toArray(new Map.Entry[cacheMap.size()]);
            }
            for(int i = 0; i < entries.length; i++){
                if(super.containsKey(entries[i].getKey())){
                    reuslt.add(entries[i]);
                }
            }
            return reuslt;
        }
    }
    
    @Override
    public boolean isMain(){
        return isMain(cluster.getMembers());
    }
    
    protected boolean isMain(List<? extends Object> members){
        if(isClient){
            return false;
        }else{
            Set<Object> targetMembers = serverConnection.getReceiveClientIds(targetMessage);
            Object myId = cluster.getUID();
            for(int i = 0, imax = members.size(); i < imax; i++){
                Object id = members.get(i);
                if(id.equals(myId)){
                    return true;
                }else if(targetMembers.contains(id)){
                    return false;
                }
            }
            return true;
        }
    }
    
    @Override
    public Object getId(){
        return cluster == null ? null :  cluster.getUID();
    }
    
    @Override
    public Object getMainId(){
        if(cluster == null){
            return null;
        }else{
            Object myId = cluster.getUID();
            List<? extends Object> members = cluster.getMembers();
            Set<Object> targetMembers = serverConnection.getReceiveClientIds(targetMessage);
            for(int i = 0, imax = members.size(); i < imax; i++){
                Object id = members.get(i);
                if(id.equals(myId)){
                    return myId;
                }else if(targetMembers.contains(id)){
                    return id;
                }
            }
            return myId;
        }
    }
    
    @Override
    public Object executeInterpretQuery(String query, Map<String, Object> variables) throws EvaluateException, SharedContextSendException, SharedContextTimeoutException{
        return executeInterpretQuery(query, variables, defaultTimeout);
    }
    
    @Override
    public Object executeInterpretQuery(String query, Map<String, Object> variables, long timeout) throws EvaluateException, SharedContextSendException, SharedContextTimeoutException{
        Object result = null;
        if(!isClient && isMain()){
            result = executeInterpretQueryLocal(query, variables);
        }else{
            try{
                Message message = serverConnection.createMessage(subject, null);
                Set<Object> receiveClients = serverConnection.getReceiveClientIds(message);
                if(receiveClients.size() != 0){
                    message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_EXECUTE_INTERPRET, query, variables));
                    Message[] responses = serverConnection.request(
                        message,
                        isClient ? clientSubject : subject,
                        null,
                        1,
                        timeout
                    );
                    if(responses != null && responses.length != 0){
                        result = responses[0].getObject();
                        if(result != null){
                            if(result instanceof Throwable){
                                throw new SharedContextSendException((Throwable)result);
                            }
                        }
                    }else{
                        throw new SharedContextTimeoutException();
                    }
                }else{
                    throw new NoConnectServerException();
                }
            }catch(MessageException e){
                throw new SharedContextSendException(e);
            }catch(MessageSendException e){
                throw new SharedContextSendException(e);
            }
        }
        return result;
    }
    
    protected Object executeInterpretQueryLocal(String evaluate, Map<String, Object> variables) throws EvaluateException{
        if(interpreter == null){
            throw new EvaluateException("Interpreter is null.");
        }
        if(variables == null){
            variables = new HashMap<String, Object>();
        }
        variables.put("context", new LocalSharedContext());
        return interpreter.evaluate(evaluate, variables);
    }
    
    @Override
    public void addSharedContextUpdateListener(SharedContextUpdateListener<K> listener){
        if(updateListeners == null){
            updateListeners = Collections.synchronizedList(new ArrayList<SharedContextUpdateListener<K>>());
        }
        if(!updateListeners.contains(listener)){
            updateListeners.add(listener);
        }
    }
    
    @Override
    public void removeSharedContextUpdateListener(SharedContextUpdateListener<K> listener){
        if(updateListeners == null){
            return;
        }
        updateListeners.remove(listener);
    }
    
    @Override
    public void memberInit(Object myId, List<? extends Object> members){
    }
    
    @Override
    public void memberChange(List<? extends Object> oldMembers, List<? extends Object> newMembers){
        if(isMain(newMembers)){
            Set<Object> deadMembers = new HashSet<Object>(oldMembers);
            deadMembers.removeAll(newMembers);
            if(deadMembers.size() != 0){
                Iterator<Object> ids = deadMembers.iterator();
                while(ids.hasNext()){
                    Object id = ids.next();
                    Set<K> keySet = idLocksMap.remove(id);
                    if(keySet == null || keySet.size() == 0){
                        continue;
                    }
                    synchronized(keySet){
                        Iterator<K> keys = keySet.iterator();
                        while(keys.hasNext()){
                            try{
                                unlock(keys.next(), true);
                            }catch(SharedContextSendException e){}
                        }
                    }
                }
            }
        }
    }
    
    @Override
    public void changeMain() throws Exception{
    }
    
    @Override
    public void changeSub(){
    }
    
    @Override
    public void removed(CachedReference<Object> ref){
        if(ref == null){
            return;
        }
        @SuppressWarnings("unchecked")
        KeyCachedReference<K, Object> kcr = (KeyCachedReference<K, Object>)ref;
        Object removed = isClient ? super.remove(kcr.getKey()) : super.put(kcr.getKey(), null);
        removed = unwrapCachedReference(removed, false, true);
        if(isClient){
            removeAllKeyIndex(kcr.getKey(), removed);
        }
    }
    
    @Override
    public void onMessage(Message message){
        SharedContextEvent event = null;
        try{
            event = (SharedContextEvent)message.getObject();
        }catch(MessageException e){
            e.printStackTrace();
            return;
        }
        switch(event.type){
        case SharedContextEvent.EVENT_PUT:
            onPut(event, null, null);
            break;
        case SharedContextEvent.EVENT_PUT_ALL:
            onPutAll(event, null, null);
            break;
        case SharedContextEvent.EVENT_REMOVE:
            onRemove(event, null, null);
            break;
        case SharedContextEvent.EVENT_CLEAR:
            onClear(event, null, null);
            break;
        case SharedContextEvent.EVENT_RELEASE_LOCK:
            onReleaseLock(event, null, null);
            break;
        case SharedContextEvent.EVENT_UPDATE:
            onUpdate(event, null, null);
            break;
        case SharedContextEvent.EVENT_PUT_INNER:
            onPutInner(event);
            break;
        default:
        }
    }
    
    @Override
    public Message onRequestMessage(Object sourceId, int sequence, Message message, String responseSubject, String responseKey){
        SharedContextEvent event = null;
        try{
            event = (SharedContextEvent)message.getObject();
        }catch(MessageException e){
            e.printStackTrace();
            return null;
        }
        Message result = null;
        switch(event.type){
        case SharedContextEvent.EVENT_PUT:
            result = onPut(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_PUT_ALL:
            result = onPutAll(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_REMOVE:
            result = onRemove(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_CLEAR:
            result = onClear(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_GET:
            result = onGet(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_GET_ALL:
            result = onGetAll(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_KEY_SET:
            result = onKeySet(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_SIZE:
            result = onSize(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_CONTAINS_KEY:
            result = onContainsKey(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_CONTAINS_VALUE:
            result = onContainsValue(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_SYNCH_ALL:
            result = onSynchronizeAll(event, sourceId, sequence, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_SYNCH:
            result = onSynchronize(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_GET_LOCK:
            result = onGetLock(event, sourceId, sequence, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_GOT_LOCK:
            result = onGotLock(event, sourceId, sequence, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_SAVE:
            result = onSave(event, sourceId, sequence, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_LOAD:
            result = onLoad(event, sourceId, sequence, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_LOAD_KEY:
            result = onLoadKey(event, sourceId, sequence, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_UPDATE:
            result = onUpdate(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_ANALYZE_KEY_INDEX:
            result = onAnalyzeKeyIndex(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_SEARCH_KEY_INDEX:
            result = onSearchKeyIndex(event, responseSubject, responseKey);
            break;
        case SharedContextEvent.EVENT_EXECUTE_INTERPRET:
            onExecuteInterpret(event, sourceId, sequence, responseSubject, responseKey);
            break;
        default:
        }
        return result;
    }
    
    protected Message createResponseMessage(String responseSubject, String responseKey, Object response){
        Message result = null;
        try{
            result = serverConnection.createMessage(responseSubject, responseKey);
            result.setObject(response);
        }catch(MessageException e){
            getLogger().write("SCS__00002", e, isClient ? clientSubject : subject, responseSubject, responseKey, response);
            result = null;
        }
        return result;
    }
    
    @SuppressWarnings("unchecked")
    protected Message onPut(SharedContextEvent event, String responseSubject, String responseKey){
        Object result = null;
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onPutBefore(this, false, (K)event.key, event.value)){
                    return createResponseMessage(responseSubject, responseKey, null);
                }
            }
        }
        Object old = null;
        if(isClient){
            if(super.containsKey(event.key)){
                old = super.put((K)event.key, wrapCachedReference((K)event.key, event.value));
                old = unwrapCachedReference(old, false, true);
            }else if(clientCacheLockMap.containsKey(event.key)){
                ClientCacheLock lock = null;
                synchronized(clientCacheLockMap){
                    lock = clientCacheLockMap.get(event.key);
                }
                if(lock != null){
                    synchronized(lock){
                        if(super.containsKey(event.key)){
                            old = super.put((K)event.key, wrapCachedReference((K)event.key, event.value));
                            old = unwrapCachedReference(old, false, true);
                        }else{
                            lock.put(event.value);
                        }
                    }
                }
            }
        }else{
            final boolean isContainsKey = super.containsKey(event.key);
            old = super.put((K)event.key, wrapCachedReference((K)event.key, event.value));
            old = unwrapCachedReference(old, false, true);
            if(isContainsKey){
                removeAllKeyIndex(event.key, old);
            }
            addAllKeyIndex((K)event.key, event.value);
            if(isMain()){
                result = old;
            }
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onPutAfter(this, false, (K)event.key, event.value, old);
            }
        }
        return createResponseMessage(responseSubject, responseKey, result);
    }
    
    protected Message onPutAll(SharedContextEvent event, String responseSubject, String responseKey){
        @SuppressWarnings("unchecked")
        Map<K, Object> map = (Map<K, Object>)event.key;
        if(map != null){
            Iterator<Map.Entry<K, Object>> entries = map.entrySet().iterator();
            while(entries.hasNext()){
                Map.Entry<K, Object> entry = entries.next();
                boolean isPut = true;
                if(updateListeners != null){
                    for(int i = 0; i < updateListeners.size(); i++){
                        if(!updateListeners.get(i).onPutBefore(this, false, entry.getKey(), entry.getValue())){
                            isPut = false;
                            break;
                        }
                    }
                }
                if(!isPut){
                    continue;
                }
                Object old = null;
                if(isClient){
                    if(super.containsKey(entry.getKey())){
                        old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                        old = unwrapCachedReference(old, false, true);
                    }else if(clientCacheLockMap.containsKey(entry.getKey())){
                        ClientCacheLock lock = null;
                        synchronized(clientCacheLockMap){
                            lock = clientCacheLockMap.get(entry.getKey());
                        }
                        if(lock != null){
                            synchronized(lock){
                                if(super.containsKey(entry.getKey())){
                                    old = super.put((K)entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                                    old = unwrapCachedReference(old, false, true);
                                }else{
                                    lock.put(entry.getValue());
                                }
                            }
                        }
                    }
                }else{
                    final boolean isContainsKey = super.containsKey(entry.getKey());
                    old = super.put(entry.getKey(), wrapCachedReference(entry.getKey(), entry.getValue()));
                    old = unwrapCachedReference(old, false, true);
                    if(isContainsKey){
                        removeAllKeyIndex(entry.getKey(), old);
                    }
                    addAllKeyIndex(entry.getKey(), entry.getValue());
                }
                if(updateListeners != null){
                    for(int i = 0; i < updateListeners.size(); i++){
                        updateListeners.get(i).onPutAfter(this, false, entry.getKey(), entry.getValue(), old);
                    }
                }
            }
        }
        return createResponseMessage(responseSubject, responseKey, null);
    }
    
    @SuppressWarnings("unchecked")
    protected void onPutInner(SharedContextEvent event){
        Object old = null;
        if(isClient){
            if(super.containsKey(event.key)){
                old = super.put((K)event.key, wrapCachedReference((K)event.key, event.value));
                old = unwrapCachedReference(old, false, true);
            }else if(clientCacheLockMap.containsKey(event.key)){
                ClientCacheLock lock = null;
                synchronized(clientCacheLockMap){
                    lock = clientCacheLockMap.get(event.key);
                }
                if(lock != null){
                    synchronized(lock){
                        if(super.containsKey(event.key)){
                            old = super.put((K)event.key, wrapCachedReference((K)event.key, event.value));
                            old = unwrapCachedReference(old, false, true);
                        }else{
                            lock.put(event.value);
                        }
                    }
                }
            }
        }else{
            final boolean isContainsKey = super.containsKey(event.key);
            old = super.put((K)event.key, wrapCachedReference((K)event.key, event.value));
            old = unwrapCachedReference(old, false, true);
            if(isContainsKey){
                removeAllKeyIndex(event.key, old);
            }
            addAllKeyIndex((K)event.key, event.value);
        }
    }
    
    @SuppressWarnings("unchecked")
    protected Message onUpdate(final SharedContextEvent event, String responseSubject, String responseKey){
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onUpdateBefore(this, false, (K)event.key, (SharedContextValueDifference)event.value)){
                    return createResponseMessage(responseSubject, responseKey, null);
                }
            }
        }
        if(!isClient || super.containsKey(event.key)){
            Object current = getRawLocal(event.key);
            Object currentValue = unwrapCachedReference(current, false, false);
            if(currentValue == null){
                if(isClient){
                    Object removed = super.remove(event.key);
                    removed = unwrapCachedReference(removed, false, true);
                }else{
                    return createResponseMessage(responseSubject, responseKey, new SharedContextUpdateException("Current value is null. key=" + event.key));
                }
            }else if(currentValue instanceof SharedContextValueDifferenceSupport){
                try{
                    boolean isSuccess = false;
                    try{
                        isSuccess = ((SharedContextValueDifferenceSupport)currentValue).update((SharedContextValueDifference)event.value);
                        if(!isSuccess){
                            if(isClient){
                                getLogger().write("SCS__00003", clientSubject, event.key);
                            }else if(!isMain()){
                                getLogger().write("SCS__00004", subject, event.key);
                            }else{
                                getLogger().write("SCS__00005", subject, event.key);
                            }
                        }
                    }catch(SharedContextUpdateException e){
                        if(isClient){
                            getLogger().write("SCS__00003", e, clientSubject, event.key);
                        }else if(!isMain()){
                            getLogger().write("SCS__00004", e, subject, event.key);
                        }else{
                            getLogger().write("SCS__00005", e, subject, event.key);
                        }
                    }
                    if(isSuccess){
                        if(current instanceof CachedReference){
                            try{
                                ((CachedReference<Object>)current).set(this, currentValue);
                            }catch(IllegalCachedReferenceException e){
                                throw new SharedContextUpdateException(e);
                            }
                        }
                    }else{
                        if(isClient){
                            Object removed = super.remove(event.key);
                            removed = unwrapCachedReference(removed, false, true);
                        }else if(!isMain()){
                            Message message = serverConnection.createMessage(subject, event.key == null ? null : event.key.toString());
                            message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_SYNCH, event.key));
                            serverConnection.request(
                                message,
                                isClient ? clientSubject : subject,
                                event.key == null ? null : event.key.toString(),
                                1,
                                0,
                                new RequestServerConnection.ResponseCallBack(){
                                    public void onResponse(Message message, boolean isLast){
                                        onMessage(message);
                                        if(SharedContextService.this.updateListeners != null){
                                            for(int i = 0; i < SharedContextService.this.updateListeners.size(); i++){
                                                SharedContextService.this.updateListeners.get(i).onUpdateAfter(
                                                    SharedContextService.this,
                                                    false,
                                                    (K)event.key,
                                                    (SharedContextValueDifference)event.value
                                                );
                                            }
                                        }
                                    }
                                }
                            );
                            return createResponseMessage(responseSubject, responseKey, null);
                        }else{
                            throw new SharedContextUpdateException(
                                "An update version is mismatching. currentVersion="
                                    + ((SharedContextValueDifferenceSupport)currentValue).getUpdateVersion()
                                    + ", updateVersion=" + ((SharedContextValueDifference)event.value).getUpdateVersion()
                            );
                        }
                    }
                }catch(Throwable th){
                    getLogger().write("SCS__00005", th, isClient ? clientSubject : subject, event.key);
                    return createResponseMessage(responseSubject, responseKey, th);
                }
            }else{
                SharedContextUpdateException e = new SharedContextUpdateException("Not support SharedContextValueDifference. key=" + event.key + ", value=" + currentValue);
                getLogger().write("SCS__00005", e, isClient ? clientSubject : subject, event.key);
                return createResponseMessage(responseSubject, responseKey, e);
            }
        }else if(isClient && clientCacheLockMap.containsKey(event.key)){
            ClientCacheLock lock = null;
            synchronized(clientCacheLockMap){
                lock = clientCacheLockMap.get(event.key);
            }
            if(lock != null){
                synchronized(lock){
                    if(super.containsKey(event.key)){
                        Object current = getRawLocal(event.key);
                        Object currentValue = unwrapCachedReference(current, false, false);
                        if(currentValue == null){
                            Object removed = super.remove(event.key);
                            removed = unwrapCachedReference(removed, false, true);
                        }else if(currentValue instanceof SharedContextValueDifferenceSupport){
                            try{
                                boolean isSuccess = false;
                                try{
                                    isSuccess = ((SharedContextValueDifferenceSupport)currentValue).update((SharedContextValueDifference)event.value);
                                    if(!isSuccess){
                                        getLogger().write("SCS__00003", clientSubject, event.key);
                                    }
                                }catch(SharedContextUpdateException e){
                                    getLogger().write("SCS__00003", e, clientSubject, event.key);
                                }
                                if(isSuccess){
                                    if(current instanceof CachedReference){
                                        try{
                                            ((CachedReference<Object>)current).set(this, currentValue);
                                        }catch(IllegalCachedReferenceException e){
                                            throw new SharedContextUpdateException(e);
                                        }
                                    }
                                }else{
                                    Object removed = super.remove(event.key);
                                    removed = unwrapCachedReference(removed, false, true);
                                }
                            }catch(Throwable th){
                                getLogger().write("SCS__00005", th , isClient ? clientSubject : subject, event.key);
                                return createResponseMessage(responseSubject, responseKey, th);
                            }
                        }else{
                            SharedContextUpdateException e = new SharedContextUpdateException("Not support SharedContextValueDifference. key=" + event.key + ", value=" + currentValue);
                            getLogger().write("SCS__00005", e, isClient ? clientSubject : subject, event.key);
                            return createResponseMessage(responseSubject, responseKey, e);
                        }
                    }else{
                        lock.update((SharedContextValueDifference)event.value);
                    }
                }
            }
        }
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onUpdateAfter(this, false, (K)event.key, (SharedContextValueDifference)event.value);
            }
        }
        return createResponseMessage(responseSubject, responseKey, null);
    }
    
    protected Message onRemove(SharedContextEvent event, String responseSubject, String responseKey){
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                if(!updateListeners.get(i).onRemoveBefore(this, false, event.key)){
                    return createResponseMessage(responseSubject, responseKey, null);
                }
            }
        }
        if(isClient && clientCacheLockMap.containsKey(event.key)){
            ClientCacheLock lock = null;
            synchronized(clientCacheLockMap){
                lock = clientCacheLockMap.get(event.key);
            }
            if(lock != null){
                synchronized(lock){
                    lock.remove();
                }
            }
        }
        Object removed = super.remove(event.key);
        removed = unwrapCachedReference(removed, false, true);
        removeAllKeyIndex(event.key, removed);
        if(updateListeners != null){
            for(int i = 0; i < updateListeners.size(); i++){
                updateListeners.get(i).onRemoveAfter(this, false, event.key, removed);
            }
        }
        return createResponseMessage(responseSubject, responseKey, isMain() ? removed : null);
    }
    
    protected Message onClear(SharedContextEvent event, String responseSubject, String responseKey){
        Object[] keys = null;
        synchronized(context){
            keys = super.keySet().toArray();
        }
        for(int i = 0; i < keys.length; i++){
            if(updateListeners != null){
                boolean isRemove = true;
                for(int j = 0; j < updateListeners.size(); j++){
                    if(!updateListeners.get(j).onRemoveBefore(this, false, keys[i])){
                        isRemove = false;
                        break;
                    }
                }
                if(!isRemove){
                    continue;
                }
            }
            if(isClient && clientCacheLockMap.containsKey(keys[i])){
                ClientCacheLock lock = null;
                synchronized(clientCacheLockMap){
                    lock = clientCacheLockMap.get(keys[i]);
                }
                if(lock != null){
                    synchronized(lock){
                        lock.remove();
                    }
                }
            }
            Object removed = super.remove(keys[i]);
            removed = unwrapCachedReference(removed, false, true);
            removeAllKeyIndex(keys[i], removed);
            if(updateListeners != null){
                for(int j = 0; j < updateListeners.size(); j++){
                    updateListeners.get(j).onRemoveAfter(this, false, event.key, removed);
                }
            }
        }
        return createResponseMessage(responseSubject, responseKey, null);
    }
    
    protected Message onGet(SharedContextEvent event, String responseSubject, String responseKey){
        if(isMain()){
            return createResponseMessage(responseSubject, responseKey, getLocal(event.key));
        }else{
            return null;
        }
    }
    
    @SuppressWarnings("unchecked")
    protected Message onGetAll(SharedContextEvent event, String responseSubject, String responseKey){
        if(isMain()){
            Map<K, Object> result = new HashMap<K, Object>();
            synchronized(context){
                if(cacheMap == null){
                    result.putAll(context);
                }else{
                    Object[] keys = null;
                    synchronized(context){
                        keys = super.keySet().toArray();
                    }
                    for(int i = 0; i < keys.length; i++){
                        if(cacheMap.containsKey(keys[i])){
                            result.put((K)keys[i], cacheMap.get(keys[i]));
                        }
                    }
                }
            }
            return createResponseMessage(responseSubject, responseKey, result);
        }else{
            return null;
        }
    }
    
    protected Message onKeySet(SharedContextEvent event, String responseSubject, String responseKey){
        if(isMain()){
            Set<K> result = new HashSet<K>();
            synchronized(context){
                result.addAll(context.keySet());
            }
            return createResponseMessage(responseSubject, responseKey, result);
        }else{
            return null;
        }
    }
    
    protected Message onSize(SharedContextEvent event, String responseSubject, String responseKey){
        if(isMain()){
            return createResponseMessage(responseSubject, responseKey, new Integer(size()));
        }else{
            return null;
        }
    }
    
    protected Message onContainsKey(SharedContextEvent event, String responseSubject, String responseKey){
        if(isMain()){
            return createResponseMessage(responseSubject, responseKey, containsKey(event.key) ? Boolean.TRUE : Boolean.FALSE);
        }else{
            return null;
        }
    }
    
    protected Message onContainsValue(SharedContextEvent event, String responseSubject, String responseKey){
        if(isMain()){
            return createResponseMessage(responseSubject, responseKey, containsValue(event.value) ? Boolean.TRUE : Boolean.FALSE);
        }else{
            return null;
        }
    }
    
    protected Message onSynchronizeAll(final SharedContextEvent event, final Object sourceId, final int sequence, final String responseSubject, final String responseKey){
        Thread synchronizeThread = new Thread(){
            public void run(){
                Message response = null;
                try{
                    synchronize(((Long)event.value).longValue());
                    response = createResponseMessage(responseSubject, responseKey, Boolean.TRUE);
                }catch(SharedContextSendException e){
                    response = createResponseMessage(responseSubject, responseKey, Boolean.FALSE);
                }catch(SharedContextTimeoutException e){
                    response = createResponseMessage(responseSubject, responseKey, Boolean.FALSE);
                }
                try{
                    serverConnection.response(sourceId, sequence, response);
                }catch(MessageSendException e){
                    getLogger().write("SCS__00006", e, isClient ? clientSubject : subject, response);
                }
            }
        };
        synchronizeThread.start();
        return null;
    }
    
    protected Message onSynchronize(SharedContextEvent event, String responseSubject, String responseKey){
        if(isClient || !isMain()){
            return null;
        }else{
            if(containsKeyLocal(event.key)){
                return createResponseMessage(responseSubject, responseKey, new SharedContextEvent(SharedContextEvent.EVENT_PUT_INNER, event.key, getLocal(event.key)));
            }else{
                return createResponseMessage(responseSubject, responseKey, new SharedContextEvent(SharedContextEvent.EVENT_REMOVE, event.key));
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    protected Message onGetLock(SharedContextEvent event, Object sourceId, int sequence, String responseSubject, String responseKey){
        if(isMain()){
            Object result = null;
            Lock lock = null;
            synchronized(keyLockMap){
                lock = keyLockMap.get(event.key);
                if(lock == null){
                    lock = new Lock((K)event.key);
                    keyLockMap.put(event.key, lock);
                }
            }
            long timeout = ((Long)event.value).longValue();
            final long start = System.currentTimeMillis();
            if(lock.acquireForReply(sourceId, timeout, sourceId, sequence, responseSubject, responseKey)){
                final boolean isNoTimeout = timeout <= 0;
                timeout = isNoTimeout ? timeout : timeout - (System.currentTimeMillis() - start);
                if(!isNoTimeout && timeout <= 0){
                    lock.release(sourceId, false);
                    result = Boolean.FALSE;
                }else{
                    try{
                        Message message = serverConnection.createMessage(subject, event.key == null ? null : event.key.toString());
                        message.setSubject(clientSubject, event.key == null ? null : event.key.toString());
                        Set<Object> receiveClients =  serverConnection.getReceiveClientIds(message);
                        receiveClients.remove(sourceId);
                        result = Boolean.TRUE;
                        if(receiveClients.size() != 0){
                            message.setDestinationIds(receiveClients);
                            message.setObject(new SharedContextEvent(SharedContextEvent.EVENT_GOT_LOCK, event.key, new Object[]{sourceId, new Long(timeout)}));
                            Message[] responses = serverConnection.request(message, 0, timeout);
                            if(responses != null && responses.length >= receiveClients.size()){
                                for(int i = 0; i < responses.length; i++){
                                    Object ret = responses[i].getObject();
                                    if(ret == null
                                        || ret instanceof Throwable
                                        || !((Boolean)ret).booleanValue()
                                    ){
                                        unlock((K)event.key);
                                        result = ret;
                                        break;
                                    }
                                }
                            }else{
                                unlock((K)event.key);
                                result = Boolean.FALSE;
                            }
                        }
                    }catch(Throwable th){
                        try{
                            unlock((K)event.key);
                        }catch(SharedContextSendException e){
                            getLogger().write("SCS__00007", e, isClient ? clientSubject : subject, event.key);
                        }
                        result = th;
                    }
                }
            }else{
                return null;
            }
            return createResponseMessage(responseSubject, responseKey, result);
        }else{
            return null;
        }
    }
    
    @SuppressWarnings("unchecked")
    protected Message onGotLock(SharedContextEvent event, Object sourceId, int sequence, String responseSubject, String responseKey){
        Lock lock = null;
        synchronized(keyLockMap){
            lock = keyLockMap.get(event.key);
            if(lock == null){
                lock = new Lock((K)event.key);
                keyLockMap.put(event.key, lock);
            }
        }
        final Object[] params = (Object[])event.value;
        final Object id = params[0];
        final long timeout = ((Long)params[1]).longValue();
        try{
            if(lock.acquireForReply(id, timeout, sourceId, sequence, responseSubject, responseKey)){
                return createResponseMessage(responseSubject, responseKey, Boolean.TRUE);
            }else{
                return null;
            }
        }catch(Throwable th){
            lock.release(id, false);
            return createResponseMessage(responseSubject, responseKey, th);
        }
    }
    
    protected Message onReleaseLock(SharedContextEvent event, String responseSubject, String responseKey){
        Lock lock = null;
        synchronized(keyLockMap){
            lock = keyLockMap.get(event.key);
        }
        if(lock != null){
            lock.release(event.value, true);
        }
        return createResponseMessage(responseSubject, responseKey, null);
    }
    
    protected Message onSave(SharedContextEvent event, final Object sourceId, final int sequence, final String responseSubject, final String responseKey){
        if(!isClient && (isMain() || !isSaveOnlyMain)){
            Thread saveThread = new Thread(){
                public void run(){
                    Message response = null;
                    try{
                        SharedContextService.super.save();
                        response = createResponseMessage(responseSubject, responseKey, null);
                    }catch(Throwable th){
                        response = createResponseMessage(responseSubject, responseKey, th);
                    }
                    try{
                        serverConnection.response(sourceId, sequence, response);
                    }catch(MessageSendException e){
                        getLogger().write("SCS__00006", e, isClient ? clientSubject : subject, response);
                    }
                }
            };
            saveThread.start();
        }
        return null;
    }
    
    protected Message onLoad(final SharedContextEvent event, final Object sourceId, final int sequence, final String responseSubject, final String responseKey){
        if(isMain()){
            Thread loadThread = new Thread(){
                public void run(){
                    Message response = null;
                    try{
                        if(event.key == null){
                            SharedContextService.super.load();
                        }else{
                            SharedContextService.super.load(event.key);
                        }
                        response = createResponseMessage(responseSubject, responseKey, null);
                    }catch(Throwable th){
                        response = createResponseMessage(responseSubject, responseKey, th);
                    }
                    try{
                        serverConnection.response(sourceId, sequence, response);
                    }catch(MessageSendException e){
                        getLogger().write("SCS__00006", e, isClient ? clientSubject : subject, response);
                    }
                }
            };
            loadThread.start();
        }
        return null;
    }
    
    protected Message onLoadKey(SharedContextEvent event, final Object sourceId, final int sequence, final String responseSubject, final String responseKey){
        if(isMain()){
            Thread loadThread = new Thread(){
                public void run(){
                    Message response = null;
                    try{
                        SharedContextService.super.loadKey();
                        response = createResponseMessage(responseSubject, responseKey, null);
                    }catch(Throwable th){
                        response = createResponseMessage(responseSubject, responseKey, th);
                    }
                    try{
                        serverConnection.response(sourceId, sequence, response);
                    }catch(MessageSendException e){
                        getLogger().write("SCS__00006", e, isClient ? clientSubject : subject, response);
                    }
                }
            };
            loadThread.start();
        }
        return null;
    }
    
    protected Message onAnalyzeKeyIndex(SharedContextEvent event, String responseSubject, String responseKey){
        if(isClient){
            return null;
        }
        try{
            analyzeKeyIndex((String)event.key);
            return createResponseMessage(responseSubject, responseKey, null);
        }catch(Throwable th){
            getLogger().write("SCS__00008", th, isClient ? clientSubject : subject, event.key);
            return createResponseMessage(responseSubject, responseKey, th);
        }
    }
    
    protected Message onSearchKeyIndex(SharedContextEvent event, String responseSubject, String responseKey){
        if(isMain()){
            return createResponseMessage(responseSubject, responseKey, searchByKeyIndex((String)event.key, event.value));
        }else{
            return null;
        }
    }
    
    protected void onExecuteInterpret(SharedContextEvent event, Object sourceId, int sequence, String responseSubject, String responseKey){
        if(isMain()){
            executeQueueHandlerContainer.push(
                new AsynchContext<Object[], Object>(
                    new Object[]{
                        event.key,
                        event.value,
                        responseSubject,
                        responseKey,
                        sourceId,
                        new Integer(sequence)
                    }
                )
            );
        }
    }
    
    protected static class SharedContextEvent implements java.io.Externalizable{
        
        public static final byte EVENT_PUT                = (byte)1;
        public static final byte EVENT_REMOVE             = (byte)2;
        public static final byte EVENT_CLEAR              = (byte)3;
        public static final byte EVENT_GET_ALL            = (byte)4;
        public static final byte EVENT_GET                = (byte)5;
        public static final byte EVENT_PUT_ALL            = (byte)6;
        public static final byte EVENT_KEY_SET            = (byte)7;
        public static final byte EVENT_SIZE               = (byte)8;
        public static final byte EVENT_CONTAINS_KEY       = (byte)9;
        public static final byte EVENT_CONTAINS_VALUE     = (byte)10;
        public static final byte EVENT_SYNCH_ALL          = (byte)11;
        public static final byte EVENT_GET_LOCK           = (byte)12;
        public static final byte EVENT_GOT_LOCK           = (byte)13;
        public static final byte EVENT_RELEASE_LOCK       = (byte)14;
        public static final byte EVENT_SAVE               = (byte)15;
        public static final byte EVENT_LOAD               = (byte)16;
        public static final byte EVENT_LOAD_KEY           = (byte)17;
        public static final byte EVENT_UPDATE             = (byte)18;
        public static final byte EVENT_SYNCH              = (byte)19;
        public static final byte EVENT_PUT_INNER          = (byte)20;
        public static final byte EVENT_ANALYZE_KEY_INDEX  = (byte)21;
        public static final byte EVENT_SEARCH_KEY_INDEX   = (byte)22;
        public static final byte EVENT_EXECUTE_INTERPRET  = (byte)23;
        
        public byte type;
        public Object key;
        public Object value;
        
        public SharedContextEvent(){
        }
        
        public SharedContextEvent(byte type){
            this(type, null, null);
        }
        
        public SharedContextEvent(byte type, Object key){
            this(type, key, null);
        }
        
        public SharedContextEvent(byte type, Object key, Object value){
            this.type = type;
            this.key = key;
            this.value = value;
        }
        
        public void writeExternal(java.io.ObjectOutput out) throws java.io.IOException{
            out.write(type);
            out.writeObject(key);
            out.writeObject(value);
        }
        
        public void readExternal(java.io.ObjectInput in) throws java.io.IOException, ClassNotFoundException{
            type = (byte)in.read();
            key = in.readObject();
            value = in.readObject();
        }
        
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.append('{');
            buf.append("type=").append(type);
            buf.append(", key=").append(key);
            buf.append(", value=").append(value);
            buf.append('}');
            return buf.toString();
        }
    }
    
    protected class Lock{
        private K key;
        private Object owner;
        private Thread ownerThread;
        private final SynchronizeMonitor monitor = new WaitSynchronizeMonitor();
        private Set<NotifyCallback> callbacks = new LinkedHashSet<NotifyCallback>();
        
        public Lock(K key){
            this.key = key;
        }
        
        public boolean acquire(Object id, long timeout){
            final boolean isLocal = id.equals(getId());
            synchronized(Lock.this){
                if(owner == null){
                    if(getState() != State.STARTED){
                        return false;
                    }
                    owner = id;
                    if(isLocal){
                        ownerThread = Thread.currentThread();
                    }
                    synchronized(idLocksMap){
                        Set<K> keySet = idLocksMap.get(id);
                        if(keySet == null){
                            keySet = new HashSet<K>();
                            idLocksMap.put(id, keySet);
                        }
                        synchronized(keySet){
                            keySet.add(key);
                        }
                    }
                    return true;
                }else if(id.equals(owner)){
                    return isLocal ? Thread.currentThread().equals(ownerThread) : true;
                }
            }
            try{
                long start = System.currentTimeMillis();
                if(monitor.initAndWaitMonitor(timeout)){
                    timeout = timeout - (System.currentTimeMillis() - start);
                    if(timeout <= 0){
                        return false;
                    }
                    return acquire(id, timeout);
                }else{
                    return false;
                }
            }catch(InterruptedException e){
                return false;
            }finally{
                monitor.releaseMonitor();
            }
        }
        
        public boolean acquireForReply(Object id, long timeout, Object sourceId, int sequence, String responseSubject, String responseKey){
            final boolean isLocal = id.equals(getId());
            synchronized(Lock.this){
                if(owner == null){
                    if(getState() != State.STARTED){
                        return false;
                    }
                    owner = id;
                    if(isLocal){
                        ownerThread = Thread.currentThread();
                    }
                    synchronized(idLocksMap){
                        Set<K> keySet = idLocksMap.get(id);
                        if(keySet == null){
                            keySet = new HashSet<K>();
                            idLocksMap.put(id, keySet);
                        }
                        synchronized(keySet){
                            keySet.add(key);
                        }
                    }
                    return true;
                }else if(id.equals(owner)){
                    return isLocal ? Thread.currentThread().equals(ownerThread) : true;
                }
                NotifyCallback callback = new NotifyCallback(id, timeout, sourceId, sequence, responseSubject, responseKey);
                callbacks.add(callback);
                if(timeout > 0){
                    lockTimeoutTimer.schedule(callback, timeout);
                }
            }
            return false;
        }
        
        public Object getOwner(){
            return owner;
        }
        
        public boolean release(Object id, boolean force){
            final boolean isLocal = id.equals(getId());
            boolean result = false;
            NotifyCallback callback = null;
            synchronized(Lock.this){
                if(owner == null || (id.equals(owner) && (force || !isLocal || Thread.currentThread().equals(ownerThread)))){
                    owner = null;
                    ownerThread = null;
                    result = true;
                }
                synchronized(idLocksMap){
                    Set<K> keySet = idLocksMap.get(id);
                    if(keySet != null){
                        synchronized(keySet){
                            keySet.remove(key);
                        }
                    }
                }
                if(monitor.isWait()){
                    monitor.notifyMonitor();
                }else if(callbacks.size() != 0){
                    Iterator<NotifyCallback> itr = callbacks.iterator();
                    callback = itr.next();
                    itr.remove();
                }
            }
            if(callback != null){
                callback.cancel();
                callback.notify(true);
            }
            return result;
        }
        
        private class NotifyCallback extends TimerTask{
            private Object id;
            private long startTime;
            private long timeout;
            private Object sourceId;
            private int sequence;
            private String responseSubject;
            private String responseKey;
            public NotifyCallback(Object id, long timeout, Object sourceId, int sequence, String responseSubject, String responseKey){
                this.id = id;
                this.sourceId = sourceId;
                this.sequence = sequence;
                this.responseSubject = responseSubject;
                this.responseKey = responseKey;
                this.timeout = timeout;
                startTime = System.currentTimeMillis();
            }
            public void notify(boolean notify){
                if(notify){
                    if(timeout <= 0){
                        if(!acquireForReply(this.id, timeout, sourceId, sequence, responseSubject, responseKey)){
                            return;
                        }
                    }else{
                        long currentTimeout = (startTime + timeout) - System.currentTimeMillis();
                        if(currentTimeout > 0){
                            if(!acquireForReply(this.id, currentTimeout, sourceId, sequence, responseSubject, responseKey)){
                                return;
                            }
                        }else{
                            notify = false;
                        }
                    }
                }
                Message response = null;
                try{
                    response = createResponseMessage(responseSubject, responseKey, notify ? Boolean.TRUE : Boolean.FALSE);
                }catch(Throwable th){
                    response = createResponseMessage(responseSubject, responseKey, th);
                }
                try{
                    serverConnection.response(sourceId, sequence, response);
                }catch(MessageSendException e){
                    getLogger().write("SCS__00006", new Object[]{isClient ? clientSubject : subject, response}, e);
                }
            }
            
            public void run(){
                boolean isRemoved = false;
                synchronized(Lock.this){
                    isRemoved = callbacks.remove(this);
                }
                if(isRemoved){
                    notify(false);
                }
            }
        }
    }
    
    protected class ClientCacheLock{
        private final SynchronizeMonitor monitor = new WaitSynchronizeMonitor();
        private Object key;
        private Object putVlaue;
        private List<SharedContextValueDifference> updateDiffList;
        private boolean isRemove;
        private boolean isNotify;
        
        public ClientCacheLock(Object key){
            this.key = key;
        }
        
        public void init(){
            monitor.initMonitor();
        }
        
        public boolean waitLock(long timeout){
            if(isNotify){
                return true;
            }
            try{
                if(monitor.waitMonitor(timeout)){
                    if(getState() != State.STARTED){
                        return false;
                    }
                    return true;
                }else{
                    return false;
                }
            }catch(InterruptedException e){
                return false;
            }finally{
                monitor.releaseMonitor();
            }
        }
        
        public void update(SharedContextValueDifference diff){
            if(updateDiffList == null){
                updateDiffList = new ArrayList<SharedContextValueDifference>();
            }
            updateDiffList.add(diff);
        }
        
        public void remove(){
            isRemove = true;
            updateDiffList = null;
            putVlaue = null;
        }
        
        public void put(Object value){
            isRemove = false;
            updateDiffList = null;
            putVlaue = value;
        }
        
        private boolean isRemove(){
            return isRemove;
        }
        
        public Object updateValue(Object value) throws SharedContextUpdateException{
            if(putVlaue != null){
                return putVlaue;
            }
            if(updateDiffList != null){
                if(value instanceof SharedContextValueDifferenceSupport){
                    for(int i = 0; i < updateDiffList.size(); i++){
                        SharedContextValueDifference diff = updateDiffList.get(i);
                        if(!((SharedContextValueDifferenceSupport)value).update(diff)){
                            throw new SharedContextUpdateException(
                                "An update version is mismatching. currentVersion="
                                    + ((SharedContextValueDifferenceSupport)value).getUpdateVersion()
                                    + ", updateVersion=" + diff.getUpdateVersion()
                            );
                        }
                    }
                }else{
                    throw new SharedContextUpdateException("Not support SharedContextValueDifference. key=" + key + ", value=" + value);
                }
            }
            return value;
        }
        
        public void notifyAllLock(){
            isNotify = true;
            monitor.notifyAllMonitor();
        }
    }
    
    protected static class IndexKey implements java.io.Serializable{
        
        private static final long serialVersionUID = -6040174467896232277L;
        
        private Object[] keys;
        
        public IndexKey(Object[] keys){
            this.keys = keys;
        }
        
        public int hashCode(){
            int hashCode = 0;
            for(int i = 0; i < keys.length; i++){
                hashCode += keys[i] == null ? 0 : keys[i].hashCode();
            }
            return hashCode;
        }
        
        public boolean equals(Object target){
            if(target == null){
                return false;
            }
            IndexKey targetKey = (IndexKey)target;
            if(targetKey == this){
                return true;
            }
            if(keys.length != targetKey.keys.length) {
                return false;
            }
            for(int i = 0; i < keys.length; i++){
                if(keys[i] == null && targetKey.keys[i] != null){
                    return false;
                }else if(keys[i] != null && !keys[i].equals(targetKey.keys[i])){
                    return false;
                }
            }
            return true;
        }
    }
    
    protected class ExecuteQueueHandler implements QueueHandler<AsynchContext<Object[], Object>>{
        public void handleDequeuedObject(AsynchContext<Object[], Object> ac) throws Throwable{
            if(ac == null){
                return;
            }
            final Object[] params = ac.getInput();
            final String query = (String)params[0];
            @SuppressWarnings("unchecked")
            final Map<String, Object> variables = (Map<String, Object>)params[1];
            final String responseSubject = (String)params[2];
            final String responseKey = (String)params[3];
            final Object sourceId = params[4];
            final int sequence = ((Integer)params[5]).intValue();
            Object ret = executeInterpretQueryLocal(query, variables);
            Message response = createResponseMessage(responseSubject, responseKey, ret);
            try{
                serverConnection.response(sourceId, sequence, response);
            }catch(MessageSendException e){
                getLogger().write("SCS__00006", e, isClient ? clientSubject : subject, response);
            }
        }
        public boolean handleError(AsynchContext<Object[], Object> ac, Throwable th) throws Throwable{
            return false;
        }
        public void handleRetryOver(AsynchContext<Object[], Object> ac, Throwable th) throws Throwable{
            final Object[] params = ac.getInput();
            final String responseSubject = (String)params[2];
            final String responseKey = (String)params[3];
            final Object sourceId = params[4];
            final int sequence = ((Integer)params[5]).intValue();
            Message response = null;
            try{
                response = createResponseMessage(responseSubject, responseKey, th);
                serverConnection.response(sourceId, sequence, response);
            }catch(MessageSendException e){
                getLogger().write("SCS__00006", e, isClient ? clientSubject : subject, response);
            }
        }
    }
    
    protected class LocalSharedContext implements SharedContext<K>{
        
        public void lock(K key) throws SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public void lock(K key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public boolean unlock(K key) throws SharedContextSendException{
            throw new UnsupportedOperationException();
        }
        
        public boolean unlock(K key, boolean force) throws SharedContextSendException{
            throw new UnsupportedOperationException();
        }
        
        public Object getLockOwner(K key){
            throw new UnsupportedOperationException();
        }
        
        public Object put(K key, Object value){
            throw new UnsupportedOperationException();
        }
        
        public Object put(K key, Object value, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public Object putLocal(K key, Object value){
            throw new UnsupportedOperationException();
        }
        
        public void putAsynch(K key, Object value) throws SharedContextSendException{
            throw new UnsupportedOperationException();
        }
        
        public void update(K key, SharedContextValueDifference diff) throws SharedContextSendException{
            throw new UnsupportedOperationException();
        }
        
        public void update(K key, SharedContextValueDifference diff, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public void updateLocal(K key, SharedContextValueDifference diff){
            throw new UnsupportedOperationException();
        }
        
        public void updateAsynch(K key, SharedContextValueDifference diff) throws SharedContextSendException{
            throw new UnsupportedOperationException();
        }
        
        public void putAll(Map<? extends K, ? extends Object> t){
            throw new UnsupportedOperationException();
        }
        
        public void putAll(Map<? extends K, ? extends Object> t, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public void putAllLocal(Map<? extends K, ? extends Object> t){
            throw new UnsupportedOperationException();
        }
        
        public void putAllAsynch(Map<? extends K, ? extends Object> t) throws SharedContextSendException{
            throw new UnsupportedOperationException();
        }
        
        public Object get(Object key){
            return SharedContextService.this.getLocal(key);
        }
        
        public Object get(Object key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.getLocal(key);
        }
        
        public Object getLocal(Object key){
            return SharedContextService.this.getLocal(key);
        }
        
        public Object remove(Object key){
            return SharedContextService.this.removeLocal(key);
        }
        
        public Object remove(Object key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.removeLocal(key);
        }
        
        public Object removeLocal(Object key){
            return SharedContextService.this.removeLocal(key);
        }
        
        public void removeAsynch(Object key) throws SharedContextSendException{
            SharedContextService.this.removeLocal(key);
        }
        
        public void clear(){
            SharedContextService.this.clearLocal();
        }
        
        public void clear(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            SharedContextService.this.clearLocal();
        }
        
        public void clearLocal(){
            SharedContextService.this.clearLocal();
        }
        
        public void clearAsynch() throws SharedContextSendException{
            SharedContextService.this.clearLocal();
        }
        
        public Set<K> keySet(){
            return SharedContextService.this.keySetLocal();
        }
        
        public Set<K> keySet(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.keySetLocal();
        }
        
        public Set<K> keySetLocal(){
            return SharedContextService.this.keySetLocal();
        }
        
        public int size(){
            return SharedContextService.this.sizeLocal();
        }
        
        public int size(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.sizeLocal();
        }
        
        public int sizeLocal(){
            return SharedContextService.this.sizeLocal();
        }
        
        public boolean isEmpty() throws SharedContextSendException{
            return SharedContextService.this.isEmptyLocal();
        }
        
        public boolean isEmptyLocal(){
            return SharedContextService.this.isEmptyLocal();
        }
        
        public boolean containsKey(Object key){
            return SharedContextService.this.containsKeyLocal(key);
        }
        
        public boolean containsKey(Object key, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.containsKeyLocal(key);
        }
        
        public boolean containsKeyLocal(Object key){
            return SharedContextService.this.containsKeyLocal(key);
        }
        
        public boolean containsValue(Object value){
            return SharedContextService.this.containsValueLocal(value);
        }
        
        public boolean containsValue(Object value, long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.containsValueLocal(value);
        }
        
        public boolean containsValueLocal(Object value){
            return SharedContextService.this.containsValueLocal(value);
        }
        
        public Map<K, Object> all(){
            return SharedContextService.this.allLocal();
        }
        
        public Map<K, Object> allLocal(){
            return SharedContextService.this.allLocal();
        }
        
        public Set<Map.Entry<K, Object>> entrySet(){
            return SharedContextService.this.entrySetLocal();
        }
        
        public Set<Map.Entry<K, Object>> entrySetLocal(){
            return SharedContextService.this.entrySetLocal();
        }
        
        public Collection<Object> values(){
            return SharedContextService.this.valuesLocal();
        }
        
        public Collection<Object> valuesLocal(){
            return SharedContextService.this.valuesLocal();
        }
        
        public void synchronize() throws SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public void synchronize(long timeout) throws SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public boolean isClient(){
            return SharedContextService.this.isClient();
        }
        
        public boolean isMain(){
            return SharedContextService.this.isMain();
        }
        
        public Object getId(){
            return SharedContextService.this.getId();
        }
        
        public Object getMainId(){
            return SharedContextService.this.getMainId();
        }
        
        public void addSharedContextUpdateListener(SharedContextUpdateListener<K> listener){
            SharedContextService.this.addSharedContextUpdateListener(listener);
        }
        
        public void removeSharedContextUpdateListener(SharedContextUpdateListener<K> listener){
            SharedContextService.this.removeSharedContextUpdateListener(listener);
        }
        
        public void analyzeKeyIndex(String name) throws SharedContextIllegalIndexException, SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public Set<K> searchByKeyIndex(String name, Object key) throws SharedContextIllegalIndexException, SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.searchByKeyIndexLocal(name, key);
        }
        
        public Set<K> searchByKeyIndex(String name, Object key, long timeout) throws SharedContextIllegalIndexException, SharedContextSendException, SharedContextTimeoutException{
            return SharedContextService.this.searchByKeyIndexLocal(name, key);
        }
        
        public Set<K> searchByKeyIndexLocal(String name, Object key) throws SharedContextIllegalIndexException{
            return SharedContextService.this.searchByKeyIndexLocal(name, key);
        }
        
        public Object executeInterpretQuery(String query, Map<String, Object> variables) throws EvaluateException, SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public Object executeInterpretQuery(String query, Map<String, Object> variables, long timeout) throws EvaluateException, SharedContextSendException, SharedContextTimeoutException{
            throw new UnsupportedOperationException();
        }
        
        public void load() throws Exception{
            throw new UnsupportedOperationException();
        }
        
        public void loadKey() throws Exception{
            throw new UnsupportedOperationException();
        }
        
        public void load(Object key) throws Exception{
            throw new UnsupportedOperationException();
        }
        
        public void save() throws Exception{
            throw new UnsupportedOperationException();
        }
    }
}