/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link PropertyEditorMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class PropertyEditorMetaDataTest{

    @DataProvider(name = "PropertyEditorMetaData")
    public Object[][] dataPropertyEditorMetaData(){
        return new Object[][]{
            {null},
            {new PropertyEditorsMetaData(null)},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorMetaData#PropertyEditorMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "PropertyEditorMetaData")
    public void testPropertyEditorMetaData(MetaData parent){
        PropertyEditorMetaData metaData = new PropertyEditorMetaData(parent);
        assertEquals(metaData.getParent(), parent);
        assertNull(metaData.getType());
        assertNull(metaData.getEditor());
    }

    @DataProvider(name = "SetType")
    public Object[][] dataSetType(){
        return new Object[][]{
            {null},
            {"java.lang.String"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorMetaData#setType(java.lang.String)",
                    "jp.ossc.nimbus.core.PropertyEditorMetaData#getType()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#PropertyEditorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetType")
    public void testSetType(String type){
        PropertyEditorMetaData metaData = new PropertyEditorMetaData(null);
        metaData.setType(type);
        assertEquals(metaData.getType(), type);
    }

    @DataProvider(name = "SetEditor")
    public Object[][] dataSetEditor(){
        return new Object[][]{
            {null},
            {"test.StringPropertyEditor"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorMetaData#setEditor(java.lang.String)",
                    "jp.ossc.nimbus.core.PropertyEditorMetaData#getEditor()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#PropertyEditorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetEditor")
    public void testSetEditor(String editor){
        PropertyEditorMetaData metaData = new PropertyEditorMetaData(null);
        metaData.setEditor(editor);
        assertEquals(metaData.getEditor(), editor);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>"), new Object[]{"java.lang.String", "test.StringPropertyEditor"}},
            {createElement("<property-editor type=\"java.lang.String\" unknown=\"hoge\">test.StringPropertyEditor</property-editor>"), new Object[]{"java.lang.String", "test.StringPropertyEditor"}},
            {createElement("<property-editor type=\"java.lang.String\">test.StringPropertyEditor<unknown/></property-editor>"), new Object[]{"java.lang.String", "test.StringPropertyEditor"}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#PropertyEditorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#getType\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#getEditor\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        PropertyEditorMetaData metaData = new PropertyEditorMetaData(null);
        metaData.importXML(element);
        assertEquals(metaData.getType(), assertValues[0]);
        assertEquals(metaData.getEditor(), assertValues[1]);
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<property-editor/>")},
            {createElement("<property-editor></property-editor>")},
            {createElement("<property-editor type=\"\">test.StringPropertyEditor</property-editor>")},
            {createElement("<property-editor type=\"java.lang.String\"></property-editor>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#PropertyEditorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        PropertyEditorMetaData metaData = new PropertyEditorMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>"), "<property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>"},
            {createElement("<property-editor type=\"java.lang.String\">\n    test.StringPropertyEditor\n</property-editor>"), "<property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>"},
            {createElement("<property-editor type=\"java.lang.String\" unknown=\"hoge\">test.StringPropertyEditor</property-editor>"), "<property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>"},
            {createElement("<property-editor type=\"java.lang.String\">test.StringPropertyEditor<unknown/></property-editor>"), "<property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyEditorMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#PropertyEditorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyEditorMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        PropertyEditorMetaData metaData = new PropertyEditorMetaData(null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
