# -*- coding: utf-8 -*-
#
#  Copyright (C) 2004-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - 「きのこ」へのステータス送信.
# - 「きのこ」の情報の参照.
# - SERIKO/1.2ベースのアニメーション
# - (スキン側の)katochan.txt
# - balloon.txt
# - surface[0/1/2]a.txt(@ゴースト)
# - 自爆イベント
# - headrect.txt : 頭の当たり判定領域データ
#   当たり領域のleft／top／right／bottomを半角カンマでセパレートして記述.
#   1行目がsurface0、2行目がsurface1の領域データ.
#   このファイルがない場合、領域は自動計算される.
# - speak.txt
# - katochan が無い場合の処理.(本体の方のpopup menuなども含めて)
# - 設定ダイアログ : [会話/反応]タブ -> [SEND SSTP/1.1] or [SHIORI]
# - 見切れ連続20[s]、もしくは画面内で静止20[s]でアニメーション記述ミスと見なし自動的に落ちる
# - 発言中にバルーンをダブルクリックで即閉じ
# - @ゴースト名は#nameと#forには使えない. もし書いても無視されすべて有効になる
# - 連続落し不可指定
#   チェックしておくと落下物を2個以上同時に落とせなくなる
# - スキンチェンジ時も起動時のトークを行う
# - ファイルセット設定機能
#   インストールされたスキン／落下物のうち使用するものだけを選択できる
# - ターゲットのアイコン化への対応
# - アイコン化されているときは自動落下しない
# - アイコン化されているときのDirectSSTP SEND/DROPリクエストはエラー(Invisible)
# - 落下物の透明化ON/OFF
# - 落下物が猫どりふ自身にも落ちてくる
#   不在時に1/2、ランダム/全員落し時に1/10の確率で自爆
# - 一定時間間隔で勝手に物を落とす
# - ターゲット指定落し、ランダム落し、全員落し
# - 出現即ヒットの場合への対応

# - 複数ゴーストでの当たり判定.
# - 透明ウィンドウ

import os
import random
import logging

from gi.repository import Gtk
from gi.repository import Gdk
from gi.repository import GLib
import cairo

import ninix.pix
import ninix.home


class Menu:

    def __init__(self, accelgroup):
        self.request_parent = lambda *a: None # dummy
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menuitem action='Settings'/>
            <menu action='Katochan'>
            </menu>
            <separator/>
            <menuitem action='Exit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'settings': [('Settings', None, _('Settings...(_O)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'edit_preferences')),
                         '/ui/popup/Settings'],
            'katochan': [('Katochan', None, _('Katochan(_K)'), None),
                         '/ui/popup/Katochan'],
            'exit':     [('Exit', None,_('Exit(_Q)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'close')),
                         '/ui/popup/Exit'],
            }
        self.__katochan_list = None
        actions = Gtk.ActionGroup('Actions')
        entry = [value[0] for value in self.__menu_list.values()]
        actions.add_actions(tuple(entry))
        ui_manager = Gtk.UIManager()
        ui_manager.insert_action_group(actions, 0)
        ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = ui_manager.get_widget('/ui/popup')
        for key in self.__menu_list:
            path = self.__menu_list[key][-1]
            self.__menu_list[key][1] = ui_manager.get_widget(path)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def popup(self, button):
        katochan_list = self.request_parent('GET', 'get_katochan_list')
        self.__set_katochan_menu(katochan_list)
        self.__popup_menu.popup(
            None, None, None, None, button, Gtk.get_current_event_time())

    def __set_katochan_menu(self, list): ## FIXME
        key = 'katochan'
        if list:
            menu = Gtk.Menu()
            for katochan in list:
                item = Gtk.MenuItem(katochan['name'])
                item.connect(
                    'activate',
                    lambda a, k: self.request_parent('NOTIFY', 'select_katochan', k),
                    (katochan))
                menu.add(item)
                item.show()
            self.__menu_list[key][1].set_submenu(menu)
            menu.show()
            self.__menu_list[key][1].show()
        else:
            self.__menu_list[key][1].hide()


class Nekoninni:

    def __init__(self):
        self.mode = 1 # 0: SEND SSTP1.1, 1: SHIORI/2.2
        self.__running = 0
        self.skin = None
        self.katochan = None

    def observer_update(self, event, args):
        if event in ['set position', 'set surface']:
            if self.skin is not None:
                self.skin.set_position()
            if self.katochan is not None and self.katochan.loaded:
                self.katochan.set_position()
        elif event == 'set scale':
            scale = self.target.get_surface_scale()
            if self.skin is not None:
                self.skin.set_scale(scale)
            if self.katochan is not None:
                self.katochan.set_scale(scale)
        elif event == 'finalize':
            self.finalize()
        else:
            ##logging.debug('OBSERVER(nekodorif): ignore - {0}'.format(event))
            pass

    def load(self, dir, katochan, target):
        if not katochan:
            return 0
        self.dir = dir
        self.target = target
        self.target.attach_observer(self)
        self.accelgroup = Gtk.AccelGroup()
        scale = self.target.get_surface_scale()
        self.skin = Skin(self.dir, self.accelgroup, scale)
        self.skin.set_responsible(self.handle_request)
        if self.skin is None:
            return 0
        self.katochan_list = katochan
        self.katochan = None
        self.launch_katochan(self.katochan_list[0])
        self.__running = 1
        GLib.timeout_add(50, self.do_idle_tasks) # 50[ms]
        return 1

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            'get_katochan_list': lambda *a: self.katochan_list,
            'get_mode': lambda *a: self.mode,
            }
        handler = handlers.get(event,
                               getattr(self, event,
                                       lambda *a: None)) ## FIXME
        result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def do_idle_tasks(self):
        if not self.__running:
            return False
        self.skin.update()
        if self.katochan is not None:
            self.katochan.update()
        #self.process_script()
        return True

    def send_event(self, event):
        if event not in ['Emerge', # 可視領域内に出現
                         'Hit',    # ヒット
                         'Drop',   # 再落下開始
                         'Vanish', # ヒットした落下物が可視領域内から消滅
                         'Dodge'   # よけられてヒットしなかった落下物が可視領域内から消滅
                         ]:
            return
        args = (self.katochan.get_name(),
                self.katochan.get_ghost_name(),
                self.katochan.get_category(),
                self.katochan.get_kinoko_flag(),
                self.katochan.get_target())
        self.target.notify_event('OnNekodorifObject{0}'.format(event), *args)

    def has_katochan(self):
        return int(self.katochan is not None)

    def select_katochan(self, args):
        self.launch_katochan(args)

    def drop_katochan(self):
        self.katochan.drop()

    def delete_katochan(self):
        self.katochan.destroy()
        self.katochan = None
        self.skin.reset()

    def launch_katochan(self, katochan):
        if self.katochan:
            self.katochan.destroy()
        self.katochan = Katochan(self.target)
        self.katochan.set_responsible(self.handle_request)
        self.katochan.load(katochan)

    def edit_preferences(self):
        pass

    def finalize(self):
        self.__running = 0
        self.target.detach_observer(self)
        if self.katochan is not None:
            self.katochan.destroy()
        if self.skin is not None:
            self.skin.destroy()
        ##if self.balloon is not None:
        ##    self.balloon.destroy()

    def close(self):
        self.finalize()


class Skin:

    def __init__(self, dir, accelgroup, scale):
        self.dir = dir
        self.accelgroup = accelgroup
        self.request_parent = lambda *a: None # dummy
        self.dragged = False
        self.x_root = None
        self.y_root = None
        self.__scale = scale
        self.__menu = Menu(self.accelgroup)
        self.__menu.set_responsible(self.handle_request)
        path = os.path.join(self.dir, b'omni.txt')
        self.omni = int(os.path.isfile(path) and os.path.getsize(path) == 0)
        self.window = ninix.pix.TransparentWindow()
        name, top_dir = ninix.home.read_profile_txt(dir) # XXX
        self.window.set_title(name)
        self.window.connect('delete_event', self.delete)
        self.window.connect('key_press_event', self.key_press)
        self.window.add_accel_group(self.accelgroup)
        self.darea = self.window.get_child()
        self.darea.set_events(Gdk.EventMask.EXPOSURE_MASK|
                              Gdk.EventMask.BUTTON_PRESS_MASK|
                              Gdk.EventMask.BUTTON_RELEASE_MASK|
                              Gdk.EventMask.POINTER_MOTION_MASK|
                              Gdk.EventMask.POINTER_MOTION_HINT_MASK|
                              Gdk.EventMask.LEAVE_NOTIFY_MASK)
        self.darea.connect('draw', self.redraw)
        self.darea.connect('button_press_event', self.button_press)
        self.darea.connect('button_release_event', self.button_release)
        self.darea.connect('motion_notify_event', self.motion_notify)
        self.darea.connect('leave_notify_event', self.leave_notify)
        self.id = [0, None]
        self.set_surface()
        self.set_position(reset=1)
        self.window.show()

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def set_scale(self, scale):
        self.__scale = scale
        self.set_surface()
        self.set_position()

    def redraw(self, widget, cr):
        scale = self.__scale
        cr.scale(scale / 100.0, scale / 100.0)
        cr.set_source_surface(self.image_surface, 0, 0)
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.paint()

    def delete(self, widget, event):
        self.request_parent('NOTIFY', 'finalize')

    def key_press(self, window, event):
        if event.get_state() & (Gdk.ModifierType.CONTROL_MASK | Gdk.ModifierType.SHIFT_MASK):
            if event.keyval == Gdk.KEY_F12:
                logging.info('reset skin position')
                self.set_position(reset=1)
        return True

    def destroy(self): ## FIXME
        self.window.destroy()

    def button_press(self,  widget, event):
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.button == 1:
            if event.type == Gdk.EventType.BUTTON_PRESS:
                pass
            elif event.type == Gdk.EventType._2BUTTON_PRESS: # double click
                if self.request_parent('GET', 'has_katochan'):
                    self.start() ## FIXME
                    self.request_parent('NOTIFY', 'drop_katochan')
        elif event.button == 3:                
            if event.type == Gdk.EventType.BUTTON_PRESS:
                self.__menu.popup(event.button)
        return True

    def set_surface(self):
        if self.id[1] is not None:
            path = os.path.join(
                self.dir,
                os.fsencode('surface{0:d}{1:d}.png'.format(self.id[0], self.id[1])))
            if not os.path.exists(path):
                self.id[1] = None
                self.set_surface()
                return
        else:
            path = os.path.join(self.dir,
                                os.fsencode('surface{0:d}.png'.format(self.id[0])))
        try:
            new_surface = ninix.pix.create_surface_from_file(path)
            w = max(8, int(new_surface.get_width() * self.__scale / 100))
            h = max(8, int(new_surface.get_height() * self.__scale / 100))
        except:
            self.request_parent('NOTIFY', 'finalize')
            return
        self.w, self.h = w, h
        self.window.update_size(self.w, self.h)
        self.image_surface = new_surface
        self.darea.queue_draw()

    def set_position(self, reset=0):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if reset:
            self.x = left
            self.y = top + scrn_h - self.h
        else:
            if not self.omni:
                self.y = top + scrn_h - self.h
        self.window.move(self.x, self.y)

    def move(self, x_delta, y_delta):
        self.x = self.x + x_delta
        if self.omni:
            self.y = self.y + y_delta
        self.set_position()

    def update(self):
        if self.id[1] is not None:
            self.id[1] += 1
        else:
            if random.choice(range(100)): ## XXX
                return
            self.id[1] = 0
        self.set_surface()

    def start(self): ## FIXME
        self.id[0] = 1
        self.set_surface()

    def reset(self): ## FIXME
        self.id[0] = 0
        self.set_surface()

    def button_release(self,  widget, event):
        if self.dragged:
            self.dragged = False
            self.set_position()
        self.x_root = None
        self.y_root = None
        return True

    def motion_notify(self, widget, event):
        if event.is_hint:
            ## FIXME: Use get_device_position() instead.
            _, x, y, state = widget.get_window().get_pointer()
        else:
            x, y, state = event.x, event.y, event.get_state()
        #x, y = self.window.winpos_to_surfacepos(x, y, self.__scale)
        if state & Gdk.ModifierType.BUTTON1_MASK:
            if self.x_root is not None and \
               self.y_root is not None:
                self.dragged = True
                x_delta = int(event.x_root - self.x_root)
                y_delta = int(event.y_root - self.y_root)
                self.move(x_delta, y_delta)
                self.x_root = event.x_root
                self.y_root = event.y_root
        return True

    def leave_notify(self,  widget, event): ## FIXME
        pass


class Balloon:

    def __init__(self):
        pass

    def destroy(self): ## FIXME
        pass


class Katochan:

    CATEGORY_LIST = ['pain',      # 痛い
                     'stab',      # 刺さる
                     'surprise',  # びっくり
                     'hate',      # 嫌い、気持ち悪い
                     'huge',      # 巨大
                     'love',      # 好き、うれしい
                     'elegant',   # 風流、優雅
                     'pretty',    # かわいい
                     'food',      # 食品
                     'reference', # 見る／読むもの
                     'other'      # 上記カテゴリに当てはまらないもの
                     ]

    def __init__(self, target):
        self.side = 0
        self.target = target
        self.request_parent = lambda *a: None # dummy
        self.settings = {}
        self.settings['state'] = 'before'
        self.settings['fall.type'] = 'gravity'
        self.settings['fall.speed'] = 1
        self.settings['slide.type'] = 'none'
        self.settings['slide.magnitude'] = 0
        self.settings['slide.sinwave.degspeed'] = 30
        self.settings['wave'] = None
        self.settings['wave.loop'] = 0
        self.__scale = 100
        self.loaded = False

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def get_name(self):
        return self.data['name']

    def get_category(self):
        return self.data['category']

    def get_kinoko_flag(self): ## FIXME
        return 0 # 0/1 = きのこに当たっていない(ない場合を含む)／当たった

    def get_target(self): ## FIXME
        if self.side == 0 :
            return self.target.get_selfname()
        else:
            return self.target.get_keroname()

    def get_ghost_name(self):
        if 'for' in self.data: # 落下物が主に対象としているゴーストの名前
            return self.data['for']
        else:
            return ''

    def destroy(self):
        self.window.destroy()

    def delete(self, widget, event):
        self.destroy()

    def redraw(self, widget, cr):
        scale = self.__scale
        cr.scale(scale / 100.0, scale / 100.0)
        cr.set_source_surface(self.image_surface, 0, 0)
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.paint()

    def set_movement(self, timing):
        key = ''.join((timing, 'fall.type'))
        if key in self.data and \
           self.data[key] in ['gravity', 'evenspeed', 'none']:
            self.settings['fall.type'] = self.data[key]
        else:
            self.settings['fall.type'] = 'gravity'
        self.settings['fall.speed'] = self.data.get(
            ''.join((timing, 'fall.speed')), 1)
        if self.settings['fall.speed'] < 1:
            self.settings['fall.speed'] = 1
        if self.settings['fall.speed'] > 100:
            self.settings['fall.speed'] = 100
        key = ''.join((timing, 'slide.type'))
        if key in self.data and \
           self.data[key] in ['none', 'sinwave', 'leaf']:
            self.settings['slide.type'] = self.data[key]
        else:
            self.settings['slide.type'] = 'none'
        self.settings['slide.magnitude'] = self.data.get(
            ''.join((timing, 'slide.magnitude')), 0)
        self.settings['slide.sinwave.degspeed'] = self.data.get(
            ''.join((timing, 'slide.sinwave.degspeed')), 30)
        self.settings['wave'] = self.data.get(''.join((timing, 'wave')), None)
        self.settings['wave.loop'] = int(
            self.data.get(''.join((timing, 'wave.loop'))) == 'on')

    def set_scale(self, scale):
        self.__scale = scale
        self.set_surface()
        self.set_position()

    def set_position(self):
        if self.settings['state'] != 'before':
            return
        target_x, target_y = self.target.get_surface_position(self.side)
        target_w, target_h = self.target.get_surface_size(self.side)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        self.x = target_x + target_w // 2 - self.w // 2 + \
                 int(self.offset_x * self.__scale / 100)
        self.y = top + int(self.offset_y * self.__scale / 100)
        self.window.move(self.x, self.y)

    def set_surface(self):
        path = os.path.join(self.data['dir'],
                            os.fsencode('surface{0:d}.png'.format(self.id)))
        try:
            new_surface = ninix.pix.create_surface_from_file(path)
            w = max(8, int(new_surface.get_width() * self.__scale / 100))
            h = max(8, int(new_surface.get_height() * self.__scale / 100))
        except:
            self.request_parent('NOTIFY', 'finalize')
            return
        self.w, self.h = w, h
        self.window.update_size(self.w, self.h)
        #self.darea.queue_draw_area(0, 0, self.w, self.h)
        self.image_surface = new_surface
        self.darea.queue_draw()

    def load(self, data):
        self.data = data
        self.__scale = self.target.get_surface_scale()
        self.set_state('before')
        if 'category' in self.data:
            category = self.data['category'].split(',')
            if category:
                if category[0] not in self.CATEGORY_LIST:
                    logging.warning('WARNING: unknown major category - {0}'.format(category[0]))
                    ##self.data['category'] = self.CATEGORY_LIST[-1]
            else:
                self.data['category'] = self.CATEGORY_LIST[-1]
        else:
            self.data['category'] = self.CATEGORY_LIST[-1]
        if 'target' in self.data:
            if self.data['target'] == 'sakura':
                self.side = 0
            elif self.data['target'] == 'kero':
                self.side = 1
            else:
                self.side = 0 # XXX
        else:
            self.side = 0 # XXX
        if self.request_parent('GET', 'get_mode') == 1:
            self.request_parent('NOTIFY', 'send_event', 'Emerge')
        else:
            if 'before.script' in self.data:
                pass ## FIXME
            else:
                pass ## FIXME
        self.set_movement('before')
        if 'before.appear.direction' in self.data:
            pass ## FIXME
        else:
            pass ## FIXME
        offset_x =  self.data.get('before.appear.ofset.x', 0)
        if offset_x < -32768:
            offset_x = -32768
        if offset_x > 32767:
            offset_x = 32767
        offset_y =  self.data.get('before.appear.ofset.y', 0)
        if offset_y < -32768:
            offset_y = -32768
        if offset_y > 32767:
            offset_y = 32767
        self.offset_x = offset_x
        self.offset_y = offset_y
        self.window = ninix.pix.TransparentWindow()
        self.window.set_title(self.data['name'])
        self.window.set_skip_taskbar_hint(True) # XXX
        self.window.connect('delete_event', self.delete)
        self.darea = self.window.get_child()
        self.darea.set_events(Gdk.EventMask.EXPOSURE_MASK)
        self.darea.connect('draw', self.redraw)
        self.window.show()
        self.id = 0
        self.set_surface()
        self.set_position()
        self.loaded = True

    def drop(self): ## FIXME
        self.set_state('fall')

    def set_state(self, state):
        self.settings['state'] = state
        self.time = 0
        self.hit = 0
        self.hit_stop = 0

    def update_surface(self): ## FIXME
        pass

    def update_position(self): ## FIXME
        if self.settings['slide.type'] == 'leaf':
            pass
        else:
            if self.settings['fall.type'] == 'gravity':
                self.y += int(self.settings['fall.speed'] * \
                              (self.time / 20.0)**2)
            elif  self.settings['fall.type'] == 'evenspeed':
                self.y += self.settings['fall.speed']
            else:
                pass
            if self.settings['slide.type'] == 'sinwave':
                pass ## FIXME
            else:
                pass
        self.window.move(self.x, self.y)

    def check_collision(self): ## FIXME: check self position
        for side in [0, 1]:
            target_x, target_y = self.target.get_surface_position(side)
            target_w, target_h = self.target.get_surface_size(side)
            center_x = self.x + self.w // 2
            center_y = self.y + self.h // 2
            if target_x < center_x < target_x + target_w and \
               target_y < center_y < target_y + target_h:
                self.side = side
                return 1
        else:
            return 0

    def check_mikire(self):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if self.x + self.w - self.w // 3 > left + scrn_w or \
                self.x + self.w // 3 < left or \
                self.y + self.h - self.h // 3 > top + scrn_h or \
                self.y + self.h // 3 < top:
            return 1
        else:
            return 0

    def update(self): ## FIXME
        if self.settings['state'] == 'fall':
            self.update_surface()
            self.update_position()
            if self.check_collision():
                self.set_state('hit')
                self.hit = 1
                if self.request_parent('GET', 'get_mode') == 1:
                    self.id = 1
                    self.set_surface()
                    self.request_parent('NOTIFY', 'send_event', 'Hit')
                else:
                    pass ## FIXME
            if self.check_mikire():
                self.set_state('dodge')
        elif self.settings['state'] == 'hit':
            if self.hit_stop >= self.data.get('hit.waittime', 0):
                self.set_state('after')
                self.set_movement('after')
                if self.request_parent('GET', 'get_mode') == 1:
                    self.id = 2
                    self.set_surface()
                    self.request_parent('NOTIFY', 'send_event', 'Drop')
                else:
                    pass ## FIXME
            else:
                self.hit_stop += 1
                self.update_surface()
        elif self.settings['state'] == 'after':
            self.update_surface()
            self.update_position()
            if self.check_mikire():
                self.set_state('end')
        elif self.settings['state'] == 'end':
            if self.request_parent('GET', 'get_mode') == 1:
                self.request_parent('NOTIFY', 'send_event', 'Vanish')
            else:
                pass ## FIXME
            self.request_parent('NOTIFY', 'delete_katochan')
            return False
        elif self.settings['state'] == 'dodge':
            if self.request_parent('GET', 'get_mode') == 1:
                self.request_parent('NOTIFY', 'send_event', 'Dodge')
            else:
                pass ## FIXME
            self.request_parent('NOTIFY', 'delete_katochan')
            return False
        else:
            pass ## check collision and mikire
        self.time += 1
        return True
