﻿using System;
using System.Collections.Generic;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace NotepadNeue
{
    public partial class MyTreeView : UserControl
    {
        public enum NodeState
        {
            none = 0,
            invisible = 1
        }
        public event TreeNodeMouseClickEventHandler NodeMouseDoubleClick;
        public event TreeNodeMouseClickEventHandler NodeMouseClick;
        public event TreeViewCancelEventHandler BeforeCollapse;
        public event TreeViewEventHandler AfterCollapse;
        public event TreeViewCancelEventHandler BeforeExpand;
        public event TreeViewEventHandler AfterExpand;
        public event ScrollEventHandler Scrolled;

        MouseButtons button;

        TreeNode parentnode;
        TreeNode SelectedNode;
        TreeNode HotNode;
        TreeNode FocusedNode;
        ImageList imagelist;
        private BufferedGraphics grafx;
        private BufferedGraphicsContext context;
        HScrollBar Hsc;
        VScrollBar Vsc;

        public MyTreeView()
        {
            InitializeComponent();
            parentnode = new TreeNode();
            ItemHeight = 15;
            Indent = 19;
            Hsc = new HScrollBar();
            Vsc = new VScrollBar();
            this.Controls.Add(Hsc);
            this.Controls.Add(Vsc);
            Hsc.Visible = false;
            Vsc.Visible = false;
            Vsc.SmallChange = 1;
            Vsc.LargeChange = 1;
            Vsc.TabStop = false;
            Hsc.TabStop = false;
            Vsc.ValueChanged += new EventHandler(Vsc_ValueChanged);
            Hsc.ValueChanged += new EventHandler(Hsc_ValueChanged);
            MouseWheel += new MouseEventHandler(MyTreeView_MouseWheel);
            LostFocus += new EventHandler(MyTreeView_LostFocus);
            GotFocus += new EventHandler(MyTreeView_GotFocus);
            imagelist = new ImageList();
            this.SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);
            context = BufferedGraphicsManager.Current;
            context.MaximumBuffer = new Size(this.Width + 1, this.Height + 1);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
            grafx = context.Allocate(this.CreateGraphics(), new Rectangle(0, 0, this.Width, this.Height));
            DrawToBuffer(grafx.Graphics);
        }

        void MyTreeView_GotFocus(object sender, EventArgs e)
        {
            if (Nodes.Count == 0) return;
            SelectedNode = FocusedNode;
            bool nu = false;
            if (Selectednode == null) nu = true;
            if (!nu)
            {
                NodeState? ns = SelectedNode.Tag as NodeState?;
                if (ns != null)
                {
                    if ((ns & NodeState.invisible) == NodeState.invisible) nu = true;
                }
            }
            if (nu)
            {
                Selectednode = Nodes[0];
                FocusedNode = Selectednode;
            }
            DrawandReflash();
        }

        void MyTreeView_LostFocus(object sender, EventArgs e)
        {
            SelectedNode = null;
            DrawandReflash();
        }

        void Hsc_ValueChanged(object sender, EventArgs e)
        {
            if (Scrolled != null)
            {
                Scrolled.Invoke(this, new ScrollEventArgs(ScrollEventType.First, 0));
            }
            DrawandReflash();
        }

        void MyTreeView_MouseWheel(object sender, MouseEventArgs e)
        {
            if ((ModifierKeys & Keys.Control) == Keys.Control)
            {
                if (Hsc.Visible)
                {
                    int result = Hsc.Value - e.Delta / 10;
                    if (result < 0) result = 0;
                    if (result > Hsc.Maximum) result = Hsc.Maximum;
                    Hsc.Value = result;
                }
            }
            else
            {
                if (Vsc.Visible)
                {
                    int result = Vsc.Value - e.Delta / 60;
                    if (result < 0) result = 0;
                    if (result > Vsc.Maximum) result = Vsc.Maximum;
                    Vsc.Value = result;
                }
            }
        }

        void Vsc_ValueChanged(object sender, EventArgs e)
        {
            if (Scrolled != null)
            {
                Scrolled.Invoke(this, new ScrollEventArgs(ScrollEventType.First, 0));
            }
            DrawandReflash();
        }
        private void MyTreeView_Paint(object sender, PaintEventArgs e)
        {
            grafx.Render(e.Graphics);
        }
        private void DrawToBuffer(Graphics g)
        {
            SolidBrush sb = new SolidBrush(this.BackColor);
            g.FillRectangle(sb, new Rectangle(0, 0, this.Width, this.Height));
            sb.Dispose();
            if (BackgroundImage != null)
            {
                g.DrawImage(BackgroundImage, 0, 0);
            }
            foreach (TreeNode node in Nodes)
            {
                int ui = 0;
                DrawNode(g, node, 0, out ui);
            }
        }
        private void DrawNode(Graphics g, TreeNode node, int treeindex, out int usedindex)
        {
            usedindex = 0;
            if (node == null) return;
            NodeState? ns = node.Tag as NodeState?;
            if (ns != null)
            {
                if ((ns & NodeState.invisible) == NodeState.invisible) return;
            }
            usedindex++;
            if (Vsc.Visible)
            {
                if (treeindex > this.Height / ItemHeight + Vsc.Value)
                {
                    return;
                }
            }
            if (treeindex < Vsc.Value)
            {
            }
            else
            {
                int zure = Indent * (node.Level - 1) - Hsc.Value;
                int y = (treeindex - Vsc.Value) * ItemHeight;
                Pen Black = new Pen(Color.FromArgb(0, 0, 0));
                Black.DashStyle = System.Drawing.Drawing2D.DashStyle.Dot;
                if (node.Parent != null)
                {
                    if (node.NextNode != null)
                    {
                        g.DrawLine(Black, new Point(zure + Indent / 2, y), new Point(zure + Indent / 2, y + ItemHeight));
                        g.DrawLine(Black, new Point(zure + Indent / 2, y + ItemHeight / 2 - 1), new Point(zure + Indent, y + ItemHeight / 2 - 1));
                    }
                    else
                    {
                        g.DrawLine(Black, new Point(zure + Indent / 2, y), new Point(zure + Indent / 2, y + ItemHeight / 2));
                        g.DrawLine(Black, new Point(zure + Indent / 2, y + ItemHeight / 2 - 1), new Point(zure + Indent, y + ItemHeight / 2 - 1));
                    }
                }
                TreeNode temp = node.Parent;
                while (temp != null)
                {
                    if (temp.Parent != null && temp.NextNode != null)
                    {
                        int zu = Indent * (temp.Level - 1) - Hsc.Value;
                        g.DrawLine(Black, new Point(zu + Indent / 2, y), new Point(zu + Indent / 2, y + ItemHeight));
                    }
                    temp = temp.Parent;
                }
                int stringwidth = (int)g.MeasureString(node.Text, Font).Width;
                int imagewidth = 18;
                if (node == SelectedNode)
                {
                    g.FillRectangle(Brushes.LightGray, new Rectangle(Indent + zure + imagewidth, y, stringwidth, Font.Height + 5));
                }
                if (node == FocusedNode)
                {
                    g.DrawRectangle(Black, new Rectangle(Indent + zure + imagewidth, y, stringwidth, Font.Height + 5));
                }
                Black.Dispose();
                if (node.Nodes.Count > 0)
                {
                    Rectangle rec = new Rectangle(zure + 4, y + 3, 10, 10);
                    LinearGradientBrush lgb = new LinearGradientBrush(rec, Color.FromArgb(240, 240, 240), Color.FromArgb(200, 200, 250), LinearGradientMode.ForwardDiagonal);
                    g.FillRectangle(lgb, rec);
                    g.DrawRectangle(Pens.DarkSlateGray, rec);
                    lgb.Dispose();
                    if (!node.IsExpanded)
                    {
                        g.DrawLine(Pens.Black, new Point(rec.X + rec.Width / 2, rec.Y + 2), new Point(rec.X + rec.Width / 2, rec.Y + rec.Height - 2));
                    }
                    g.DrawLine(Pens.Black, new Point(rec.X + 2, rec.Y + rec.Height / 2), new Point(rec.X + rec.Width - 2, rec.Y + rec.Height / 2));
                }
                int index = ImageList.Images.IndexOfKey(node.ImageKey);
                if (index >= 0)
                {
                    Image image = ImageList.Images[index];
                    if ((node.Tag as string) != "forcut")
                    {
                        g.DrawImage(image, Indent + zure, y);
                    }
                    else
                    {
                        System.Drawing.Imaging.ColorMatrix cm =
                            new System.Drawing.Imaging.ColorMatrix();
                        //ColorMatrixの行列の値を変更して、アルファ値が0.5に変更されるようにする
                        cm.Matrix00 = 1;
                        cm.Matrix11 = 1;
                        cm.Matrix22 = 1;
                        cm.Matrix33 = 0.4f;
                        cm.Matrix44 = 1;

                        //ImageAttributesオブジェクトの作成
                        System.Drawing.Imaging.ImageAttributes ia =
                            new System.Drawing.Imaging.ImageAttributes();
                        //ColorMatrixを設定する
                        ia.SetColorMatrix(cm);
                        g.DrawImage(image, new Rectangle(Indent + zure, y, image.Width, image.Height), 0, 0, image.Width, image.Height, GraphicsUnit.Pixel, ia);
                    }
                }
                if (node == HotNode)
                {
                    bool selected = (node == SelectedNode);
                    g.DrawString(node.Text, Font, (selected ? Brushes.Black : Brushes.SkyBlue), new PointF(Indent + zure + imagewidth, y + 2));
                    g.DrawLine((selected ? Pens.Black : Pens.SkyBlue), new Point(Indent + zure + imagewidth + 2, y + Font.Height + 2), new Point(Indent + zure + imagewidth + stringwidth - 3, y + Font.Height + 2));
                }
                else
                {
                    g.DrawString(node.Text, Font, Brushes.Black, new PointF(Indent + zure + imagewidth, y + 2));
                }
            }
            if (node.IsExpanded)
            {
                foreach (TreeNode tn in node.Nodes)
                {
                    int ui = 0;
                    DrawNode(g, tn, treeindex + usedindex, out ui);
                    usedindex += ui;
                }
            }
        }
        public Point GetNodePoint(TreeNode node)
        {
            Point ret;
            if (node == Nodes[0])
            {
                ret = new Point(0, -ItemHeight * Vsc.Value);
            }
            else
            {
                bool f;
                int index = RecursiveGetNodeIndex(Nodes[0], node, out f);
                ret = new Point((node.Level + 1) * Indent, ItemHeight * (index - Vsc.Value + 1));
            }
            return ret;
        }
        private int RecursiveGetNodeIndex(TreeNode serarchnode, TreeNode targetnode, out bool found)
        {
            found = false;
            int ret = 0;
            foreach (TreeNode tn in serarchnode.Nodes)
            {
                if (targetnode == tn)
                {
                    found = true;
                    break;
                }
                else
                {
                    ret++;
                    if (tn.IsExpanded)
                    {
                        bool f = false;
                        int temp = RecursiveGetNodeIndex(tn, targetnode, out f);
                        if (f)
                        {
                            ret += temp;
                            found = true;
                            break;
                        }
                        ret += temp;
                    }
                }
            }
            return ret;
        }
        private TreeNode GetTreeNodeFromPointY(int Y)
        {
            TreeNode ret = null;
            int startindex = Vsc.Value;
            int targetindex = Y / ItemHeight + startindex;
            int ui = 0;
            foreach (TreeNode node in Nodes)
            {
                ret = RecusiveGetTreeNode(node, targetindex - ui, out ui);
                if (ret != null) break;
            }
            return ret;
        }
        private TreeNode RecusiveGetTreeNode(TreeNode node, int index, out int useindex)
        {
            useindex = 0;
            NodeState? ns = node.Tag as NodeState?;
            if (ns != null)
            {
                if ((ns & NodeState.invisible) == NodeState.invisible) return null;
            }
            useindex++;
            if (index == 0)
            {
                return node;
            }
            if (node.IsExpanded)
            {
                foreach (TreeNode tn in node.Nodes)
                {
                    int ui = 0;
                    TreeNode temp = RecusiveGetTreeNode(tn, index - useindex, out ui);
                    if (temp != null)
                    {
                        return temp;
                    }
                    else
                    {
                        useindex += ui;
                    }
                }
            }
            return null;
        }
        private void MyTreeView_SizeChanged(object sender, EventArgs e)
        {
            if (this.Width > 0 && this.Height > 0)
            {
                context = BufferedGraphicsManager.Current;
                context.MaximumBuffer = new Size(this.Width + 1, this.Height + 1);
                grafx = context.Allocate(this.CreateGraphics(), new Rectangle(0, 0, this.Width, this.Height));
                DrawandReflash();
            }
            AdjustSize();
        }
        public void DrawandReflash()
        {
            DrawToBuffer(grafx.Graphics);
            Refresh();
        }
        public TreeNodeCollection Nodes
        {
            get
            {
                return parentnode.Nodes;
            }
        }
        public int ItemHeight
        {
            get;
            set;
        }
        public int Indent
        {
            get;
            set;
        }
        public ImageList ImageList
        {
            get
            {
                return imagelist;
            }
            set
            {
                imagelist = value;
            }
        }
        public TreeViewDrawMode DrawMode
        {
            get;
            set;
        }
        public bool HotTracking
        {
            get;
            set;
        }
        public string GetFullPath(TreeNode node)
        {
            if (node.Level <= 1) return "";
            string ret = node.Text;
            TreeNode temp = node.Parent;
            while (temp != null)
            {
                if (temp.Level < 2) break;
                ret = temp.Text + "\\" + ret;
                temp = temp.Parent;
            }
            return ret;
        }
        private void MyTreeView_Click(object sender, EventArgs e)
        {
            Point pos = this.PointToClient(Cursor.Position);
            pos = new Point(pos.X + Hsc.Value, pos.Y);
            TreeNode node = GetTreeNodeFromPointY(pos.Y);
            if (node != null)
            {
                if (pos.X >= Indent * (node.Level - 1) && pos.X <= Indent * node.Level)
                {
                    if (node.Nodes.Count > 0)
                    {
                        if (node.IsExpanded)
                        {
                            if (BeforeCollapse != null)
                            {
                                BeforeCollapse.Invoke(this, new TreeViewCancelEventArgs(node, false, TreeViewAction.Collapse));
                            }
                            node.Collapse();
                            if (AfterCollapse != null)
                            {
                                AfterCollapse.Invoke(this, new TreeViewEventArgs(node));
                            }
                        }
                        else
                        {
                            if (BeforeExpand != null)
                            {
                                BeforeExpand.Invoke(this, new TreeViewCancelEventArgs(node, false, TreeViewAction.Expand));
                            }
                            node.Expand();
                            if (AfterExpand != null)
                            {
                                AfterExpand.Invoke(this, new TreeViewEventArgs(node));
                            }
                        }
                    }
                }
                else if (pos.X > Indent * (node.Level) + 18 && pos.X <= Indent * (node.Level) + 16 + NodeTextWidth(node))
                {
                    SelectedNode = node;
                    FocusedNode = node;
                    if (NodeMouseClick != null)
                    {
                        this.NodeMouseClick.Invoke(this, new TreeNodeMouseClickEventArgs(node, button, 1, pos.X, pos.Y));
                    }
                }
                AdjustSize();
                DrawandReflash();
            }
        }
        public void RightClick()
        {
            if (Selectednode != null && this.NodeMouseClick != null)
            {
                Point pos = GetNodePoint(Selectednode);
                this.NodeMouseClick.Invoke(this, new TreeNodeMouseClickEventArgs(Selectednode, MouseButtons.Right, 1, pos.X, pos.Y + ItemHeight));
            }
        }
        private void AdjustSize()
        {
            int width = 0, height = 0;
            foreach (TreeNode tn in Nodes)
            {
                GetMaxSize(tn, out width, out height);
                break;
            }
            if (this.Height < height * ItemHeight)
            {
                Vsc.Visible = true;
                Vsc.Maximum = height - this.Height / ItemHeight;
                Vsc.Height = this.Height;
                Vsc.Location = new Point(this.Width - Vsc.Width, 0);
            }
            else
            {
                Vsc.Visible = false;
                Vsc.Value = 0;
            }
            if (this.Width < width)
            {
                Hsc.Visible = true;
                Hsc.Maximum = width - this.Width;
                Hsc.Width = Vsc.Visible ? this.Width - Vsc.Width : this.Width;
                Hsc.Location = new Point(0, this.Height - Hsc.Height);
            }
            else
            {
                Hsc.Visible = false;
                Hsc.Value = 0;
            }
            if (Hsc.Visible)
            {
                if (this.Height - Hsc.Height < height * ItemHeight)
                {
                    Vsc.Visible = true;
                    Vsc.Maximum = height + (Hsc.Height / ItemHeight == 0 ? 1 : Hsc.Height / ItemHeight) - this.Height / ItemHeight;
                    Vsc.Height = this.Height;
                    Vsc.Location = new Point(this.Width - Vsc.Width, 0);
                }
            }
        }
        private void GetMaxSize(TreeNode node, out int width, out int index)
        {
            index = 0;
            NodeState? ns = node.Tag as NodeState?;
            if (ns != null)
            {
                if ((ns & NodeState.invisible) == NodeState.invisible)
                {
                    width = 0;
                    return;
                }
            }
            index++;
            width = (node.Level + 1) * Indent + NodeTextWidth(node) + 30;
            if (node.IsExpanded)
            {
                foreach (TreeNode tn in node.Nodes)
                {
                    int outwidth, outindex;
                    GetMaxSize(tn, out outwidth, out outindex);
                    width = Math.Max(width, outwidth);
                    index += outindex;
                }
            }
        }
        public void ToggleSelectedNode()
        {
            if (SelectedNode != null && SelectedNode.Nodes.Count > 0)
            {
                if (SelectedNode.IsExpanded)
                {
                    if (BeforeCollapse != null)
                    {
                        BeforeCollapse.Invoke(this, new TreeViewCancelEventArgs(SelectedNode, false, TreeViewAction.Collapse));
                    }
                    SelectedNode.Collapse();
                    if (AfterCollapse != null)
                    {
                        AfterCollapse.Invoke(this, new TreeViewEventArgs(SelectedNode));
                    }
                }
                else
                {
                    if (BeforeExpand != null)
                    {
                        BeforeExpand.Invoke(this, new TreeViewCancelEventArgs(SelectedNode, false, TreeViewAction.Expand));
                    }
                    SelectedNode.Expand();
                    if (AfterExpand != null)
                    {
                        AfterExpand.Invoke(this, new TreeViewEventArgs(SelectedNode));
                    }
                }
            }
        }
        public int NodeTextWidth(TreeNode node)
        {
            return (int)grafx.Graphics.MeasureString(node.Text, Font).Width;
        }
        public void DownSelectedNode()
        {
            if (SelectedNode != null)
            {
                TreeNode prev = SelectedNode;
                if (SelectedNode.IsExpanded)
                {
                    if (SelectedNode.Nodes.Count > 0)
                    {
                        SelectedNode = SelectedNode.Nodes[0];
                    }
                }
                else if (SelectedNode.NextNode != null)
                {
                    SelectedNode = SelectedNode.NextNode;
                }
                else if (SelectedNode.Parent != null)
                {
                    TreeNode temp = GetNextNodeFromParent(SelectedNode);
                    if (temp != null)
                    {
                        SelectedNode = temp;
                    }
                }
                if (prev != SelectedNode && Vsc.Value < Vsc.Maximum) Vsc.Value++;
            }
            FocusedNode = SelectedNode;
            HotNode = null;
        }
        private TreeNode GetNextNodeFromParent(TreeNode node)
        {
            if (node.Parent == null) return null;
            if (node.Parent.Parent == null) return null;
            if (node.Parent.NextNode == null) return GetNextNodeFromParent(node.Parent);
            return node.Parent.NextNode;
        }
        public void UpSelectedNode()
        {
            if (SelectedNode != null)
            {
                TreeNode prev = SelectedNode;
                if (SelectedNode.Parent.Parent == null)
                {

                }
                else if (SelectedNode.PrevNode != null)
                {
                    TreeNode temp = GetOpenLastTreeNode(SelectedNode.PrevNode);
                    if (temp != null) SelectedNode = temp;
                }
                else if (SelectedNode.Parent != null)
                {
                    SelectedNode = SelectedNode.Parent;
                }
                if (prev != SelectedNode && Vsc.Value > Vsc.Minimum) Vsc.Value--;
            }
            FocusedNode = SelectedNode;
            HotNode = null;
        }
        private TreeNode GetOpenLastTreeNode(TreeNode node)
        {
            if (!node.IsExpanded || node.Nodes.Count == 0) return node;
            return GetOpenLastTreeNode(node.Nodes[node.Nodes.Count - 1]);
        }
        public TreeNode Selectednode
        {
            get
            {
                return SelectedNode;
            }
            set
            {
                SelectedNode = value;
            }
        }
        public TreeNode Focusednode
        {
            get
            {
                return FocusedNode;
            }
        }
        public TreeNode Hotnode
        {
            get
            {
                return HotNode;
            }
        }
        private int CountCharInString(string st, string target)
        {
            int before = st.Length;
            st = st.Replace(target, "");
            int after = st.Length;
            return before - after;
        }
        private void MyTreeView_DoubleClick(object sender, EventArgs e)
        {
            Point pos = this.PointToClient(Cursor.Position);
            TreeNode node = GetTreeNodeFromPointY(pos.Y);
            if (node != null)
            {
                if (NodeMouseDoubleClick != null)
                {
                    NodeMouseDoubleClick.Invoke(this, new TreeNodeMouseClickEventArgs(node, button, 2, pos.X, pos.Y));
                }
            }
        }

        private void MyTreeView_MouseMove(object sender, MouseEventArgs e)
        {
            Point pos = this.PointToClient(Cursor.Position);
            pos = new Point(pos.X + Hsc.Value, pos.Y);
            TreeNode node = GetTreeNodeFromPointY(pos.Y);
            if (node != null)
            {
                if (pos.X > Indent * (node.Level) + 18 && pos.X <= Indent * (node.Level) + 16 + NodeTextWidth(node))
                {
                    HotNode = node;
                }
                else
                {
                    HotNode = null;
                }
                DrawandReflash();
            }
        }

        private void MyTreeView_BackgroundImageChanged(object sender, EventArgs e)
        {
            DrawandReflash();
        }

        private void MyTreeView_MouseLeave(object sender, EventArgs e)
        {
            HotNode = null;
            DrawandReflash();
        }

        private void MyTreeView_MouseDown(object sender, MouseEventArgs e)
        {
            button = e.Button;
        }

    }
}
