﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NotepadNeueExtension;
using ParseExtension.RegularExpressions;

namespace ParseExtension
{
    class CParser : ParserBase
    {
        private static ParserRegex includeRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexTextContent("#include"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit}),
                1, ParserRegexCountType.Fixed),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexTextContent("<"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.Symbol}),
                1, ParserRegexCountType.Fixed),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexExcludeType(new TokenType[]{TokenType.NewLine}),
                1, ParserRegexCountType.MoreThanEqual, "FileName"),
            new ParserRegexExpression(new ParserRegexTextContent(">"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.Symbol}),
                1, ParserRegexCountType.Fixed),
        });

        private static ParserRegex includeRegexQuote = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexTextContent("#include"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit}),
                1, ParserRegexCountType.Fixed),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexFuncContent(s=>s.StartsWith("\"")),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Literal}),
                1, ParserRegexCountType.MoreThanEqual, "FileName"),
        });

        private static ParserRegex defineRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexTextContent("#define"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed, "Key"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexExcludeType(new TokenType[]{ TokenType.NewLine}),
                0, ParserRegexCountType.MoreThanEqual, "Value"),
        });

        private static ParserRegex typedefRegex = new ParserRegex(new ParserRegexExpression[]{            
            new ParserRegexExpression(new ParserRegexTextContent("typedef"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit, TokenType.WhiteSpace}),
                1, ParserRegexCountType.MoreThanEqual, "Type"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit}),
                1, ParserRegexCountType.Fixed, "NewType"),
            new ParserRegexExpression(new ParserRegexTextContent(";"),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol}),
                1, ParserRegexCountType.Fixed),
        });

        private static ParserRegex typedefNoTypeRegex = new ParserRegex(new ParserRegexExpression[]{            
            new ParserRegexExpression(new ParserRegexTextContent("typedef"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit, TokenType.WhiteSpace, TokenType.NewLine}),
                1, ParserRegexCountType.MoreThanEqual, "Type"),
            new ParserRegexExpression(new ParserRegexTextContent("{"),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol}),
                1, ParserRegexCountType.Fixed),
        });

        private static ParserRegex enumValueRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed, "Key"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexTextContent("="),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol}),
                1, ParserRegexCountType.Fixed),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed, "Value"),
        });

        private static ParserRegex structRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexTextContent("struct"),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed, "Struct"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed, "Name"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace, TokenType.NewLine}),
                0, ParserRegexCountType.MoreThanEqual),
            new ParserRegexExpression(new ParserRegexTextContent("{"),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol}),
                1, ParserRegexCountType.Fixed),
        });

        private static ParserRegex funcRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexFuncContent(s=>!s.Contains("=") && !s.Contains("return")),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit, TokenType.Symbol, TokenType.WhiteSpace}),
                1, ParserRegexCountType.MoreThanEqual,"Type"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{ TokenType.LetterOrDigit }),
                1, ParserRegexCountType.Fixed, "Name"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.WhiteSpace}),
                0, ParserRegexCountType.ZeroOrOne),
            new ParserRegexExpression(new ParserRegexTextContent("("),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol}),
                1, ParserRegexCountType.Fixed),
        });

        private static ParserRegex variableRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit, TokenType.WhiteSpace}),
            1, ParserRegexCountType.MoreThanEqual, "Type"),
            new ParserRegexExpression(new ParserRegexTokenContent(t=> (t.TokenType == TokenType.Symbol && t.Token==  "*") || t.TokenType == TokenType.WhiteSpace),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol, TokenType.WhiteSpace}),
            0, ParserRegexCountType.MoreThanEqual, "Pointer"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit}),
            1, ParserRegexCountType.Fixed, "Name"),
        });

        private static ParserRegex variableDefineRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit, TokenType.WhiteSpace}),
            1, ParserRegexCountType.MoreThanEqual, "Type"),
            new ParserRegexExpression(new ParserRegexTokenContent(t=> (t.TokenType == TokenType.Symbol && t.Token==  "*") || t.TokenType == TokenType.WhiteSpace),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol, TokenType.WhiteSpace}),
            0, ParserRegexCountType.MoreThanEqual, "Pointer"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit}),
            1, ParserRegexCountType.Fixed, "Name"),
        });

        private static ParserRegex variableDefineContinuousRegex = new ParserRegex(new ParserRegexExpression[]{
            new ParserRegexExpression(new ParserRegexTokenContent(t=> (t.TokenType == TokenType.Symbol && t.Token==  "*") || t.TokenType == TokenType.WhiteSpace),
                new ParserRegexIncludeType(new TokenType[]{TokenType.Symbol, TokenType.WhiteSpace}),
            0, ParserRegexCountType.MoreThanEqual, "Pointer"),
            new ParserRegexExpression(new ParserRegexAnyContent(),
                new ParserRegexIncludeType(new TokenType[]{TokenType.LetterOrDigit}),
            1, ParserRegexCountType.Fixed, "Name"),
        });

        public override TokenizerBase Tokenizer
        {
            get { return new CTokenizer(); }
        }

        public CParser(IExtensionHost host)
            : base(host)
        {
        }

        protected override int ParseImpl(int i, ParseToken[] tokens, NotepadNeueExtension.BasicTree tree,
            NotepadNeueExtension.ObjectInfo objectInfo, NotepadNeueExtension.ParseEventArgs e)
        {
            for (; i < tokens.Length; i++)
            {
                switch (tokens[i].TokenType)
                {
                    case TokenType.LetterOrDigit:
                        if (tokens[i].Token.StartsWith("#"))
                        {
                            var match = includeRegex.Match(tokens, i);
                            if (!match.IsSuccess)
                            {
                                match = includeRegexQuote.Match(tokens, i);
                            }
                            if (match.IsSuccess)
                            {
                                string fileName = match["FileName"].Join();
                                if (fileName.StartsWith("\""))
                                {
                                    fileName = fileName.Trim('"', ' ');
                                }
                                Host.ReadBuffer(fileName, e.Filename);
                                e.AssistData.LoadedBufferName.Add(fileName);
                                i = match.StartIndex + match.Length - 1;
                                continue;
                            }
                            match = defineRegex.Match(tokens, i);
                            if (match.IsSuccess)
                            {
                                string key = match["Key"].Join(),
                                    value = match["Value"].Join();
                                e.AssistData.GlobalDefineValue.Add(new DefineValueInfo()
                                {
                                    DefineLocation = new DefineLocation()
                                    {
                                        DefineIndex = match["Key"][0].Index,
                                        Editor = e.AssistData.Editor,
                                    },
                                    Name = key,
                                    Value = value,
                                });
                                i = match.StartIndex + match.Length - 1;
                                continue;
                            }
                            i = SkipTo(TokenType.NewLine, i + 1, tokens);
                        }
                        else
                        {
                            switch (tokens[i].Token)
                            {
                                case "typedef":
                                    var match = typedefRegex.Match(tokens, i);
                                    if (match.IsSuccess)
                                    {
                                        string type = match["Type"].Join().Trim(),
                                            newType = match["NewType"].Join();
                                        i = match.StartIndex + match.Length - 1;

                                        ObjectInfo target = e.AssistData.GlobalClass.FirstOrDefault(o => o.Name == type);
                                        if (target != null)
                                        {
                                            ObjectInfo newObject = new ObjectInfo()
                                            {
                                                DefineLocation = new DefineLocation()
                                                {
                                                    DefineIndex = match["Type"][0].Index,
                                                    Editor = e.AssistData.Editor,
                                                },
                                                Name = newType,
                                            };
                                            foreach (VariableInfo variableInfo in target.GetAllVariable())
                                            {
                                                newObject.AddVariable(variableInfo, variableInfo.Name);
                                            }
                                            e.AssistData.GlobalClass.Add(newObject);
                                        }
                                        continue;
                                    }
                                    match = typedefNoTypeRegex.Match(tokens, i);
                                    if (match.IsSuccess)
                                    {
                                        string type = match["Type"].Join().Trim();
                                        if (type.Contains("struct"))
                                        {
                                            BasicTree newTree = new BasicTree()
                                            {
                                                Start = tokens[match.StartIndex + match.Length - 1].Index,
                                            };
                                            tree.AddData(new EitherVariableOrTree()
                                            {
                                                BasicTree = newTree
                                            }, tokens[match.StartIndex + match.Length - 1].Index);
                                            ObjectInfo newObject = new ObjectInfo();
                                            i = ParseImpl(match.StartIndex + match.Length, tokens, newTree, newObject, e);
                                            int next = SkipTo(TokenType.LetterOrDigit, i, tokens);
                                            if (next >= tokens.Length)
                                            {
                                                return next;
                                            }
                                            string newType = tokens[next].Token;
                                            int newTypeIndex = tokens[next].Index;
                                            newObject.DefineLocation = new DefineLocation()
                                            {
                                                DefineIndex = newTypeIndex,
                                                Editor = e.AssistData.Editor,
                                            };
                                            newObject.Name = newType;
                                            e.AssistData.GlobalClass.Add(newObject);
                                            i = next;
                                        }
                                        else if (type.Contains("enum"))
                                        {
                                            int currentIndex = match.StartIndex + match.Length;
                                            currentIndex = SkipTo(TokenType.LetterOrDigit, currentIndex, tokens);
                                            int semiBraceEndIndex = SkipTo("}", TokenType.Symbol, currentIndex, tokens);
                                            if (semiBraceEndIndex >= tokens.Length)
                                            {
                                                return semiBraceEndIndex;
                                            }
                                            while (currentIndex < semiBraceEndIndex)
                                            {
                                                match = enumValueRegex.Match(tokens, currentIndex);
                                                if (!match.IsSuccess)
                                                {
                                                    break;
                                                }

                                                e.AssistData.GlobalDefineValue.Add(new DefineValueInfo()
                                                {
                                                    DefineLocation = new DefineLocation()
                                                    {
                                                        DefineIndex = tokens[match.StartIndex].Index,
                                                        Editor = e.AssistData.Editor,
                                                    },
                                                    Name = match["Key"].Join(),
                                                    Value = match["Value"].Join(),
                                                });

                                                currentIndex = match.StartIndex + match.Length + 1;
                                                currentIndex = SkipTo(TokenType.LetterOrDigit, currentIndex, tokens);
                                            }
                                            int newTypeIndex = SkipTo(TokenType.LetterOrDigit, semiBraceEndIndex + 1, tokens);
                                            if (newTypeIndex >= tokens.Length)
                                            {
                                                return newTypeIndex;
                                            }
                                            e.AssistData.GlobalClass.Add(new ObjectInfo()
                                            {
                                                DefineLocation = new DefineLocation()
                                                {
                                                    DefineIndex = newTypeIndex,
                                                    Editor = e.AssistData.Editor,
                                                },
                                                Name = tokens[newTypeIndex].Token.Trim(),
                                            });
                                            i = newTypeIndex;
                                        }
                                    }
                                    break;
                                case "struct":
                                    match = structRegex.Match(tokens, i);
                                    if (match.IsSuccess)
                                    {
                                        string name = String.Format("struct {0}", match["Name"].Join());
                                        ObjectInfo newObject = new ObjectInfo()
                                        {
                                            Name = name,
                                            DefineLocation = new DefineLocation()
                                            {
                                                DefineIndex = match["Struct"][0].Index,
                                                Editor = e.AssistData.Editor,
                                            },
                                        };
                                        BasicTree newTree = new BasicTree()
                                        {
                                            Start = tokens[match.StartIndex + match.Length - 1].Index,
                                        };
                                        tree.AddData(new EitherVariableOrTree()
                                        {
                                            BasicTree = newTree
                                        }, newTree.Start);
                                        i = ParseImpl(match.StartIndex + match.Length, tokens, newTree, newObject, e);
                                        e.AssistData.GlobalClass.Add(newObject);
                                        continue;
                                    }
                                    goto default;
                                case "if":
                                case "for":
                                case "while":
                                case "switch":
                                case "else if":
                                    int braceStartIndex = SkipTo("(", TokenType.Symbol, i + 1, tokens);
                                    if (braceStartIndex >= tokens.Length)
                                    {
                                        return braceStartIndex;
                                    }
                                    int braceEndIndex = SkipToMatchedBrace(braceStartIndex, tokens);
                                    if (braceEndIndex >= tokens.Length)
                                    {
                                        return braceEndIndex;
                                    }
                                    i = braceEndIndex;
                                    break;
                                case "else":
                                    break;
                                default:
                                    match = funcRegex.Match(tokens, i);
                                    if (match.IsSuccess)
                                    {
                                        braceStartIndex = match.StartIndex + match.Length - 1;
                                        braceEndIndex = SkipToMatchedBrace(braceStartIndex, tokens);
                                        if (braceEndIndex >= tokens.Length)
                                        {
                                            return braceEndIndex;
                                        }

                                        int semiBraceStartIndex = SkipTo(new String[] { "{", ";" }, TokenType.Symbol, braceEndIndex + 1, tokens);
                                        if (semiBraceStartIndex >= tokens.Length)
                                        {
                                            return semiBraceStartIndex;
                                        }

                                        BasicTree newTree = null;

                                        if (tokens[semiBraceStartIndex].Token == "{")
                                        {
                                            newTree = new BasicTree()
                                            {
                                                Start = tokens[semiBraceStartIndex].Index,
                                            };
                                            tree.AddData(new EitherVariableOrTree()
                                            {
                                                BasicTree = newTree,
                                            }, newTree.Start);
                                        }

                                        List<VariableInfo> variables = new List<VariableInfo>();
                                        int currentVariableIndex = braceStartIndex + 1;
                                        while (currentVariableIndex < braceEndIndex)
                                        {
                                            var tempMatch = variableRegex.Match(tokens, currentVariableIndex);
                                            if (!tempMatch.IsSuccess)
                                            {
                                                break;
                                            }

                                            variables.Add(new VariableInfo()
                                            {
                                                DefineLocation = new DefineLocation()
                                                {
                                                    DefineIndex = tempMatch["Name"][0].Index,
                                                    Editor = e.AssistData.Editor,
                                                },
                                                Name = tempMatch["Name"].Join(),
                                                Type = String.Format("{0}{1}", tempMatch["Type"].Join().Trim(),
                                                    tempMatch["Pointer"].Join().Replace(" ", "")),
                                            });

                                            currentVariableIndex = SkipTo(new string[] { ",", ")" }, TokenType.Symbol,
                                                tempMatch.StartIndex + tempMatch.Length, tokens) + 1;
                                        }

                                        if (newTree != null)
                                        {
                                            int iter = 0;
                                            foreach (VariableInfo variable in variables)
                                            {
                                                newTree.AddData(new EitherVariableOrTree()
                                                {
                                                    VariableInfo = variable,
                                                }, tokens[semiBraceStartIndex].Index + ++iter);
                                            }
                                        }

                                        string joinedText = tokens.Skip(braceStartIndex + 1).
                                        Take(braceEndIndex - braceStartIndex - 1).Aggregate("", (prev, p2) =>
                                        {
                                            switch (p2.TokenType)
                                            {
                                                case TokenType.Symbol:
                                                case TokenType.LetterOrDigit:
                                                    if (String.IsNullOrEmpty(prev))
                                                    {
                                                        return p2.Token;
                                                    }
                                                    else if (p2.Token == ",")
                                                    {
                                                        return String.Format("{0}{1}", prev, p2.Token);
                                                    }
                                                    return String.Format("{0} {1}", prev, p2.Token);
                                                default:
                                                    return prev;
                                            }
                                        });

                                        int functionDefinitionIndex = match["Name"][0].Index;
                                        FunctionInfo functionInfo = new FunctionInfo()
                                        {
                                            DefineLocation = new DefineLocation()
                                            {
                                                DefineIndex = functionDefinitionIndex,
                                                Editor = e.AssistData.Editor,
                                            },
                                            Name = match["Name"].Join(),
                                            Type = match["Type"].Join().Trim(),
                                        };
                                        string[] arguments = joinedText.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries);
                                        if (arguments.Length > 0)
                                        {
                                            functionInfo.AddArgument(arguments);
                                        }
                                        functionInfo.AddContent(String.Format("{0} {1}({2})", functionInfo.Type, functionInfo.Name, joinedText));
                                        tree.AddFunctionData(functionInfo, functionDefinitionIndex);
                                        if (newTree != null)
                                        {
                                            i = ParseImpl(semiBraceStartIndex + 1, tokens, newTree, null, e);
                                        }
                                        else
                                        {
                                            i = semiBraceStartIndex;
                                        }
                                        continue;
                                    }
                                    match = variableDefineRegex.Match(tokens, i);
                                    if (match.IsSuccess)
                                    {
                                        string type = match["Type"].Join().Trim(),
                                            name = match["Name"].Join();
                                        if (type == "return")
                                        {
                                            i = SkipTo(";", TokenType.Symbol, i + 1, tokens);
                                            break;
                                        }
                                        string pointer = match["Pointer"].Join().Replace(" ", "");
                                        int definitionIndex = match["Name"][0].Index;
                                        VariableInfo variableInfo = new VariableInfo()
                                        {
                                            Type = String.Format("{0}{1}", type, pointer),
                                            Name = name,
                                            DefineLocation = new DefineLocation
                                            {
                                                DefineIndex = definitionIndex,
                                                Editor = e.AssistData.Editor,
                                            },
                                        };
                                        tree.AddData(new EitherVariableOrTree()
                                        {
                                            VariableInfo = variableInfo,
                                        }, definitionIndex);
                                        if (objectInfo != null)
                                        {
                                            objectInfo.AddVariable(variableInfo, variableInfo.Name);
                                        }
                                        i = SkipTo(new string[] { ";", "," }, TokenType.Symbol,
                                            match.StartIndex + match.Length, tokens);
                                        while (i < tokens.Length)
                                        {
                                            if (tokens[i].Token == ";")
                                            {
                                                break;
                                            }

                                            match = variableDefineContinuousRegex.Match(tokens, i + 1);
                                            if (!match.IsSuccess)
                                            {
                                                break;
                                            }

                                            pointer = match["Pointer"].Join().Replace(" ", "");
                                            variableInfo = new VariableInfo()
                                            {
                                                Type = String.Format("{0}{1}", type, pointer),
                                                Name = match["Name"].Join(),
                                                DefineLocation = new DefineLocation
                                                {
                                                    DefineIndex = match["Name"][0].Index,
                                                    Editor = e.AssistData.Editor,
                                                },
                                            };
                                            tree.AddData(new EitherVariableOrTree()
                                            {
                                                VariableInfo = variableInfo,
                                            }, tokens[i].Index);
                                            if (objectInfo != null)
                                            {
                                                objectInfo.AddVariable(variableInfo, variableInfo.Name);
                                            }

                                            i = SkipTo(new string[] { ";", "," }, TokenType.Symbol,
                                                match.StartIndex + match.Length, tokens);
                                        }
                                        continue;
                                    }
                                    i = SkipTo(new string[] { "}", ";", ":" }, TokenType.Symbol, i, tokens);
                                    if (i >= tokens.Length)
                                    {
                                        return i;
                                    }
                                    switch (tokens[i].Token)
                                    {
                                        case "}":
                                            tree.End = tokens[i].Index;
                                            return i;
                                    }
                                    break;
                            }
                        }
                        break;
                    case TokenType.Symbol:
                        switch (tokens[i].Token)
                        {
                            case "}":
                                tree.End = tokens[i].Index;
                                return i;
                            case "{":
                                BasicTree newTree = new BasicTree()
                                {
                                    Start = tokens[i].Index,
                                };
                                tree.AddData(new EitherVariableOrTree()
                                {
                                    BasicTree = newTree
                                }, newTree.Start);
                                i = ParseImpl(i + 1, tokens, newTree, null, e);
                                break;
                        }
                        break;
                }
            }

            return i;
        }

        protected override void ParseEnd(NotepadNeueExtension.ParseEventArgs e)
        {
            foreach (string bufferName in e.AssistData.LoadedBufferName)
            {
                NameSpaceInfo nsi = Host.GetBuffer(bufferName, e.Filename);
                if (nsi == null)
                {
                    continue;
                }
                foreach (VariableInfo vi in nsi.GetAllVariable())
                {
                    e.AssistData.GlobalVariable.Add(vi);
                }
                foreach (FunctionInfo fi in nsi.GetAllFunction())
                {
                    e.AssistData.GlobalFunction.Add(fi);
                }
                foreach (DefineValueInfo dvi in nsi.GetAllDefineValue())
                {
                    e.AssistData.GlobalDefineValue.Add(dvi);
                }
                foreach (ObjectInfo oi in nsi.GetAllObjectType())
                {
                    e.AssistData.GlobalClass.Add(oi);
                }
            }

            AutoClose(e.AssistData.VariableDatas, e.ParseText.Length);
        }

        private void AutoClose(BasicTree tree, int length)
        {
            if (tree.End == 0)
            {
                tree.End = length;
            }

            foreach (BasicTree childTree in tree.Trees)
            {
                AutoClose(childTree, length);
            }
        }
    }
}
