﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

using NotepadNeueExtension;

namespace ParseExtension
{
    class JavaScriptParser : IParser
    {
        enum State
        {
            none = 0,
            linecomment = 1,
            areacomment = 2,
            charssingle = 3,
            charsdouble = 4,
            regex = 5
        }
        Regex FunctionSearchRegex = new Regex("function *(?<name>[\\w_]*)");
        Regex VariableSearchRegex = new Regex("var +(?<name>\\w+)");
        Regex ObjectCreateRegex = new Regex("(?<name>\\w+)-\\>new");
        List<KeyValuePair<ObjectInfo, BasicTree>> pool = new List<KeyValuePair<ObjectInfo, BasicTree>>();
        const string areacommentstart = "=comment";
        const string areacommentend = "=cut";
        const string defaultpackage = "__PACKAGE__";
        IExtensionHost Host;
        public JavaScriptParser(IExtensionHost host)
        {
            this.Host = host;
        }

        public void Parse(NotepadNeueExtension.ParseEventArgs e)
        {
            EditorAssistData assistdata = e.AssistData;
            string text = e.ParseText;
            char[] array = e.ParseText.ToCharArray();
            State state = State.none;
            foreach (string bn in assistdata.LoadedBufferName)
            {
                NameSpaceInfo nsi = Host.GetBuffer(bn, e.Filename);
                if (nsi == null) continue;
                foreach (VariableInfo vi in nsi.GetAllVariable())
                {
                    assistdata.GlobalVariable.Add(vi);
                }
                foreach (FunctionInfo fi in nsi.GetAllFunction())
                {
                    assistdata.GlobalFunction.Add(fi);
                }
                foreach (DefineValueInfo dvi in nsi.GetAllDefineValue())
                {
                    assistdata.GlobalDefineValue.Add(dvi);
                }
                foreach (ObjectInfo oi in nsi.GetAllObjectType())
                {
                    assistdata.GlobalClass.Add(oi);
                }
            }
            int anchor = 0;
            BasicTree currenttree = assistdata.VariableDatas;
            currenttree.Start = 0;
            currenttree.End = array.Length;
            BasicTreeAndLevel btalevel = new BasicTreeAndLevel(currenttree);
            List<BasicTreeAndLevel> parents = new List<BasicTreeAndLevel>();
            parents.Add(btalevel);
            state = State.none;
            for (int i = 0; i < array.Length; i++)
            {
                char c = array[i];
                if (state == State.none)
                {
                    if (c == '/' && i + 1 < array.Length)
                    {
                        if (array[i + 1] == '/') state = State.linecomment;
                        else if (array[i + 1] == '*') state = State.areacomment;
                    }
                    else if (c == '\'') state = State.charssingle;
                    else if (c == '"') state = State.charsdouble;
#if FINDREGEX
                    else if (c == '/')
                    {
                        regexchar = array[i + 1];
                        state = State.regex;
                    }
#endif
                    else
                    {
                        while (i < array.Length && (char.IsLetterOrDigit(c = array[i]) || c == ' ' || c == '_')) i++;
                        if (c == '{')
                        {
                            if (parents.Count > 0)
                            {
                                parents[parents.Count - 1].LevelCount++;
                            }
                        }
                        else if (c == '}')
                        {
                            if (parents.Count > 0)
                            {
                                parents[parents.Count - 1].LevelCount--;
                                if (parents[parents.Count - 1].LevelCount == 0)
                                {
                                    parents.RemoveAt(parents.Count - 1);
                                    currenttree = parents[parents.Count - 1].BasicTree;
                                }
                            }
                        }
                        else if (c == '(')
                        {
                            string temp = SafeSubstring(ref text, anchor, i - anchor);
                            string name = "";
                            if (IsFunction(temp, out name))
                            {
                                //function
                                int rec = i + 1;
                                while (i < array.Length && array[i] != ')') i++;
                                string content = SafeSubstring(ref text, anchor, i - anchor) + ")";
                                if (content.Contains("\r") || content.Contains("\n")) continue;
                                content = content.Trim();
                                FunctionInfo fi = new FunctionInfo();
                                fi.Name = name;
                                fi.Type = "function";
                                fi.DefineLocation.DefineIndex = anchor + temp.IndexOf(name);
                                fi.DefineLocation.Editor = assistdata.Editor;
                                fi.AddContent(content);
                                currenttree.AddFunctionData(fi, fi.DefineLocation.DefineIndex);
                                BasicTree bt = new BasicTree();
                                bt.Start = i;
                                EitherVariableOrTree funcevot = new EitherVariableOrTree();
                                funcevot.BasicTree = bt;
                                currenttree.AddData(funcevot, i);
                                BasicTreeAndLevel btal = new BasicTreeAndLevel(bt);
                                parents.Add(btal);
                                currenttree = bt;
                                string args = SafeSubstring(ref text, rec, i - rec);
                                string[] spargs = args.Split(',');
                                int sum = 0;
                                foreach (string arg in spargs)
                                {
                                    VariableInfo vi = GetVariable(arg, assistdata);
                                    if (vi != null)
                                    {
                                        vi.DefineLocation.DefineIndex += rec + sum;
                                        vi.DefineLocation.Editor = assistdata.Editor;
                                        EitherVariableOrTree evot = new EitherVariableOrTree();
                                        evot.VariableInfo = vi;
                                        currenttree.AddData(evot, evot.VariableInfo.DefineLocation.DefineIndex);
                                    }
                                    sum += arg.Length + 1;
                                }
                                while (i < array.Length && ((c = array[i]) != '{' && c != ';')) i++;

                            }
                        }
                        else if (c == '=' || c == ';' || c == ',')
                        {
                            string temp = SafeSubstring(ref text, anchor, i - anchor);
                            temp = temp.Trim();
                            Match m = VariableSearchRegex.Match(temp);
                            if (m.Success)
                            {
                                AddAndCreate(m.Groups["name"].Value, "var", anchor + m.Groups["name"].Index, currenttree, assistdata.Editor);
                                if (c == ';') continue;
                                if (c == '=') i = SkipUntilCommaOrSemiColon(i, array); 
                                if (i == array.Length || array[i] == ';') continue;
                                i++; while (true)
                                {
                                    int rec = i;
                                    i = SkipUntilCommaOrSemiColon(i, array);
                                    temp = SafeSubstring(ref text, rec, i - rec);
                                    temp = temp.Trim();
                                    int begin;
                                    string name = GetName(temp, out begin);
                                    if (name != "" && begin >= 0)
                                    {
                                        AddAndCreate(name, "var", i - begin, currenttree, assistdata.Editor);
                                    }
                                    if (i == array.Length || array[i] == ';') break;
                                    i++;
                                }
                            }
                        }
                        else
                        {
                            anchor = i + 1;
                        }
                    }
                }
                else if (state == State.linecomment)
                {
                    if (c == '\r' || c == '\n') state = State.none;
                }
                else if (state == State.areacomment)
                {
                    if (c == '*' && i + 1 < array.Length)
                    {
                        state = array[i + 1] == '/' ? state = State.none : State.areacomment;
                        i++;
                    }
                }
                else if (state == State.charssingle)
                {
                    if (c == '\'' && array[i - 1] != '\\') state = State.none;
                }
                else if (state == State.charsdouble)
                {
                    if (c == '"' && array[i - 1] != '\\') state = State.none;
                }
#if FINDREGEX
                else if (state == State.regex)
                {
                    if (c == '/' && array[i - 1] != '\\') state = State.none;
                }
#endif
            }
            foreach (KeyValuePair<ObjectInfo, BasicTree> pair in pool)
            {
                foreach (FunctionInfo fi in pair.Value.Functions)
                {
                    pair.Key.AddMethod(fi, fi.Name);
                }
                assistdata.GlobalClass.Add(pair.Key);
            }
            currenttree = assistdata.VariableDatas;
        }
        private void SetObjectType(string str, VariableInfo vi)
        {
            str = str.Trim();
            Match m = ObjectCreateRegex.Match(str);
            if (m.Success)
            {
                string name = m.Groups["name"].Value;
                vi.Type = name;
            }
        }
        private bool FindVariable(string str, BasicTree bt, BasicTree topbt, int index, IEditorControl editor, bool restrict)
        {
            str = str.Trim();
            Match m = VariableSearchRegex.Match(str);
            if (m.Success)
            {
                string restriction = m.Groups["attri"].Value;
                string name = m.Groups["name"].Value;
                string type = "";
                if (name == "self")
                {
                    type = "SELF";
                }
                else
                {
                    type = m.Groups["type"].Value;
                }
                foreach (VariableInfo vi in bt.GetVariableData(index))
                {
                    if (vi.Name == name && vi.Type == type) return false;
                }
                restrict |= (restriction != null || restriction != "");
                AddAndCreate(name, type, restrict ? index : 0, restrict ? bt : topbt, editor);
                return true;
            }
            return false;
        }
        private bool FindVariable(string str, BasicTree bt, BasicTree topbt, int index, IEditorControl editor, bool restrict, out VariableInfo vi)
        {
            vi = null;
            str = str.Trim();
            Match m = VariableSearchRegex.Match(str);
            if (m.Success)
            {
                string restriction = m.Groups["attri"].Value;
                string name = m.Groups["name"].Value;
                string type = "";
                if (name == "self")
                {
                    type = "SELF";
                }
                else
                {
                    type = m.Groups["type"].Value;
                }
                foreach (VariableInfo vinfo in bt.GetVariableData(index))
                {
                    if (vinfo.Name == name && vinfo.Type == type) return false;
                }
                restrict |= (restriction != null || restriction != "");
                vi = AddAndCreate(name, type, restrict ? index : 0, restrict ? bt : topbt, editor);
                return true;
            }
            return false;
        }
        private int SkipUntilCommaOrSemiColon(int start, char[] array)
        {
            int i;
            State state = State.none;
            int largekakkocount = 0, smallkakkocount = 0;
            for (i = start; i < array.Length; i++)
            {
                char c = array[i];
                if (state == State.none)
                {
                    if (c == '/' && i + 1 < array.Length)
                    {
                        if (array[i + 1] == '/') state = State.linecomment;
                        else if (array[i + 1] == '*') state = State.areacomment;
                    }
                    else if (c == '\'') state = State.charssingle;
                    else if (c == '"') state = State.charsdouble;
                    else if (c == '{') largekakkocount++;
                    else if (c == '}') largekakkocount--;
                    else if (c == '(') smallkakkocount++;
                    else if (c == ')') smallkakkocount--;
                    else if ((c == ',' || c == ';') && largekakkocount == 0 && smallkakkocount == 0) break;
                }
                else if (state == State.linecomment)
                {
                    if (c == '\r' || c == '\n') state = State.none;
                }
                else if (state == State.areacomment)
                {
                    if (c == '*' && i + 1 < array.Length)
                    {
                        state = array[i + 1] == '/' ? state = State.none : State.areacomment;
                        i++;
                    }
                }
                else if (state == State.charssingle)
                {
                    if (c == '\'' && array[i - 1] != '\\') state = State.none;
                }
                else if (state == State.charsdouble)
                {
                    if (c == '"' && array[i - 1] != '\\') state = State.none;
                }
            }
            return i;
        }
        private string GetName(string data, out int begin)
        {
            //name = とかname
            string ret = null;
            begin = 0;
            char[] array = data.ToCharArray();
            if (array.Length == 1 && char.IsLetterOrDigit(array[0])) return data;
            int start = -1, end = -1;
            for (int i = 0; i < array.Length; i++)
            {
                char c = array[i];
                if (start < 0)
                {
                    if (char.IsLetterOrDigit(c)) start = i;
                }
                else if (end < 0)
                {
                    if (!char.IsLetterOrDigit(c))
                    {
                        end = i;
                        break;
                    }
                }
            }
            begin = start;
            if (end < 0) end = array.Length;
            ret = SafeSubstring(ref data, start, end - start);
            return ret;
        }
        private VariableInfo AddAndCreate(string name, string type, int defineindex, BasicTree bt, IEditorControl editor)
        {
            VariableInfo vi = new VariableInfo();
            vi.DefineLocation.DefineIndex = defineindex;
            vi.DefineLocation.Editor = editor;
            vi.Name = name;
            vi.Type = type;
            EitherVariableOrTree evot = new EitherVariableOrTree();
            evot.VariableInfo = vi;
            bt.AddData(evot, vi.DefineLocation.DefineIndex);
            return vi;
        }
        private FunctionInfo FindFunction(BasicTree bt, string name)
        {
            foreach (FunctionInfo fi in bt.Functions)
            {
                if (fi.Name == name) return fi;
            }
            return null;
        }

        private bool IsNotWord(int i, char[] target)
        {
            if (i < 0) return true;
            return !Char.IsLetterOrDigit(target[i]);
        }
        private bool IsNotWordAndNotWhiteSpace(int i, char[] target)
        {
            if (i < 0) return true;
            return !Char.IsLetterOrDigit(target[i]) && !Char.IsWhiteSpace(target[i]);
        }
        private string SafeSubstring(ref string src, int begin)
        {
            if (begin < 0) begin = 0;
            if (begin >= src.Length) begin = src.Length;
            return src.Substring(begin);
        }
        private string SafeSubstring(ref string src, int begin, int length)
        {
            if (begin < 0) begin = 0;
            if (length < 0) length = 0;
            if (begin >= src.Length) begin = src.Length;
            if (begin + length > src.Length) length = src.Length - begin;
            return src.Substring(begin, length);
        }
        private bool IsFunction(string str, out string name)
        {
            name = "";
            str = str.Trim();
            Match m = FunctionSearchRegex.Match(str);
            if (!m.Success) return false;
            name = m.Groups["name"].Value;
            if (name == "") name = "noname" + System.Environment.TickCount;
            return true;
        }
        private bool IsMy(string str)
        {
            str = str.Trim();
            return str == "my";
        }
        private void CopyData(EditorAssistData src, EditorAssistData dest)
        {
            /*foreach (DefineValueInfo dvi in src.GlobalDefineValue)
            {
                dest.GlobalDefineValue.Add(dvi);
            }*/
            foreach (ObjectInfo oi in src.GlobalClass)
            {
                dest.GlobalClass.Add(oi);
            }
            /*foreach (VariableInfo vi in src.VariableDatas.vinfos)
            {
                dest.GlobalVariable.Add(vi);
            }
            foreach (FunctionInfo fi in src.VariableDatas.finfos)
            {
                dest.GlobalFunction.Add(fi);
            }*/
        }
        private VariableInfo GetVariable(string data, EditorAssistData assistdata)
        {
            // data とか
            VariableInfo ret = new VariableInfo();
            ret.Name = data.TrimStart();
            ret.DefineLocation.DefineIndex = data.Length - ret.Name.Length;
            ret.Name = ret.Name.TrimEnd();
            ret.Type = "var";
            return ret;
        }
    }

    class BasicTreeAndLevel
    {
        public BasicTree BasicTree { get; set; }
        public int LevelCount { get; set; }
        public BasicTreeAndLevel(BasicTree bt)
        {
            this.BasicTree = bt;
            this.LevelCount = 1;
        }
    }
}