﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace ParseExtension.RegularExpressions
{
    class ParserRegex
    {
        private ParserRegexNode start;
        private ParserRegexNode end;

        public ParserRegexExpression[] Expressions
        {
            get;
            private set;
        }

        public ParserRegex(ParserRegexExpression[] expressions)
        {
            Expressions = expressions;
            Initialize();
        }

        private void Initialize()
        {
            start = new ParserRegexNode();
            ParserRegexNode current = start;
            foreach (ParserRegexExpression expression in Expressions)
            {
                switch (expression.CountType)
                {
                    case ParserRegexCountType.Fixed:
                        current = AddNode(expression, expression.Count, current);
                        break;
                    case ParserRegexCountType.MoreThanEqual:
                        current = AddNode(expression, expression.Count, current);
                        current = AddAstariskNode(expression, current);
                        break;
                    case ParserRegexCountType.ZeroOrOne:
                        current = AddZeroOrOneNode(expression, current);
                        break;
                }
            }
            end = current;
        }

        private ParserRegexNode AddZeroOrOneNode(ParserRegexExpression expression, ParserRegexNode current)
        {
            ParserRegexNode next = new ParserRegexNode();
            current.AddNext(next, expression);
            current.AddNext(next, null);
            return next;
        }

        private ParserRegexNode AddNode(ParserRegexExpression expression, int count, ParserRegexNode current)
        {
            for (int i = 0; i < count; i++)
            {
                ParserRegexNode next = new ParserRegexNode();
                current.AddNext(next, expression);
                current = next;
            }
            return current;
        }

        private ParserRegexNode AddAstariskNode(ParserRegexExpression expression, ParserRegexNode current)
        {
            ParserRegexNode node1 = new ParserRegexNode(),
                        node2 = new ParserRegexNode(),
                        node3 = new ParserRegexNode();
            current.AddNext(node1, null);
            current.AddNext(node3, null);
            node1.AddNext(node2, expression);
            node2.AddNext(node1, null);
            node2.AddNext(node3, null);
            return node3;
        }

        public ParserRegexMatch Match(ParseToken[] tokens, int startIndex)
        {
            ParserRegexBackTrack backTrack = new ParserRegexBackTrack(start, null);
            backTrack = Process(start, new ParserRegexContext(tokens, startIndex, end), backTrack);
            return new ParserRegexMatch(this, tokens, backTrack);
        }

        private ParserRegexBackTrack Process(ParserRegexNode node, ParserRegexContext context, ParserRegexBackTrack backTrack)
        {
            if (node == context.End)
            {
                return backTrack;
            }

            List<ParserRegexBackTrack> results = new List<ParserRegexBackTrack>();
            object lockObject = new object();
            Parallel.ForEach(node.Nexts, flow =>
            {
                if (flow.Expression == null)
                {
                    var temp = Process(flow.Dest, new ParserRegexContext(context.Tokens, context.Index, context.End),
                        new ParserRegexBackTrack(flow.Dest, backTrack));
                    if (temp != null)
                    {
                        lock (lockObject)
                        {
                            results.Add(temp);
                        }
                    }
                }
                else if (context.Index < context.Tokens.Length)
                {
                    if (flow.Expression.IsMatch(context.Tokens[context.Index]))
                    {
                        var temp = Process(flow.Dest, new ParserRegexContext(context.Tokens, context.Index + 1, context.End),
                            new ParserRegexBackTrack(flow.Dest, backTrack, context.Index, flow.Expression));
                        if (temp != null)
                        {
                            lock (lockObject)
                            {
                                results.Add(temp);
                            }
                        }
                    }
                }
            });

            switch (results.Count)
            {
                case 1:
                    return results[0];
                case 2:
                    return results[0].Length > results[1].Length ? results[0] : results[1];
                default:
                    return null;
            }
        }
    }
}
